#!/usr/bin/env node
var define = (function amdefine(module, requireFn) {
    'use strict';
    var defineCache = {},
        loaderCache = {},
        alreadyCalled = false,
        path = require('path'),
        makeRequire, stringRequire;

    /**
     * Trims the . and .. from an array of path segments.
     * It will keep a leading path segment if a .. will become
     * the first path segment, to help with module name lookups,
     * which act like paths, but can be remapped. But the end result,
     * all paths that use this function should look normalized.
     * NOTE: this method MODIFIES the input array.
     * @param {Array} ary the array of path segments.
     */
    function trimDots(ary) {
        var i, part;
        for (i = 0; ary[i]; i+= 1) {
            part = ary[i];
            if (part === '.') {
                ary.splice(i, 1);
                i -= 1;
            } else if (part === '..') {
                if (i === 1 && (ary[2] === '..' || ary[0] === '..')) {
                    //End of the line. Keep at least one non-dot
                    //path segment at the front so it can be mapped
                    //correctly to disk. Otherwise, there is likely
                    //no path mapping for a path starting with '..'.
                    //This can still fail, but catches the most reasonable
                    //uses of ..
                    break;
                } else if (i > 0) {
                    ary.splice(i - 1, 2);
                    i -= 2;
                }
            }
        }
    }

    function normalize(name, baseName) {
        var baseParts;

        //Adjust any relative paths.
        if (name && name.charAt(0) === '.') {
            //If have a base name, try to normalize against it,
            //otherwise, assume it is a top-level require that will
            //be relative to baseUrl in the end.
            if (baseName) {
                baseParts = baseName.split('/');
                baseParts = baseParts.slice(0, baseParts.length - 1);
                baseParts = baseParts.concat(name.split('/'));
                trimDots(baseParts);
                name = baseParts.join('/');
            }
        }

        return name;
    }

    /**
     * Create the normalize() function passed to a loader plugin's
     * normalize method.
     */
    function makeNormalize(relName) {
        return function (name) {
            return normalize(name, relName);
        };
    }

    function makeLoad(id) {
        function load(value) {
            loaderCache[id] = value;
        }

        load.fromText = function (id, text) {
            //This one is difficult because the text can/probably uses
            //define, and any relative paths and requires should be relative
            //to that id was it would be found on disk. But this would require
            //bootstrapping a module/require fairly deeply from node core.
            //Not sure how best to go about that yet.
            throw new Error('amdefine does not implement load.fromText');
        };

        return load;
    }

    makeRequire = function (systemRequire, exports, module, relId) {
        function amdRequire(deps, callback) {
            if (typeof deps === 'string') {
                //Synchronous, single module require('')
                return stringRequire(systemRequire, exports, module, deps, relId);
            } else {
                //Array of dependencies with a callback.

                //Convert the dependencies to modules.
                deps = deps.map(function (depName) {
                    return stringRequire(systemRequire, exports, module, depName, relId);
                });

                //Wait for next tick to call back the require call.
                process.nextTick(function () {
                    callback.apply(null, deps);
                });
            }
        }

        amdRequire.toUrl = function (filePath) {
            if (filePath.indexOf('.') === 0) {
                return normalize(filePath, path.dirname(module.filename));
            } else {
                return filePath;
            }
        };

        return amdRequire;
    };

    //Favor explicit value, passed in if the module wants to support Node 0.4.
    requireFn = requireFn || function req() {
        return module.require.apply(module, arguments);
    };

    function runFactory(id, deps, factory) {
        var r, e, m, result;

        if (id) {
            e = loaderCache[id] = {};
            m = {
                id: id,
                uri: __filename,
                exports: e
            };
            r = makeRequire(requireFn, e, m, id);
        } else {
            //Only support one define call per file
            if (alreadyCalled) {
                throw new Error('amdefine with no module ID cannot be called more than once per file.');
            }
            alreadyCalled = true;

            //Use the real variables from node
            //Use module.exports for exports, since
            //the exports in here is amdefine exports.
            e = module.exports;
            m = module;
            r = makeRequire(requireFn, e, m, module.id);
        }

        //If there are dependencies, they are strings, so need
        //to convert them to dependency values.
        if (deps) {
            deps = deps.map(function (depName) {
                return r(depName);
            });
        }

        //Call the factory with the right dependencies.
        if (typeof factory === 'function') {
            result = factory.apply(m.exports, deps);
        } else {
            result = factory;
        }

        if (result !== undefined) {
            m.exports = result;
            if (id) {
                loaderCache[id] = m.exports;
            }
        }
    }

    stringRequire = function (systemRequire, exports, module, id, relId) {
        //Split the ID by a ! so that
        var index = id.indexOf('!'),
            originalId = id,
            prefix, plugin;

        if (index === -1) {
            id = normalize(id, relId);

            //Straight module lookup. If it is one of the special dependencies,
            //deal with it, otherwise, delegate to node.
            if (id === 'require') {
                return makeRequire(systemRequire, exports, module, relId);
            } else if (id === 'exports') {
                return exports;
            } else if (id === 'module') {
                return module;
            } else if (loaderCache.hasOwnProperty(id)) {
                return loaderCache[id];
            } else if (defineCache[id]) {
                runFactory.apply(null, defineCache[id]);
                return loaderCache[id];
            } else {
                if(systemRequire) {
                    return systemRequire(originalId);
                } else {
                    throw new Error('No module with ID: ' + id);
                }
            }
        } else {
            //There is a plugin in play.
            prefix = id.substring(0, index);
            id = id.substring(index + 1, id.length);

            plugin = stringRequire(systemRequire, exports, module, prefix, relId);

            if (plugin.normalize) {
                id = plugin.normalize(id, makeNormalize(relId));
            } else {
                //Normalize the ID normally.
                id = normalize(id, relId);
            }

            if (loaderCache[id]) {
                return loaderCache[id];
            } else {
                plugin.load(id, makeRequire(systemRequire, exports, module, relId), makeLoad(id), {});

                return loaderCache[id];
            }
        }
    };

    //Create a define function specific to the module asking for amdefine.
    function define(id, deps, factory) {
        if (Array.isArray(id)) {
            factory = deps;
            deps = id;
            id = undefined;
        } else if (typeof id !== 'string') {
            factory = id;
            id = deps = undefined;
        }

        if (deps && !Array.isArray(deps)) {
            factory = deps;
            deps = undefined;
        }

        if (!deps) {
            deps = ['require', 'exports', 'module'];
        }

        //Set up properties for this module. If an ID, then use
        //internal cache. If no ID, then use the external variables
        //for this node module.
        if (id) {
            //Put the module in deep freeze until there is a
            //require call for it.
            defineCache[id] = [id, deps, factory];
        } else {
            runFactory(id, deps, factory);
        }
    }

    //define.require, which has access to all the values in the
    //cache. Useful for AMD modules that all have IDs in the file,
    //but need to finally export a value to node based on one of those
    //IDs.
    define.require = function (id) {
        if (loaderCache[id]) {
            return loaderCache[id];
        }

        if (defineCache[id]) {
            runFactory.apply(null, defineCache[id]);
            return loaderCache[id];
        }
    };

    define.amd = {};

    return define;
})(null, function (id) { throw new Error("Dependency not found: " +  id); }), requirejs = define.require;
define("amber_vm/browser-compatibility", [], {});
/* ====================================================================
 |
 |   Amber Smalltalk
 |   http://amber-lang.net
 |
 ======================================================================

 ======================================================================
 |
 | Copyright (c) 2010-2014
 | Nicolas Petton <petton.nicolas@gmail.com>
 |
 | Copyright (c) 2012-2014
 | The Amber team https://github.com/amber-smalltalk?tab=members
 | Amber contributors https://github.com/amber-smalltalk/amber/graphs/contributors
 |
 | Amber is released under the MIT license
 |
 | Permission is hereby granted, free of charge, to any person obtaining
 | a copy of this software and associated documentation files (the
 | 'Software'), to deal in the Software without restriction, including
 | without limitation the rights to use, copy, modify, merge, publish,
 | distribute, sublicense, and/or sell copies of the Software, and to
 | permit persons to whom the Software is furnished to do so, subject to
 | the following conditions:
 |
 | The above copyright notice and this permission notice shall be
 | included in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND,
 | EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 | IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 | CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 | TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 | SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 ==================================================================== */


define("amber_vm/boot", [ 'require', './browser-compatibility' ], function (require) {

	/* Reconfigurable micro composition system, https://github.com/amber-smalltalk/brikz */

	function Brikz(api, apiKey, initKey) {
		var brikz = this, backup = {};
		apiKey = apiKey || 'exports';
		initKey = initKey || '__init__';

		function mixin(src, target, what) {
			for (var keys = Object.keys(what||src), l=keys.length, i=0; i<l; ++i) {
				var value = src[keys[i]];
				if (typeof value !== "undefined") { target[keys[i]] = value; }
			}
			return target;
		}

		var d={value: null, enumerable: false, configurable: true, writable: true};
		Object.defineProperties(this, { ensure: d, rebuild: d });
		var exclude = mixin(this, {});

		this.rebuild = function () {
			Object.keys(backup).forEach(function (key) {
				mixin({}, api, (backup[key]||0)[apiKey]);
			});
			var oapi = mixin(api, {}), order = [], chk = {};
			brikz.ensure = function(key) {
				if (key in exclude) { return null; }
				var b = brikz[key], bak = backup[key];
				mixin({}, api, api);
				while (typeof b === "function") { b = new b(brikz, api, bak); }
				if (b && !chk[key]) { chk[key]=true; order.push(b); }
				if (b && !b[apiKey]) { b[apiKey] = mixin(api, {}); }
				return brikz[key] = b;
			};
			Object.keys(brikz).forEach(function (key) { brikz.ensure(key); });
			brikz.ensure = null;
			mixin(oapi, mixin({}, api, api));
			order.forEach(function(brik) { mixin(brik[apiKey] || {}, api); });
			order.forEach(function(brik) { brik[initKey] && brik[initKey](); });
			backup = mixin(brikz, {});
		};
	}

	/* Brikz end */

	function inherits(child, parent) {
		child.prototype = Object.create(parent.prototype, {
			constructor: { value: child,
						   enumerable: false, configurable: true, writable: true }
		});
		return child;
	}

	var globals = {};
    globals.SmalltalkSettings = {};
	var api = Object.create(globals);
	var brikz = new Brikz(api);

	function RootBrik(brikz, st) {

		/* Smalltalk foundational objects */

		/* SmalltalkRoot is the hidden root of the Amber hierarchy.
		 All objects including `Object` inherit from SmalltalkRoot */
		function SmalltalkRoot() {}
		function SmalltalkProtoObject() {}
		inherits(SmalltalkProtoObject, SmalltalkRoot);
		function SmalltalkObject() {}
		inherits(SmalltalkObject, SmalltalkProtoObject);
		function SmalltalkNil() {}
		inherits(SmalltalkNil, SmalltalkObject);

		this.Object = SmalltalkObject;
		this.nil = new SmalltalkNil();

		// Hidden root class of the system.
		this.rootAsClass = {fn: SmalltalkRoot};

		this.__init__ = function () {
			st.addPackage("Kernel-Objects");
			st.addPackage("Kernel-Infrastructure");
			st.wrapClassName("ProtoObject", "Kernel-Objects", SmalltalkProtoObject, undefined, false);
			st.wrapClassName("Object", "Kernel-Objects", SmalltalkObject, globals.ProtoObject, false);
			st.wrapClassName("UndefinedObject", "Kernel-Objects", SmalltalkNil, globals.Object, false);
		};
	}

	function OrganizeBrik(brikz, st) {

		brikz.ensure("augments");
		var SmalltalkObject = brikz.ensure("root").Object;

		function SmalltalkOrganizer () {}
		function SmalltalkPackageOrganizer () {
			this.elements = [];
		}
		function SmalltalkClassOrganizer () {
			this.elements = [];
		}

		inherits(SmalltalkOrganizer, SmalltalkObject);
		inherits(SmalltalkPackageOrganizer, SmalltalkOrganizer);
		inherits(SmalltalkClassOrganizer, SmalltalkOrganizer);

		this.__init__ = function () {
			st.wrapClassName("Organizer", "Kernel-Infrastructure", SmalltalkOrganizer, globals.Object, false);
			st.wrapClassName("PackageOrganizer", "Kernel-Infrastructure", SmalltalkPackageOrganizer, globals.Organizer, false);
			st.wrapClassName("ClassOrganizer", "Kernel-Infrastructure", SmalltalkClassOrganizer, globals.Organizer, false);
		};

		this.setupClassOrganization = function (klass) {
			klass.organization = new SmalltalkClassOrganizer();
			klass.organization.theClass = klass;
		};

		this.setupPackageOrganization = function (pkg) {
			pkg.organization = new SmalltalkPackageOrganizer();
		};

		this.addOrganizationElement = function (owner, element) {
			owner.organization.elements.addElement(element);
		};

		this.removeOrganizationElement = function (owner, element) {
			owner.organization.elements.removeElement(element);
		};
	}

	function DNUBrik(brikz, st) {

		brikz.ensure("selectorConversion");
		brikz.ensure("messageSend");
		var manip = brikz.ensure("manipulation");
		var rootAsClass = brikz.ensure("root").rootAsClass;

		/* Method not implemented handlers */

		var methods = [], checker = Object.create(null);
		this.selectors = [];

		this.get = function (string) {
			var index = this.selectors.indexOf(string);
			if(index !== -1) {
				return methods[index];
			}
			this.selectors.push(string);
			var selector = st.selector(string);
			checker[selector] = true;
			var method = {jsSelector: selector, fn: createHandler(selector)};
			methods.push(method);
			manip.installMethod(method, rootAsClass);
			return method;
		};

		this.isSelector = function (selector) {
			return checker[selector];
		};

		/* Dnu handler method */

		function createHandler(selector) {
			return function() {
				var args = Array.prototype.slice.call(arguments);
				return brikz.messageSend.messageNotUnderstood(this, selector, args);
			};
		}

		this.installHandlers = function (klass) {
			for(var i=0; i<methods.length; i++) {
				manip.installMethodIfAbsent(methods[i], klass);
			}
		};
	}

	function ClassInitBrik(brikz, st) {

		var dnu = brikz.ensure("dnu");
		var manip = brikz.ensure("manipulation");
		var nil = brikz.ensure("root").nil;

		/* Initialize a class in its class hierarchy. Handle both classes and
		 metaclasses. */

		st.init = function(klass) {
			st.initClass(klass);
			if(klass.klass && !klass.meta) {
				st.initClass(klass.klass);
			}
		};

		st.initClass = function(klass) {
			if(klass.wrapped) {
				copySuperclass(klass);
			}

			if(klass.wrapped) {
				dnu.installHandlers(klass);
			}
		};

		function copySuperclass(klass, superclass) {
			var inheritedMethods = Object.create(null);
			deinstallAllMethods(klass);
			for (superclass = superclass || klass.superclass;
				 superclass && superclass !== nil;
				 superclass = superclass.superclass) {
				for (var keys = Object.keys(superclass.methods), i = 0; i < keys.length; i++) {
					inheritMethodIfAbsent(superclass.methods[keys[i]]);
				}
			}
			manip.reinstallMethods(klass);

			function inheritMethodIfAbsent(method) {
				var selector = method.selector;

				//TODO: prepare klass methods into inheritedMethods to only test once
				if(klass.methods[selector] || inheritedMethods[selector]) {
					return;
				}

				manip.installMethod(method, klass);
				inheritedMethods[method.selector] = true;
			}

		}

		function deinstallAllMethods(klass) {
			var proto = klass.fn.prototype;
			for(var keys = Object.getOwnPropertyNames(proto), i=0; i<keys.length; i++) {
				var key = keys[i];
				if (dnu.isSelector(key)) {
					proto[key] = null;
				}
			}
		}
	}

	function ManipulationBrik(brikz, st) {

		this.installMethodIfAbsent = function (handler, klass) {
			if(!klass.fn.prototype[handler.jsSelector]) {
				installMethod(handler, klass);
			}
		};

		function installMethod (method, klass) {
			Object.defineProperty(klass.fn.prototype, method.jsSelector, {
				value: method.fn,
				enumerable: false, configurable: true, writable: true
			});
		}
		this.installMethod = installMethod;

		this.reinstallMethods = function (klass) {
			var methods = klass.methods;
			for(var keys = Object.keys(methods), i=0; i<keys.length; i++) {
				installMethod(methods[keys[i]], klass);
			}
		};
	}

	function ClassesBrik(brikz, st) {

		var org = brikz.ensure("organize");
		var root = brikz.ensure("root");
		var nil = root.nil;
		var rootAsClass = root.rootAsClass;
		var SmalltalkObject = root.Object;
		rootAsClass.klass = {fn: SmalltalkClass};

		function SmalltalkPackage() {}
		function SmalltalkBehavior() {}
		function SmalltalkClass() {}
		function SmalltalkMetaclass() {}

		inherits(SmalltalkPackage, SmalltalkObject);
		inherits(SmalltalkBehavior, SmalltalkObject);
		inherits(SmalltalkClass, SmalltalkBehavior);
		inherits(SmalltalkMetaclass, SmalltalkBehavior);

		SmalltalkMetaclass.prototype.meta = true;

		this.__init__ = function () {
			st.addPackage("Kernel-Classes");
			st.wrapClassName("Behavior", "Kernel-Classes", SmalltalkBehavior, globals.Object, false);
			st.wrapClassName("Metaclass", "Kernel-Classes", SmalltalkMetaclass, globals.Behavior, false);
			st.wrapClassName("Class", "Kernel-Classes", SmalltalkClass, globals.Behavior, false);

			// Manually bootstrap the metaclass hierarchy
			globals.ProtoObject.klass.superclass = rootAsClass.klass = globals.Class;
			addSubclass(globals.ProtoObject.klass);

			st.wrapClassName("Package", "Kernel-Infrastructure", SmalltalkPackage, globals.Object, false);
		};

		/* Smalltalk classes */

		var classes = [];
		var wrappedClasses = [];

		/* We hold all Packages in a separate Object */

		st.packages = {};

		/* Smalltalk package creation. To add a Package, use smalltalk.addPackage() */

		function pkg(spec) {
			var that = new SmalltalkPackage();
			that.pkgName = spec.pkgName;
			org.setupPackageOrganization(that);
			that.properties = spec.properties || {};
			return that;
		}

		/* Smalltalk class creation. A class is an instance of an automatically
		 created metaclass object. Newly created classes (not their metaclass)
		 should be added to the smalltalk object, see smalltalk.addClass().
		 Superclass linking is *not* handled here, see smalltalk.init()  */

		function klass(spec) {
			spec = spec || {};
			var setSuperClass = spec.superclass;
			if(!spec.superclass) {
				spec.superclass = rootAsClass;
			}

			var meta = metaclass(spec);
			var that = meta.instanceClass;

			that.superclass = setSuperClass;

			that.fn = spec.fn || inherits(function () {}, spec.superclass.fn);
			that.subclasses = [];

			setupClass(that, spec);

			that.className = spec.className;
			that.wrapped   = spec.wrapped || false;
			meta.className = spec.className + ' class';
			meta.superclass = spec.superclass.klass;
			return that;
		}

		function metaclass(spec) {
			spec = spec || {};
			var that = new SmalltalkMetaclass();
			that.fn = inherits(function () {}, spec.superclass.klass.fn);
			that.instanceClass = new that.fn();
			setupClass(that);
			return that;
		}

		SmalltalkBehavior.prototype.toString = function () {
			return 'Smalltalk ' + this.className;
		};

		function setupClass(klass, spec) {
			spec = spec || {};
			klass.iVarNames = spec.iVarNames || [];
			klass.pkg = spec.pkg;

			org.setupClassOrganization(klass);
			Object.defineProperty(klass, "methods", {
				value: Object.create(null),
				enumerable: false, configurable: true, writable: true
			});
			Object.defineProperty(klass.fn.prototype, "klass", {
				value: klass,
				enumerable: false, configurable: true, writable: true
			});
		}

		/* Add a package to the smalltalk.packages object, creating a new one if needed.
		 If pkgName is null or empty we return nil, which is an allowed package for a class.
		 If package already exists we still update the properties of it. */

		st.addPackage = function(pkgName, properties) {
			if(!pkgName) {return nil;}
			if(!(st.packages[pkgName])) {
				st.packages[pkgName] = pkg({
					pkgName: pkgName,
					properties: properties
				});
			} else {
				if(properties) {
					st.packages[pkgName].properties = properties;
				}
			}
			return st.packages[pkgName];
		};

		/* Add a class to the smalltalk object, creating a new one if needed.
		 A Package is lazily created if it does not exist with given name. */

		st.addClass = function(className, superclass, iVarNames, pkgName) {
			if (superclass == nil) { superclass = null; }

			// While subclassing nil is allowed, it might be an error, so
			// warn about it.
			if (superclass === null) {
				console.warn('Compiling ' + className + ' as a subclass of `nil`. A dependency might be missing.');
			}
			rawAddClass(pkgName, className, superclass, iVarNames, false, null);
		};

		function rawAddClass(pkgName, className, superclass, iVarNames, wrapped, fn) {
			var pkg = st.packages[pkgName];

			if (!pkg) {
				throw new Error("Missing package "+pkgName);
			}

			if(globals[className] && globals[className].superclass == superclass) {
				//            globals[className].superclass = superclass;
				globals[className].iVarNames = iVarNames || [];
				if (pkg) globals[className].pkg = pkg;
				if (fn) {
					fn.prototype = globals[className].fn.prototype;
					globals[className].fn = fn;
					fn.prototype.constructor = fn;
				}
			} else {
				if(globals[className]) {
					st.removeClass(globals[className]);
				}
				globals[className] = klass({
					className: className,
					superclass: superclass,
					pkg: pkg,
					iVarNames: iVarNames,
					fn: fn,
					wrapped: wrapped
				});

				addSubclass(globals[className]);
			}

			classes.addElement(globals[className]);
			org.addOrganizationElement(pkg, globals[className]);
		}

		st.removeClass = function(klass) {
			org.removeOrganizationElement(klass.pkg, klass);
			classes.removeElement(klass);
			removeSubclass(klass);
			delete globals[klass.className];
		};

		function addSubclass(klass) {
			if(klass.superclass) {
				klass.superclass.subclasses.addElement(klass);
			}
		}

		function removeSubclass(klass) {
			if(klass.superclass) {
				klass.superclass.subclasses.removeElement(klass);
			}
		}

		/* Create a new class wrapping a JavaScript constructor, and add it to the
		 global smalltalk object. Package is lazily created if it does not exist with given name. */

		st.wrapClassName = function(className, pkgName, fn, superclass, wrapped) {
			wrapped = wrapped !== false;
			rawAddClass(pkgName, className, superclass, globals[className] && globals[className].iVarNames, wrapped, fn);
			if(wrapped) {
				wrappedClasses.addElement(globals[className]);
			}
		};

		/* Manually set the constructor of an existing Smalltalk klass, making it a wrapped class. */

		st.setClassConstructor = function(klass, constructor) {
			wrappedClasses.addElement(klass);
			klass.wrapped = true;
			klass.fn = constructor;

			// The fn property changed. We need to add back the klass property to the prototype
			Object.defineProperty(klass.fn.prototype, "klass", {
				value: klass,
				enumerable: false, configurable: true, writable: true
			});

			st.initClass(klass);
		};

		/* Create an alias for an existing class */

		st.alias = function(klass, alias) {
			globals[alias] = klass;
		};

		/* Answer all registered Smalltalk classes */
		//TODO: remove the function and make smalltalk.classes an array

		st.classes = function() {
			return classes;
		};

		st.wrappedClasses = function() {
			return wrappedClasses;
		};

		// Still used, but could go away now that subclasses are stored
		// into classes directly.
		st.allSubclasses = function(klass) {
			return klass._allSubclasses();
		};

	}

	function MethodsBrik(brikz, st) {

		var manip = brikz.ensure("manipulation");
		var org = brikz.ensure("organize");
		var stInit = brikz.ensure("stInit");
		var dnu = brikz.ensure("dnu");
		var SmalltalkObject = brikz.ensure("root").Object;
		brikz.ensure("selectorConversion");
		brikz.ensure("classes");
		brikz.ensure("classInit");

		function SmalltalkMethod() {}
		inherits(SmalltalkMethod, SmalltalkObject);

		this.__init__ = function () {
			st.addPackage("Kernel-Methods");
			st.wrapClassName("CompiledMethod", "Kernel-Methods", SmalltalkMethod, globals.Object, false);
		};

		/* Smalltalk method object. To add a method to a class,
		 use smalltalk.addMethod() */

		st.method = function(spec) {
			var that = new SmalltalkMethod();
			that.selector          = spec.selector;
			that.jsSelector        = spec.jsSelector;
			that.args              = spec.args || {};
			// spec.category is kept for compatibility
			that.protocol          = spec.protocol || spec.category;
			that.source            = spec.source;
			that.messageSends      = spec.messageSends || [];
			that.referencedClasses = spec.referencedClasses || [];
			that.fn                = spec.fn;
			return that;
		};

		function installNewDnuHandler(newHandler) {
			var wrappedClasses = st.wrappedClasses();
			for(var i = 0; i < wrappedClasses.length; i++) {
				manip.installMethodIfAbsent(newHandler, wrappedClasses[i]);
			}
		}

		/* Add/remove a method to/from a class */

		st.addMethod = function (method, klass) {
			if (!(method.jsSelector)) {
				method.jsSelector = st.selector(method.selector);
			}
			manip.installMethod(method, klass);
			klass.methods[method.selector] = method;
			method.methodClass = klass;

			// During the bootstrap, #addCompiledMethod is not used.
			// Therefore we populate the organizer here too
			org.addOrganizationElement(klass, method.protocol);

			propagateMethodChange(klass);

			for(var i=0; i<method.messageSends.length; i++) {
				var dnuHandler = dnu.get(method.messageSends[i]);
				if(stInit.initialized()) {
					installNewDnuHandler(dnuHandler);
				}
			}
		};

		function propagateMethodChange(klass) {
			// If already initialized (else it will be done later anyway),
			// re-initialize all subclasses to ensure the method change
			// propagation (for wrapped classes, not using the prototype
			// chain).

			//TODO: optimize, only one method need to be updated, not all of them
			if (stInit.initialized()) {
				st.allSubclasses(klass).forEach(function (subclass) {
					st.initClass(subclass);
				});
			}
		}

		st.removeMethod = function(method, klass) {
			if (klass !== method.methodClass) {
				throw new Error(
					"Refusing to remove method "
						+ method.methodClass.className+">>"+method.selector
						+ " from different class "
						+ klass.className);
			}

			delete klass.fn.prototype[st.selector(method.selector)];
			delete klass.methods[method.selector];

			st.initClass(klass);
			propagateMethodChange(klass);

			// Do *not* delete protocols from here.
			// This is handled by #removeCompiledMethod
		};

		/* Answer all method selectors based on dnu handlers */

		st.allSelectors = function() {
			return dnu.selectors;
		};

	}

	function AugmentsBrik(brikz, st) {

		/* Make sure that console is defined */

		if(typeof console === "undefined") {
			this.console = {
				log: function() {},
				warn: function() {},
				info: function() {},
				debug: function() {},
				error: function() {}
			};
		}

		/* Array extensions */

		Array.prototype.addElement = function(el) {
			if(typeof el === 'undefined') { return; }
			if(this.indexOf(el) == -1) {
				this.push(el);
			}
		};

		Array.prototype.removeElement = function(el) {
			var i = this.indexOf(el);
			if (i !== -1) { this.splice(i, 1); }
		};
	}

	function SmalltalkInitBrik(brikz, st) {

		brikz.ensure("classInit");
		brikz.ensure("classes");
		var nil = brikz.ensure("root").nil;

		var initialized = false;

		/* Smalltalk initialization. Called on page load */

		st.initialize = function() {
			if(initialized) { return; }

			st.classes().forEach(function(klass) {
				st.init(klass);
			});

			runnable();

			st.classes().forEach(function(klass) {
				klass._initialize();
			});

			initialized = true;
		};

		this.initialized = function () {
			return initialized;
		};

		this.__init__ = function () {
			st.addPackage("Kernel-Methods");
			st.wrapClassName("Number", "Kernel-Objects", Number, globals.Object);
			st.wrapClassName("BlockClosure", "Kernel-Methods", Function, globals.Object);
			st.wrapClassName("Boolean", "Kernel-Objects", Boolean, globals.Object);
			st.wrapClassName("Date", "Kernel-Objects", Date, globals.Object);

			st.addPackage("Kernel-Collections");
			st.addClass("Collection", globals.Object, null, "Kernel-Collections");
			st.addClass("IndexableCollection", globals.Collection, null, "Kernel-Collections");
			st.addClass("SequenceableCollection", globals.IndexableCollection, null, "Kernel-Collections");
			st.addClass("CharacterArray", globals.SequenceableCollection, null, "Kernel-Collections");
			st.wrapClassName("String", "Kernel-Collections", String, globals.CharacterArray);
			st.wrapClassName("Array", "Kernel-Collections", Array, globals.SequenceableCollection);
			st.wrapClassName("RegularExpression", "Kernel-Collections", RegExp, globals.Object);

			st.addPackage("Kernel-Exceptions");
			st.wrapClassName("Error", "Kernel-Exceptions", Error, globals.Object);

			/* Alias definitions */

			st.alias(globals.Array, "OrderedCollection");
			st.alias(globals.Date, "Time");

		};
	}

	function PrimitivesBrik(brikz, st) {

		/* Unique ID number generator */

		var oid = 0;
		st.nextId = function() {
			oid += 1;
			return oid;
		};

		/* Converts a JavaScript object to valid Smalltalk Object */
		st.readJSObject = function(js) {
			var object = js;
			var readObject = (js.constructor === Object);
			var readArray = (js.constructor === Array);

			if(readObject) {
				object = globals.Dictionary._new();
			}
			for(var i in js) {
				if(readObject) {
					object._at_put_(i, st.readJSObject(js[i]));
				}
				if(readArray) {
					object[i] = st.readJSObject(js[i]);
				}
			}
			return object;
		};

		/* Boolean assertion */
		st.assert = function(shouldBeBoolean) {
			if (undefined !== shouldBeBoolean && shouldBeBoolean.klass === globals.Boolean) {
				return shouldBeBoolean == true;
			} else {
				globals.NonBooleanReceiver._new()._object_(shouldBeBoolean)._signal();
			}
		};

		/* List of all reserved words in JavaScript. They may not be used as variables
		 in Smalltalk. */

		// list of reserved JavaScript keywords as of
		//   http://es5.github.com/#x7.6.1.1
		// and
		//   http://people.mozilla.org/~jorendorff/es6-draft.html#sec-7.6.1
		st.reservedWords = ['break', 'case', 'catch', 'continue', 'debugger',
							'default', 'delete', 'do', 'else', 'finally', 'for', 'function',
							'if', 'in', 'instanceof', 'new', 'return', 'switch', 'this', 'throw',
							'try', 'typeof', 'var', 'void', 'while', 'with',
							// ES5: future use: http://es5.github.com/#x7.6.1.2
							'class', 'const', 'enum', 'export', 'extends', 'import', 'super',
							// ES5: future use in strict mode
							'implements', 'interface', 'let', 'package', 'private', 'protected',
							'public', 'static', 'yield'];

		st.globalJsVariables = ['jQuery', 'window', 'document', 'process', 'global'];

	}

	function RuntimeBrik(brikz, st) {

		brikz.ensure("selectorConversion");
		var root = brikz.ensure("root");
		var nil = root.nil;
		var SmalltalkObject = root.Object;

		function SmalltalkMethodContext(home, setup) {
			this.sendIdx     = {};
			this.homeContext = home;
			this.setup       = setup || function() {};
		}

		inherits(SmalltalkMethodContext, SmalltalkObject);

		this.__init__ = function () {
			st.addPackage("Kernel-Methods");
			st.wrapClassName("MethodContext", "Kernel-Methods", SmalltalkMethodContext, globals.Object, false);

			// Fallbacks
			SmalltalkMethodContext.prototype.locals = {};
			SmalltalkMethodContext.prototype.receiver = null;
			SmalltalkMethodContext.prototype.selector = null;
			SmalltalkMethodContext.prototype.lookupClass = null;

			SmalltalkMethodContext.prototype.fill = function(receiver, selector, locals, lookupClass) {
				this.receiver    = receiver;
				this.selector    = selector;
				this.locals      = locals || {};
				this.lookupClass = lookupClass;
				if(this.homeContext) {
					this.homeContext.evaluatedSelector = selector;
				}
			};

			SmalltalkMethodContext.prototype.fillBlock = function(locals, ctx, index) {
				this.locals        = locals || {};
				this.outerContext  = ctx;
				this.index         = index || 0;
			};

			SmalltalkMethodContext.prototype.init = function() {
				var home = this.homeContext;
				if(home) {
					home.init();
				}

				this.setup(this);
			};

			SmalltalkMethodContext.prototype.method = function() {
				var method;
				var lookup = this.lookupClass || this.receiver.klass;
				while(!method && lookup) {
					method = lookup.methods[st.convertSelector(this.selector)];
					lookup = lookup.superclass;
				}
				return method;
			};
		};

		/* This is the current call context object. While it is publicly available,
		 Use smalltalk.getThisContext() instead which will answer a safe copy of
		 the current context */

		st.thisContext = undefined;

		st.withContext = function(worker, setup) {
			if(st.thisContext) {
				return inContext(worker, setup);
			} else {
				try {
					return inContext(worker, setup);
				} catch(error) {
					handleError(error);
					st.thisContext = null;
					// Rethrow the error in any case.
					error.amberHandled = true;
					throw error;
				}
			}
		};

		function inContext(worker, setup) {
			var context = pushContext(setup);
			var result = worker(context);
			popContext(context);
			return result;
		}

		/* Wrap a JavaScript exception in a Smalltalk Exception. 

		 In case of a RangeError, stub the stack after 100 contexts to
		 avoid another RangeError later when the stack is manipulated. */
		function wrappedError(error) {
			var errorWrapper = globals.JavaScriptException._on_(error);
			// Add the error to the context, so it is visible in the stack
			try { errorWrapper._signal(); } catch (ex) {}
			var context = st.getThisContext();
			if(isRangeError(error)) {
				stubContextStack(context);
			}
			errorWrapper._context_(context);
			return errorWrapper;
		}

		/* Stub the context stack after 100 contexts */
		function stubContextStack(context) {
			var currentContext = context;
			var contexts = 0;
			while(contexts < 100) {
				if(currentContext) {
					currentContext = currentContext.homeContext;
				}
				contexts++;
			}
			if(currentContext) {
				currentContext.homeContext = undefined;
			}
		}

		function isRangeError(error) {
			return error instanceof RangeError;
		}


		/* Handles Smalltalk errors. Triggers the registered ErrorHandler
		 (See the Smalltalk class ErrorHandler and its subclasses */

		function handleError(error) {
			if (!error.smalltalkError) {
				error = wrappedError(error);
			}
			globals.ErrorHandler._handleError_(error);
		}

		/* Handle thisContext pseudo variable */

		st.getThisContext = function() {
			if(st.thisContext) {
				st.thisContext.init();
				return st.thisContext;
			} else {
				return nil;
			}
		};

		function pushContext(setup) {
			return st.thisContext = new SmalltalkMethodContext(st.thisContext, setup);
		}

		function popContext(context) {
			st.thisContext = context.homeContext;
		}

	}

	function MessageSendBrik(brikz, st) {

		brikz.ensure("selectorConversion");
		var nil = brikz.ensure("root").nil;

		/* Handles unhandled errors during message sends */
		// simply send the message and handle #dnu:

		st.send = function(receiver, selector, args, klass) {
			var method;
			if(receiver === null) {
				receiver = nil;
			}
			method = klass ? klass.fn.prototype[selector] : receiver.klass && receiver[selector];
			if(method) {
				return method.apply(receiver, args);
			} else {
				return messageNotUnderstood(receiver, selector, args);
			}
		};

		/* Handles #dnu: *and* JavaScript method calls.
		 if the receiver has no klass, we consider it a JS object (outside of the
		 Amber system). Else assume that the receiver understands #doesNotUnderstand: */

		function messageNotUnderstood(receiver, selector, args) {
			/* Handles JS method calls. */
			if(receiver.klass === undefined || receiver.allowJavaScriptCalls) {
				return callJavaScriptMethod(receiver, selector, args);
			}

			/* Handles not understood messages. Also see the Amber counter-part
			 Object>>doesNotUnderstand: */

			return receiver._doesNotUnderstand_(
				globals.Message._new()
					._selector_(st.convertSelector(selector))
					._arguments_(args)
			);
		}

		/* Call a method of a JS object, or answer a property if it exists.
		 Else try wrapping a JSObjectProxy around the receiver.

		 If the object property is a function, then call it, except if it starts with
		 an uppercase character (we probably want to answer the function itself in this
		 case and send it #new from Amber).

		 Converts keyword-based selectors by using the first
		 keyword only, but keeping all message arguments.

		 Example:
		 "self do: aBlock with: anObject" -> "self.do(aBlock, anObject)" */

		function callJavaScriptMethod(receiver, selector, args) {
			var jsSelector = selector._asJavaScriptSelector();
			var jsProperty = receiver[jsSelector];
			if(typeof jsProperty === "function" && !/^[A-Z]/.test(jsSelector)) {
				return jsProperty.apply(receiver, args);
			} else if(jsProperty !== undefined) {
				if(args[0]) {
					receiver[jsSelector] = args[0];
					return nil;
				} else {
					return jsProperty;
				}
			}

			return st.send(globals.JSObjectProxy._on_(receiver), selector, args);
		}

		this.messageNotUnderstood = messageNotUnderstood;
	}

	function SelectorConversionBrik(brikz, st) {
		/* Convert a Smalltalk selector into a JS selector */

		st.selector = function(string) {
			var selector = '_' + string;
			selector = selector.replace(/:/g, '_');
			selector = selector.replace(/[\&]/g, '_and');
			selector = selector.replace(/[\|]/g, '_or');
			selector = selector.replace(/[+]/g, '_plus');
			selector = selector.replace(/-/g, '_minus');
			selector = selector.replace(/[*]/g ,'_star');
			selector = selector.replace(/[\/]/g ,'_slash');
			selector = selector.replace(/[\\]/g ,'_backslash');
			selector = selector.replace(/[\~]/g ,'_tild');
			selector = selector.replace(/>/g ,'_gt');
			selector = selector.replace(/</g ,'_lt');
			selector = selector.replace(/=/g ,'_eq');
			selector = selector.replace(/,/g ,'_comma');
			selector = selector.replace(/[@]/g ,'_at');
			return selector;
		};

		/* Convert a string to a valid smalltalk selector.
		 if you modify the following functions, also change String>>asSelector
		 accordingly */

		st.convertSelector = function(selector) {
			if(selector.match(/__/)) {
				return convertBinarySelector(selector);
			} else {
				return convertKeywordSelector(selector);
			}
		};

		function convertKeywordSelector(selector) {
			return selector.replace(/^_/, '').replace(/_/g, ':');
		}

		function convertBinarySelector(selector) {
			return selector
				.replace(/^_/, '')
				.replace(/_and/g, '&')
				.replace(/_or/g, '|')
				.replace(/_plus/g, '+')
				.replace(/_minus/g, '-')
				.replace(/_star/g, '*')
				.replace(/_slash/g, '/')
				.replace(/_backslash/g, '\\')
				.replace(/_tild/g, '~')
				.replace(/_gt/g, '>')
				.replace(/_lt/g, '<')
				.replace(/_eq/g, '=')
				.replace(/_comma/g, ',')
				.replace(/_at/g, '@');
		}
	}

	/* Adds AMD and requirejs related methods to the smalltalk object */
	function AMDBrik(brikz, st) {
		this.__init__ = function () {
			st.amdRequire = require;
			st.defaultTransportType = st.defaultTransportType || "amd";
			st.defaultAmdNamespace = st.defaultAmdNamespace || "amber_core";
		};
	}


	/* Making smalltalk that can load */

	brikz.root = RootBrik;
	brikz.dnu = DNUBrik;
	brikz.organize = OrganizeBrik;
	brikz.selectorConversion = SelectorConversionBrik;
	brikz.classInit = ClassInitBrik;
	brikz.manipulation = ManipulationBrik;
	brikz.classes = ClassesBrik;
	brikz.methods = MethodsBrik;
	brikz.stInit = SmalltalkInitBrik;
	brikz.augments = AugmentsBrik;
	brikz.amdBrik = AMDBrik;

	brikz.rebuild();

	/* Making smalltalk that can run */

	function runnable () {
		brikz.messageSend = MessageSendBrik;
		brikz.runtime = RuntimeBrik;
		brikz.primitives = PrimitivesBrik;

		brikz.rebuild();
	};

	return { vm: api, nil: brikz.root.nil, globals: globals };
});

define("amber_vm/smalltalk", ["./boot"], function (boot) {
    return boot.vm;
});


define("amber_vm/globals", ["./boot"], function (boot) {
    return boot.globals;
});

define("amber_vm/nil", ["./boot"], function (boot) {
    return boot.nil;
});


/**
 * _st is a function used all over the compiled amber code that
 * takes any value (JavaScript or Smalltalk)
 * and returns a proper Amber Smalltalk receiver.
 *
 * null or undefined -> nil,
 * plain JS object -> wrapped JS object,
 * otherwise unchanged
 */

define("amber_vm/_st", ["./globals", "./nil"], function (globals, nil) {
    return function (o) {
        if (o == null) { return nil; }
        if (o.klass) { return o; }
        return globals.JSObjectProxy._on_(o);
    };
});

define("amber_core/Kernel-Objects", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Objects');
smalltalk.packages["Kernel-Objects"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('ProtoObject', globals.nil, [], 'Kernel-Objects');
globals.ProtoObject.comment="I implement the basic behavior required for any object in Amber.\x0a\x0aIn most cases, subclassing `ProtoObject` is wrong and `Object` should be used instead. However subclassing `ProtoObject` can be useful in some special cases like proxy implementations.";
smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__eq_eq(anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"=",{anObject:anObject},globals.ProtoObject)})},
args: ["anObject"],
source: "= anObject\x0a\x09^ self == anObject",
messageSends: ["=="],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "==",
protocol: 'comparing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._identityHash();
$ctx1.sendIdx["identityHash"]=1;
$1=_st($2).__eq(_st(anObject)._identityHash());
return $1;
}, function($ctx1) {$ctx1.fill(self,"==",{anObject:anObject},globals.ProtoObject)})},
args: ["anObject"],
source: "== anObject\x0a\x09^ self identityHash = anObject identityHash",
messageSends: ["=", "identityHash"],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._printString();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asString",{},globals.ProtoObject)})},
args: [],
source: "asString\x0a\x09^ self printString",
messageSends: ["printString"],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "class",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.klass;
return self}, function($ctx1) {$ctx1.fill(self,"class",{},globals.ProtoObject)})},
args: [],
source: "class\x0a\x09<return self.klass>",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "doesNotUnderstand:",
protocol: 'error handling',
fn: function (aMessage){
var self=this;
function $MessageNotUnderstood(){return globals.MessageNotUnderstood||(typeof MessageNotUnderstood=="undefined"?nil:MessageNotUnderstood)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st($MessageNotUnderstood())._new();
_st($1)._receiver_(self);
_st($1)._message_(aMessage);
$2=_st($1)._signal();
return self}, function($ctx1) {$ctx1.fill(self,"doesNotUnderstand:",{aMessage:aMessage},globals.ProtoObject)})},
args: ["aMessage"],
source: "doesNotUnderstand: aMessage\x0a\x09MessageNotUnderstood new\x0a\x09\x09receiver: self;\x0a\x09\x09message: aMessage;\x0a\x09\x09signal",
messageSends: ["receiver:", "new", "message:", "signal"],
referencedClasses: ["MessageNotUnderstood"]
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "identityHash",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var hash=self.identityHash;
		if (hash) return hash;
		hash=smalltalk.nextId();
		Object.defineProperty(self, 'identityHash', {value:hash});
		return hash;
	;
return self}, function($ctx1) {$ctx1.fill(self,"identityHash",{},globals.ProtoObject)})},
args: [],
source: "identityHash\x0a\x09<\x0a\x09\x09var hash=self.identityHash;\x0a\x09\x09if (hash) return hash;\x0a\x09\x09hash=smalltalk.nextId();\x0a\x09\x09Object.defineProperty(self, 'identityHash', {value:hash});\x0a\x09\x09return hash;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return self},
args: [],
source: "initialize",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "inspect",
protocol: 'inspecting',
fn: function (){
var self=this;
function $Inspector(){return globals.Inspector||(typeof Inspector=="undefined"?nil:Inspector)}
return smalltalk.withContext(function($ctx1) { 
_st($Inspector())._inspect_(self);
return self}, function($ctx1) {$ctx1.fill(self,"inspect",{},globals.ProtoObject)})},
args: [],
source: "inspect\x0a\x09Inspector inspect: self",
messageSends: ["inspect:"],
referencedClasses: ["Inspector"]
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "inspectOn:",
protocol: 'inspecting',
fn: function (anInspector){
var self=this;
return self},
args: ["anInspector"],
source: "inspectOn: anInspector",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "instVarAt:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 return self['@'+aString] ;
return self}, function($ctx1) {$ctx1.fill(self,"instVarAt:",{aString:aString},globals.ProtoObject)})},
args: ["aString"],
source: "instVarAt: aString\x0a\x09< return self['@'+aString] >",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "instVarAt:put:",
protocol: 'accessing',
fn: function (aString,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 self['@' + aString] = anObject ;
return self}, function($ctx1) {$ctx1.fill(self,"instVarAt:put:",{aString:aString,anObject:anObject},globals.ProtoObject)})},
args: ["aString", "anObject"],
source: "instVarAt: aString put: anObject\x0a\x09< self['@' + aString] = anObject >",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "perform:",
protocol: 'message handling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._perform_withArguments_(aString,[]);
return $1;
}, function($ctx1) {$ctx1.fill(self,"perform:",{aString:aString},globals.ProtoObject)})},
args: ["aString"],
source: "perform: aString\x0a\x09^ self perform: aString withArguments: #()",
messageSends: ["perform:withArguments:"],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "perform:withArguments:",
protocol: 'message handling',
fn: function (aString,aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.send(self, aString._asSelector(), aCollection);
return self}, function($ctx1) {$ctx1.fill(self,"perform:withArguments:",{aString:aString,aCollection:aCollection},globals.ProtoObject)})},
args: ["aString", "aCollection"],
source: "perform: aString withArguments: aCollection\x0a\x09<return smalltalk.send(self, aString._asSelector(), aCollection)>",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $5,$4,$3,$2,$1;
$5=self._class();
$ctx1.sendIdx["class"]=1;
$4=_st($5)._name();
$ctx1.sendIdx["name"]=1;
$3=_st($4)._first();
$2=_st($3)._isVowel();
if(smalltalk.assert($2)){
$1="an ";
} else {
$1="a ";
};
_st(aStream)._nextPutAll_($1);
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aStream)._nextPutAll_(_st(self._class())._name());
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.ProtoObject)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: (self class name first isVowel\x0a\x09\x09ifTrue: [ 'an ' ]\x0a\x09\x09ifFalse: [ 'a ' ]).\x0a\x09aStream nextPutAll: self class name",
messageSends: ["nextPutAll:", "ifTrue:ifFalse:", "isVowel", "first", "name", "class"],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "printString",
protocol: 'printing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($String())._streamContents_((function(str){
return smalltalk.withContext(function($ctx2) {
return self._printOn_(str);
}, function($ctx2) {$ctx2.fillBlock({str:str},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"printString",{},globals.ProtoObject)})},
args: [],
source: "printString\x0a\x09^ String streamContents: [ :str | \x0a\x09\x09self printOn: str ]",
messageSends: ["streamContents:", "printOn:"],
referencedClasses: ["String"]
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "yourself",
protocol: 'accessing',
fn: function (){
var self=this;
return self;
},
args: [],
source: "yourself\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "~=",
protocol: 'comparing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self.__eq(anObject)).__eq(false);
$ctx1.sendIdx["="]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"~=",{anObject:anObject},globals.ProtoObject)})},
args: ["anObject"],
source: "~= anObject\x0a\x09^ (self = anObject) = false",
messageSends: ["="],
referencedClasses: []
}),
globals.ProtoObject);

smalltalk.addMethod(
smalltalk.method({
selector: "~~",
protocol: 'comparing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self.__eq_eq(anObject)).__eq(false);
return $1;
}, function($ctx1) {$ctx1.fill(self,"~~",{anObject:anObject},globals.ProtoObject)})},
args: ["anObject"],
source: "~~ anObject\x0a\x09^ (self == anObject) = false",
messageSends: ["=", "=="],
referencedClasses: []
}),
globals.ProtoObject);


smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'accessing',
fn: function (){
var self=this;
return "class";
},
args: [],
source: "heliosClass\x0a\x09\x22Should be an Helios extension. Unfortunately, since helios can browse remote\x0a\x09environments, we can't extend base classes\x22\x0a\x09\x0a\x09^ 'class'",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return self},
args: [],
source: "initialize",
messageSends: [],
referencedClasses: []
}),
globals.ProtoObject.klass);


smalltalk.addClass('Object', globals.ProtoObject, [], 'Kernel-Objects');
globals.Object.comment="**I am the root of the Smalltalk class system**. With the exception of unual subclasses of `ProtoObject`, all other classes in the system are subclasses of me.\x0a\x0aI provide default behavior common to all normal objects (some of it inherited from `ProtoObject`), such as:\x0a\x0a- accessing\x0a- copying\x0a- comparison\x0a- error handling\x0a- message sending\x0a- reflection\x0a\x0aAlso utility messages that all objects should respond to are defined here.\x0a\x0aI have no instance variable.\x0a\x0a##Access\x0a\x0aInstance variables can be accessed with `#instVarAt:` and `#instVarAt:put:`. `#instanceVariableNames` answers a collection of all instance variable names.\x0aAccessing JavaScript properties of an object is done through `#basicAt:`, `#basicAt:put:` and `basicDelete:`.\x0a\x0a##Copying\x0a\x0aCopying an object is handled by `#copy` and `#deepCopy`. The first one performs a shallow copy of the receiver, while the second one performs a deep copy.\x0aThe hook method `#postCopy` can be overriden in subclasses to copy fields as necessary to complete the full copy. It will be sent by the copy of the receiver.\x0a\x0a##Comparison\x0a\x0aI understand equality `#=` and identity `#==` comparison.\x0a\x0a##Error handling\x0a\x0a- `#halt` is the typical message to use for inserting breakpoints during debugging.\x0a- `#error:` throws a generic error exception\x0a- `#doesNotUnderstand:` handles the fact that there was an attempt to send the given message to the receiver but the receiver does not understand this message.\x0a\x09Overriding this message can be useful to implement proxies for example.";
smalltalk.addMethod(
smalltalk.method({
selector: "->",
protocol: 'converting',
fn: function (anObject){
var self=this;
function $Association(){return globals.Association||(typeof Association=="undefined"?nil:Association)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Association())._key_value_(self,anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"->",{anObject:anObject},globals.Object)})},
args: ["anObject"],
source: "-> anObject\x0a\x09^ Association key: self value: anObject",
messageSends: ["key:value:"],
referencedClasses: ["Association"]
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
var variables;
function $HashedCollection(){return globals.HashedCollection||(typeof HashedCollection=="undefined"?nil:HashedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $1;
variables=_st($HashedCollection())._new();
_st(_st(self._class())._allInstanceVariableNames())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(variables)._at_put_(each,_st(self._instVarAt_(each))._asJSON());
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=variables;
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJSON",{variables:variables},globals.Object)})},
args: [],
source: "asJSON\x0a\x09| variables |\x0a\x09variables := HashedCollection new.\x0a\x09self class allInstanceVariableNames do: [ :each |\x0a\x09\x09variables at: each put: (self instVarAt: each) asJSON ].\x0a\x09^ variables",
messageSends: ["new", "do:", "allInstanceVariableNames", "class", "at:put:", "asJSON", "instVarAt:"],
referencedClasses: ["HashedCollection"]
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSONString",
protocol: 'converting',
fn: function (){
var self=this;
function $JSON(){return globals.JSON||(typeof JSON=="undefined"?nil:JSON)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($JSON())._stringify_(self._asJSON());
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJSONString",{},globals.Object)})},
args: [],
source: "asJSONString\x0a\x09^ JSON stringify: self asJSON",
messageSends: ["stringify:", "asJSON"],
referencedClasses: ["JSON"]
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "asJavascript",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._asString();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJavascript",{},globals.Object)})},
args: [],
source: "asJavascript\x0a\x09^ self asString",
messageSends: ["asString"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "basicAt:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self[aString];
return self}, function($ctx1) {$ctx1.fill(self,"basicAt:",{aString:aString},globals.Object)})},
args: ["aString"],
source: "basicAt: aString\x0a\x09<return self[aString]>",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "basicAt:put:",
protocol: 'accessing',
fn: function (aString,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self[aString] = anObject;
return self}, function($ctx1) {$ctx1.fill(self,"basicAt:put:",{aString:aString,anObject:anObject},globals.Object)})},
args: ["aString", "anObject"],
source: "basicAt: aString put: anObject\x0a\x09<return self[aString] = anObject>",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "basicDelete:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
delete self[aString]; return aString;
return self}, function($ctx1) {$ctx1.fill(self,"basicDelete:",{aString:aString},globals.Object)})},
args: ["aString"],
source: "basicDelete: aString\x0a\x09<delete self[aString]; return aString>",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "basicPerform:",
protocol: 'message handling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicPerform_withArguments_(aString,[]);
return $1;
}, function($ctx1) {$ctx1.fill(self,"basicPerform:",{aString:aString},globals.Object)})},
args: ["aString"],
source: "basicPerform: aString\x0a\x09^ self basicPerform: aString withArguments: #()",
messageSends: ["basicPerform:withArguments:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "basicPerform:withArguments:",
protocol: 'message handling',
fn: function (aString,aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self[aString].apply(self, aCollection);;
return self}, function($ctx1) {$ctx1.fill(self,"basicPerform:withArguments:",{aString:aString,aCollection:aCollection},globals.Object)})},
args: ["aString", "aCollection"],
source: "basicPerform: aString withArguments: aCollection\x0a\x09<return self[aString].apply(self, aCollection);>",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "browse",
protocol: 'browsing',
fn: function (){
var self=this;
function $Finder(){return globals.Finder||(typeof Finder=="undefined"?nil:Finder)}
return smalltalk.withContext(function($ctx1) { 
_st($Finder())._findClass_(self._class());
return self}, function($ctx1) {$ctx1.fill(self,"browse",{},globals.Object)})},
args: [],
source: "browse\x0a\x09Finder findClass: self class",
messageSends: ["findClass:", "class"],
referencedClasses: ["Finder"]
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "copy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._shallowCopy())._postCopy();
return $1;
}, function($ctx1) {$ctx1.fill(self,"copy",{},globals.Object)})},
args: [],
source: "copy\x0a\x09^ self shallowCopy postCopy",
messageSends: ["postCopy", "shallowCopy"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var copy = self.klass._new();
		Object.keys(self).forEach(function (i) {
		if(/^@.+/.test(i)) {
			copy[i] = self[i]._deepCopy();
		}
		});
		return copy;
	;
return self}, function($ctx1) {$ctx1.fill(self,"deepCopy",{},globals.Object)})},
args: [],
source: "deepCopy\x0a\x09<\x0a\x09\x09var copy = self.klass._new();\x0a\x09\x09Object.keys(self).forEach(function (i) {\x0a\x09\x09if(/^@.+/.test(i)) {\x0a\x09\x09\x09copy[i] = self[i]._deepCopy();\x0a\x09\x09}\x0a\x09\x09});\x0a\x09\x09return copy;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "deprecatedAPI",
protocol: 'error handling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$6,$5,$4,$8,$7,$3,$2;
$1=console;
$6=smalltalk.getThisContext()._home();
$ctx1.sendIdx["home"]=1;
$5=_st($6)._asString();
$ctx1.sendIdx["asString"]=1;
$4=_st($5).__comma(" is deprecated! (in ");
$8=_st(smalltalk.getThisContext()._home())._home();
$ctx1.sendIdx["home"]=2;
$7=_st($8)._asString();
$3=_st($4).__comma($7);
$ctx1.sendIdx[","]=2;
$2=_st($3).__comma(")");
$ctx1.sendIdx[","]=1;
_st($1)._warn_($2);
return self}, function($ctx1) {$ctx1.fill(self,"deprecatedAPI",{},globals.Object)})},
args: [],
source: "deprecatedAPI\x0a\x09\x22Just a simple way to deprecate methods.\x0a\x09#deprecatedAPI is in the 'error handling' protocol even if it doesn't throw an error,\x0a\x09but it could in the future.\x22\x0a\x09console warn: thisContext home asString, ' is deprecated! (in ', thisContext home home asString, ')'",
messageSends: ["warn:", ",", "asString", "home"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "error:",
protocol: 'error handling',
fn: function (aString){
var self=this;
function $Error(){return globals.Error||(typeof Error=="undefined"?nil:Error)}
return smalltalk.withContext(function($ctx1) { 
_st($Error())._signal_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"error:",{aString:aString},globals.Object)})},
args: ["aString"],
source: "error: aString\x0a\x09Error signal: aString",
messageSends: ["signal:"],
referencedClasses: ["Error"]
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "halt",
protocol: 'error handling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("Halt encountered");
return self}, function($ctx1) {$ctx1.fill(self,"halt",{},globals.Object)})},
args: [],
source: "halt\x0a\x09self error: 'Halt encountered'",
messageSends: ["error:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNil:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return self;
},
args: ["aBlock"],
source: "ifNil: aBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNil:ifNotNil:",
protocol: 'testing',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(anotherBlock)._value_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNil:ifNotNil:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Object)})},
args: ["aBlock", "anotherBlock"],
source: "ifNil: aBlock ifNotNil: anotherBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ anotherBlock value: self",
messageSends: ["value:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNotNil:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aBlock)._value_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNotNil:",{aBlock:aBlock},globals.Object)})},
args: ["aBlock"],
source: "ifNotNil: aBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ aBlock value: self",
messageSends: ["value:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNotNil:ifNil:",
protocol: 'testing',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aBlock)._value_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNotNil:ifNil:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Object)})},
args: ["aBlock", "anotherBlock"],
source: "ifNotNil: aBlock ifNil: anotherBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ aBlock value: self",
messageSends: ["value:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "inspectOn:",
protocol: 'inspecting',
fn: function (anInspector){
var self=this;
var variables;
function $Dictionary(){return globals.Dictionary||(typeof Dictionary=="undefined"?nil:Dictionary)}
return smalltalk.withContext(function($ctx1) { 
var $1;
variables=_st($Dictionary())._new();
_st(variables)._at_put_("#self",self);
$ctx1.sendIdx["at:put:"]=1;
_st(_st(self._class())._allInstanceVariableNames())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(variables)._at_put_(each,self._instVarAt_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
_st(anInspector)._setLabel_(self._printString());
$1=_st(anInspector)._setVariables_(variables);
return self}, function($ctx1) {$ctx1.fill(self,"inspectOn:",{anInspector:anInspector,variables:variables},globals.Object)})},
args: ["anInspector"],
source: "inspectOn: anInspector\x0a\x09| variables |\x0a\x09variables := Dictionary new.\x0a\x09variables at: '#self' put: self.\x0a\x09self class allInstanceVariableNames do: [ :each |\x0a\x09\x09variables at: each put: (self instVarAt: each) ].\x0a\x09anInspector\x0a\x09\x09setLabel: self printString;\x0a\x09\x09setVariables: variables",
messageSends: ["new", "at:put:", "do:", "allInstanceVariableNames", "class", "instVarAt:", "setLabel:", "printString", "setVariables:"],
referencedClasses: ["Dictionary"]
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isBehavior",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isBehavior\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isBoolean",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isBoolean\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isClass",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isClass\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isCompiledMethod",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isCompiledMethod\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isImmutable\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isKindOf:",
protocol: 'testing',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._isMemberOf_(aClass);
if(smalltalk.assert($2)){
$1=true;
} else {
$1=_st(self._class())._inheritsFrom_(aClass);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"isKindOf:",{aClass:aClass},globals.Object)})},
args: ["aClass"],
source: "isKindOf: aClass\x0a\x09^ (self isMemberOf: aClass)\x0a\x09\x09ifTrue: [ true ]\x0a\x09\x09ifFalse: [ self class inheritsFrom: aClass ]",
messageSends: ["ifTrue:ifFalse:", "isMemberOf:", "inheritsFrom:", "class"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isMemberOf:",
protocol: 'testing',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class()).__eq(aClass);
return $1;
}, function($ctx1) {$ctx1.fill(self,"isMemberOf:",{aClass:aClass},globals.Object)})},
args: ["aClass"],
source: "isMemberOf: aClass\x0a\x09^ self class = aClass",
messageSends: ["=", "class"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isMetaclass",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isMetaclass\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isNil",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isNil\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isNumber",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isNumber\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isPackage",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isPackage\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isParseFailure",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isParseFailure\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isString",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isString\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "isSymbol",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isSymbol\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "notNil",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._isNil())._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"notNil",{},globals.Object)})},
args: [],
source: "notNil\x0a\x09^ self isNil not",
messageSends: ["not", "isNil"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "postCopy",
protocol: 'copying',
fn: function (){
var self=this;
return self},
args: [],
source: "postCopy",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "putOn:",
protocol: 'streaming',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPut_(self);
return self}, function($ctx1) {$ctx1.fill(self,"putOn:",{aStream:aStream},globals.Object)})},
args: ["aStream"],
source: "putOn: aStream\x0a\x09aStream nextPut: self",
messageSends: ["nextPut:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "respondsTo:",
protocol: 'testing',
fn: function (aSelector){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._canUnderstand_(aSelector);
return $1;
}, function($ctx1) {$ctx1.fill(self,"respondsTo:",{aSelector:aSelector},globals.Object)})},
args: ["aSelector"],
source: "respondsTo: aSelector\x0a\x09^ self class canUnderstand: aSelector",
messageSends: ["canUnderstand:", "class"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "shallowCopy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var copy = self.klass._new();
		Object.keys(self).forEach(function(i) {
		if(/^@.+/.test(i)) {
			copy[i] = self[i];
		}
		});
		return copy;
	;
return self}, function($ctx1) {$ctx1.fill(self,"shallowCopy",{},globals.Object)})},
args: [],
source: "shallowCopy\x0a\x09<\x0a\x09\x09var copy = self.klass._new();\x0a\x09\x09Object.keys(self).forEach(function(i) {\x0a\x09\x09if(/^@.+/.test(i)) {\x0a\x09\x09\x09copy[i] = self[i];\x0a\x09\x09}\x0a\x09\x09});\x0a\x09\x09return copy;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldNotImplement",
protocol: 'error handling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("This method should not be implemented in ".__comma(_st(self._class())._name()));
return self}, function($ctx1) {$ctx1.fill(self,"shouldNotImplement",{},globals.Object)})},
args: [],
source: "shouldNotImplement\x0a\x09self error: 'This method should not be implemented in ', self class name",
messageSends: ["error:", ",", "name", "class"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("Object not indexable");
return self}, function($ctx1) {$ctx1.fill(self,"size",{},globals.Object)})},
args: [],
source: "size\x0a\x09self error: 'Object not indexable'",
messageSends: ["error:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "subclassResponsibility",
protocol: 'error handling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("This method is a responsibility of a subclass");
return self}, function($ctx1) {$ctx1.fill(self,"subclassResponsibility",{},globals.Object)})},
args: [],
source: "subclassResponsibility\x0a\x09self error: 'This method is a responsibility of a subclass'",
messageSends: ["error:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "throw:",
protocol: 'error handling',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 throw anObject ;
return self}, function($ctx1) {$ctx1.fill(self,"throw:",{anObject:anObject},globals.Object)})},
args: ["anObject"],
source: "throw: anObject\x0a\x09< throw anObject >",
messageSends: [],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "try:catch:",
protocol: 'error handling',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=_st(aBlock)._tryCatch_(anotherBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"try:catch:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Object)})},
args: ["aBlock", "anotherBlock"],
source: "try: aBlock catch: anotherBlock\x0a\x09self deprecatedAPI.\x0a\x09\x0a\x09^ aBlock tryCatch: anotherBlock",
messageSends: ["deprecatedAPI", "tryCatch:"],
referencedClasses: []
}),
globals.Object);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.valueOf();
return self}, function($ctx1) {$ctx1.fill(self,"value",{},globals.Object)})},
args: [],
source: "value\x0a\x09<return self.valueOf()>",
messageSends: [],
referencedClasses: []
}),
globals.Object);


smalltalk.addMethod(
smalltalk.method({
selector: "accessorProtocolWith:",
protocol: 'helios',
fn: function (aGenerator){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aGenerator)._accessorProtocolForObject();
return self}, function($ctx1) {$ctx1.fill(self,"accessorProtocolWith:",{aGenerator:aGenerator},globals.Object.klass)})},
args: ["aGenerator"],
source: "accessorProtocolWith: aGenerator\x0a\x09aGenerator accessorProtocolForObject",
messageSends: ["accessorProtocolForObject"],
referencedClasses: []
}),
globals.Object.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "accessorsSourceCodesWith:",
protocol: 'helios',
fn: function (aGenerator){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aGenerator)._accessorsForObject();
return self}, function($ctx1) {$ctx1.fill(self,"accessorsSourceCodesWith:",{aGenerator:aGenerator},globals.Object.klass)})},
args: ["aGenerator"],
source: "accessorsSourceCodesWith: aGenerator\x0a\x09aGenerator accessorsForObject",
messageSends: ["accessorsForObject"],
referencedClasses: []
}),
globals.Object.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "class";
},
args: [],
source: "heliosClass\x0a\x09\x22Should be an Helios extension. Unfortunately, since helios can browse remote\x0a\x09environments, we can't extend base classes\x22\x0a\x09\x0a\x09^ 'class'",
messageSends: [],
referencedClasses: []
}),
globals.Object.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return self},
args: [],
source: "initialize\x0a\x09\x22no op\x22",
messageSends: [],
referencedClasses: []
}),
globals.Object.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "initializeProtocolWith:",
protocol: 'helios',
fn: function (aGenerator){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aGenerator)._initializeProtocolForObject();
return self}, function($ctx1) {$ctx1.fill(self,"initializeProtocolWith:",{aGenerator:aGenerator},globals.Object.klass)})},
args: ["aGenerator"],
source: "initializeProtocolWith: aGenerator\x0a\x09aGenerator initializeProtocolForObject",
messageSends: ["initializeProtocolForObject"],
referencedClasses: []
}),
globals.Object.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "initializeSourceCodesWith:",
protocol: 'helios',
fn: function (aGenerator){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aGenerator)._initializeForObject();
return self}, function($ctx1) {$ctx1.fill(self,"initializeSourceCodesWith:",{aGenerator:aGenerator},globals.Object.klass)})},
args: ["aGenerator"],
source: "initializeSourceCodesWith: aGenerator\x0a\x09aGenerator initializeForObject",
messageSends: ["initializeForObject"],
referencedClasses: []
}),
globals.Object.klass);


smalltalk.addClass('Boolean', globals.Object, [], 'Kernel-Objects');
globals.Boolean.comment="I define the protocol for logic testing operations and conditional control structures for the logical values (see the `controlling` protocol).\x0a\x0aI have two instances, `true` and `false`.\x0a\x0aI am directly mapped to JavaScript Boolean. The `true` and `false` objects are the JavaScript boolean objects.\x0a\x0a## Usage Example:\x0a\x0a    aBoolean not ifTrue: [ ... ] ifFalse: [ ... ]";
smalltalk.addMethod(
smalltalk.method({
selector: "&",
protocol: 'controlling',
fn: function (aBoolean){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if(self == true) {
		return aBoolean;
		} else {
		return false;
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"&",{aBoolean:aBoolean},globals.Boolean)})},
args: ["aBoolean"],
source: "& aBoolean\x0a\x09<\x0a\x09\x09if(self == true) {\x0a\x09\x09return aBoolean;\x0a\x09\x09} else {\x0a\x09\x09return false;\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (aBoolean){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return aBoolean != null &&
			typeof aBoolean._isBoolean === "function" &&
			aBoolean._isBoolean() &&
			Boolean(self == true) == aBoolean
	;
return self}, function($ctx1) {$ctx1.fill(self,"=",{aBoolean:aBoolean},globals.Boolean)})},
args: ["aBoolean"],
source: "= aBoolean\x0a\x09<\x0a\x09\x09return aBoolean != null &&\x0a\x09\x09\x09typeof aBoolean._isBoolean === \x22function\x22 &&\x0a\x09\x09\x09aBoolean._isBoolean() &&\x0a\x09\x09\x09Boolean(self == true) == aBoolean\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "==",
protocol: 'comparing',
fn: function (aBoolean){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__eq(aBoolean);
return $1;
}, function($ctx1) {$ctx1.fill(self,"==",{aBoolean:aBoolean},globals.Boolean)})},
args: ["aBoolean"],
source: "== aBoolean\x0a\x09^ self = aBoolean",
messageSends: ["="],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "and:",
protocol: 'controlling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self.__eq(true);
$1=_st($2)._ifTrue_ifFalse_(aBlock,(function(){
return false;
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"and:",{aBlock:aBlock},globals.Boolean)})},
args: ["aBlock"],
source: "and: aBlock\x0a\x09^ self = true\x0a\x09\x09ifTrue: aBlock\x0a\x09\x09ifFalse: [ false ]",
messageSends: ["ifTrue:ifFalse:", "="],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "asBit",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
if(smalltalk.assert(self)){
$1=(1);
} else {
$1=(0);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"asBit",{},globals.Boolean)})},
args: [],
source: "asBit\x0a\x09^ self ifTrue: [ 1 ] ifFalse: [ 0 ]",
messageSends: ["ifTrue:ifFalse:"],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asJSON\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 return self.toString() ;
return self}, function($ctx1) {$ctx1.fill(self,"asString",{},globals.Boolean)})},
args: [],
source: "asString\x0a\x09< return self.toString() >",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
return self;
},
args: [],
source: "deepCopy\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "ifFalse:",
protocol: 'controlling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._ifTrue_ifFalse_((function(){
}),aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifFalse:",{aBlock:aBlock},globals.Boolean)})},
args: ["aBlock"],
source: "ifFalse: aBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ self ifTrue: [] ifFalse: aBlock",
messageSends: ["ifTrue:ifFalse:"],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "ifFalse:ifTrue:",
protocol: 'controlling',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._ifTrue_ifFalse_(anotherBlock,aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifFalse:ifTrue:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Boolean)})},
args: ["aBlock", "anotherBlock"],
source: "ifFalse: aBlock ifTrue: anotherBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ self ifTrue: anotherBlock ifFalse: aBlock",
messageSends: ["ifTrue:ifFalse:"],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "ifTrue:",
protocol: 'controlling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._ifTrue_ifFalse_(aBlock,(function(){
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifTrue:",{aBlock:aBlock},globals.Boolean)})},
args: ["aBlock"],
source: "ifTrue: aBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ self ifTrue: aBlock ifFalse: []",
messageSends: ["ifTrue:ifFalse:"],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "ifTrue:ifFalse:",
protocol: 'controlling',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if(self == true) {
		return aBlock._value();
		} else {
		return anotherBlock._value();
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"ifTrue:ifFalse:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Boolean)})},
args: ["aBlock", "anotherBlock"],
source: "ifTrue: aBlock ifFalse: anotherBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09<\x0a\x09\x09if(self == true) {\x0a\x09\x09return aBlock._value();\x0a\x09\x09} else {\x0a\x09\x09return anotherBlock._value();\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "isBoolean",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isBoolean\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isImmutable\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "not",
protocol: 'controlling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__eq(false);
return $1;
}, function($ctx1) {$ctx1.fill(self,"not",{},globals.Boolean)})},
args: [],
source: "not\x0a\x09^ self = false",
messageSends: ["="],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "or:",
protocol: 'controlling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self.__eq(true);
$1=_st($2)._ifTrue_ifFalse_((function(){
return true;
}),aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"or:",{aBlock:aBlock},globals.Boolean)})},
args: ["aBlock"],
source: "or: aBlock\x0a\x09^ self = true\x0a\x09\x09ifTrue: [ true ]\x0a\x09\x09ifFalse: aBlock",
messageSends: ["ifTrue:ifFalse:", "="],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutAll_(self._asString());
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Boolean)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: self asString",
messageSends: ["nextPutAll:", "asString"],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "shallowCopy",
protocol: 'copying',
fn: function (){
var self=this;
return self;
},
args: [],
source: "shallowCopy\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);

smalltalk.addMethod(
smalltalk.method({
selector: "|",
protocol: 'controlling',
fn: function (aBoolean){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if(self == true) {
		return true;
		} else {
		return aBoolean;
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"|",{aBoolean:aBoolean},globals.Boolean)})},
args: ["aBoolean"],
source: "| aBoolean\x0a\x09<\x0a\x09\x09if(self == true) {\x0a\x09\x09return true;\x0a\x09\x09} else {\x0a\x09\x09return aBoolean;\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Boolean);



smalltalk.addClass('Date', globals.Object, [], 'Kernel-Objects');
globals.Date.comment="I am used to work with both dates and times. Therefore `Date today` and `Date now` are both valid in\x0aAmber and answer the same date object.\x0a\x0aDate directly maps to the `Date()` JavaScript constructor, and Amber date objects are JavaScript date objects.\x0a\x0a## API\x0a\x0aThe class-side `instance creation` protocol contains some convenience methods for creating date/time objects such as `#fromSeconds:`.\x0a\x0aArithmetic and comparison is supported (see the `comparing` and `arithmetic` protocols).\x0a\x0aThe `converting` protocol provides convenience methods for various convertions (to numbers, strings, etc.).";
smalltalk.addMethod(
smalltalk.method({
selector: "+",
protocol: 'arithmetic',
fn: function (aDate){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self + aDate;
return self}, function($ctx1) {$ctx1.fill(self,"+",{aDate:aDate},globals.Date)})},
args: ["aDate"],
source: "+ aDate\x0a\x09<return self + aDate>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "-",
protocol: 'arithmetic',
fn: function (aDate){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self - aDate;
return self}, function($ctx1) {$ctx1.fill(self,"-",{aDate:aDate},globals.Date)})},
args: ["aDate"],
source: "- aDate\x0a\x09<return self - aDate>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "<",
protocol: 'comparing',
fn: function (aDate){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self < aDate;
return self}, function($ctx1) {$ctx1.fill(self,"<",{aDate:aDate},globals.Date)})},
args: ["aDate"],
source: "< aDate\x0a\x09<return self < aDate>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "<=",
protocol: 'comparing',
fn: function (aDate){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self <= aDate;
return self}, function($ctx1) {$ctx1.fill(self,"<=",{aDate:aDate},globals.Date)})},
args: ["aDate"],
source: "<= aDate\x0a\x09<return self <= aDate>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: ">",
protocol: 'comparing',
fn: function (aDate){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self > aDate;
return self}, function($ctx1) {$ctx1.fill(self,">",{aDate:aDate},globals.Date)})},
args: ["aDate"],
source: "> aDate\x0a\x09<return self >> aDate>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: ">=",
protocol: 'comparing',
fn: function (aDate){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self >= aDate;
return self}, function($ctx1) {$ctx1.fill(self,">=",{aDate:aDate},globals.Date)})},
args: ["aDate"],
source: ">= aDate\x0a\x09<return self >>= aDate>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "asDateString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toDateString();
return self}, function($ctx1) {$ctx1.fill(self,"asDateString",{},globals.Date)})},
args: [],
source: "asDateString\x0a\x09<return self.toDateString()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "asLocaleString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toLocaleString();
return self}, function($ctx1) {$ctx1.fill(self,"asLocaleString",{},globals.Date)})},
args: [],
source: "asLocaleString\x0a\x09<return self.toLocaleString()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "asMilliseconds",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._time();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asMilliseconds",{},globals.Date)})},
args: [],
source: "asMilliseconds\x0a\x09^ self time",
messageSends: ["time"],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "asNumber",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._asMilliseconds();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asNumber",{},globals.Date)})},
args: [],
source: "asNumber\x0a\x09^ self asMilliseconds",
messageSends: ["asMilliseconds"],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toString();
return self}, function($ctx1) {$ctx1.fill(self,"asString",{},globals.Date)})},
args: [],
source: "asString\x0a\x09<return self.toString()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "asTimeString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toTimeString();
return self}, function($ctx1) {$ctx1.fill(self,"asTimeString",{},globals.Date)})},
args: [],
source: "asTimeString\x0a\x09<return self.toTimeString()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "day",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._dayOfWeek();
return $1;
}, function($ctx1) {$ctx1.fill(self,"day",{},globals.Date)})},
args: [],
source: "day\x0a\x09^ self dayOfWeek",
messageSends: ["dayOfWeek"],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "day:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._dayOfWeek_(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"day:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "day: aNumber\x0a\x09self dayOfWeek: aNumber",
messageSends: ["dayOfWeek:"],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "dayOfMonth",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getDate();
return self}, function($ctx1) {$ctx1.fill(self,"dayOfMonth",{},globals.Date)})},
args: [],
source: "dayOfMonth\x0a\x09<return self.getDate()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "dayOfMonth:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setDate(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"dayOfMonth:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "dayOfMonth: aNumber\x0a\x09<self.setDate(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "dayOfWeek",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getDay() + 1;
return self}, function($ctx1) {$ctx1.fill(self,"dayOfWeek",{},globals.Date)})},
args: [],
source: "dayOfWeek\x0a\x09<return self.getDay() + 1>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "dayOfWeek:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.setDay(aNumber - 1);
return self}, function($ctx1) {$ctx1.fill(self,"dayOfWeek:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "dayOfWeek: aNumber\x0a\x09<return self.setDay(aNumber - 1)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "hours",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getHours();
return self}, function($ctx1) {$ctx1.fill(self,"hours",{},globals.Date)})},
args: [],
source: "hours\x0a\x09<return self.getHours()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "hours:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setHours(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"hours:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "hours: aNumber\x0a\x09<self.setHours(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "milliseconds",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getMilliseconds();
return self}, function($ctx1) {$ctx1.fill(self,"milliseconds",{},globals.Date)})},
args: [],
source: "milliseconds\x0a\x09<return self.getMilliseconds()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "milliseconds:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setMilliseconds(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"milliseconds:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "milliseconds: aNumber\x0a\x09<self.setMilliseconds(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "minutes",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getMinutes();
return self}, function($ctx1) {$ctx1.fill(self,"minutes",{},globals.Date)})},
args: [],
source: "minutes\x0a\x09<return self.getMinutes()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "minutes:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setMinutes(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"minutes:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "minutes: aNumber\x0a\x09<self.setMinutes(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "month",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getMonth() + 1;
return self}, function($ctx1) {$ctx1.fill(self,"month",{},globals.Date)})},
args: [],
source: "month\x0a\x09<return self.getMonth() + 1>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "month:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setMonth(aNumber - 1);
return self}, function($ctx1) {$ctx1.fill(self,"month:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "month: aNumber\x0a\x09<self.setMonth(aNumber - 1)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutAll_(self._asString());
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Date)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: self asString",
messageSends: ["nextPutAll:", "asString"],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "seconds",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getSeconds();
return self}, function($ctx1) {$ctx1.fill(self,"seconds",{},globals.Date)})},
args: [],
source: "seconds\x0a\x09<return self.getSeconds()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "seconds:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setSeconds(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"seconds:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "seconds: aNumber\x0a\x09<self.setSeconds(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "time",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getTime();
return self}, function($ctx1) {$ctx1.fill(self,"time",{},globals.Date)})},
args: [],
source: "time\x0a\x09<return self.getTime()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "time:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setTime(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"time:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "time: aNumber\x0a\x09<self.setTime(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "year",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.getFullYear();
return self}, function($ctx1) {$ctx1.fill(self,"year",{},globals.Date)})},
args: [],
source: "year\x0a\x09<return self.getFullYear()>",
messageSends: [],
referencedClasses: []
}),
globals.Date);

smalltalk.addMethod(
smalltalk.method({
selector: "year:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.setFullYear(aNumber);
return self}, function($ctx1) {$ctx1.fill(self,"year:",{aNumber:aNumber},globals.Date)})},
args: ["aNumber"],
source: "year: aNumber\x0a\x09<self.setFullYear(aNumber)>",
messageSends: [],
referencedClasses: []
}),
globals.Date);


smalltalk.addMethod(
smalltalk.method({
selector: "fromMilliseconds:",
protocol: 'instance creation',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._new_(aNumber);
return $1;
}, function($ctx1) {$ctx1.fill(self,"fromMilliseconds:",{aNumber:aNumber},globals.Date.klass)})},
args: ["aNumber"],
source: "fromMilliseconds: aNumber\x0a\x09^ self new: aNumber",
messageSends: ["new:"],
referencedClasses: []
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "fromSeconds:",
protocol: 'instance creation',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._fromMilliseconds_(_st(aNumber).__star((1000)));
return $1;
}, function($ctx1) {$ctx1.fill(self,"fromSeconds:",{aNumber:aNumber},globals.Date.klass)})},
args: ["aNumber"],
source: "fromSeconds: aNumber\x0a\x09^ self fromMilliseconds: aNumber * 1000",
messageSends: ["fromMilliseconds:", "*"],
referencedClasses: []
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "fromString:",
protocol: 'instance creation',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._new_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"fromString:",{aString:aString},globals.Date.klass)})},
args: ["aString"],
source: "fromString: aString\x0a\x09\x22Example: Date fromString('2011/04/15 00:00:00')\x22\x0a\x09^ self new: aString",
messageSends: ["new:"],
referencedClasses: []
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "magnitude";
},
args: [],
source: "heliosClass\x0a\x09^ 'magnitude'",
messageSends: [],
referencedClasses: []
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "millisecondsToRun:",
protocol: 'instance creation',
fn: function (aBlock){
var self=this;
var t;
function $Date(){return globals.Date||(typeof Date=="undefined"?nil:Date)}
return smalltalk.withContext(function($ctx1) { 
var $1;
t=_st($Date())._now();
$ctx1.sendIdx["now"]=1;
_st(aBlock)._value();
$1=_st(_st($Date())._now()).__minus(t);
return $1;
}, function($ctx1) {$ctx1.fill(self,"millisecondsToRun:",{aBlock:aBlock,t:t},globals.Date.klass)})},
args: ["aBlock"],
source: "millisecondsToRun: aBlock\x0a\x09| t |\x0a\x09t := Date now.\x0a\x09aBlock value.\x0a\x09^ Date now - t",
messageSends: ["now", "value", "-"],
referencedClasses: ["Date"]
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "new:",
protocol: 'instance creation',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return new Date(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"new:",{anObject:anObject},globals.Date.klass)})},
args: ["anObject"],
source: "new: anObject\x0a\x09<return new Date(anObject)>",
messageSends: [],
referencedClasses: []
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "now",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._today();
return $1;
}, function($ctx1) {$ctx1.fill(self,"now",{},globals.Date.klass)})},
args: [],
source: "now\x0a\x09^ self today",
messageSends: ["today"],
referencedClasses: []
}),
globals.Date.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "today",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._new();
return $1;
}, function($ctx1) {$ctx1.fill(self,"today",{},globals.Date.klass)})},
args: [],
source: "today\x0a\x09^ self new",
messageSends: ["new"],
referencedClasses: []
}),
globals.Date.klass);


smalltalk.addClass('Number', globals.Object, [], 'Kernel-Objects');
globals.Number.comment="I am the Amber representation for all numbers.\x0aI am directly mapped to JavaScript Number.\x0a\x0a## API\x0a\x0aI provide all necessary methods for arithmetic operations, comparison, conversion and so on with numbers.\x0a\x0aMy instances can also be used to evaluate a block a fixed number of times:\x0a\x0a\x095 timesRepeat: [ Transcript show: 'This will be printed 5 times'; cr ].\x0a\x09\x0a\x091 to: 5 do: [ :aNumber| Transcript show: aNumber asString; cr ].\x0a\x09\x0a\x091 to: 10 by: 2 do: [ :aNumber| Transcript show: aNumber asString; cr ].";
smalltalk.addMethod(
smalltalk.method({
selector: "&",
protocol: 'converting',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self & aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"&",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "& aNumber\x0a\x09<return self & aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "*",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self * aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"*",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "* aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self * aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "**",
protocol: 'mathematical functions',
fn: function (exponent){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._raisedTo_(exponent);
return $1;
}, function($ctx1) {$ctx1.fill(self,"**",{exponent:exponent},globals.Number)})},
args: ["exponent"],
source: "** exponent\x0a\x09^ self raisedTo: exponent",
messageSends: ["raisedTo:"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "+",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self + aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"+",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "+ aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self + aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "-",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self - aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"-",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "- aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self - aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "/",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self / aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"/",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "/ aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self / aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "//",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self.__slash(aNumber))._floor();
return $1;
}, function($ctx1) {$ctx1.fill(self,"//",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "// aNumber\x0a\x09^ (self / aNumber) floor",
messageSends: ["floor", "/"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "<",
protocol: 'comparing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self < aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"<",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "< aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self < aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "<=",
protocol: 'comparing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self <= aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"<=",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "<= aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self <= aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return aNumber != null &&
			typeof aNumber._isNumber === "function" &&
			aNumber._isNumber() &&
			Number(self) == aNumber
	;
return self}, function($ctx1) {$ctx1.fill(self,"=",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "= aNumber\x0a\x09<\x0a\x09\x09return aNumber != null &&\x0a\x09\x09\x09typeof aNumber._isNumber === \x22function\x22 &&\x0a\x09\x09\x09aNumber._isNumber() &&\x0a\x09\x09\x09Number(self) == aNumber\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: ">",
protocol: 'comparing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self > aNumber;
return self}, function($ctx1) {$ctx1.fill(self,">",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "> aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self >> aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: ">=",
protocol: 'comparing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self >= aNumber;
return self}, function($ctx1) {$ctx1.fill(self,">=",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: ">= aNumber\x0a\x09\x22Inlined in the Compiler\x22\x0a\x09<return self >>= aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "@",
protocol: 'converting',
fn: function (aNumber){
var self=this;
function $Point(){return globals.Point||(typeof Point=="undefined"?nil:Point)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Point())._x_y_(self,aNumber);
return $1;
}, function($ctx1) {$ctx1.fill(self,"@",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "@ aNumber\x0a\x09^ Point x: self y: aNumber",
messageSends: ["x:y:"],
referencedClasses: ["Point"]
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "\x5c\x5c",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self % aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"\x5c\x5c",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "\x5c\x5c aNumber\x0a\x09<return self % aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "abs",
protocol: 'arithmetic',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.abs(self);;
return self}, function($ctx1) {$ctx1.fill(self,"abs",{},globals.Number)})},
args: [],
source: "abs\x0a\x09<return Math.abs(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "arcCos",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.acos(self);;
return self}, function($ctx1) {$ctx1.fill(self,"arcCos",{},globals.Number)})},
args: [],
source: "arcCos\x0a\x09<return Math.acos(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "arcSin",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.asin(self);;
return self}, function($ctx1) {$ctx1.fill(self,"arcSin",{},globals.Number)})},
args: [],
source: "arcSin\x0a\x09<return Math.asin(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "arcTan",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.atan(self);;
return self}, function($ctx1) {$ctx1.fill(self,"arcTan",{},globals.Number)})},
args: [],
source: "arcTan\x0a\x09<return Math.atan(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asJSON\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "asJavascript",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st("(".__comma(self._printString())).__comma(")");
$ctx1.sendIdx[","]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJavascript",{},globals.Number)})},
args: [],
source: "asJavascript\x0a\x09^ '(', self printString, ')'",
messageSends: [",", "printString"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "asNumber",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asNumber\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "asPoint",
protocol: 'converting',
fn: function (){
var self=this;
function $Point(){return globals.Point||(typeof Point=="undefined"?nil:Point)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Point())._x_y_(self,self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"asPoint",{},globals.Number)})},
args: [],
source: "asPoint\x0a\x09^ Point x: self y: self",
messageSends: ["x:y:"],
referencedClasses: ["Point"]
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 return String(self) ;
return self}, function($ctx1) {$ctx1.fill(self,"asString",{},globals.Number)})},
args: [],
source: "asString\x0a\x09< return String(self) >",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "atRandom",
protocol: 'converting',
fn: function (){
var self=this;
function $Random(){return globals.Random||(typeof Random=="undefined"?nil:Random)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(_st(_st(_st($Random())._new())._next()).__star(self))._truncated()).__plus((1));
return $1;
}, function($ctx1) {$ctx1.fill(self,"atRandom",{},globals.Number)})},
args: [],
source: "atRandom\x0a\x09^ (Random new next * self) truncated + 1",
messageSends: ["+", "truncated", "*", "next", "new"],
referencedClasses: ["Random"]
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "ceiling",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.ceil(self);;
return self}, function($ctx1) {$ctx1.fill(self,"ceiling",{},globals.Number)})},
args: [],
source: "ceiling\x0a\x09<return Math.ceil(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "copy",
protocol: 'copying',
fn: function (){
var self=this;
return self;
},
args: [],
source: "copy\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "cos",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.cos(self);;
return self}, function($ctx1) {$ctx1.fill(self,"cos",{},globals.Number)})},
args: [],
source: "cos\x0a\x09<return Math.cos(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._copy();
return $1;
}, function($ctx1) {$ctx1.fill(self,"deepCopy",{},globals.Number)})},
args: [],
source: "deepCopy\x0a\x09^ self copy",
messageSends: ["copy"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "even",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=(0).__eq(self.__backslash_backslash((2)));
return $1;
}, function($ctx1) {$ctx1.fill(self,"even",{},globals.Number)})},
args: [],
source: "even\x0a\x09^ 0 = (self \x5c\x5c 2)",
messageSends: ["=", "\x5c\x5c"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "floor",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.floor(self);;
return self}, function($ctx1) {$ctx1.fill(self,"floor",{},globals.Number)})},
args: [],
source: "floor\x0a\x09<return Math.floor(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "identityHash",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._asString()).__comma("n");
return $1;
}, function($ctx1) {$ctx1.fill(self,"identityHash",{},globals.Number)})},
args: [],
source: "identityHash\x0a\x09^ self asString, 'n'",
messageSends: [",", "asString"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isImmutable\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "isNumber",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isNumber\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "isZero",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__eq((0));
return $1;
}, function($ctx1) {$ctx1.fill(self,"isZero",{},globals.Number)})},
args: [],
source: "isZero\x0a\x09^ self = 0",
messageSends: ["="],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "ln",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.log(self);;
return self}, function($ctx1) {$ctx1.fill(self,"ln",{},globals.Number)})},
args: [],
source: "ln\x0a\x09<return Math.log(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "log",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.log(self) / Math.LN10;;
return self}, function($ctx1) {$ctx1.fill(self,"log",{},globals.Number)})},
args: [],
source: "log\x0a\x09<return Math.log(self) / Math.LN10;>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "log:",
protocol: 'mathematical functions',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.log(self) / Math.log(aNumber);;
return self}, function($ctx1) {$ctx1.fill(self,"log:",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "log: aNumber\x0a\x09<return Math.log(self) / Math.log(aNumber);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "max:",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.max(self, aNumber);;
return self}, function($ctx1) {$ctx1.fill(self,"max:",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "max: aNumber\x0a\x09<return Math.max(self, aNumber);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "min:",
protocol: 'arithmetic',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.min(self, aNumber);;
return self}, function($ctx1) {$ctx1.fill(self,"min:",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "min: aNumber\x0a\x09<return Math.min(self, aNumber);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "negated",
protocol: 'arithmetic',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=(0).__minus(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"negated",{},globals.Number)})},
args: [],
source: "negated\x0a\x09^ 0 - self",
messageSends: ["-"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "negative",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__lt((0));
return $1;
}, function($ctx1) {$ctx1.fill(self,"negative",{},globals.Number)})},
args: [],
source: "negative\x0a\x09\x22Answer whether the receiver is mathematically negative.\x22\x0a\x0a\x09^ self < 0",
messageSends: ["<"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "odd",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._even())._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"odd",{},globals.Number)})},
args: [],
source: "odd\x0a\x09^ self even not",
messageSends: ["not", "even"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "positive",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__gt_eq((0));
return $1;
}, function($ctx1) {$ctx1.fill(self,"positive",{},globals.Number)})},
args: [],
source: "positive\x0a\x09\x22Answer whether the receiver is positive or equal to 0. (ST-80 protocol).\x22\x0a\x0a\x09^ self >= 0",
messageSends: [">="],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutAll_(self._asString());
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Number)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: self asString",
messageSends: ["nextPutAll:", "asString"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "printShowingDecimalPlaces:",
protocol: 'printing',
fn: function (placesDesired){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toFixed(placesDesired);
return self}, function($ctx1) {$ctx1.fill(self,"printShowingDecimalPlaces:",{placesDesired:placesDesired},globals.Number)})},
args: ["placesDesired"],
source: "printShowingDecimalPlaces: placesDesired\x0a\x09<return self.toFixed(placesDesired)>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "raisedTo:",
protocol: 'mathematical functions',
fn: function (exponent){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.pow(self, exponent);;
return self}, function($ctx1) {$ctx1.fill(self,"raisedTo:",{exponent:exponent},globals.Number)})},
args: ["exponent"],
source: "raisedTo: exponent\x0a\x09<return Math.pow(self, exponent);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "rounded",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.round(self);;
return self}, function($ctx1) {$ctx1.fill(self,"rounded",{},globals.Number)})},
args: [],
source: "rounded\x0a\x09<return Math.round(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "sign",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self._isZero();
if(smalltalk.assert($1)){
return (0);
};
$2=self._positive();
if(smalltalk.assert($2)){
return (1);
} else {
return (-1);
};
return self}, function($ctx1) {$ctx1.fill(self,"sign",{},globals.Number)})},
args: [],
source: "sign\x0a\x09self isZero \x0a\x09\x09ifTrue: [ ^ 0 ].\x0a\x09self positive\x0a\x09\x09ifTrue: [ ^ 1 ]\x0a\x09\x09ifFalse: [ ^ -1 ].",
messageSends: ["ifTrue:", "isZero", "ifTrue:ifFalse:", "positive"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "sin",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.sin(self);;
return self}, function($ctx1) {$ctx1.fill(self,"sin",{},globals.Number)})},
args: [],
source: "sin\x0a\x09<return Math.sin(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "sqrt",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.sqrt(self);
return self}, function($ctx1) {$ctx1.fill(self,"sqrt",{},globals.Number)})},
args: [],
source: "sqrt\x0a\x09<return Math.sqrt(self)>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "squared",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__star(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"squared",{},globals.Number)})},
args: [],
source: "squared\x0a\x09^ self * self",
messageSends: ["*"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "tan",
protocol: 'mathematical functions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.tan(self);;
return self}, function($ctx1) {$ctx1.fill(self,"tan",{},globals.Number)})},
args: [],
source: "tan\x0a\x09<return Math.tan(self);>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "timesRepeat:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var count;
return smalltalk.withContext(function($ctx1) { 
count=(1);
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(count).__gt(self);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileFalse_((function(){
return smalltalk.withContext(function($ctx2) {
_st(aBlock)._value();
count=_st(count).__plus((1));
return count;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"timesRepeat:",{aBlock:aBlock,count:count},globals.Number)})},
args: ["aBlock"],
source: "timesRepeat: aBlock\x0a\x09| count |\x0a\x09count := 1.\x0a\x09[ count > self ] whileFalse: [\x0a\x09\x09aBlock value.\x0a\x09\x09count := count + 1 ]",
messageSends: ["whileFalse:", ">", "value", "+"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "to:",
protocol: 'converting',
fn: function (aNumber){
var self=this;
var array,first,last,count;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1;
first=self._truncated();
$ctx1.sendIdx["truncated"]=1;
last=_st(_st(aNumber)._truncated()).__plus((1));
$ctx1.sendIdx["+"]=1;
count=(1);
array=_st($Array())._new();
_st(_st(last).__minus(first))._timesRepeat_((function(){
return smalltalk.withContext(function($ctx2) {
_st(array)._at_put_(count,first);
count=_st(count).__plus((1));
$ctx2.sendIdx["+"]=2;
count;
first=_st(first).__plus((1));
return first;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$1=array;
return $1;
}, function($ctx1) {$ctx1.fill(self,"to:",{aNumber:aNumber,array:array,first:first,last:last,count:count},globals.Number)})},
args: ["aNumber"],
source: "to: aNumber\x0a\x09| array first last count |\x0a\x09first := self truncated.\x0a\x09last := aNumber truncated + 1.\x0a\x09count := 1.\x0a\x09array := Array new.\x0a\x09(last - first) timesRepeat: [\x0a\x09\x09array at: count put: first.\x0a\x09\x09count := count + 1.\x0a\x09\x09first := first + 1 ].\x0a\x09^ array",
messageSends: ["truncated", "+", "new", "timesRepeat:", "-", "at:put:"],
referencedClasses: ["Array"]
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "to:by:",
protocol: 'converting',
fn: function (stop,step){
var self=this;
var array,value,pos;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
value=self;
array=_st($Array())._new();
pos=(1);
$1=_st(step).__eq((0));
if(smalltalk.assert($1)){
self._error_("step must be non-zero");
};
$2=_st(step).__lt((0));
if(smalltalk.assert($2)){
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(value).__gt_eq(stop);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}))._whileTrue_((function(){
return smalltalk.withContext(function($ctx2) {
_st(array)._at_put_(pos,value);
$ctx2.sendIdx["at:put:"]=1;
pos=_st(pos).__plus((1));
$ctx2.sendIdx["+"]=1;
pos;
value=_st(value).__plus(step);
$ctx2.sendIdx["+"]=2;
return value;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,4)})}));
$ctx1.sendIdx["whileTrue:"]=1;
} else {
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(value).__lt_eq(stop);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,6)})}))._whileTrue_((function(){
return smalltalk.withContext(function($ctx2) {
_st(array)._at_put_(pos,value);
pos=_st(pos).__plus((1));
$ctx2.sendIdx["+"]=3;
pos;
value=_st(value).__plus(step);
return value;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,7)})}));
};
$3=array;
return $3;
}, function($ctx1) {$ctx1.fill(self,"to:by:",{stop:stop,step:step,array:array,value:value,pos:pos},globals.Number)})},
args: ["stop", "step"],
source: "to: stop by: step\x0a\x09| array value pos |\x0a\x09value := self.\x0a\x09array := Array new.\x0a\x09pos := 1.\x0a\x09step = 0 ifTrue: [ self error: 'step must be non-zero' ].\x0a\x09step < 0\x0a\x09\x09ifTrue: [ [ value >= stop ] whileTrue: [\x0a\x09\x09\x09\x09\x09array at: pos put: value.\x0a\x09\x09\x09\x09\x09pos := pos + 1.\x0a\x09\x09\x09\x09\x09value := value + step ]]\x0a\x09\x09ifFalse: [ [ value <= stop ] whileTrue: [\x0a\x09\x09\x09\x09\x09array at: pos put: value.\x0a\x09\x09\x09\x09pos := pos + 1.\x0a\x09\x09\x09\x09\x09value := value + step ]].\x0a\x09^ array",
messageSends: ["new", "ifTrue:", "=", "error:", "ifTrue:ifFalse:", "<", "whileTrue:", ">=", "at:put:", "+", "<="],
referencedClasses: ["Array"]
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "to:by:do:",
protocol: 'enumerating',
fn: function (stop,step,aBlock){
var self=this;
var value;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
value=self;
$1=_st(step).__eq((0));
if(smalltalk.assert($1)){
self._error_("step must be non-zero");
};
$2=_st(step).__lt((0));
if(smalltalk.assert($2)){
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(value).__gt_eq(stop);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}))._whileTrue_((function(){
return smalltalk.withContext(function($ctx2) {
_st(aBlock)._value_(value);
$ctx2.sendIdx["value:"]=1;
value=_st(value).__plus(step);
$ctx2.sendIdx["+"]=1;
return value;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,4)})}));
$ctx1.sendIdx["whileTrue:"]=1;
} else {
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(value).__lt_eq(stop);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,6)})}))._whileTrue_((function(){
return smalltalk.withContext(function($ctx2) {
_st(aBlock)._value_(value);
value=_st(value).__plus(step);
return value;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,7)})}));
};
return self}, function($ctx1) {$ctx1.fill(self,"to:by:do:",{stop:stop,step:step,aBlock:aBlock,value:value},globals.Number)})},
args: ["stop", "step", "aBlock"],
source: "to: stop by: step do: aBlock\x0a\x09| value |\x0a\x09value := self.\x0a\x09step = 0 ifTrue: [ self error: 'step must be non-zero' ].\x0a\x09step < 0\x0a\x09\x09ifTrue: [ [ value >= stop ] whileTrue: [\x0a\x09\x09\x09\x09\x09aBlock value: value.\x0a\x09\x09\x09\x09\x09value := value + step ]]\x0a\x09\x09ifFalse: [ [ value <= stop ] whileTrue: [\x0a\x09\x09\x09\x09\x09aBlock value: value.\x0a\x09\x09\x09\x09\x09value := value + step ]]",
messageSends: ["ifTrue:", "=", "error:", "ifTrue:ifFalse:", "<", "whileTrue:", ">=", "value:", "+", "<="],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "to:do:",
protocol: 'enumerating',
fn: function (stop,aBlock){
var self=this;
var nextValue;
return smalltalk.withContext(function($ctx1) { 
nextValue=self;
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(nextValue).__lt_eq(stop);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileTrue_((function(){
return smalltalk.withContext(function($ctx2) {
_st(aBlock)._value_(nextValue);
nextValue=_st(nextValue).__plus((1));
return nextValue;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"to:do:",{stop:stop,aBlock:aBlock,nextValue:nextValue},globals.Number)})},
args: ["stop", "aBlock"],
source: "to: stop do: aBlock\x0a\x09\x22Evaluate aBlock for each number from self to aNumber.\x22\x0a\x09| nextValue |\x0a\x09nextValue := self.\x0a\x09[ nextValue <= stop ]\x0a\x09\x09whileTrue:\x0a\x09\x09\x09[ aBlock value: nextValue.\x0a\x09\x09\x09nextValue := nextValue + 1 ]",
messageSends: ["whileTrue:", "<=", "value:", "+"],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "truncated",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if(self >= 0) {
			return Math.floor(self);
		} else {
			return Math.floor(self * (-1)) * (-1);
		};
	;
return self}, function($ctx1) {$ctx1.fill(self,"truncated",{},globals.Number)})},
args: [],
source: "truncated\x0a\x09<\x0a\x09\x09if(self >>= 0) {\x0a\x09\x09\x09return Math.floor(self);\x0a\x09\x09} else {\x0a\x09\x09\x09return Math.floor(self * (-1)) * (-1);\x0a\x09\x09};\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Number);

smalltalk.addMethod(
smalltalk.method({
selector: "|",
protocol: 'converting',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self | aNumber;
return self}, function($ctx1) {$ctx1.fill(self,"|",{aNumber:aNumber},globals.Number)})},
args: ["aNumber"],
source: "| aNumber\x0a\x09<return self | aNumber>",
messageSends: [],
referencedClasses: []
}),
globals.Number);


smalltalk.addMethod(
smalltalk.method({
selector: "e",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.E;;
return self}, function($ctx1) {$ctx1.fill(self,"e",{},globals.Number.klass)})},
args: [],
source: "e\x0a\x09<return Math.E;>",
messageSends: [],
referencedClasses: []
}),
globals.Number.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "magnitude";
},
args: [],
source: "heliosClass\x0a\x09^ 'magnitude'",
messageSends: [],
referencedClasses: []
}),
globals.Number.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "pi",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.PI;
return self}, function($ctx1) {$ctx1.fill(self,"pi",{},globals.Number.klass)})},
args: [],
source: "pi\x0a\x09<return Math.PI>",
messageSends: [],
referencedClasses: []
}),
globals.Number.klass);


smalltalk.addClass('Point', globals.Object, ['x', 'y'], 'Kernel-Objects');
globals.Point.comment="I represent an x-y pair of numbers usually designating a geometric coordinate.\x0a\x0a## API\x0a\x0aInstances are traditionally created using the binary `#@` message to a number:\x0a\x0a\x09100@120\x0a\x0aPoints can then be arithmetically manipulated:\x0a\x0a\x09100@100 + (10@10)\x0a\x0a...or for example:\x0a\x0a\x09(100@100) * 2\x0a\x0a**NOTE:** Creating a point with a negative y-value will need a space after `@` in order to avoid a parsing error:\x0a\x0a\x09100@ -100 \x22but 100@-100 would not parse\x22";
smalltalk.addMethod(
smalltalk.method({
selector: "*",
protocol: 'arithmetic',
fn: function (aPoint){
var self=this;
function $Point(){return globals.Point||(typeof Point=="undefined"?nil:Point)}
return smalltalk.withContext(function($ctx1) { 
var $3,$5,$4,$2,$7,$6,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$5=_st(aPoint)._asPoint();
$ctx1.sendIdx["asPoint"]=1;
$4=_st($5)._x();
$2=_st($3).__star($4);
$ctx1.sendIdx["*"]=1;
$7=self._y();
$ctx1.sendIdx["y"]=1;
$6=_st($7).__star(_st(_st(aPoint)._asPoint())._y());
$1=_st($Point())._x_y_($2,$6);
return $1;
}, function($ctx1) {$ctx1.fill(self,"*",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "* aPoint\x0a\x09^ Point x: self x * aPoint asPoint x y: self y * aPoint asPoint y",
messageSends: ["x:y:", "*", "x", "asPoint", "y"],
referencedClasses: ["Point"]
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "+",
protocol: 'arithmetic',
fn: function (aPoint){
var self=this;
function $Point(){return globals.Point||(typeof Point=="undefined"?nil:Point)}
return smalltalk.withContext(function($ctx1) { 
var $3,$5,$4,$2,$7,$6,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$5=_st(aPoint)._asPoint();
$ctx1.sendIdx["asPoint"]=1;
$4=_st($5)._x();
$2=_st($3).__plus($4);
$ctx1.sendIdx["+"]=1;
$7=self._y();
$ctx1.sendIdx["y"]=1;
$6=_st($7).__plus(_st(_st(aPoint)._asPoint())._y());
$1=_st($Point())._x_y_($2,$6);
return $1;
}, function($ctx1) {$ctx1.fill(self,"+",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "+ aPoint\x0a\x09^ Point x: self x + aPoint asPoint x y: self y + aPoint asPoint y",
messageSends: ["x:y:", "+", "x", "asPoint", "y"],
referencedClasses: ["Point"]
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "-",
protocol: 'arithmetic',
fn: function (aPoint){
var self=this;
function $Point(){return globals.Point||(typeof Point=="undefined"?nil:Point)}
return smalltalk.withContext(function($ctx1) { 
var $3,$5,$4,$2,$7,$6,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$5=_st(aPoint)._asPoint();
$ctx1.sendIdx["asPoint"]=1;
$4=_st($5)._x();
$2=_st($3).__minus($4);
$ctx1.sendIdx["-"]=1;
$7=self._y();
$ctx1.sendIdx["y"]=1;
$6=_st($7).__minus(_st(_st(aPoint)._asPoint())._y());
$1=_st($Point())._x_y_($2,$6);
return $1;
}, function($ctx1) {$ctx1.fill(self,"-",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "- aPoint\x0a\x09^ Point x: self x - aPoint asPoint x y: self y - aPoint asPoint y",
messageSends: ["x:y:", "-", "x", "asPoint", "y"],
referencedClasses: ["Point"]
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "/",
protocol: 'arithmetic',
fn: function (aPoint){
var self=this;
function $Point(){return globals.Point||(typeof Point=="undefined"?nil:Point)}
return smalltalk.withContext(function($ctx1) { 
var $3,$5,$4,$2,$7,$6,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$5=_st(aPoint)._asPoint();
$ctx1.sendIdx["asPoint"]=1;
$4=_st($5)._x();
$2=_st($3).__slash($4);
$ctx1.sendIdx["/"]=1;
$7=self._y();
$ctx1.sendIdx["y"]=1;
$6=_st($7).__slash(_st(_st(aPoint)._asPoint())._y());
$1=_st($Point())._x_y_($2,$6);
return $1;
}, function($ctx1) {$ctx1.fill(self,"/",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "/ aPoint\x0a\x09^ Point x: self x / aPoint asPoint x y: self y / aPoint asPoint y",
messageSends: ["x:y:", "/", "x", "asPoint", "y"],
referencedClasses: ["Point"]
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "<",
protocol: 'comparing',
fn: function (aPoint){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$4,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$2=_st($3).__lt(_st(aPoint)._x());
$ctx1.sendIdx["<"]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=self._y();
$ctx2.sendIdx["y"]=1;
return _st($4).__lt(_st(aPoint)._y());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"<",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "< aPoint\x0a\x09^ self x < aPoint x and: [\x0a\x09\x09self y < aPoint y ]",
messageSends: ["and:", "<", "x", "y"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "<=",
protocol: 'comparing',
fn: function (aPoint){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$4,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$2=_st($3).__lt_eq(_st(aPoint)._x());
$ctx1.sendIdx["<="]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=self._y();
$ctx2.sendIdx["y"]=1;
return _st($4).__lt_eq(_st(aPoint)._y());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"<=",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "<= aPoint\x0a\x09^ self x <= aPoint x and: [\x0a\x09\x09self y <= aPoint y ]",
messageSends: ["and:", "<=", "x", "y"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (aPoint){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$5,$4,$7,$6,$1;
$3=_st(aPoint)._class();
$ctx1.sendIdx["class"]=1;
$2=_st($3).__eq(self._class());
$ctx1.sendIdx["="]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$5=_st(aPoint)._x();
$ctx2.sendIdx["x"]=1;
$4=_st($5).__eq(self._x());
$ctx2.sendIdx["="]=2;
$7=_st(aPoint)._y();
$ctx2.sendIdx["y"]=1;
$6=_st($7).__eq(self._y());
return _st($4).__and($6);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"=",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "= aPoint\x0a\x09^ aPoint class = self class and: [\x0a\x09\x09(aPoint x = self x) & (aPoint y = self y) ]",
messageSends: ["and:", "=", "class", "&", "x", "y"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: ">",
protocol: 'comparing',
fn: function (aPoint){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$4,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$2=_st($3).__gt(_st(aPoint)._x());
$ctx1.sendIdx[">"]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=self._y();
$ctx2.sendIdx["y"]=1;
return _st($4).__gt(_st(aPoint)._y());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,">",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: "> aPoint\x0a\x09^ self x > aPoint x and: [\x0a\x09\x09self y > aPoint y ]",
messageSends: ["and:", ">", "x", "y"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: ">=",
protocol: 'comparing',
fn: function (aPoint){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$4,$1;
$3=self._x();
$ctx1.sendIdx["x"]=1;
$2=_st($3).__gt_eq(_st(aPoint)._x());
$ctx1.sendIdx[">="]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=self._y();
$ctx2.sendIdx["y"]=1;
return _st($4).__gt_eq(_st(aPoint)._y());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,">=",{aPoint:aPoint},globals.Point)})},
args: ["aPoint"],
source: ">= aPoint\x0a\x09^ self x >= aPoint x and: [\x0a\x09\x09self y >= aPoint y ]",
messageSends: ["and:", ">=", "x", "y"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "asPoint",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asPoint\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "dist:",
protocol: 'transforming',
fn: function (aPoint){
var self=this;
var dx,dy;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1;
dx=_st(_st(aPoint)._x()).__minus(self["@x"]);
$ctx1.sendIdx["-"]=1;
dy=_st(_st(aPoint)._y()).__minus(self["@y"]);
$3=_st(dx).__star(dx);
$ctx1.sendIdx["*"]=1;
$2=_st($3).__plus(_st(dy).__star(dy));
$1=_st($2)._sqrt();
return $1;
}, function($ctx1) {$ctx1.fill(self,"dist:",{aPoint:aPoint,dx:dx,dy:dy},globals.Point)})},
args: ["aPoint"],
source: "dist: aPoint \x0a\x09\x22Answer the distance between aPoint and the receiver.\x22\x0a\x09| dx dy |\x0a\x09dx := aPoint x - x.\x0a\x09dy := aPoint y - y.\x0a\x09^ (dx * dx + (dy * dy)) sqrt",
messageSends: ["-", "x", "y", "sqrt", "+", "*"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(self["@x"])._printOn_(aStream);
$ctx1.sendIdx["printOn:"]=1;
_st(aStream)._nextPutAll_("@");
$1=_st(_st(self["@y"])._notNil())._and_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(self["@y"])._negative();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
if(smalltalk.assert($1)){
_st(aStream)._space();
};
_st(self["@y"])._printOn_(aStream);
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Point)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09\x22Print receiver in classic x@y notation.\x22\x0a\x0a\x09x printOn: aStream.\x0a\x09\x0a\x09aStream nextPutAll: '@'.\x0a\x09(y notNil and: [ y negative ]) ifTrue: [\x0a\x09\x09\x09\x22Avoid ambiguous @- construct\x22\x0a\x09\x09\x09aStream space ].\x0a\x09\x0a\x09y printOn: aStream",
messageSends: ["printOn:", "nextPutAll:", "ifTrue:", "and:", "notNil", "negative", "space"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "translateBy:",
protocol: 'transforming',
fn: function (delta){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(_st(delta)._x()).__plus(self["@x"]);
$ctx1.sendIdx["+"]=1;
$1=_st($2).__at(_st(_st(delta)._y()).__plus(self["@y"]));
return $1;
}, function($ctx1) {$ctx1.fill(self,"translateBy:",{delta:delta},globals.Point)})},
args: ["delta"],
source: "translateBy: delta\x0a\x09\x22Answer a Point translated by delta (an instance of Point).\x22\x0a\x09^ (delta x + x) @ (delta y + y)",
messageSends: ["@", "+", "x", "y"],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "x",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@x"];
return $1;
},
args: [],
source: "x\x0a\x09^ x",
messageSends: [],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "x:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
self["@x"]=aNumber;
return self},
args: ["aNumber"],
source: "x: aNumber\x0a\x09x := aNumber",
messageSends: [],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "y",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@y"];
return $1;
},
args: [],
source: "y\x0a\x09^ y",
messageSends: [],
referencedClasses: []
}),
globals.Point);

smalltalk.addMethod(
smalltalk.method({
selector: "y:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
self["@y"]=aNumber;
return self},
args: ["aNumber"],
source: "y: aNumber\x0a\x09y := aNumber",
messageSends: [],
referencedClasses: []
}),
globals.Point);


smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "magnitude";
},
args: [],
source: "heliosClass\x0a\x09^ 'magnitude'",
messageSends: [],
referencedClasses: []
}),
globals.Point.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "x:y:",
protocol: 'instance creation',
fn: function (aNumber,anotherNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._x_(aNumber);
_st($2)._y_(anotherNumber);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"x:y:",{aNumber:aNumber,anotherNumber:anotherNumber},globals.Point.klass)})},
args: ["aNumber", "anotherNumber"],
source: "x: aNumber y: anotherNumber\x0a\x09^ self new\x0a\x09\x09x: aNumber;\x0a\x09\x09y: anotherNumber;\x0a\x09\x09yourself",
messageSends: ["x:", "new", "y:", "yourself"],
referencedClasses: []
}),
globals.Point.klass);


smalltalk.addClass('Random', globals.Object, [], 'Kernel-Objects');
globals.Random.comment="I an used to generate a random number and I am implemented as a trivial wrapper around javascript `Math.random()`.\x0a\x0a## API\x0a\x0aThe typical use case it to use the `#next` method like the following:\x0a\x0a\x09Random new next\x0a\x0aThis will return a float x where x < 1 and x > 0. If you want a random integer from 1 to 10 you can use `#atRandom`\x0a\x0a\x0910 atRandom\x0a\x0aA random number in a specific interval can be obtained with the following:\x0a\x0a\x09(3 to: 7) atRandom\x0a\x0aBe aware that `#to:` does not create an Interval as in other Smalltalk implementations but in fact an `Array` of numbers, so it's better to use:\x0a\x0a\x095 atRandom + 2\x0a\x0aSince `#atRandom` is implemented in `SequencableCollection` you can easy pick an element at random:\x0a\x0a\x09#('a' 'b' 'c') atRandom\x0a\x0aAs well as letter from a `String`:\x0a\x0a\x09'abc' atRandom\x0a\x0aSince Amber does not have Characters this will return a `String` of length 1 like for example `'b'`.";
smalltalk.addMethod(
smalltalk.method({
selector: "next",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Math.random();
return self}, function($ctx1) {$ctx1.fill(self,"next",{},globals.Random)})},
args: [],
source: "next\x0a\x09<return Math.random()>",
messageSends: [],
referencedClasses: []
}),
globals.Random);

smalltalk.addMethod(
smalltalk.method({
selector: "next:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st((1)._to_(anInteger))._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._next();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"next:",{anInteger:anInteger},globals.Random)})},
args: ["anInteger"],
source: "next: anInteger\x0a\x09^ (1 to: anInteger) collect: [ :each | self next ]",
messageSends: ["collect:", "to:", "next"],
referencedClasses: []
}),
globals.Random);



smalltalk.addClass('UndefinedObject', globals.Object, [], 'Kernel-Objects');
globals.UndefinedObject.comment="I describe the behavior of my sole instance, `nil`. `nil` represents a prior value for variables that have not been initialized, or for results which are meaningless.\x0a\x0a`nil` is the Smalltalk equivalent of the `undefined` JavaScript object.\x0a\x0a__note:__ When sending messages to the `undefined` JavaScript object, it will be replaced by `nil`.";
smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
var $1;
$1=null;
return $1;
},
args: [],
source: "asJSON\x0a\x09^ null",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
return self;
},
args: [],
source: "deepCopy\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNil:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._ifNil_ifNotNil_(aBlock,(function(){
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNil:",{aBlock:aBlock},globals.UndefinedObject)})},
args: ["aBlock"],
source: "ifNil: aBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ self ifNil: aBlock ifNotNil: []",
messageSends: ["ifNil:ifNotNil:"],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNil:ifNotNil:",
protocol: 'testing',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aBlock)._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNil:ifNotNil:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.UndefinedObject)})},
args: ["aBlock", "anotherBlock"],
source: "ifNil: aBlock ifNotNil: anotherBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ aBlock value",
messageSends: ["value"],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNotNil:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return self;
},
args: ["aBlock"],
source: "ifNotNil: aBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNotNil:ifNil:",
protocol: 'testing',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(anotherBlock)._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNotNil:ifNil:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.UndefinedObject)})},
args: ["aBlock", "anotherBlock"],
source: "ifNotNil: aBlock ifNil: anotherBlock\x0a\x09\x22inlined in the Compiler\x22\x0a\x09^ anotherBlock value",
messageSends: ["value"],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isImmutable\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "isNil",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isNil\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "notNil",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "notNil\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutAll_("nil");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.UndefinedObject)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: 'nil'",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "shallowCopy",
protocol: 'copying',
fn: function (){
var self=this;
return self;
},
args: [],
source: "shallowCopy\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:",
protocol: 'class creation',
fn: function (aString,anotherString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._subclass_instanceVariableNames_package_(aString,anotherString,nil);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:",{aString:aString,anotherString:anotherString},globals.UndefinedObject)})},
args: ["aString", "anotherString"],
source: "subclass: aString instanceVariableNames: anotherString\x0a\x09^ self subclass: aString instanceVariableNames: anotherString package: nil",
messageSends: ["subclass:instanceVariableNames:package:"],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:category:",
protocol: 'class creation',
fn: function (aString,aString2,aString3){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=self._subclass_instanceVariableNames_package_(aString,aString2,aString3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:category:",{aString:aString,aString2:aString2,aString3:aString3},globals.UndefinedObject)})},
args: ["aString", "aString2", "aString3"],
source: "subclass: aString instanceVariableNames: aString2 category: aString3\x0a\x09\x22Kept for compatibility.\x22\x0a\x09self deprecatedAPI.\x0a\x09^ self subclass: aString instanceVariableNames: aString2 package: aString3",
messageSends: ["deprecatedAPI", "subclass:instanceVariableNames:package:"],
referencedClasses: []
}),
globals.UndefinedObject);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:package:",
protocol: 'class creation',
fn: function (aString,aString2,aString3){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($ClassBuilder())._new())._superclass_subclass_instanceVariableNames_package_(self,_st(aString)._asString(),aString2,aString3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:package:",{aString:aString,aString2:aString2,aString3:aString3},globals.UndefinedObject)})},
args: ["aString", "aString2", "aString3"],
source: "subclass: aString instanceVariableNames: aString2 package: aString3\x0a\x09^ ClassBuilder new\x0a\x09\x09superclass: self subclass: aString asString instanceVariableNames: aString2 package: aString3",
messageSends: ["superclass:subclass:instanceVariableNames:package:", "new", "asString"],
referencedClasses: ["ClassBuilder"]
}),
globals.UndefinedObject);


smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("You cannot create new instances of UndefinedObject. Use nil");
return self}, function($ctx1) {$ctx1.fill(self,"new",{},globals.UndefinedObject.klass)})},
args: [],
source: "new\x0a\x09\x09self error: 'You cannot create new instances of UndefinedObject. Use nil'",
messageSends: ["error:"],
referencedClasses: []
}),
globals.UndefinedObject.klass);

});

define("amber_core/Kernel-Classes", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Classes');
smalltalk.packages["Kernel-Classes"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('Behavior', globals.Object, [], 'Kernel-Classes');
globals.Behavior.comment="I am the superclass of all class objects.\x0a\x0aI define the protocol for creating instances of a class with `#basicNew` and `#new` (see `boot.js` for class constructors details).\x0a\x0aMy instances know about the subclass/superclass relationships between classes, contain the description that instances are created from,\x0aand hold the method dictionary that's associated with each class.\x0a\x0aI also provides methods for compiling methods, examining the method dictionary, and iterating over the class hierarchy.";
smalltalk.addMethod(
smalltalk.method({
selector: ">>",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._methodAt_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,">>",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: ">> aString\x0a\x09^ self methodAt: aString",
messageSends: ["methodAt:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "addCompiledMethod:",
protocol: 'compiling',
fn: function (aMethod){
var self=this;
var oldMethod,announcement;
function $MethodAdded(){return globals.MethodAdded||(typeof MethodAdded=="undefined"?nil:MethodAdded)}
function $MethodModified(){return globals.MethodModified||(typeof MethodModified=="undefined"?nil:MethodModified)}
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1,$4,$5,$6,$7,$8,$9,$10,$11;
oldMethod=_st(self._methodDictionary())._at_ifAbsent_(_st(aMethod)._selector(),(function(){
return nil;
}));
$2=self._protocols();
$3=_st(aMethod)._protocol();
$ctx1.sendIdx["protocol"]=1;
$1=_st($2)._includes_($3);
if(! smalltalk.assert($1)){
$4=self._organization();
$5=_st(aMethod)._protocol();
$ctx1.sendIdx["protocol"]=2;
_st($4)._addElement_($5);
};
self._basicAddCompiledMethod_(aMethod);
$6=oldMethod;
if(($receiver = $6) == nil || $receiver == null){
$6;
} else {
self._removeProtocolIfEmpty_(_st(oldMethod)._protocol());
};
$7=oldMethod;
if(($receiver = $7) == nil || $receiver == null){
$8=_st($MethodAdded())._new();
$ctx1.sendIdx["new"]=1;
_st($8)._method_(aMethod);
$ctx1.sendIdx["method:"]=1;
$9=_st($8)._yourself();
$ctx1.sendIdx["yourself"]=1;
announcement=$9;
} else {
$10=_st($MethodModified())._new();
_st($10)._oldMethod_(oldMethod);
_st($10)._method_(aMethod);
$11=_st($10)._yourself();
announcement=$11;
};
_st(_st($SystemAnnouncer())._current())._announce_(announcement);
return self}, function($ctx1) {$ctx1.fill(self,"addCompiledMethod:",{aMethod:aMethod,oldMethod:oldMethod,announcement:announcement},globals.Behavior)})},
args: ["aMethod"],
source: "addCompiledMethod: aMethod\x0a\x09| oldMethod announcement |\x0a\x09\x0a\x09oldMethod := self methodDictionary\x0a\x09\x09at: aMethod selector\x0a\x09\x09ifAbsent: [ nil ].\x0a\x09\x0a\x09(self protocols includes: aMethod protocol)\x0a\x09\x09ifFalse: [ self organization addElement: aMethod protocol ].\x0a\x0a\x09self basicAddCompiledMethod: aMethod.\x0a\x09\x0a\x09oldMethod ifNotNil: [\x0a\x09\x09self removeProtocolIfEmpty: oldMethod protocol ].\x0a\x09\x0a\x09announcement := oldMethod\x0a\x09\x09ifNil: [\x0a\x09\x09\x09MethodAdded new\x0a\x09\x09\x09\x09\x09method: aMethod;\x0a\x09\x09\x09\x09\x09yourself ]\x0a\x09\x09ifNotNil: [\x0a\x09\x09\x09MethodModified new\x0a\x09\x09\x09\x09\x09oldMethod: oldMethod;\x0a\x09\x09\x09\x09\x09method: aMethod;\x0a\x09\x09\x09\x09\x09yourself ].\x0a\x09\x09\x09\x09\x09\x0a\x09\x09\x09\x09\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09\x09\x09announce: announcement",
messageSends: ["at:ifAbsent:", "methodDictionary", "selector", "ifFalse:", "includes:", "protocols", "protocol", "addElement:", "organization", "basicAddCompiledMethod:", "ifNotNil:", "removeProtocolIfEmpty:", "ifNil:ifNotNil:", "method:", "new", "yourself", "oldMethod:", "announce:", "current"],
referencedClasses: ["MethodAdded", "MethodModified", "SystemAnnouncer"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "allInstanceVariableNames",
protocol: 'accessing',
fn: function (){
var self=this;
var result;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
result=_st(self._instanceVariableNames())._copy();
$1=self._superclass();
$ctx1.sendIdx["superclass"]=1;
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
_st(result)._addAll_(_st(self._superclass())._allInstanceVariableNames());
};
$2=result;
return $2;
}, function($ctx1) {$ctx1.fill(self,"allInstanceVariableNames",{result:result},globals.Behavior)})},
args: [],
source: "allInstanceVariableNames\x0a\x09| result |\x0a\x09result := self instanceVariableNames copy.\x0a\x09self superclass ifNotNil: [\x0a\x09\x09result addAll: self superclass allInstanceVariableNames ].\x0a\x09^ result",
messageSends: ["copy", "instanceVariableNames", "ifNotNil:", "superclass", "addAll:", "allInstanceVariableNames"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "allSelectors",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$4,$1;
$2=self._allSuperclasses();
$3=self._selectors();
$ctx1.sendIdx["selectors"]=1;
$1=_st($2)._inject_into_($3,(function(acc,each){
return smalltalk.withContext(function($ctx2) {
_st(acc)._addAll_(_st(each)._selectors());
$4=_st(acc)._yourself();
return $4;
}, function($ctx2) {$ctx2.fillBlock({acc:acc,each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"allSelectors",{},globals.Behavior)})},
args: [],
source: "allSelectors\x0a\x09^ self allSuperclasses\x0a\x09\x09inject: self selectors\x0a\x09\x09into: [ :acc :each | acc addAll: each selectors; yourself ]",
messageSends: ["inject:into:", "allSuperclasses", "selectors", "addAll:", "yourself"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "allSubclasses",
protocol: 'accessing',
fn: function (){
var self=this;
var subclasses,index;
return smalltalk.withContext(function($ctx1) { 
var $1;
subclasses=self._subclasses();
$ctx1.sendIdx["subclasses"]=1;
index=(1);
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(index).__gt(_st(subclasses)._size());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileFalse_((function(){
return smalltalk.withContext(function($ctx2) {
_st(subclasses)._addAll_(_st(_st(subclasses)._at_(index))._subclasses());
index=_st(index).__plus((1));
return index;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
$1=subclasses;
return $1;
}, function($ctx1) {$ctx1.fill(self,"allSubclasses",{subclasses:subclasses,index:index},globals.Behavior)})},
args: [],
source: "allSubclasses\x0a\x09\x22Answer an collection of the receiver's and the receiver's descendent's subclasses. \x22\x0a\x0a\x09| subclasses index |\x0a\x09\x0a\x09subclasses := self subclasses.\x0a\x09index := 1.\x0a\x09[ index > subclasses size ]\x0a\x09\x09whileFalse: [ subclasses addAll: (subclasses at: index) subclasses.\x0a\x09\x09\x09index := index + 1 ].\x0a\x0a\x09^ subclasses",
messageSends: ["subclasses", "whileFalse:", ">", "size", "addAll:", "at:", "+"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "allSubclassesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._allSubclasses())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"allSubclassesDo:",{aBlock:aBlock},globals.Behavior)})},
args: ["aBlock"],
source: "allSubclassesDo: aBlock\x0a\x09\x22Evaluate the argument, aBlock, for each of the receiver's subclasses.\x22\x0a\x0a\x09self allSubclasses do: [ :each |\x0a    \x09aBlock value: each ]",
messageSends: ["do:", "allSubclasses", "value:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "allSuperclasses",
protocol: 'accessing',
fn: function (){
var self=this;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$5,$4,$6,$3;
$1=self._superclass();
$ctx1.sendIdx["superclass"]=1;
if(($receiver = $1) == nil || $receiver == null){
$2=[];
return $2;
} else {
$1;
};
$5=self._superclass();
$ctx1.sendIdx["superclass"]=2;
$4=_st($OrderedCollection())._with_($5);
_st($4)._addAll_(_st(self._superclass())._allSuperclasses());
$6=_st($4)._yourself();
$3=$6;
return $3;
}, function($ctx1) {$ctx1.fill(self,"allSuperclasses",{},globals.Behavior)})},
args: [],
source: "allSuperclasses\x0a\x09\x0a\x09self superclass ifNil: [ ^ #() ].\x0a\x09\x0a\x09^ (OrderedCollection with: self superclass)\x0a\x09\x09addAll: self superclass allSuperclasses;\x0a\x09\x09yourself",
messageSends: ["ifNil:", "superclass", "addAll:", "with:", "allSuperclasses", "yourself"],
referencedClasses: ["OrderedCollection"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "basicAddCompiledMethod:",
protocol: 'private',
fn: function (aMethod){
var self=this;
return smalltalk.withContext(function($ctx1) { 
smalltalk.addMethod(aMethod, self);
return self}, function($ctx1) {$ctx1.fill(self,"basicAddCompiledMethod:",{aMethod:aMethod},globals.Behavior)})},
args: ["aMethod"],
source: "basicAddCompiledMethod: aMethod\x0a\x09<smalltalk.addMethod(aMethod, self)>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "basicNew",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return new self.fn();
return self}, function($ctx1) {$ctx1.fill(self,"basicNew",{},globals.Behavior)})},
args: [],
source: "basicNew\x0a\x09<return new self.fn()>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "basicRemoveCompiledMethod:",
protocol: 'private',
fn: function (aMethod){
var self=this;
return smalltalk.withContext(function($ctx1) { 
smalltalk.removeMethod(aMethod,self);
return self}, function($ctx1) {$ctx1.fill(self,"basicRemoveCompiledMethod:",{aMethod:aMethod},globals.Behavior)})},
args: ["aMethod"],
source: "basicRemoveCompiledMethod: aMethod\x0a\x09<smalltalk.removeMethod(aMethod,self)>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "canUnderstand:",
protocol: 'testing',
fn: function (aSelector){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1;
$1=_st(self._includesSelector_(_st(aSelector)._asString()))._or_((function(){
return smalltalk.withContext(function($ctx2) {
$3=self._superclass();
$ctx2.sendIdx["superclass"]=1;
$2=_st($3)._notNil();
return _st($2)._and_((function(){
return smalltalk.withContext(function($ctx3) {
return _st(self._superclass())._canUnderstand_(aSelector);
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"canUnderstand:",{aSelector:aSelector},globals.Behavior)})},
args: ["aSelector"],
source: "canUnderstand: aSelector\x0a\x09^ (self includesSelector: aSelector asString) or: [\x0a\x09\x09self superclass notNil and: [ self superclass canUnderstand: aSelector ]]",
messageSends: ["or:", "includesSelector:", "asString", "and:", "notNil", "superclass", "canUnderstand:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "comment",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._basicAt_("comment");
if(($receiver = $2) == nil || $receiver == null){
$1="";
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"comment",{},globals.Behavior)})},
args: [],
source: "comment\x0a\x09^ (self basicAt: 'comment') ifNil: [ '' ]",
messageSends: ["ifNil:", "basicAt:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "comment:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassCommentChanged(){return globals.ClassCommentChanged||(typeof ClassCommentChanged=="undefined"?nil:ClassCommentChanged)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._basicAt_put_("comment",aString);
$1=_st($ClassCommentChanged())._new();
_st($1)._theClass_(self);
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
return self}, function($ctx1) {$ctx1.fill(self,"comment:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "comment: aString\x0a\x09self basicAt: 'comment' put: aString.\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (ClassCommentChanged new\x0a\x09\x09\x09theClass: self;\x0a\x09\x09\x09yourself)",
messageSends: ["basicAt:put:", "announce:", "current", "theClass:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassCommentChanged"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "commentStamp",
protocol: 'accessing',
fn: function (){
var self=this;
function $ClassCommentReader(){return globals.ClassCommentReader||(typeof ClassCommentReader=="undefined"?nil:ClassCommentReader)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($ClassCommentReader())._new();
_st($2)._class_(self);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"commentStamp",{},globals.Behavior)})},
args: [],
source: "commentStamp\x0a\x09^ ClassCommentReader new\x0a\x09class: self;\x0a\x09yourself",
messageSends: ["class:", "new", "yourself"],
referencedClasses: ["ClassCommentReader"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "commentStamp:prior:",
protocol: 'accessing',
fn: function (aStamp,prior){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._commentStamp();
return $1;
}, function($ctx1) {$ctx1.fill(self,"commentStamp:prior:",{aStamp:aStamp,prior:prior},globals.Behavior)})},
args: ["aStamp", "prior"],
source: "commentStamp: aStamp prior: prior\x0a\x09\x09^ self commentStamp",
messageSends: ["commentStamp"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "compile:",
protocol: 'compiling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._compile_protocol_(aString,"");
return $1;
}, function($ctx1) {$ctx1.fill(self,"compile:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "compile: aString\x0a\x09^ self compile: aString protocol: ''",
messageSends: ["compile:protocol:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "compile:protocol:",
protocol: 'compiling',
fn: function (aString,anotherString){
var self=this;
function $Compiler(){return globals.Compiler||(typeof Compiler=="undefined"?nil:Compiler)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Compiler())._new())._install_forClass_protocol_(aString,self,anotherString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"compile:protocol:",{aString:aString,anotherString:anotherString},globals.Behavior)})},
args: ["aString", "anotherString"],
source: "compile: aString protocol: anotherString\x0a\x09^ Compiler new\x0a\x09\x09install: aString\x0a\x09\x09forClass: self\x0a\x09\x09protocol: anotherString",
messageSends: ["install:forClass:protocol:", "new"],
referencedClasses: ["Compiler"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "definition",
protocol: 'accessing',
fn: function (){
var self=this;
return "";
},
args: [],
source: "definition\x0a\x09^ ''",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "includesBehavior:",
protocol: 'testing',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self.__eq_eq(aClass))._or_((function(){
return smalltalk.withContext(function($ctx2) {
return self._inheritsFrom_(aClass);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"includesBehavior:",{aClass:aClass},globals.Behavior)})},
args: ["aClass"],
source: "includesBehavior: aClass\x0a\x09^ self == aClass or: [\x0a\x09\x09\x09self inheritsFrom: aClass ]",
messageSends: ["or:", "==", "inheritsFrom:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "includesSelector:",
protocol: 'testing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._methodDictionary())._includesKey_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"includesSelector:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "includesSelector: aString\x0a\x09^ self methodDictionary includesKey: aString",
messageSends: ["includesKey:", "methodDictionary"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "inheritsFrom:",
protocol: 'testing',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$4,$3,$2;
$1=self._superclass();
$ctx1.sendIdx["superclass"]=1;
if(($receiver = $1) == nil || $receiver == null){
return false;
} else {
$1;
};
$4=self._superclass();
$ctx1.sendIdx["superclass"]=2;
$3=_st(aClass).__eq_eq($4);
$2=_st($3)._or_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._superclass())._inheritsFrom_(aClass);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return $2;
}, function($ctx1) {$ctx1.fill(self,"inheritsFrom:",{aClass:aClass},globals.Behavior)})},
args: ["aClass"],
source: "inheritsFrom: aClass\x0a\x09self superclass ifNil: [ ^ false ].\x0a\x0a\x09^ aClass == self superclass or: [ \x0a\x09\x09self superclass inheritsFrom: aClass ]",
messageSends: ["ifNil:", "superclass", "or:", "==", "inheritsFrom:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "instanceVariableNames",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.iVarNames;
return self}, function($ctx1) {$ctx1.fill(self,"instanceVariableNames",{},globals.Behavior)})},
args: [],
source: "instanceVariableNames\x0a\x09<return self.iVarNames>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "isBehavior",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isBehavior\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "javascriptConstructor",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.fn;
return self}, function($ctx1) {$ctx1.fill(self,"javascriptConstructor",{},globals.Behavior)})},
args: [],
source: "javascriptConstructor\x0a\x09\x22Answer the JS constructor used to instantiate. See boot.js\x22\x0a\x09\x0a\x09<return self.fn>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "javascriptConstructor:",
protocol: 'accessing',
fn: function (aJavaScriptFunction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
smalltalk.setClassConstructor(self, aJavaScriptFunction);;
return self}, function($ctx1) {$ctx1.fill(self,"javascriptConstructor:",{aJavaScriptFunction:aJavaScriptFunction},globals.Behavior)})},
args: ["aJavaScriptFunction"],
source: "javascriptConstructor: aJavaScriptFunction\x0a\x09\x22Set the JS constructor used to instantiate.\x0a\x09See the JS counter-part in boot.js `smalltalk.setClassConstructor'\x22\x0a\x09\x0a\x09<smalltalk.setClassConstructor(self, aJavaScriptFunction);>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "lookupSelector:",
protocol: 'accessing',
fn: function (selector){
var self=this;
var lookupClass;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
var $early={};
try {
lookupClass=self;
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(lookupClass).__eq(nil);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileFalse_((function(){
return smalltalk.withContext(function($ctx2) {
$1=_st(lookupClass)._includesSelector_(selector);
if(smalltalk.assert($1)){
$2=_st(lookupClass)._methodAt_(selector);
throw $early=[$2];
};
lookupClass=_st(lookupClass)._superclass();
return lookupClass;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return nil;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"lookupSelector:",{selector:selector,lookupClass:lookupClass},globals.Behavior)})},
args: ["selector"],
source: "lookupSelector: selector\x0a\x09\x22Look up the given selector in my methodDictionary.\x0a\x09Return the corresponding method if found.\x0a\x09Otherwise chase the superclass chain and try again.\x0a\x09Return nil if no method is found.\x22\x0a\x09\x0a\x09| lookupClass |\x0a\x09\x0a\x09lookupClass := self.\x0a\x09[ lookupClass = nil ] whileFalse: [\x0a\x09\x09(lookupClass includesSelector: selector)\x0a\x09\x09\x09\x09ifTrue: [ ^ lookupClass methodAt: selector ].\x0a\x09\x09\x09lookupClass := lookupClass superclass ].\x0a\x09^ nil",
messageSends: ["whileFalse:", "=", "ifTrue:", "includesSelector:", "methodAt:", "superclass"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methodAt:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._methodDictionary())._at_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"methodAt:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "methodAt: aString\x0a\x09^ self methodDictionary at: aString",
messageSends: ["at:", "methodDictionary"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methodDictionary",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var dict = globals.HashedCollection._new();
	var methods = self.methods;
	Object.keys(methods).forEach(function(i) {
		if(methods[i].selector) {
			dict._at_put_(methods[i].selector, methods[i]);
		}
	});
	return dict;
return self}, function($ctx1) {$ctx1.fill(self,"methodDictionary",{},globals.Behavior)})},
args: [],
source: "methodDictionary\x0a\x09<var dict = globals.HashedCollection._new();\x0a\x09var methods = self.methods;\x0a\x09Object.keys(methods).forEach(function(i) {\x0a\x09\x09if(methods[i].selector) {\x0a\x09\x09\x09dict._at_put_(methods[i].selector, methods[i]);\x0a\x09\x09}\x0a\x09});\x0a\x09return dict>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methodTemplate",
protocol: 'accessing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $3,$4,$2,$7,$8,$6,$9,$5,$10,$1;
$1=_st($String())._streamContents_((function(stream){
return smalltalk.withContext(function($ctx2) {
_st(stream)._nextPutAll_("messageSelectorAndArgumentNames");
$ctx2.sendIdx["nextPutAll:"]=1;
$3=_st($String())._lf();
$ctx2.sendIdx["lf"]=1;
$4=_st($String())._tab();
$ctx2.sendIdx["tab"]=1;
$2=_st($3).__comma($4);
$ctx2.sendIdx[","]=1;
_st(stream)._nextPutAll_($2);
$ctx2.sendIdx["nextPutAll:"]=2;
_st(stream)._nextPutAll_("\x22comment stating purpose of message\x22");
$ctx2.sendIdx["nextPutAll:"]=3;
$7=_st($String())._lf();
$ctx2.sendIdx["lf"]=2;
$8=_st($String())._lf();
$ctx2.sendIdx["lf"]=3;
$6=_st($7).__comma($8);
$ctx2.sendIdx[","]=3;
$9=_st($String())._tab();
$ctx2.sendIdx["tab"]=2;
$5=_st($6).__comma($9);
$ctx2.sendIdx[","]=2;
_st(stream)._nextPutAll_($5);
$ctx2.sendIdx["nextPutAll:"]=4;
_st(stream)._nextPutAll_("| temporary variable names |");
$ctx2.sendIdx["nextPutAll:"]=5;
_st(stream)._nextPutAll_(_st(_st($String())._lf()).__comma(_st($String())._tab()));
$ctx2.sendIdx["nextPutAll:"]=6;
$10=_st(stream)._nextPutAll_("statements");
return $10;
}, function($ctx2) {$ctx2.fillBlock({stream:stream},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"methodTemplate",{},globals.Behavior)})},
args: [],
source: "methodTemplate\x0a\x09^ String streamContents: [ :stream |\x0a\x09\x09stream \x0a\x09\x09\x09nextPutAll: 'messageSelectorAndArgumentNames';\x0a\x09\x09\x09nextPutAll: String lf, String tab;\x0a\x09\x09\x09nextPutAll: '\x22comment stating purpose of message\x22';\x0a\x09\x09\x09nextPutAll: String lf, String lf, String tab;\x0a\x09\x09\x09nextPutAll: '| temporary variable names |';\x0a\x09\x09\x09nextPutAll: String lf, String tab;\x0a\x09\x09\x09nextPutAll: 'statements' ]",
messageSends: ["streamContents:", "nextPutAll:", ",", "lf", "tab"],
referencedClasses: ["String"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methods",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._methodDictionary())._values();
return $1;
}, function($ctx1) {$ctx1.fill(self,"methods",{},globals.Behavior)})},
args: [],
source: "methods\x0a\x09^ self methodDictionary values",
messageSends: ["values", "methodDictionary"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methodsFor:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $ClassCategoryReader(){return globals.ClassCategoryReader||(typeof ClassCategoryReader=="undefined"?nil:ClassCategoryReader)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($ClassCategoryReader())._new();
_st($2)._class_category_(self,aString);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"methodsFor:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "methodsFor: aString\x0a\x09^ ClassCategoryReader new\x0a\x09\x09class: self category: aString;\x0a\x09\x09yourself",
messageSends: ["class:category:", "new", "yourself"],
referencedClasses: ["ClassCategoryReader"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methodsFor:stamp:",
protocol: 'accessing',
fn: function (aString,aStamp){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._methodsFor_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"methodsFor:stamp:",{aString:aString,aStamp:aStamp},globals.Behavior)})},
args: ["aString", "aStamp"],
source: "methodsFor: aString stamp: aStamp\x0a\x09\x22Added for compatibility, right now ignores stamp.\x22\x0a\x09^ self methodsFor: aString",
messageSends: ["methodsFor:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "methodsInProtocol:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._methods())._select_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(each)._protocol()).__eq(aString);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"methodsInProtocol:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "methodsInProtocol: aString\x0a\x09^ self methods select: [ :each | each protocol = aString ]",
messageSends: ["select:", "methods", "=", "protocol"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "name",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.className || nil;
return self}, function($ctx1) {$ctx1.fill(self,"name",{},globals.Behavior)})},
args: [],
source: "name\x0a\x09<return self.className || nil>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._basicNew())._initialize();
return $1;
}, function($ctx1) {$ctx1.fill(self,"new",{},globals.Behavior)})},
args: [],
source: "new\x0a\x09^ self basicNew initialize",
messageSends: ["initialize", "basicNew"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "organization",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("organization");
return $1;
}, function($ctx1) {$ctx1.fill(self,"organization",{},globals.Behavior)})},
args: [],
source: "organization\x0a\x09^ self basicAt: 'organization'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "ownMethods",
protocol: 'accessing',
fn: function (){
var self=this;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$1=_st(_st(self._ownProtocols())._inject_into_(_st($OrderedCollection())._new(),(function(acc,each){
return smalltalk.withContext(function($ctx2) {
return _st(acc).__comma(self._methodsInProtocol_(each));
}, function($ctx2) {$ctx2.fillBlock({acc:acc,each:each},$ctx1,1)})})))._sorted_((function(a,b){
return smalltalk.withContext(function($ctx2) {
$2=_st(a)._selector();
$ctx2.sendIdx["selector"]=1;
return _st($2).__lt_eq(_st(b)._selector());
}, function($ctx2) {$ctx2.fillBlock({a:a,b:b},$ctx1,2)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"ownMethods",{},globals.Behavior)})},
args: [],
source: "ownMethods\x0a\x09\x22Answer the methods of the receiver that are not package extensions\x22\x0a\x0a\x09^ (self ownProtocols \x0a\x09\x09inject: OrderedCollection new\x0a\x09\x09into: [ :acc :each | acc, (self methodsInProtocol: each) ])\x0a\x09\x09\x09sorted: [ :a :b | a selector <= b selector ]",
messageSends: ["sorted:", "inject:into:", "ownProtocols", "new", ",", "methodsInProtocol:", "<=", "selector"],
referencedClasses: ["OrderedCollection"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "ownProtocols",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._protocols())._reject_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._match_("^\x5c*");
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"ownProtocols",{},globals.Behavior)})},
args: [],
source: "ownProtocols\x0a\x09\x22Answer the protocols of the receiver that are not package extensions\x22\x0a\x0a\x09^ self protocols reject: [ :each |\x0a\x09\x09each match: '^\x5c*' ]",
messageSends: ["reject:", "protocols", "match:"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "protocols",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._organization())._elements())._sorted();
return $1;
}, function($ctx1) {$ctx1.fill(self,"protocols",{},globals.Behavior)})},
args: [],
source: "protocols\x0a\x09^ self organization elements sorted",
messageSends: ["sorted", "elements", "organization"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "protocolsDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var methodsByProtocol;
function $HashedCollection(){return globals.HashedCollection||(typeof HashedCollection=="undefined"?nil:HashedCollection)}
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
methodsByProtocol=_st($HashedCollection())._new();
$ctx1.sendIdx["new"]=1;
_st(self._methodDictionary())._valuesDo_((function(m){
return smalltalk.withContext(function($ctx2) {
return _st(_st(methodsByProtocol)._at_ifAbsentPut_(_st(m)._protocol(),(function(){
return smalltalk.withContext(function($ctx3) {
return _st($Array())._new();
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})})))._add_(m);
}, function($ctx2) {$ctx2.fillBlock({m:m},$ctx1,1)})}));
_st(self._protocols())._do_((function(protocol){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_value_(protocol,_st(methodsByProtocol)._at_(protocol));
}, function($ctx2) {$ctx2.fillBlock({protocol:protocol},$ctx1,3)})}));
return self}, function($ctx1) {$ctx1.fill(self,"protocolsDo:",{aBlock:aBlock,methodsByProtocol:methodsByProtocol},globals.Behavior)})},
args: ["aBlock"],
source: "protocolsDo: aBlock\x0a\x09\x22Execute aBlock for each method protocol with\x0a\x09its collection of methods in the sort order of protocol name.\x22\x0a\x0a\x09| methodsByProtocol |\x0a\x09methodsByProtocol := HashedCollection new.\x0a\x09self methodDictionary valuesDo: [ :m |\x0a\x09\x09(methodsByProtocol at: m protocol ifAbsentPut: [ Array new ])\x0a\x09\x09\x09add: m ].\x0a\x09self protocols do: [ :protocol |\x0a\x09\x09aBlock value: protocol value: (methodsByProtocol at: protocol) ]",
messageSends: ["new", "valuesDo:", "methodDictionary", "add:", "at:ifAbsentPut:", "protocol", "do:", "protocols", "value:value:", "at:"],
referencedClasses: ["HashedCollection", "Array"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "prototype",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.fn.prototype;
return self}, function($ctx1) {$ctx1.fill(self,"prototype",{},globals.Behavior)})},
args: [],
source: "prototype\x0a\x09<return self.fn.prototype>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "recompile",
protocol: 'compiling',
fn: function (){
var self=this;
function $Compiler(){return globals.Compiler||(typeof Compiler=="undefined"?nil:Compiler)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Compiler())._new())._recompile_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"recompile",{},globals.Behavior)})},
args: [],
source: "recompile\x0a\x09^ Compiler new recompile: self",
messageSends: ["recompile:", "new"],
referencedClasses: ["Compiler"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "removeCompiledMethod:",
protocol: 'compiling',
fn: function (aMethod){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $MethodRemoved(){return globals.MethodRemoved||(typeof MethodRemoved=="undefined"?nil:MethodRemoved)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._basicRemoveCompiledMethod_(aMethod);
self._removeProtocolIfEmpty_(_st(aMethod)._protocol());
$1=_st($MethodRemoved())._new();
_st($1)._method_(aMethod);
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
return self}, function($ctx1) {$ctx1.fill(self,"removeCompiledMethod:",{aMethod:aMethod},globals.Behavior)})},
args: ["aMethod"],
source: "removeCompiledMethod: aMethod\x0a\x09self basicRemoveCompiledMethod: aMethod.\x0a\x09\x0a\x09self removeProtocolIfEmpty: aMethod protocol.\x0a\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (MethodRemoved new\x0a\x09\x09\x09method: aMethod;\x0a\x09\x09\x09yourself)",
messageSends: ["basicRemoveCompiledMethod:", "removeProtocolIfEmpty:", "protocol", "announce:", "current", "method:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "MethodRemoved"]
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "removeProtocolIfEmpty:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._methods())._detect_ifNone_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(each)._protocol()).__eq(aString);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._organization())._removeElement_(aString);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"removeProtocolIfEmpty:",{aString:aString},globals.Behavior)})},
args: ["aString"],
source: "removeProtocolIfEmpty: aString\x0a\x09self methods\x0a\x09\x09detect: [ :each | each protocol = aString ]\x0a\x09\x09ifNone: [ self organization removeElement: aString ]",
messageSends: ["detect:ifNone:", "methods", "=", "protocol", "removeElement:", "organization"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "selectors",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._methodDictionary())._keys();
return $1;
}, function($ctx1) {$ctx1.fill(self,"selectors",{},globals.Behavior)})},
args: [],
source: "selectors\x0a\x09^ self methodDictionary keys",
messageSends: ["keys", "methodDictionary"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "subclasses",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"subclasses",{},globals.Behavior)})},
args: [],
source: "subclasses\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "superclass",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.superclass || nil;
return self}, function($ctx1) {$ctx1.fill(self,"superclass",{},globals.Behavior)})},
args: [],
source: "superclass\x0a\x09<return self.superclass || nil>",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "theMetaClass",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._class();
return $1;
}, function($ctx1) {$ctx1.fill(self,"theMetaClass",{},globals.Behavior)})},
args: [],
source: "theMetaClass\x0a\x09^ self class",
messageSends: ["class"],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "theNonMetaClass",
protocol: 'accessing',
fn: function (){
var self=this;
return self;
},
args: [],
source: "theNonMetaClass\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Behavior);

smalltalk.addMethod(
smalltalk.method({
selector: "withAllSubclasses",
protocol: 'accessing',
fn: function (){
var self=this;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($Array())._with_(self);
_st($2)._addAll_(self._allSubclasses());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"withAllSubclasses",{},globals.Behavior)})},
args: [],
source: "withAllSubclasses\x0a\x09^ (Array with: self) addAll: self allSubclasses; yourself",
messageSends: ["addAll:", "with:", "allSubclasses", "yourself"],
referencedClasses: ["Array"]
}),
globals.Behavior);



smalltalk.addClass('Class', globals.Behavior, [], 'Kernel-Classes');
globals.Class.comment="I am __the__ class object.\x0a\x0aMy instances are the classes of the system.\x0aClass creation is done throught a `ClassBuilder` instance.";
smalltalk.addMethod(
smalltalk.method({
selector: "asJavascript",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1="globals.".__comma(self._name());
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJavascript",{},globals.Class)})},
args: [],
source: "asJavascript\x0a\x09^ 'globals.', self name",
messageSends: [",", "name"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "browse",
protocol: 'browsing',
fn: function (){
var self=this;
function $Finder(){return globals.Finder||(typeof Finder=="undefined"?nil:Finder)}
return smalltalk.withContext(function($ctx1) { 
_st($Finder())._findClass_(self);
return self}, function($ctx1) {$ctx1.fill(self,"browse",{},globals.Class)})},
args: [],
source: "browse\x0a\x09Finder findClass: self",
messageSends: ["findClass:"],
referencedClasses: ["Finder"]
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "category",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._package();
$ctx1.sendIdx["package"]=1;
if(($receiver = $2) == nil || $receiver == null){
$1="Unclassified";
} else {
$1=_st(self._package())._name();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"category",{},globals.Class)})},
args: [],
source: "category\x0a\x09^ self package ifNil: [ 'Unclassified' ] ifNotNil: [ self package name ]",
messageSends: ["ifNil:ifNotNil:", "package", "name"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "definition",
protocol: 'accessing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $3,$4,$2,$5,$6,$7,$1;
$1=_st($String())._streamContents_((function(stream){
return smalltalk.withContext(function($ctx2) {
_st(stream)._nextPutAll_(_st(self._superclass())._asString());
$ctx2.sendIdx["nextPutAll:"]=1;
_st(stream)._nextPutAll_(" subclass: #");
$ctx2.sendIdx["nextPutAll:"]=2;
_st(stream)._nextPutAll_(self._name());
$ctx2.sendIdx["nextPutAll:"]=3;
$3=_st($String())._lf();
$ctx2.sendIdx["lf"]=1;
$4=_st($String())._tab();
$ctx2.sendIdx["tab"]=1;
$2=_st($3).__comma($4);
$ctx2.sendIdx[","]=1;
_st(stream)._nextPutAll_($2);
$ctx2.sendIdx["nextPutAll:"]=4;
$5=_st(stream)._nextPutAll_("instanceVariableNames: '");
$ctx2.sendIdx["nextPutAll:"]=5;
$5;
_st(self._instanceVariableNames())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx3) {
return _st(stream)._nextPutAll_(each);
$ctx3.sendIdx["nextPutAll:"]=6;
}, function($ctx3) {$ctx3.fillBlock({each:each},$ctx2,2)})}),(function(){
return smalltalk.withContext(function($ctx3) {
return _st(stream)._nextPutAll_(" ");
$ctx3.sendIdx["nextPutAll:"]=7;
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,3)})}));
$6=_st("'".__comma(_st($String())._lf())).__comma(_st($String())._tab());
$ctx2.sendIdx[","]=2;
_st(stream)._nextPutAll_($6);
$ctx2.sendIdx["nextPutAll:"]=8;
_st(stream)._nextPutAll_("package: '");
$ctx2.sendIdx["nextPutAll:"]=9;
_st(stream)._nextPutAll_(self._category());
$ctx2.sendIdx["nextPutAll:"]=10;
$7=_st(stream)._nextPutAll_("'");
return $7;
}, function($ctx2) {$ctx2.fillBlock({stream:stream},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"definition",{},globals.Class)})},
args: [],
source: "definition\x0a\x09^ String streamContents: [ :stream |\x0a\x09\x09stream\x0a\x09\x09\x09nextPutAll: self superclass asString;\x0a\x09\x09\x09nextPutAll: ' subclass: #';\x0a\x09\x09\x09nextPutAll: self name;\x0a\x09\x09\x09nextPutAll: String lf, String tab;\x0a\x09\x09\x09nextPutAll: 'instanceVariableNames: '''.\x0a\x09\x09self instanceVariableNames\x0a\x09\x09\x09do: [ :each | stream nextPutAll: each ]\x0a\x09\x09\x09separatedBy: [ stream nextPutAll: ' ' ].\x0a\x09\x09stream\x0a\x09\x09\x09nextPutAll: '''', String lf, String tab;\x0a\x09\x09\x09nextPutAll: 'package: ''';\x0a\x09\x09\x09nextPutAll: self category;\x0a\x09\x09\x09nextPutAll: '''' ]",
messageSends: ["streamContents:", "nextPutAll:", "asString", "superclass", "name", ",", "lf", "tab", "do:separatedBy:", "instanceVariableNames", "category"],
referencedClasses: ["String"]
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "isClass",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isClass\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "package",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("pkg");
return $1;
}, function($ctx1) {$ctx1.fill(self,"package",{},globals.Class)})},
args: [],
source: "package\x0a\x09^ self basicAt: 'pkg'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "package:",
protocol: 'accessing',
fn: function (aPackage){
var self=this;
var oldPackage;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassMoved(){return globals.ClassMoved||(typeof ClassMoved=="undefined"?nil:ClassMoved)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3,$4,$5;
$2=self._package();
$ctx1.sendIdx["package"]=1;
$1=_st($2).__eq(aPackage);
if(smalltalk.assert($1)){
return self;
};
oldPackage=self._package();
self._basicAt_put_("pkg",aPackage);
$3=_st(oldPackage)._organization();
$ctx1.sendIdx["organization"]=1;
_st($3)._removeElement_(self);
_st(_st(aPackage)._organization())._addElement_(self);
$4=_st($ClassMoved())._new();
_st($4)._theClass_(self);
_st($4)._oldPackage_(oldPackage);
$5=_st($4)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($5);
return self}, function($ctx1) {$ctx1.fill(self,"package:",{aPackage:aPackage,oldPackage:oldPackage},globals.Class)})},
args: ["aPackage"],
source: "package: aPackage\x0a\x09| oldPackage |\x0a\x09\x0a\x09self package = aPackage ifTrue: [ ^ self ].\x0a\x09\x0a\x09oldPackage := self package.\x0a\x09\x0a\x09self basicAt: 'pkg' put: aPackage.\x0a\x09oldPackage organization removeElement: self.\x0a\x09aPackage organization addElement: self.\x0a\x0a\x09SystemAnnouncer current announce: (ClassMoved new\x0a\x09\x09theClass: self;\x0a\x09\x09oldPackage: oldPackage;\x0a\x09\x09yourself)",
messageSends: ["ifTrue:", "=", "package", "basicAt:put:", "removeElement:", "organization", "addElement:", "announce:", "current", "theClass:", "new", "oldPackage:", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassMoved"]
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutAll_(self._name());
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Class)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: self name",
messageSends: ["nextPutAll:", "name"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "rename:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
_st(_st($ClassBuilder())._new())._renameClass_to_(self,aString);
return self}, function($ctx1) {$ctx1.fill(self,"rename:",{aString:aString},globals.Class)})},
args: ["aString"],
source: "rename: aString\x0a\x09ClassBuilder new renameClass: self to: aString",
messageSends: ["renameClass:to:", "new"],
referencedClasses: ["ClassBuilder"]
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:",
protocol: 'class creation',
fn: function (aString,anotherString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._subclass_instanceVariableNames_package_(aString,anotherString,nil);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:",{aString:aString,anotherString:anotherString},globals.Class)})},
args: ["aString", "anotherString"],
source: "subclass: aString instanceVariableNames: anotherString\x0a\x09\x22Kept for compatibility.\x22\x0a\x09^ self subclass: aString instanceVariableNames: anotherString package: nil",
messageSends: ["subclass:instanceVariableNames:package:"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:category:",
protocol: 'class creation',
fn: function (aString,aString2,aString3){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=self._subclass_instanceVariableNames_package_(aString,aString2,aString3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:category:",{aString:aString,aString2:aString2,aString3:aString3},globals.Class)})},
args: ["aString", "aString2", "aString3"],
source: "subclass: aString instanceVariableNames: aString2 category: aString3\x0a\x09\x22Kept for compatibility.\x22\x0a\x09self deprecatedAPI.\x0a\x09^ self subclass: aString instanceVariableNames: aString2 package: aString3",
messageSends: ["deprecatedAPI", "subclass:instanceVariableNames:package:"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:classVariableNames:poolDictionaries:category:",
protocol: 'class creation',
fn: function (aString,aString2,classVars,pools,aString3){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._subclass_instanceVariableNames_package_(aString,aString2,aString3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:classVariableNames:poolDictionaries:category:",{aString:aString,aString2:aString2,classVars:classVars,pools:pools,aString3:aString3},globals.Class)})},
args: ["aString", "aString2", "classVars", "pools", "aString3"],
source: "subclass: aString instanceVariableNames: aString2 classVariableNames: classVars poolDictionaries: pools category: aString3\x0a\x09\x22Just ignore class variables and pools. Added for compatibility.\x22\x0a\x09^ self subclass: aString instanceVariableNames: aString2 package: aString3",
messageSends: ["subclass:instanceVariableNames:package:"],
referencedClasses: []
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "subclass:instanceVariableNames:package:",
protocol: 'class creation',
fn: function (aString,aString2,aString3){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($ClassBuilder())._new())._superclass_subclass_instanceVariableNames_package_(self,_st(aString)._asString(),aString2,aString3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclass:instanceVariableNames:package:",{aString:aString,aString2:aString2,aString3:aString3},globals.Class)})},
args: ["aString", "aString2", "aString3"],
source: "subclass: aString instanceVariableNames: aString2 package: aString3\x0a\x09^ ClassBuilder new\x0a\x09\x09superclass: self subclass: aString asString instanceVariableNames: aString2 package: aString3",
messageSends: ["superclass:subclass:instanceVariableNames:package:", "new", "asString"],
referencedClasses: ["ClassBuilder"]
}),
globals.Class);

smalltalk.addMethod(
smalltalk.method({
selector: "subclasses",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.subclasses._copy();
return self}, function($ctx1) {$ctx1.fill(self,"subclasses",{},globals.Class)})},
args: [],
source: "subclasses\x0a\x09<return self.subclasses._copy()>",
messageSends: [],
referencedClasses: []
}),
globals.Class);



smalltalk.addClass('Metaclass', globals.Behavior, [], 'Kernel-Classes');
globals.Metaclass.comment="I am the root of the class hierarchy.\x0a\x0aMy instances are metaclasses, one for each real class, and have a single instance, which they hold onto: the class that they are the metaclass of.";
smalltalk.addMethod(
smalltalk.method({
selector: "asJavascript",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st("globals.".__comma(_st(self._instanceClass())._name())).__comma(".klass");
$ctx1.sendIdx[","]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJavascript",{},globals.Metaclass)})},
args: [],
source: "asJavascript\x0a\x09^ 'globals.', self instanceClass name, '.klass'",
messageSends: [",", "name", "instanceClass"],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "definition",
protocol: 'accessing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$1=_st($String())._streamContents_((function(stream){
return smalltalk.withContext(function($ctx2) {
_st(stream)._nextPutAll_(self._asString());
$ctx2.sendIdx["nextPutAll:"]=1;
$2=_st(stream)._nextPutAll_(" instanceVariableNames: '");
$ctx2.sendIdx["nextPutAll:"]=2;
$2;
_st(self._instanceVariableNames())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx3) {
return _st(stream)._nextPutAll_(each);
$ctx3.sendIdx["nextPutAll:"]=3;
}, function($ctx3) {$ctx3.fillBlock({each:each},$ctx2,2)})}),(function(){
return smalltalk.withContext(function($ctx3) {
return _st(stream)._nextPutAll_(" ");
$ctx3.sendIdx["nextPutAll:"]=4;
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,3)})}));
return _st(stream)._nextPutAll_("'");
}, function($ctx2) {$ctx2.fillBlock({stream:stream},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"definition",{},globals.Metaclass)})},
args: [],
source: "definition\x0a\x09^ String streamContents: [ :stream |\x0a\x09\x09stream\x0a\x09\x09\x09nextPutAll: self asString;\x0a\x09\x09\x09nextPutAll: ' instanceVariableNames: '''.\x0a\x09\x09self instanceVariableNames\x0a\x09\x09\x09do: [ :each | stream nextPutAll: each ]\x0a\x09\x09\x09separatedBy: [ stream nextPutAll: ' ' ].\x0a\x09\x09stream nextPutAll: '''' ]",
messageSends: ["streamContents:", "nextPutAll:", "asString", "do:separatedBy:", "instanceVariableNames"],
referencedClasses: ["String"]
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "instanceClass",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.instanceClass;
return self}, function($ctx1) {$ctx1.fill(self,"instanceClass",{},globals.Metaclass)})},
args: [],
source: "instanceClass\x0a\x09<return self.instanceClass>",
messageSends: [],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "instanceVariableNames:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
_st(_st($ClassBuilder())._new())._class_instanceVariableNames_(self,aCollection);
return self}, function($ctx1) {$ctx1.fill(self,"instanceVariableNames:",{aCollection:aCollection},globals.Metaclass)})},
args: ["aCollection"],
source: "instanceVariableNames: aCollection\x0a\x09ClassBuilder new\x0a\x09\x09class: self instanceVariableNames: aCollection",
messageSends: ["class:instanceVariableNames:", "new"],
referencedClasses: ["ClassBuilder"]
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "isMetaclass",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isMetaclass\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "package",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._instanceClass())._package();
return $1;
}, function($ctx1) {$ctx1.fill(self,"package",{},globals.Metaclass)})},
args: [],
source: "package\x0a\x09^ self instanceClass package",
messageSends: ["package", "instanceClass"],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(aStream)._nextPutAll_(_st(self._instanceClass())._name());
$ctx1.sendIdx["nextPutAll:"]=1;
$1=_st(aStream)._nextPutAll_(" class");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Metaclass)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream\x0a\x09\x09nextPutAll: self instanceClass name;\x0a\x09\x09nextPutAll: ' class'",
messageSends: ["nextPutAll:", "name", "instanceClass"],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "subclasses",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(_st(self._instanceClass())._subclasses())._select_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(each)._isMetaclass())._not();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})})))._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._theMetaClass();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"subclasses",{},globals.Metaclass)})},
args: [],
source: "subclasses\x0a\x09^ (self instanceClass subclasses \x0a\x09\x09select: [ :each | each isMetaclass not ])\x0a\x09\x09collect: [ :each | each theMetaClass ]",
messageSends: ["collect:", "select:", "subclasses", "instanceClass", "not", "isMetaclass", "theMetaClass"],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "theMetaClass",
protocol: 'accessing',
fn: function (){
var self=this;
return self;
},
args: [],
source: "theMetaClass\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Metaclass);

smalltalk.addMethod(
smalltalk.method({
selector: "theNonMetaClass",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._instanceClass();
return $1;
}, function($ctx1) {$ctx1.fill(self,"theNonMetaClass",{},globals.Metaclass)})},
args: [],
source: "theNonMetaClass\x0a\x09^ self instanceClass",
messageSends: ["instanceClass"],
referencedClasses: []
}),
globals.Metaclass);



smalltalk.addClass('ClassBuilder', globals.Object, [], 'Kernel-Classes');
globals.ClassBuilder.comment="I am responsible for compiling new classes or modifying existing classes in the system.\x0a\x0aRather than using me directly to compile a class, use `Class >> subclass:instanceVariableNames:package:`.";
smalltalk.addMethod(
smalltalk.method({
selector: "addSubclassOf:named:instanceVariableNames:package:",
protocol: 'class definition',
fn: function (aClass,className,aCollection,packageName){
var self=this;
var theClass,thePackage;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
function $Package(){return globals.Package||(typeof Package=="undefined"?nil:Package)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4;
theClass=_st(_st($Smalltalk())._globals())._at_(className);
thePackage=_st($Package())._named_(packageName);
$1=theClass;
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
_st(theClass)._package_(thePackage);
$2=_st(_st(theClass)._superclass()).__eq_eq(aClass);
if(! smalltalk.assert($2)){
$3=self._migrateClassNamed_superclass_instanceVariableNames_package_(className,aClass,aCollection,packageName);
return $3;
};
};
$4=self._basicAddSubclassOf_named_instanceVariableNames_package_(aClass,className,aCollection,packageName);
return $4;
}, function($ctx1) {$ctx1.fill(self,"addSubclassOf:named:instanceVariableNames:package:",{aClass:aClass,className:className,aCollection:aCollection,packageName:packageName,theClass:theClass,thePackage:thePackage},globals.ClassBuilder)})},
args: ["aClass", "className", "aCollection", "packageName"],
source: "addSubclassOf: aClass named: className instanceVariableNames: aCollection package: packageName\x0a\x09| theClass thePackage |\x0a\x09\x0a\x09theClass := Smalltalk globals at: className.\x0a\x09thePackage := Package named: packageName.\x0a\x09\x0a\x09theClass ifNotNil: [\x0a\x09\x09theClass package: thePackage.\x0a\x09\x09theClass superclass == aClass ifFalse: [\x0a\x09\x09\x09^ self\x0a\x09\x09\x09\x09migrateClassNamed: className\x0a\x09\x09\x09\x09superclass: aClass\x0a\x09\x09\x09\x09instanceVariableNames: aCollection\x0a\x09\x09\x09\x09package: packageName ] ].\x0a\x09\x09\x0a\x09^ self\x0a\x09\x09basicAddSubclassOf: aClass\x0a\x09\x09named: className\x0a\x09\x09instanceVariableNames: aCollection\x0a\x09\x09package: packageName",
messageSends: ["at:", "globals", "named:", "ifNotNil:", "package:", "ifFalse:", "==", "superclass", "migrateClassNamed:superclass:instanceVariableNames:package:", "basicAddSubclassOf:named:instanceVariableNames:package:"],
referencedClasses: ["Smalltalk", "Package"]
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "basicAddSubclassOf:named:instanceVariableNames:package:",
protocol: 'private',
fn: function (aClass,aString,aCollection,packageName){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		smalltalk.addClass(aString, aClass, aCollection, packageName);
		return globals[aString]
	;
return self}, function($ctx1) {$ctx1.fill(self,"basicAddSubclassOf:named:instanceVariableNames:package:",{aClass:aClass,aString:aString,aCollection:aCollection,packageName:packageName},globals.ClassBuilder)})},
args: ["aClass", "aString", "aCollection", "packageName"],
source: "basicAddSubclassOf: aClass named: aString instanceVariableNames: aCollection package: packageName\x0a\x09<\x0a\x09\x09smalltalk.addClass(aString, aClass, aCollection, packageName);\x0a\x09\x09return globals[aString]\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "basicClass:instanceVariableNames:",
protocol: 'private',
fn: function (aClass,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._basicClass_instanceVariables_(aClass,self._instanceVariableNamesFor_(aString));
return self}, function($ctx1) {$ctx1.fill(self,"basicClass:instanceVariableNames:",{aClass:aClass,aString:aString},globals.ClassBuilder)})},
args: ["aClass", "aString"],
source: "basicClass: aClass instanceVariableNames: aString\x0a\x09self basicClass: aClass instanceVariables: (self instanceVariableNamesFor: aString)",
messageSends: ["basicClass:instanceVariables:", "instanceVariableNamesFor:"],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "basicClass:instanceVariables:",
protocol: 'private',
fn: function (aClass,aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aClass)._isMetaclass();
if(! smalltalk.assert($1)){
self._error_(_st(_st(aClass)._name()).__comma(" is not a metaclass"));
};
_st(aClass)._basicAt_put_("iVarNames",aCollection);
return self}, function($ctx1) {$ctx1.fill(self,"basicClass:instanceVariables:",{aClass:aClass,aCollection:aCollection},globals.ClassBuilder)})},
args: ["aClass", "aCollection"],
source: "basicClass: aClass instanceVariables: aCollection\x0a\x0a\x09aClass isMetaclass ifFalse: [ self error: aClass name, ' is not a metaclass' ].\x0a\x09aClass basicAt: 'iVarNames' put: aCollection",
messageSends: ["ifFalse:", "isMetaclass", "error:", ",", "name", "basicAt:put:"],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "basicRemoveClass:",
protocol: 'private',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
smalltalk.removeClass(aClass);
return self}, function($ctx1) {$ctx1.fill(self,"basicRemoveClass:",{aClass:aClass},globals.ClassBuilder)})},
args: ["aClass"],
source: "basicRemoveClass: aClass\x0a\x09<smalltalk.removeClass(aClass)>",
messageSends: [],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "basicRenameClass:to:",
protocol: 'private',
fn: function (aClass,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		globals[aString] = aClass;
		delete globals[aClass.className];
		aClass.className = aString;
	;
return self}, function($ctx1) {$ctx1.fill(self,"basicRenameClass:to:",{aClass:aClass,aString:aString},globals.ClassBuilder)})},
args: ["aClass", "aString"],
source: "basicRenameClass: aClass to: aString\x0a\x09<\x0a\x09\x09globals[aString] = aClass;\x0a\x09\x09delete globals[aClass.className];\x0a\x09\x09aClass.className = aString;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "basicSwapClassNames:with:",
protocol: 'private',
fn: function (aClass,anotherClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var tmp = aClass.className;
		aClass.className = anotherClass.className;
		anotherClass.className = tmp;
	;
return self}, function($ctx1) {$ctx1.fill(self,"basicSwapClassNames:with:",{aClass:aClass,anotherClass:anotherClass},globals.ClassBuilder)})},
args: ["aClass", "anotherClass"],
source: "basicSwapClassNames: aClass with: anotherClass\x0a\x09<\x0a\x09\x09var tmp = aClass.className;\x0a\x09\x09aClass.className = anotherClass.className;\x0a\x09\x09anotherClass.className = tmp;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "class:instanceVariableNames:",
protocol: 'class definition',
fn: function (aClass,ivarNames){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassDefinitionChanged(){return globals.ClassDefinitionChanged||(typeof ClassDefinitionChanged=="undefined"?nil:ClassDefinitionChanged)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._basicClass_instanceVariableNames_(aClass,ivarNames);
self._setupClass_(aClass);
$1=_st($ClassDefinitionChanged())._new();
_st($1)._theClass_(aClass);
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
return self}, function($ctx1) {$ctx1.fill(self,"class:instanceVariableNames:",{aClass:aClass,ivarNames:ivarNames},globals.ClassBuilder)})},
args: ["aClass", "ivarNames"],
source: "class: aClass instanceVariableNames: ivarNames\x0a\x09self basicClass: aClass instanceVariableNames: ivarNames.\x0a\x09self setupClass: aClass.\x0a\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (ClassDefinitionChanged new\x0a\x09\x09\x09theClass: aClass;\x0a\x09\x09\x09yourself)",
messageSends: ["basicClass:instanceVariableNames:", "setupClass:", "announce:", "current", "theClass:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassDefinitionChanged"]
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "copyClass:named:",
protocol: 'copying',
fn: function (aClass,className){
var self=this;
var newClass;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassAdded(){return globals.ClassAdded||(typeof ClassAdded=="undefined"?nil:ClassAdded)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
newClass=self._addSubclassOf_named_instanceVariableNames_package_(_st(aClass)._superclass(),className,_st(aClass)._instanceVariableNames(),_st(_st(aClass)._package())._name());
self._copyClass_to_(aClass,newClass);
$1=_st($ClassAdded())._new();
_st($1)._theClass_(newClass);
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
$3=newClass;
return $3;
}, function($ctx1) {$ctx1.fill(self,"copyClass:named:",{aClass:aClass,className:className,newClass:newClass},globals.ClassBuilder)})},
args: ["aClass", "className"],
source: "copyClass: aClass named: className\x0a\x09| newClass |\x0a\x0a\x09newClass := self\x0a\x09\x09addSubclassOf: aClass superclass\x0a\x09\x09named: className\x0a\x09\x09instanceVariableNames: aClass instanceVariableNames\x0a\x09\x09package: aClass package name.\x0a\x0a\x09self copyClass: aClass to: newClass.\x0a\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (ClassAdded new\x0a\x09\x09\x09theClass: newClass;\x0a\x09\x09\x09yourself).\x0a\x09\x0a\x09^ newClass",
messageSends: ["addSubclassOf:named:instanceVariableNames:package:", "superclass", "instanceVariableNames", "name", "package", "copyClass:to:", "announce:", "current", "theClass:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassAdded"]
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "copyClass:to:",
protocol: 'copying',
fn: function (aClass,anotherClass){
var self=this;
function $Compiler(){return globals.Compiler||(typeof Compiler=="undefined"?nil:Compiler)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$5,$7,$6,$9,$8;
_st(anotherClass)._comment_(_st(aClass)._comment());
$1=_st(aClass)._methodDictionary();
$ctx1.sendIdx["methodDictionary"]=1;
_st($1)._valuesDo_((function(each){
return smalltalk.withContext(function($ctx2) {
$2=_st($Compiler())._new();
$ctx2.sendIdx["new"]=1;
$3=_st(each)._source();
$ctx2.sendIdx["source"]=1;
$4=_st(each)._protocol();
$ctx2.sendIdx["protocol"]=1;
return _st($2)._install_forClass_protocol_($3,anotherClass,$4);
$ctx2.sendIdx["install:forClass:protocol:"]=1;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$ctx1.sendIdx["valuesDo:"]=1;
$5=_st(anotherClass)._class();
$ctx1.sendIdx["class"]=1;
$7=_st(aClass)._class();
$ctx1.sendIdx["class"]=2;
$6=_st($7)._instanceVariableNames();
self._basicClass_instanceVariables_($5,$6);
$9=_st(aClass)._class();
$ctx1.sendIdx["class"]=3;
$8=_st($9)._methodDictionary();
_st($8)._valuesDo_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st($Compiler())._new())._install_forClass_protocol_(_st(each)._source(),_st(anotherClass)._class(),_st(each)._protocol());
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
self._setupClass_(anotherClass);
return self}, function($ctx1) {$ctx1.fill(self,"copyClass:to:",{aClass:aClass,anotherClass:anotherClass},globals.ClassBuilder)})},
args: ["aClass", "anotherClass"],
source: "copyClass: aClass to: anotherClass\x0a\x0a\x09anotherClass comment: aClass comment.\x0a\x0a\x09aClass methodDictionary valuesDo: [ :each |\x0a\x09\x09Compiler new install: each source forClass: anotherClass protocol: each protocol ].\x0a\x0a\x09self basicClass: anotherClass class instanceVariables: aClass class instanceVariableNames.\x0a\x0a\x09aClass class methodDictionary valuesDo: [ :each |\x0a\x09\x09Compiler new install: each source forClass: anotherClass class protocol: each protocol ].\x0a\x0a\x09self setupClass: anotherClass",
messageSends: ["comment:", "comment", "valuesDo:", "methodDictionary", "install:forClass:protocol:", "new", "source", "protocol", "basicClass:instanceVariables:", "class", "instanceVariableNames", "setupClass:"],
referencedClasses: ["Compiler"]
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "installMethod:forClass:protocol:",
protocol: 'method definition',
fn: function (aCompiledMethod,aBehavior,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aCompiledMethod)._protocol_(aString);
_st(aBehavior)._addCompiledMethod_(aCompiledMethod);
self._setupClass_(aBehavior);
return aCompiledMethod;
}, function($ctx1) {$ctx1.fill(self,"installMethod:forClass:protocol:",{aCompiledMethod:aCompiledMethod,aBehavior:aBehavior,aString:aString},globals.ClassBuilder)})},
args: ["aCompiledMethod", "aBehavior", "aString"],
source: "installMethod: aCompiledMethod forClass: aBehavior protocol: aString\x0a\x09aCompiledMethod protocol: aString.\x0a\x09aBehavior addCompiledMethod: aCompiledMethod.\x0a\x09self setupClass: aBehavior.\x0a\x09^ aCompiledMethod",
messageSends: ["protocol:", "addCompiledMethod:", "setupClass:"],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "instanceVariableNamesFor:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(aString)._tokenize_(" "))._reject_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._isEmpty();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"instanceVariableNamesFor:",{aString:aString},globals.ClassBuilder)})},
args: ["aString"],
source: "instanceVariableNamesFor: aString\x0a\x09^ (aString tokenize: ' ') reject: [ :each | each isEmpty ]",
messageSends: ["reject:", "tokenize:", "isEmpty"],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "migrateClass:superclass:",
protocol: 'class migration',
fn: function (aClass,anotherClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(aClass)._name();
$ctx1.sendIdx["name"]=1;
$1=self._migrateClassNamed_superclass_instanceVariableNames_package_($2,anotherClass,_st(aClass)._instanceVariableNames(),_st(_st(aClass)._package())._name());
return $1;
}, function($ctx1) {$ctx1.fill(self,"migrateClass:superclass:",{aClass:aClass,anotherClass:anotherClass},globals.ClassBuilder)})},
args: ["aClass", "anotherClass"],
source: "migrateClass: aClass superclass: anotherClass\x0a\x09^ self\x0a\x09\x09migrateClassNamed: aClass name\x0a\x09\x09superclass: anotherClass\x0a\x09\x09instanceVariableNames: aClass instanceVariableNames\x0a\x09\x09package: aClass package name",
messageSends: ["migrateClassNamed:superclass:instanceVariableNames:package:", "name", "instanceVariableNames", "package"],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "migrateClassNamed:superclass:instanceVariableNames:package:",
protocol: 'class migration',
fn: function (className,aClass,aCollection,packageName){
var self=this;
var oldClass,newClass,tmp;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
function $Error(){return globals.Error||(typeof Error=="undefined"?nil:Error)}
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassMigrated(){return globals.ClassMigrated||(typeof ClassMigrated=="undefined"?nil:ClassMigrated)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$5,$6,$7,$8;
tmp="new*".__comma(className);
$ctx1.sendIdx[","]=1;
oldClass=_st(_st($Smalltalk())._globals())._at_(className);
newClass=self._addSubclassOf_named_instanceVariableNames_package_(aClass,tmp,aCollection,packageName);
self._basicSwapClassNames_with_(oldClass,newClass);
$ctx1.sendIdx["basicSwapClassNames:with:"]=1;
_st((function(){
return smalltalk.withContext(function($ctx2) {
return self._copyClass_to_(oldClass,newClass);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._on_do_($Error(),(function(exception){
return smalltalk.withContext(function($ctx2) {
self._basicSwapClassNames_with_(oldClass,newClass);
$1=self._basicRemoveClass_(newClass);
$ctx2.sendIdx["basicRemoveClass:"]=1;
$1;
return _st(exception)._signal();
}, function($ctx2) {$ctx2.fillBlock({exception:exception},$ctx1,2)})}));
self._rawRenameClass_to_(oldClass,tmp);
$ctx1.sendIdx["rawRenameClass:to:"]=1;
$2=self._rawRenameClass_to_(newClass,className);
$3=_st(oldClass)._subclasses();
$4=(function(each){
return smalltalk.withContext(function($ctx2) {
return self._migrateClass_superclass_(each,newClass);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,3)})});
$5=_st("Recompiling ".__comma(_st(newClass)._name())).__comma("...");
$ctx1.sendIdx[","]=2;
_st($3)._do_displayingProgress_($4,$5);
self._basicRemoveClass_(oldClass);
$6=_st($ClassMigrated())._new();
_st($6)._theClass_(newClass);
_st($6)._oldClass_(oldClass);
$7=_st($6)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($7);
$8=newClass;
return $8;
}, function($ctx1) {$ctx1.fill(self,"migrateClassNamed:superclass:instanceVariableNames:package:",{className:className,aClass:aClass,aCollection:aCollection,packageName:packageName,oldClass:oldClass,newClass:newClass,tmp:tmp},globals.ClassBuilder)})},
args: ["className", "aClass", "aCollection", "packageName"],
source: "migrateClassNamed: className superclass: aClass instanceVariableNames: aCollection package: packageName\x0a\x09| oldClass newClass tmp |\x0a\x09\x0a\x09tmp := 'new*', className.\x0a\x09oldClass := Smalltalk globals at: className.\x0a\x09\x0a\x09newClass := self\x0a\x09\x09addSubclassOf: aClass\x0a\x09\x09named: tmp\x0a\x09\x09instanceVariableNames: aCollection\x0a\x09\x09package: packageName.\x0a\x0a\x09self basicSwapClassNames: oldClass with: newClass.\x0a\x0a\x09[ self copyClass: oldClass to: newClass ]\x0a\x09\x09on: Error\x0a\x09\x09do: [ :exception |\x0a\x09\x09\x09self\x0a\x09\x09\x09\x09basicSwapClassNames: oldClass with: newClass;\x0a\x09\x09\x09\x09basicRemoveClass: newClass.\x0a\x09\x09\x09exception signal ].\x0a\x0a\x09self\x0a\x09\x09rawRenameClass: oldClass to: tmp;\x0a\x09\x09rawRenameClass: newClass to: className.\x0a\x0a\x09oldClass subclasses \x0a\x09\x09do: [ :each | self migrateClass: each superclass: newClass ]\x0a\x09\x09displayingProgress: 'Recompiling ', newClass name, '...'.\x0a\x0a\x09self basicRemoveClass: oldClass.\x0a\x09\x0a\x09SystemAnnouncer current announce: (ClassMigrated new\x0a\x09\x09theClass: newClass;\x0a\x09\x09oldClass: oldClass;\x0a\x09\x09yourself).\x0a\x09\x0a\x09^ newClass",
messageSends: [",", "at:", "globals", "addSubclassOf:named:instanceVariableNames:package:", "basicSwapClassNames:with:", "on:do:", "copyClass:to:", "basicRemoveClass:", "signal", "rawRenameClass:to:", "do:displayingProgress:", "subclasses", "migrateClass:superclass:", "name", "announce:", "current", "theClass:", "new", "oldClass:", "yourself"],
referencedClasses: ["Smalltalk", "Error", "SystemAnnouncer", "ClassMigrated"]
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "rawRenameClass:to:",
protocol: 'private',
fn: function (aClass,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		globals[aString] = aClass;
	;
return self}, function($ctx1) {$ctx1.fill(self,"rawRenameClass:to:",{aClass:aClass,aString:aString},globals.ClassBuilder)})},
args: ["aClass", "aString"],
source: "rawRenameClass: aClass to: aString\x0a\x09<\x0a\x09\x09globals[aString] = aClass;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "renameClass:to:",
protocol: 'class migration',
fn: function (aClass,className){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassRenamed(){return globals.ClassRenamed||(typeof ClassRenamed=="undefined"?nil:ClassRenamed)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._basicRenameClass_to_(aClass,className);
_st(aClass)._recompile();
$1=_st($ClassRenamed())._new();
_st($1)._theClass_(aClass);
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
return self}, function($ctx1) {$ctx1.fill(self,"renameClass:to:",{aClass:aClass,className:className},globals.ClassBuilder)})},
args: ["aClass", "className"],
source: "renameClass: aClass to: className\x0a\x09self basicRenameClass: aClass to: className.\x0a\x09\x0a\x09\x22Recompile the class to fix potential issues with super sends\x22\x0a\x09aClass recompile.\x0a\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (ClassRenamed new\x0a\x09\x09\x09theClass: aClass;\x0a\x09\x09\x09yourself)",
messageSends: ["basicRenameClass:to:", "recompile", "announce:", "current", "theClass:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassRenamed"]
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "setupClass:",
protocol: 'public',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
smalltalk.init(aClass);;
return self}, function($ctx1) {$ctx1.fill(self,"setupClass:",{aClass:aClass},globals.ClassBuilder)})},
args: ["aClass"],
source: "setupClass: aClass\x0a\x09<smalltalk.init(aClass);>",
messageSends: [],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "superclass:subclass:",
protocol: 'class definition',
fn: function (aClass,className){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._superclass_subclass_instanceVariableNames_package_(aClass,className,"",nil);
return $1;
}, function($ctx1) {$ctx1.fill(self,"superclass:subclass:",{aClass:aClass,className:className},globals.ClassBuilder)})},
args: ["aClass", "className"],
source: "superclass: aClass subclass: className\x0a\x09^ self superclass: aClass subclass: className instanceVariableNames: '' package: nil",
messageSends: ["superclass:subclass:instanceVariableNames:package:"],
referencedClasses: []
}),
globals.ClassBuilder);

smalltalk.addMethod(
smalltalk.method({
selector: "superclass:subclass:instanceVariableNames:package:",
protocol: 'class definition',
fn: function (aClass,className,ivarNames,packageName){
var self=this;
var newClass;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassAdded(){return globals.ClassAdded||(typeof ClassAdded=="undefined"?nil:ClassAdded)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$5;
$1=self._instanceVariableNamesFor_(ivarNames);
if(($receiver = packageName) == nil || $receiver == null){
$2="unclassified";
} else {
$2=packageName;
};
newClass=self._addSubclassOf_named_instanceVariableNames_package_(aClass,className,$1,$2);
self._setupClass_(newClass);
$3=_st($ClassAdded())._new();
_st($3)._theClass_(newClass);
$4=_st($3)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($4);
$5=newClass;
return $5;
}, function($ctx1) {$ctx1.fill(self,"superclass:subclass:instanceVariableNames:package:",{aClass:aClass,className:className,ivarNames:ivarNames,packageName:packageName,newClass:newClass},globals.ClassBuilder)})},
args: ["aClass", "className", "ivarNames", "packageName"],
source: "superclass: aClass subclass: className instanceVariableNames: ivarNames package: packageName\x0a\x09| newClass |\x0a\x09\x0a\x09newClass := self addSubclassOf: aClass\x0a\x09\x09named: className instanceVariableNames: (self instanceVariableNamesFor: ivarNames)\x0a\x09\x09package: (packageName ifNil: [ 'unclassified' ]).\x0a\x09self setupClass: newClass.\x0a\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (ClassAdded new\x0a\x09\x09\x09theClass: newClass;\x0a\x09\x09\x09yourself).\x0a\x09\x0a\x09^ newClass",
messageSends: ["addSubclassOf:named:instanceVariableNames:package:", "instanceVariableNamesFor:", "ifNil:", "setupClass:", "announce:", "current", "theClass:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassAdded"]
}),
globals.ClassBuilder);



smalltalk.addClass('ClassCategoryReader', globals.Object, ['class', 'category'], 'Kernel-Classes');
globals.ClassCategoryReader.comment="I provide a mechanism for retrieving class descriptions stored on a file in the Smalltalk chunk format.";
smalltalk.addMethod(
smalltalk.method({
selector: "class:category:",
protocol: 'accessing',
fn: function (aClass,aString){
var self=this;
self["@class"]=aClass;
self["@category"]=aString;
return self},
args: ["aClass", "aString"],
source: "class: aClass category: aString\x0a\x09class := aClass.\x0a\x09category := aString",
messageSends: [],
referencedClasses: []
}),
globals.ClassCategoryReader);

smalltalk.addMethod(
smalltalk.method({
selector: "compileMethod:",
protocol: 'private',
fn: function (aString){
var self=this;
function $Compiler(){return globals.Compiler||(typeof Compiler=="undefined"?nil:Compiler)}
return smalltalk.withContext(function($ctx1) { 
_st(_st($Compiler())._new())._install_forClass_protocol_(aString,self["@class"],self["@category"]);
return self}, function($ctx1) {$ctx1.fill(self,"compileMethod:",{aString:aString},globals.ClassCategoryReader)})},
args: ["aString"],
source: "compileMethod: aString\x0a\x09Compiler new install: aString forClass: class protocol: category",
messageSends: ["install:forClass:protocol:", "new"],
referencedClasses: ["Compiler"]
}),
globals.ClassCategoryReader);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.ClassCategoryReader.superclass.fn.prototype._initialize.apply(_st(self), []);
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.ClassCategoryReader)})},
args: [],
source: "initialize\x0a\x09super initialize.",
messageSends: ["initialize"],
referencedClasses: []
}),
globals.ClassCategoryReader);

smalltalk.addMethod(
smalltalk.method({
selector: "scanFrom:",
protocol: 'fileIn',
fn: function (aChunkParser){
var self=this;
var chunk;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
_st((function(){
return smalltalk.withContext(function($ctx2) {
chunk=_st(aChunkParser)._nextChunk();
chunk;
return _st(chunk)._isEmpty();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileFalse_((function(){
return smalltalk.withContext(function($ctx2) {
return self._compileMethod_(chunk);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(_st($ClassBuilder())._new())._setupClass_(self["@class"]);
return self}, function($ctx1) {$ctx1.fill(self,"scanFrom:",{aChunkParser:aChunkParser,chunk:chunk},globals.ClassCategoryReader)})},
args: ["aChunkParser"],
source: "scanFrom: aChunkParser\x0a\x09| chunk |\x0a\x09[ chunk := aChunkParser nextChunk.\x0a\x09chunk isEmpty ] whileFalse: [\x0a\x09\x09self compileMethod: chunk ].\x0a\x09ClassBuilder new setupClass: class",
messageSends: ["whileFalse:", "nextChunk", "isEmpty", "compileMethod:", "setupClass:", "new"],
referencedClasses: ["ClassBuilder"]
}),
globals.ClassCategoryReader);



smalltalk.addClass('ClassCommentReader', globals.Object, ['class'], 'Kernel-Classes');
globals.ClassCommentReader.comment="I provide a mechanism for retrieving class comments stored on a file.\x0a\x0aSee also `ClassCategoryReader`.";
smalltalk.addMethod(
smalltalk.method({
selector: "class:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@class"]=aClass;
return self},
args: ["aClass"],
source: "class: aClass\x0a\x09class := aClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassCommentReader);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.ClassCommentReader.superclass.fn.prototype._initialize.apply(_st(self), []);
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.ClassCommentReader)})},
args: [],
source: "initialize\x0a\x09super initialize.",
messageSends: ["initialize"],
referencedClasses: []
}),
globals.ClassCommentReader);

smalltalk.addMethod(
smalltalk.method({
selector: "scanFrom:",
protocol: 'fileIn',
fn: function (aChunkParser){
var self=this;
var chunk;
return smalltalk.withContext(function($ctx1) { 
var $1;
chunk=_st(aChunkParser)._nextChunk();
$1=_st(chunk)._isEmpty();
if(! smalltalk.assert($1)){
self._setComment_(chunk);
};
return self}, function($ctx1) {$ctx1.fill(self,"scanFrom:",{aChunkParser:aChunkParser,chunk:chunk},globals.ClassCommentReader)})},
args: ["aChunkParser"],
source: "scanFrom: aChunkParser\x0a\x09| chunk |\x0a\x09chunk := aChunkParser nextChunk.\x0a\x09chunk isEmpty ifFalse: [\x0a\x09\x09self setComment: chunk ].",
messageSends: ["nextChunk", "ifFalse:", "isEmpty", "setComment:"],
referencedClasses: []
}),
globals.ClassCommentReader);

smalltalk.addMethod(
smalltalk.method({
selector: "setComment:",
protocol: 'private',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@class"])._comment_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"setComment:",{aString:aString},globals.ClassCommentReader)})},
args: ["aString"],
source: "setComment: aString\x0a\x09class comment: aString",
messageSends: ["comment:"],
referencedClasses: []
}),
globals.ClassCommentReader);



smalltalk.addClass('ClassSorterNode', globals.Object, ['theClass', 'level', 'nodes'], 'Kernel-Classes');
globals.ClassSorterNode.comment="I provide an algorithm for sorting classes alphabetically.\x0a\x0aSee [Issue #143](https://github.com/amber-smalltalk/amber/issues/143) on GitHub.";
smalltalk.addMethod(
smalltalk.method({
selector: "getNodesFrom:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
var children,others;
function $ClassSorterNode(){return globals.ClassSorterNode||(typeof ClassSorterNode=="undefined"?nil:ClassSorterNode)}
return smalltalk.withContext(function($ctx1) { 
var $1;
children=[];
others=[];
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(_st(each)._superclass()).__eq(self._theClass());
if(smalltalk.assert($1)){
return _st(children)._add_(each);
$ctx2.sendIdx["add:"]=1;
} else {
return _st(others)._add_(each);
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
self["@nodes"]=_st(children)._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st($ClassSorterNode())._on_classes_level_(each,others,_st(self._level()).__plus((1)));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,4)})}));
return self}, function($ctx1) {$ctx1.fill(self,"getNodesFrom:",{aCollection:aCollection,children:children,others:others},globals.ClassSorterNode)})},
args: ["aCollection"],
source: "getNodesFrom: aCollection\x0a\x09| children others |\x0a\x09children := #().\x0a\x09others := #().\x0a\x09aCollection do: [ :each |\x0a\x09\x09(each superclass = self theClass)\x0a\x09\x09\x09ifTrue: [ children add: each ]\x0a\x09\x09\x09ifFalse: [ others add: each ]].\x0a\x09nodes:= children collect: [ :each |\x0a\x09\x09ClassSorterNode on: each classes: others level: self level + 1 ]",
messageSends: ["do:", "ifTrue:ifFalse:", "=", "superclass", "theClass", "add:", "collect:", "on:classes:level:", "+", "level"],
referencedClasses: ["ClassSorterNode"]
}),
globals.ClassSorterNode);

smalltalk.addMethod(
smalltalk.method({
selector: "level",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@level"];
return $1;
},
args: [],
source: "level\x0a\x09^ level",
messageSends: [],
referencedClasses: []
}),
globals.ClassSorterNode);

smalltalk.addMethod(
smalltalk.method({
selector: "level:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
self["@level"]=anInteger;
return self},
args: ["anInteger"],
source: "level: anInteger\x0a\x09level := anInteger",
messageSends: [],
referencedClasses: []
}),
globals.ClassSorterNode);

smalltalk.addMethod(
smalltalk.method({
selector: "nodes",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@nodes"];
return $1;
},
args: [],
source: "nodes\x0a\x09^ nodes",
messageSends: [],
referencedClasses: []
}),
globals.ClassSorterNode);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@theClass"];
return $1;
},
args: [],
source: "theClass\x0a\x09^ theClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassSorterNode);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@theClass"]=aClass;
return self},
args: ["aClass"],
source: "theClass: aClass\x0a\x09theClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassSorterNode);

smalltalk.addMethod(
smalltalk.method({
selector: "traverseClassesWith:",
protocol: 'visiting',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$2;
$1=self._theClass();
$ctx1.sendIdx["theClass"]=1;
_st(aCollection)._add_($1);
_st(_st(self._nodes())._sorted_((function(a,b){
return smalltalk.withContext(function($ctx2) {
$3=_st(a)._theClass();
$ctx2.sendIdx["theClass"]=2;
$2=_st($3)._name();
$ctx2.sendIdx["name"]=1;
return _st($2).__lt_eq(_st(_st(b)._theClass())._name());
}, function($ctx2) {$ctx2.fillBlock({a:a,b:b},$ctx1,1)})})))._do_((function(aNode){
return smalltalk.withContext(function($ctx2) {
return _st(aNode)._traverseClassesWith_(aCollection);
}, function($ctx2) {$ctx2.fillBlock({aNode:aNode},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"traverseClassesWith:",{aCollection:aCollection},globals.ClassSorterNode)})},
args: ["aCollection"],
source: "traverseClassesWith: aCollection\x0a\x09\x22sort classes alphabetically Issue #143\x22\x0a\x0a\x09aCollection add: self theClass.\x0a\x09(self nodes sorted: [ :a :b | a theClass name <= b theClass name ]) do: [ :aNode |\x0a\x09\x09aNode traverseClassesWith: aCollection ].",
messageSends: ["add:", "theClass", "do:", "sorted:", "nodes", "<=", "name", "traverseClassesWith:"],
referencedClasses: []
}),
globals.ClassSorterNode);


smalltalk.addMethod(
smalltalk.method({
selector: "on:classes:level:",
protocol: 'instance creation',
fn: function (aClass,aCollection,anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._theClass_(aClass);
_st($2)._level_(anInteger);
_st($2)._getNodesFrom_(aCollection);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:classes:level:",{aClass:aClass,aCollection:aCollection,anInteger:anInteger},globals.ClassSorterNode.klass)})},
args: ["aClass", "aCollection", "anInteger"],
source: "on: aClass classes: aCollection level: anInteger\x0a\x09^ self new\x0a\x09\x09theClass: aClass;\x0a\x09\x09level: anInteger;\x0a\x09\x09getNodesFrom: aCollection;\x0a\x09\x09yourself",
messageSends: ["theClass:", "new", "level:", "getNodesFrom:", "yourself"],
referencedClasses: []
}),
globals.ClassSorterNode.klass);

});

define("amber_core/Kernel-Methods", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Methods');
smalltalk.packages["Kernel-Methods"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('BlockClosure', globals.Object, [], 'Kernel-Methods');
globals.BlockClosure.comment="I represent a lexical closure.\x0aI am is directly mapped to JavaScript Function.\x0a\x0a## API\x0a\x0a1. Evaluation\x0a\x0a    My instances get evaluated with the `#value*` methods in the 'evaluating' protocol.\x0a\x0a    Example: ` [ :x | x + 1 ] value: 3 \x22Answers 4\x22 `\x0a\x0a2. Control structures\x0a\x0a    Blocks are used (together with `Boolean`) for control structures (methods in the `controlling` protocol).\x0a\x0a    Example: `aBlock whileTrue: [ ... ]`\x0a\x0a3. Error handling\x0a\x0a    I provide the `#on:do:` method for handling exceptions.\x0a\x0a    Example: ` aBlock on: MessageNotUnderstood do: [ :ex | ... ] `";
smalltalk.addMethod(
smalltalk.method({
selector: "applyTo:arguments:",
protocol: 'evaluating',
fn: function (anObject,aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.apply(anObject, aCollection);
return self}, function($ctx1) {$ctx1.fill(self,"applyTo:arguments:",{anObject:anObject,aCollection:aCollection},globals.BlockClosure)})},
args: ["anObject", "aCollection"],
source: "applyTo: anObject arguments: aCollection\x0a\x09<return self.apply(anObject, aCollection)>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "asCompiledMethod:",
protocol: 'converting',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.method({selector:aString, fn:self});;
return self}, function($ctx1) {$ctx1.fill(self,"asCompiledMethod:",{aString:aString},globals.BlockClosure)})},
args: ["aString"],
source: "asCompiledMethod: aString\x0a\x09<return smalltalk.method({selector:aString, fn:self});>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "compiledSource",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toString();
return self}, function($ctx1) {$ctx1.fill(self,"compiledSource",{},globals.BlockClosure)})},
args: [],
source: "compiledSource\x0a\x09<return self.toString()>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "currySelf",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return function () {
			var args = [ this ];
			args.push.apply(args, arguments);
			return self.apply(null, args);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"currySelf",{},globals.BlockClosure)})},
args: [],
source: "currySelf\x0a\x09\x22Transforms [ :selfarg :x :y | stcode ] block\x0a\x09which represents JS function (selfarg, x, y, ...) {jscode}\x0a\x09into function (x, y, ...) {jscode} that takes selfarg from 'this'.\x0a\x09IOW, it is usable as JS method and first arg takes the receiver.\x22\x0a\x09\x0a\x09<\x0a\x09\x09return function () {\x0a\x09\x09\x09var args = [ this ];\x0a\x09\x09\x09args.push.apply(args, arguments);\x0a\x09\x09\x09return self.apply(null, args);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "ensure:",
protocol: 'evaluating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
try{return self._value()}finally{aBlock._value()};
return self}, function($ctx1) {$ctx1.fill(self,"ensure:",{aBlock:aBlock},globals.BlockClosure)})},
args: ["aBlock"],
source: "ensure: aBlock\x0a\x09<try{return self._value()}finally{aBlock._value()}>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "fork",
protocol: 'timeout/interval',
fn: function (){
var self=this;
function $ForkPool(){return globals.ForkPool||(typeof ForkPool=="undefined"?nil:ForkPool)}
return smalltalk.withContext(function($ctx1) { 
_st(_st($ForkPool())._default())._fork_(self);
return self}, function($ctx1) {$ctx1.fill(self,"fork",{},globals.BlockClosure)})},
args: [],
source: "fork\x0a\x09ForkPool default fork: self",
messageSends: ["fork:", "default"],
referencedClasses: ["ForkPool"]
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'evaluating',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return new self();
return self}, function($ctx1) {$ctx1.fill(self,"new",{},globals.BlockClosure)})},
args: [],
source: "new\x0a\x09\x22Use the receiver as a JS constructor.\x0a\x09*Do not* use this method to instanciate Smalltalk objects!\x22\x0a\x09<return new self()>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "newValue:",
protocol: 'evaluating',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._newWithValues_([anObject]);
return $1;
}, function($ctx1) {$ctx1.fill(self,"newValue:",{anObject:anObject},globals.BlockClosure)})},
args: ["anObject"],
source: "newValue: anObject\x0a\x09^ self newWithValues: { anObject }",
messageSends: ["newWithValues:"],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "newValue:value:",
protocol: 'evaluating',
fn: function (anObject,anObject2){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._newWithValues_([anObject,anObject2]);
return $1;
}, function($ctx1) {$ctx1.fill(self,"newValue:value:",{anObject:anObject,anObject2:anObject2},globals.BlockClosure)})},
args: ["anObject", "anObject2"],
source: "newValue: anObject value: anObject2\x0a\x09^ self newWithValues: { anObject. anObject2 }.",
messageSends: ["newWithValues:"],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "newValue:value:value:",
protocol: 'evaluating',
fn: function (anObject,anObject2,anObject3){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._newWithValues_([anObject,anObject2,anObject3]);
return $1;
}, function($ctx1) {$ctx1.fill(self,"newValue:value:value:",{anObject:anObject,anObject2:anObject2,anObject3:anObject3},globals.BlockClosure)})},
args: ["anObject", "anObject2", "anObject3"],
source: "newValue: anObject value: anObject2 value: anObject3\x0a\x09^ self newWithValues: { anObject. anObject2. anObject3 }.",
messageSends: ["newWithValues:"],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "newWithValues:",
protocol: 'evaluating',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var object = Object.create(self.prototype);
		var result = self.apply(object, aCollection);
		return typeof result === "object" ? result : object;
	;
return self}, function($ctx1) {$ctx1.fill(self,"newWithValues:",{aCollection:aCollection},globals.BlockClosure)})},
args: ["aCollection"],
source: "newWithValues: aCollection\x0a\x09\x22Simulates JS new operator by combination of Object.create and .apply\x22\x0a\x09<\x0a\x09\x09var object = Object.create(self.prototype);\x0a\x09\x09var result = self.apply(object, aCollection);\x0a\x09\x09return typeof result === \x22object\x22 ? result : object;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "numArgs",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.length;
return self}, function($ctx1) {$ctx1.fill(self,"numArgs",{},globals.BlockClosure)})},
args: [],
source: "numArgs\x0a\x09<return self.length>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "on:do:",
protocol: 'error handling',
fn: function (anErrorClass,aBlock){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$1=self._tryCatch_((function(error){
var smalltalkError;
return smalltalk.withContext(function($ctx2) {
smalltalkError=_st($Smalltalk())._asSmalltalkException_(error);
smalltalkError;
$2=_st(smalltalkError)._isKindOf_(anErrorClass);
if(smalltalk.assert($2)){
return _st(aBlock)._value_(smalltalkError);
} else {
return _st(smalltalkError)._resignal();
};
}, function($ctx2) {$ctx2.fillBlock({error:error,smalltalkError:smalltalkError},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:do:",{anErrorClass:anErrorClass,aBlock:aBlock},globals.BlockClosure)})},
args: ["anErrorClass", "aBlock"],
source: "on: anErrorClass do: aBlock\x0a\x09\x22All exceptions thrown in the Smalltalk stack are cought.\x0a\x09Convert all JS exceptions to JavaScriptException instances.\x22\x0a\x09\x0a\x09^ self tryCatch: [ :error | | smalltalkError |\x0a\x09\x09smalltalkError := Smalltalk asSmalltalkException: error.\x0a\x09\x09(smalltalkError isKindOf: anErrorClass)\x0a\x09\x09ifTrue: [ aBlock value: smalltalkError ]\x0a\x09\x09ifFalse: [ smalltalkError resignal ] ]",
messageSends: ["tryCatch:", "asSmalltalkException:", "ifTrue:ifFalse:", "isKindOf:", "value:", "resignal"],
referencedClasses: ["Smalltalk"]
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
return nil;
},
args: [],
source: "receiver\x0a\x09^ nil",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "timeToRun",
protocol: 'evaluating',
fn: function (){
var self=this;
function $Date(){return globals.Date||(typeof Date=="undefined"?nil:Date)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Date())._millisecondsToRun_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"timeToRun",{},globals.BlockClosure)})},
args: [],
source: "timeToRun\x0a\x09\x22Answer the number of milliseconds taken to execute this block.\x22\x0a\x0a\x09^ Date millisecondsToRun: self",
messageSends: ["millisecondsToRun:"],
referencedClasses: ["Date"]
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "tryCatch:",
protocol: 'error handling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		try {
			return self._value();
		} catch(error) {
			return aBlock._value_(error);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"tryCatch:",{aBlock:aBlock},globals.BlockClosure)})},
args: ["aBlock"],
source: "tryCatch: aBlock\x0a\x09<\x0a\x09\x09try {\x0a\x09\x09\x09return self._value();\x0a\x09\x09} catch(error) {\x0a\x09\x09\x09return aBlock._value_(error);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'evaluating',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self();;
return self}, function($ctx1) {$ctx1.fill(self,"value",{},globals.BlockClosure)})},
args: [],
source: "value\x0a\x09<return self();>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'evaluating',
fn: function (anArg){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self(anArg);;
return self}, function($ctx1) {$ctx1.fill(self,"value:",{anArg:anArg},globals.BlockClosure)})},
args: ["anArg"],
source: "value: anArg\x0a\x09<return self(anArg);>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "value:value:",
protocol: 'evaluating',
fn: function (firstArg,secondArg){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self(firstArg, secondArg);;
return self}, function($ctx1) {$ctx1.fill(self,"value:value:",{firstArg:firstArg,secondArg:secondArg},globals.BlockClosure)})},
args: ["firstArg", "secondArg"],
source: "value: firstArg value: secondArg\x0a\x09<return self(firstArg, secondArg);>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "value:value:value:",
protocol: 'evaluating',
fn: function (firstArg,secondArg,thirdArg){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self(firstArg, secondArg, thirdArg);;
return self}, function($ctx1) {$ctx1.fill(self,"value:value:value:",{firstArg:firstArg,secondArg:secondArg,thirdArg:thirdArg},globals.BlockClosure)})},
args: ["firstArg", "secondArg", "thirdArg"],
source: "value: firstArg value: secondArg value: thirdArg\x0a\x09<return self(firstArg, secondArg, thirdArg);>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "valueWithInterval:",
protocol: 'timeout/interval',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var interval = setInterval(self, aNumber);
		return globals.Timeout._on_(interval);
	;
return self}, function($ctx1) {$ctx1.fill(self,"valueWithInterval:",{aNumber:aNumber},globals.BlockClosure)})},
args: ["aNumber"],
source: "valueWithInterval: aNumber\x0a\x09<\x0a\x09\x09var interval = setInterval(self, aNumber);\x0a\x09\x09return globals.Timeout._on_(interval);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "valueWithPossibleArguments:",
protocol: 'evaluating',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.apply(null, aCollection);;
return self}, function($ctx1) {$ctx1.fill(self,"valueWithPossibleArguments:",{aCollection:aCollection},globals.BlockClosure)})},
args: ["aCollection"],
source: "valueWithPossibleArguments: aCollection\x0a\x09<return self.apply(null, aCollection);>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "valueWithTimeout:",
protocol: 'timeout/interval',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var timeout = setTimeout(self, aNumber);
		return globals.Timeout._on_(timeout);
	;
return self}, function($ctx1) {$ctx1.fill(self,"valueWithTimeout:",{aNumber:aNumber},globals.BlockClosure)})},
args: ["aNumber"],
source: "valueWithTimeout: aNumber\x0a\x09<\x0a\x09\x09var timeout = setTimeout(self, aNumber);\x0a\x09\x09return globals.Timeout._on_(timeout);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "whileFalse",
protocol: 'controlling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._whileFalse_((function(){
}));
return self}, function($ctx1) {$ctx1.fill(self,"whileFalse",{},globals.BlockClosure)})},
args: [],
source: "whileFalse\x0a\x09self whileFalse: []",
messageSends: ["whileFalse:"],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "whileFalse:",
protocol: 'controlling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
while(!smalltalk.assert(self._value())) {aBlock._value()};
return self}, function($ctx1) {$ctx1.fill(self,"whileFalse:",{aBlock:aBlock},globals.BlockClosure)})},
args: ["aBlock"],
source: "whileFalse: aBlock\x0a\x09<while(!smalltalk.assert(self._value())) {aBlock._value()}>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "whileTrue",
protocol: 'controlling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._whileTrue_((function(){
}));
return self}, function($ctx1) {$ctx1.fill(self,"whileTrue",{},globals.BlockClosure)})},
args: [],
source: "whileTrue\x0a\x09self whileTrue: []",
messageSends: ["whileTrue:"],
referencedClasses: []
}),
globals.BlockClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "whileTrue:",
protocol: 'controlling',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
while(smalltalk.assert(self._value())) {aBlock._value()};
return self}, function($ctx1) {$ctx1.fill(self,"whileTrue:",{aBlock:aBlock},globals.BlockClosure)})},
args: ["aBlock"],
source: "whileTrue: aBlock\x0a\x09<while(smalltalk.assert(self._value())) {aBlock._value()}>",
messageSends: [],
referencedClasses: []
}),
globals.BlockClosure);



smalltalk.addClass('CompiledMethod', globals.Object, [], 'Kernel-Methods');
globals.CompiledMethod.comment="I represent a class method of the system. I hold the source and compiled code of a class method.\x0a\x0a## API\x0aMy instances can be accessed using `Behavior >> #methodAt:`\x0a\x0a    Object methodAt: 'asString'\x0a\x0aSource code access:\x0a\x0a\x09(String methodAt: 'lines') source\x0a\x0aReferenced classes:\x0a\x0a\x09(String methodAt: 'lines') referencedClasses\x0a\x0aMessages sent from an instance:\x0a\x09\x0a\x09(String methodAt: 'lines') messageSends";
smalltalk.addMethod(
smalltalk.method({
selector: "arguments",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.args || [];
return self}, function($ctx1) {$ctx1.fill(self,"arguments",{},globals.CompiledMethod)})},
args: [],
source: "arguments\x0a\x09<return self.args || []>",
messageSends: [],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "browse",
protocol: 'browsing',
fn: function (){
var self=this;
function $Finder(){return globals.Finder||(typeof Finder=="undefined"?nil:Finder)}
return smalltalk.withContext(function($ctx1) { 
_st($Finder())._findMethod_(self);
return self}, function($ctx1) {$ctx1.fill(self,"browse",{},globals.CompiledMethod)})},
args: [],
source: "browse\x0a\x09Finder findMethod: self",
messageSends: ["findMethod:"],
referencedClasses: ["Finder"]
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "category",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._protocol();
return $1;
}, function($ctx1) {$ctx1.fill(self,"category",{},globals.CompiledMethod)})},
args: [],
source: "category\x0a\x09^ self protocol",
messageSends: ["protocol"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "defaultProtocol",
protocol: 'defaults',
fn: function (){
var self=this;
return "as yet unclassified";
},
args: [],
source: "defaultProtocol\x0a\x09^ 'as yet unclassified'",
messageSends: [],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "fn",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("fn");
return $1;
}, function($ctx1) {$ctx1.fill(self,"fn",{},globals.CompiledMethod)})},
args: [],
source: "fn\x0a\x09^ self basicAt: 'fn'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "fn:",
protocol: 'accessing',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._basicAt_put_("fn",aBlock);
return self}, function($ctx1) {$ctx1.fill(self,"fn:",{aBlock:aBlock},globals.CompiledMethod)})},
args: ["aBlock"],
source: "fn: aBlock\x0a\x09self basicAt: 'fn' put: aBlock",
messageSends: ["basicAt:put:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "isCompiledMethod",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isCompiledMethod\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "isOverridden",
protocol: 'testing',
fn: function (){
var self=this;
var selector;
return smalltalk.withContext(function($ctx1) { 
var $1;
var $early={};
try {
selector=self._selector();
_st(self._methodClass())._allSubclassesDo_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(each)._includesSelector_(selector);
if(smalltalk.assert($1)){
throw $early=[true];
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return false;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"isOverridden",{selector:selector},globals.CompiledMethod)})},
args: [],
source: "isOverridden\x0a\x09| selector |\x0a    \x0a    selector := self selector.\x0a    self methodClass allSubclassesDo: [ :each |\x0a\x09    (each includesSelector: selector)\x0a        \x09ifTrue: [ ^ true ] ].\x0a\x09^ false",
messageSends: ["selector", "allSubclassesDo:", "methodClass", "ifTrue:", "includesSelector:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "isOverride",
protocol: 'testing',
fn: function (){
var self=this;
var superclass;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
$1=self._methodClass();
$ctx1.sendIdx["methodClass"]=1;
superclass=_st($1)._superclass();
$ctx1.sendIdx["superclass"]=1;
$2=superclass;
if(($receiver = $2) == nil || $receiver == null){
return false;
} else {
$2;
};
$3=_st(_st(_st(self._methodClass())._superclass())._lookupSelector_(self._selector()))._notNil();
return $3;
}, function($ctx1) {$ctx1.fill(self,"isOverride",{superclass:superclass},globals.CompiledMethod)})},
args: [],
source: "isOverride\x0a\x09| superclass |\x0a    \x0a    superclass := self methodClass superclass.\x0a\x09superclass ifNil: [ ^ false ].\x0a\x09\x0a    ^ (self methodClass superclass lookupSelector: self selector) notNil",
messageSends: ["superclass", "methodClass", "ifNil:", "notNil", "lookupSelector:", "selector"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "messageSends",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("messageSends");
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageSends",{},globals.CompiledMethod)})},
args: [],
source: "messageSends\x0a\x09^ self basicAt: 'messageSends'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "methodClass",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("methodClass");
return $1;
}, function($ctx1) {$ctx1.fill(self,"methodClass",{},globals.CompiledMethod)})},
args: [],
source: "methodClass\x0a\x09^ self basicAt: 'methodClass'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "package",
protocol: 'accessing',
fn: function (){
var self=this;
function $Package(){return globals.Package||(typeof Package=="undefined"?nil:Package)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$4,$3,$5;
$2=self._protocol();
$ctx1.sendIdx["protocol"]=1;
$1=_st($2)._beginsWith_("*");
if(! smalltalk.assert($1)){
$4=self._methodClass();
$ctx1.sendIdx["methodClass"]=1;
$3=_st($4)._package();
$ctx1.sendIdx["package"]=1;
return $3;
};
$5=_st($Package())._named_ifAbsent_(_st(self._protocol())._allButFirst(),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._methodClass())._package();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return $5;
}, function($ctx1) {$ctx1.fill(self,"package",{},globals.CompiledMethod)})},
args: [],
source: "package\x0a\x09\x22Answer the package the receiver belongs to:\x0a\x09- if it is an extension method, answer the corresponding package\x0a\x09- else answer the `methodClass` package\x22\x0a\x09\x0a\x09(self protocol beginsWith: '*') ifFalse: [\x0a\x09\x09^ self methodClass package ].\x0a\x09\x09\x0a\x09^ Package \x0a\x09\x09named: self protocol allButFirst\x0a\x09\x09ifAbsent: [ self methodClass package ]",
messageSends: ["ifFalse:", "beginsWith:", "protocol", "package", "methodClass", "named:ifAbsent:", "allButFirst"],
referencedClasses: ["Package"]
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "protocol",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._basicAt_("protocol");
if(($receiver = $2) == nil || $receiver == null){
$1=self._defaultProtocol();
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"protocol",{},globals.CompiledMethod)})},
args: [],
source: "protocol\x0a\x09^ (self basicAt: 'protocol') ifNil: [ self defaultProtocol ]",
messageSends: ["ifNil:", "basicAt:", "defaultProtocol"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "protocol:",
protocol: 'accessing',
fn: function (aString){
var self=this;
var oldProtocol;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $MethodMoved(){return globals.MethodMoved||(typeof MethodMoved=="undefined"?nil:MethodMoved)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
oldProtocol=self._protocol();
self._basicAt_put_("protocol",aString);
$1=_st($MethodMoved())._new();
_st($1)._method_(self);
_st($1)._oldProtocol_(oldProtocol);
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
$3=self._methodClass();
if(($receiver = $3) == nil || $receiver == null){
$3;
} else {
var methodClass;
methodClass=$receiver;
_st(_st(methodClass)._organization())._addElement_(aString);
_st(methodClass)._removeProtocolIfEmpty_(oldProtocol);
};
return self}, function($ctx1) {$ctx1.fill(self,"protocol:",{aString:aString,oldProtocol:oldProtocol},globals.CompiledMethod)})},
args: ["aString"],
source: "protocol: aString\x0a\x09| oldProtocol |\x0a\x09oldProtocol := self protocol.\x0a\x09self basicAt: 'protocol' put: aString.\x0a\x0a\x09SystemAnnouncer current announce: (MethodMoved new\x0a\x09\x09method: self;\x0a\x09\x09oldProtocol: oldProtocol;\x0a\x09\x09yourself).\x0a\x0a\x09self methodClass ifNotNil: [ :methodClass |\x0a\x09\x09methodClass organization addElement: aString.\x0a\x09\x09methodClass removeProtocolIfEmpty: oldProtocol ]",
messageSends: ["protocol", "basicAt:put:", "announce:", "current", "method:", "new", "oldProtocol:", "yourself", "ifNotNil:", "methodClass", "addElement:", "organization", "removeProtocolIfEmpty:"],
referencedClasses: ["SystemAnnouncer", "MethodMoved"]
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "referencedClasses",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("referencedClasses");
return $1;
}, function($ctx1) {$ctx1.fill(self,"referencedClasses",{},globals.CompiledMethod)})},
args: [],
source: "referencedClasses\x0a\x09^ self basicAt: 'referencedClasses'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("selector");
return $1;
}, function($ctx1) {$ctx1.fill(self,"selector",{},globals.CompiledMethod)})},
args: [],
source: "selector\x0a\x09^ self basicAt: 'selector'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._basicAt_put_("selector",aString);
return self}, function($ctx1) {$ctx1.fill(self,"selector:",{aString:aString},globals.CompiledMethod)})},
args: ["aString"],
source: "selector: aString\x0a\x09self basicAt: 'selector' put: aString",
messageSends: ["basicAt:put:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "sendTo:arguments:",
protocol: 'evaluating',
fn: function (anObject,aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._fn())._applyTo_arguments_(anObject,aCollection);
return $1;
}, function($ctx1) {$ctx1.fill(self,"sendTo:arguments:",{anObject:anObject,aCollection:aCollection},globals.CompiledMethod)})},
args: ["anObject", "aCollection"],
source: "sendTo: anObject arguments: aCollection\x0a\x09^ self fn applyTo: anObject arguments: aCollection",
messageSends: ["applyTo:arguments:", "fn"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._basicAt_("source");
if(($receiver = $2) == nil || $receiver == null){
$1="";
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"source",{},globals.CompiledMethod)})},
args: [],
source: "source\x0a\x09^ (self basicAt: 'source') ifNil: [ '' ]",
messageSends: ["ifNil:", "basicAt:"],
referencedClasses: []
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._basicAt_put_("source",aString);
return self}, function($ctx1) {$ctx1.fill(self,"source:",{aString:aString},globals.CompiledMethod)})},
args: ["aString"],
source: "source: aString\x0a\x09self basicAt: 'source' put: aString",
messageSends: ["basicAt:put:"],
referencedClasses: []
}),
globals.CompiledMethod);



smalltalk.addClass('ForkPool', globals.Object, ['poolSize', 'maxPoolSize', 'queue', 'worker'], 'Kernel-Methods');
globals.ForkPool.comment="I am responsible for handling forked blocks.\x0aThe pool size sets the maximum concurrent forked blocks.\x0a\x0a## API\x0a\x0aThe default instance is accessed with `#default`.\x0aThe maximum concurrent forked blocks can be set with `#maxPoolSize:`.\x0a\x0aForking is done via `BlockClosure >> #fork`";
smalltalk.addMethod(
smalltalk.method({
selector: "addWorker",
protocol: 'private',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@worker"])._valueWithTimeout_((0));
self["@poolSize"]=_st(self["@poolSize"]).__plus((1));
return self}, function($ctx1) {$ctx1.fill(self,"addWorker",{},globals.ForkPool)})},
args: [],
source: "addWorker\x0a\x09worker valueWithTimeout: 0.\x0a\x09poolSize := poolSize + 1",
messageSends: ["valueWithTimeout:", "+"],
referencedClasses: []
}),
globals.ForkPool);

smalltalk.addMethod(
smalltalk.method({
selector: "defaultMaxPoolSize",
protocol: 'defaults',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._defaultMaxPoolSize();
return $1;
}, function($ctx1) {$ctx1.fill(self,"defaultMaxPoolSize",{},globals.ForkPool)})},
args: [],
source: "defaultMaxPoolSize\x0a\x09^ self class defaultMaxPoolSize",
messageSends: ["defaultMaxPoolSize", "class"],
referencedClasses: []
}),
globals.ForkPool);

smalltalk.addMethod(
smalltalk.method({
selector: "fork:",
protocol: 'actions',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@poolSize"]).__lt(self._maxPoolSize());
if(smalltalk.assert($1)){
self._addWorker();
};
_st(self["@queue"])._nextPut_(aBlock);
return self}, function($ctx1) {$ctx1.fill(self,"fork:",{aBlock:aBlock},globals.ForkPool)})},
args: ["aBlock"],
source: "fork: aBlock\x0a\x09poolSize < self maxPoolSize ifTrue: [ self addWorker ].\x0a\x09queue nextPut: aBlock",
messageSends: ["ifTrue:", "<", "maxPoolSize", "addWorker", "nextPut:"],
referencedClasses: []
}),
globals.ForkPool);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $Queue(){return globals.Queue||(typeof Queue=="undefined"?nil:Queue)}
return smalltalk.withContext(function($ctx1) { 
globals.ForkPool.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@poolSize"]=(0);
self["@queue"]=_st($Queue())._new();
self["@worker"]=self._makeWorker();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.ForkPool)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09\x0a\x09poolSize := 0.\x0a\x09queue := Queue new.\x0a\x09worker := self makeWorker",
messageSends: ["initialize", "new", "makeWorker"],
referencedClasses: ["Queue"]
}),
globals.ForkPool);

smalltalk.addMethod(
smalltalk.method({
selector: "makeWorker",
protocol: 'initialization',
fn: function (){
var self=this;
var sentinel;
function $Object(){return globals.Object||(typeof Object=="undefined"?nil:Object)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
sentinel=_st($Object())._new();
$1=(function(){
var block;
return smalltalk.withContext(function($ctx2) {
self["@poolSize"]=_st(self["@poolSize"]).__minus((1));
self["@poolSize"];
block=_st(self["@queue"])._nextIfAbsent_((function(){
return sentinel;
}));
block;
$2=_st(block).__eq_eq(sentinel);
if(! smalltalk.assert($2)){
return _st((function(){
return smalltalk.withContext(function($ctx3) {
return _st(block)._value();
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,4)})}))._ensure_((function(){
return smalltalk.withContext(function($ctx3) {
return self._addWorker();
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,5)})}));
};
}, function($ctx2) {$ctx2.fillBlock({block:block},$ctx1,1)})});
return $1;
}, function($ctx1) {$ctx1.fill(self,"makeWorker",{sentinel:sentinel},globals.ForkPool)})},
args: [],
source: "makeWorker\x0a\x09| sentinel |\x0a\x09sentinel := Object new.\x0a\x09^ [ | block |\x0a\x09\x09poolSize := poolSize - 1.\x0a\x09\x09block := queue nextIfAbsent: [ sentinel ].\x0a\x09\x09block == sentinel ifFalse: [\x0a\x09\x09\x09[ block value ] ensure: [ self addWorker ] ]]",
messageSends: ["new", "-", "nextIfAbsent:", "ifFalse:", "==", "ensure:", "value", "addWorker"],
referencedClasses: ["Object"]
}),
globals.ForkPool);

smalltalk.addMethod(
smalltalk.method({
selector: "maxPoolSize",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@maxPoolSize"];
if(($receiver = $2) == nil || $receiver == null){
$1=self._defaultMaxPoolSize();
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"maxPoolSize",{},globals.ForkPool)})},
args: [],
source: "maxPoolSize\x0a\x09^ maxPoolSize ifNil: [ self defaultMaxPoolSize ]",
messageSends: ["ifNil:", "defaultMaxPoolSize"],
referencedClasses: []
}),
globals.ForkPool);

smalltalk.addMethod(
smalltalk.method({
selector: "maxPoolSize:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
self["@maxPoolSize"]=anInteger;
return self},
args: ["anInteger"],
source: "maxPoolSize: anInteger\x0a\x09maxPoolSize := anInteger",
messageSends: [],
referencedClasses: []
}),
globals.ForkPool);


globals.ForkPool.klass.iVarNames = ['default'];
smalltalk.addMethod(
smalltalk.method({
selector: "default",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@default"];
if(($receiver = $2) == nil || $receiver == null){
self["@default"]=self._new();
$1=self["@default"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"default",{},globals.ForkPool.klass)})},
args: [],
source: "default\x0a\x09^ default ifNil: [ default := self new ]",
messageSends: ["ifNil:", "new"],
referencedClasses: []
}),
globals.ForkPool.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "defaultMaxPoolSize",
protocol: 'accessing',
fn: function (){
var self=this;
return (100);
},
args: [],
source: "defaultMaxPoolSize\x0a\x09^ 100",
messageSends: [],
referencedClasses: []
}),
globals.ForkPool.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "resetDefault",
protocol: 'accessing',
fn: function (){
var self=this;
self["@default"]=nil;
return self},
args: [],
source: "resetDefault\x0a\x09default := nil",
messageSends: [],
referencedClasses: []
}),
globals.ForkPool.klass);


smalltalk.addClass('Message', globals.Object, ['selector', 'arguments'], 'Kernel-Methods');
globals.Message.comment="In general, the system does not use instances of me for efficiency reasons.\x0aHowever, when a message is not understood by its receiver, the interpreter will make up an instance of it in order to capture the information involved in an actual message transmission.\x0aThis instance is sent it as an argument with the message `#doesNotUnderstand:` to the receiver.\x0a\x0aSee boot.js, `messageNotUnderstood` and its counterpart `Object >> #doesNotUnderstand:`\x0a\x0a## API\x0a\x0aBesides accessing methods, `#sendTo:` provides a convenient way to send a message to an object.";
smalltalk.addMethod(
smalltalk.method({
selector: "arguments",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@arguments"];
return $1;
},
args: [],
source: "arguments\x0a\x09^ arguments",
messageSends: [],
referencedClasses: []
}),
globals.Message);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments:",
protocol: 'accessing',
fn: function (anArray){
var self=this;
self["@arguments"]=anArray;
return self},
args: ["anArray"],
source: "arguments: anArray\x0a\x09arguments := anArray",
messageSends: [],
referencedClasses: []
}),
globals.Message);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
globals.Message.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
_st(aStream)._nextPutAll_("(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aStream)._nextPutAll_(self._selector());
$ctx1.sendIdx["nextPutAll:"]=2;
$1=_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Message)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09aStream\x0a\x09\x09nextPutAll: '(';\x0a\x09\x09nextPutAll: self selector;\x0a\x09\x09nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "selector"],
referencedClasses: []
}),
globals.Message);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@selector"];
return $1;
},
args: [],
source: "selector\x0a\x09^ selector",
messageSends: [],
referencedClasses: []
}),
globals.Message);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@selector"]=aString;
return self},
args: ["aString"],
source: "selector: aString\x0a\x09selector := aString",
messageSends: [],
referencedClasses: []
}),
globals.Message);

smalltalk.addMethod(
smalltalk.method({
selector: "sendTo:",
protocol: 'actions',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(anObject)._perform_withArguments_(self._selector(),self._arguments());
return $1;
}, function($ctx1) {$ctx1.fill(self,"sendTo:",{anObject:anObject},globals.Message)})},
args: ["anObject"],
source: "sendTo: anObject\x0a\x09^ anObject perform: self selector withArguments: self arguments",
messageSends: ["perform:withArguments:", "selector", "arguments"],
referencedClasses: []
}),
globals.Message);


smalltalk.addMethod(
smalltalk.method({
selector: "selector:arguments:",
protocol: 'instance creation',
fn: function (aString,anArray){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._selector_(aString);
_st($2)._arguments_(anArray);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"selector:arguments:",{aString:aString,anArray:anArray},globals.Message.klass)})},
args: ["aString", "anArray"],
source: "selector: aString arguments: anArray\x0a\x09^ self new\x0a\x09\x09selector: aString;\x0a\x09\x09arguments: anArray;\x0a\x09\x09yourself",
messageSends: ["selector:", "new", "arguments:", "yourself"],
referencedClasses: []
}),
globals.Message.klass);


smalltalk.addClass('MessageSend', globals.Object, ['receiver', 'message'], 'Kernel-Methods');
globals.MessageSend.comment="I encapsulate message sends to objects. Arguments can be either predefined or supplied when the message send is performed. \x0a\x0a## API\x0a\x0aUse `#value` to perform a message send with its predefined arguments and `#value:*` if additonal arguments have to supplied.";
smalltalk.addMethod(
smalltalk.method({
selector: "arguments",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@message"])._arguments();
return $1;
}, function($ctx1) {$ctx1.fill(self,"arguments",{},globals.MessageSend)})},
args: [],
source: "arguments\x0a\x09^ message arguments",
messageSends: ["arguments"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@message"])._arguments_(aCollection);
return self}, function($ctx1) {$ctx1.fill(self,"arguments:",{aCollection:aCollection},globals.MessageSend)})},
args: ["aCollection"],
source: "arguments: aCollection\x0a\x09message arguments: aCollection",
messageSends: ["arguments:"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $Message(){return globals.Message||(typeof Message=="undefined"?nil:Message)}
return smalltalk.withContext(function($ctx1) { 
globals.MessageSend.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@message"]=_st($Message())._new();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.MessageSend)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09message := Message new",
messageSends: ["initialize", "new"],
referencedClasses: ["Message"]
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
globals.MessageSend.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
_st(aStream)._nextPutAll_("(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aStream)._nextPutAll_(self._receiver());
$ctx1.sendIdx["nextPutAll:"]=2;
_st(aStream)._nextPutAll_(" >> ");
$ctx1.sendIdx["nextPutAll:"]=3;
_st(aStream)._nextPutAll_(self._selector());
$ctx1.sendIdx["nextPutAll:"]=4;
$1=_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.MessageSend)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09aStream\x0a\x09\x09nextPutAll: '(';\x0a\x09\x09nextPutAll: self receiver;\x0a\x09\x09nextPutAll: ' >> ';\x0a\x09\x09nextPutAll: self selector;\x0a\x09\x09nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "receiver", "selector"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@receiver"];
return $1;
},
args: [],
source: "receiver\x0a\x09^ receiver",
messageSends: [],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@receiver"]=anObject;
return self},
args: ["anObject"],
source: "receiver: anObject\x0a\x09receiver := anObject",
messageSends: [],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@message"])._selector();
return $1;
}, function($ctx1) {$ctx1.fill(self,"selector",{},globals.MessageSend)})},
args: [],
source: "selector\x0a\x09^ message selector",
messageSends: ["selector"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@message"])._selector_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"selector:",{aString:aString},globals.MessageSend)})},
args: ["aString"],
source: "selector: aString\x0a\x09message selector: aString",
messageSends: ["selector:"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'evaluating',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@message"])._sendTo_(self._receiver());
return $1;
}, function($ctx1) {$ctx1.fill(self,"value",{},globals.MessageSend)})},
args: [],
source: "value\x0a\x09^ message sendTo: self receiver",
messageSends: ["sendTo:", "receiver"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'evaluating',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self["@message"];
_st($2)._arguments_([anObject]);
$3=_st($2)._sendTo_(self._receiver());
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"value:",{anObject:anObject},globals.MessageSend)})},
args: ["anObject"],
source: "value: anObject\x0a\x09^ message \x0a\x09\x09arguments: { anObject };\x0a\x09\x09sendTo: self receiver",
messageSends: ["arguments:", "sendTo:", "receiver"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "value:value:",
protocol: 'evaluating',
fn: function (firstArgument,secondArgument){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self["@message"];
_st($2)._arguments_([firstArgument,secondArgument]);
$3=_st($2)._sendTo_(self._receiver());
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"value:value:",{firstArgument:firstArgument,secondArgument:secondArgument},globals.MessageSend)})},
args: ["firstArgument", "secondArgument"],
source: "value: firstArgument value: secondArgument\x0a\x09^ message \x0a\x09\x09arguments: { firstArgument. secondArgument };\x0a\x09\x09sendTo: self receiver",
messageSends: ["arguments:", "sendTo:", "receiver"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "value:value:value:",
protocol: 'evaluating',
fn: function (firstArgument,secondArgument,thirdArgument){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self["@message"];
_st($2)._arguments_([firstArgument,secondArgument,thirdArgument]);
$3=_st($2)._sendTo_(self._receiver());
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"value:value:value:",{firstArgument:firstArgument,secondArgument:secondArgument,thirdArgument:thirdArgument},globals.MessageSend)})},
args: ["firstArgument", "secondArgument", "thirdArgument"],
source: "value: firstArgument value: secondArgument value: thirdArgument\x0a\x09^ message \x0a\x09\x09arguments: { firstArgument. secondArgument. thirdArgument };\x0a\x09\x09sendTo: self receiver",
messageSends: ["arguments:", "sendTo:", "receiver"],
referencedClasses: []
}),
globals.MessageSend);

smalltalk.addMethod(
smalltalk.method({
selector: "valueWithPossibleArguments:",
protocol: 'evaluating',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._arguments_(aCollection);
$1=self._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"valueWithPossibleArguments:",{aCollection:aCollection},globals.MessageSend)})},
args: ["aCollection"],
source: "valueWithPossibleArguments: aCollection\x0a\x09self arguments: aCollection.\x0a\x09^ self value",
messageSends: ["arguments:", "value"],
referencedClasses: []
}),
globals.MessageSend);



smalltalk.addClass('MethodContext', globals.Object, [], 'Kernel-Methods');
globals.MethodContext.comment="I hold all the dynamic state associated with the execution of either a method activation resulting from a message send. I am used to build the call stack while debugging.\x0a\x0aMy instances are JavaScript `SmalltalkMethodContext` objects defined in `boot.js`.";
smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._isBlockContext();
if(smalltalk.assert($2)){
$3="a block (in ".__comma(_st(self._methodContext())._asString());
$ctx1.sendIdx[","]=2;
$1=_st($3).__comma(")");
$ctx1.sendIdx[","]=1;
} else {
$1=_st(_st(_st(_st(self._receiver())._class())._name()).__comma(" >> ")).__comma(self._selector());
$ctx1.sendIdx[","]=3;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"asString",{},globals.MethodContext)})},
args: [],
source: "asString\x0a\x09^ self isBlockContext\x0a\x09\x09ifTrue: [ 'a block (in ', self methodContext asString, ')' ]\x0a\x09\x09ifFalse: [ self receiver class name, ' >> ', self selector ]",
messageSends: ["ifTrue:ifFalse:", "isBlockContext", ",", "asString", "methodContext", "name", "class", "receiver", "selector"],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "evaluatedSelector",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.evaluatedSelector;
return self}, function($ctx1) {$ctx1.fill(self,"evaluatedSelector",{},globals.MethodContext)})},
args: [],
source: "evaluatedSelector\x0a\x09<return self.evaluatedSelector>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "home",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.homeContext;
return self}, function($ctx1) {$ctx1.fill(self,"home",{},globals.MethodContext)})},
args: [],
source: "home\x0a\x09<return self.homeContext>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "index",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.index || 0;
return self}, function($ctx1) {$ctx1.fill(self,"index",{},globals.MethodContext)})},
args: [],
source: "index\x0a\x09<return self.index || 0>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockContext",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._selector())._isNil();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isBlockContext",{},globals.MethodContext)})},
args: [],
source: "isBlockContext\x0a\x09\x22Block context do not have selectors.\x22\x0a\x09\x0a\x09^ self selector isNil",
messageSends: ["isNil", "selector"],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "locals",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.locals || {};
return self}, function($ctx1) {$ctx1.fill(self,"locals",{},globals.MethodContext)})},
args: [],
source: "locals\x0a\x09<return self.locals || {}>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$5,$4,$3,$1;
$2=self._methodContext();
$ctx1.sendIdx["methodContext"]=1;
if(($receiver = $2) == nil || $receiver == null){
$1=$2;
} else {
$5=self._methodContext();
$ctx1.sendIdx["methodContext"]=2;
$4=_st($5)._receiver();
$3=_st($4)._class();
$1=_st($3)._lookupSelector_(_st(self._methodContext())._selector());
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"method",{},globals.MethodContext)})},
args: [],
source: "method\x0a\x09^ self methodContext ifNotNil: [\x0a\x09\x09self methodContext receiver class lookupSelector: self methodContext selector ]",
messageSends: ["ifNotNil:", "methodContext", "lookupSelector:", "class", "receiver", "selector"],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "methodContext",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$2;
$1=self._isBlockContext();
if(! smalltalk.assert($1)){
return self;
};
$3=self._outerContext();
if(($receiver = $3) == nil || $receiver == null){
$2=$3;
} else {
var outer;
outer=$receiver;
$2=_st(outer)._methodContext();
};
return $2;
}, function($ctx1) {$ctx1.fill(self,"methodContext",{},globals.MethodContext)})},
args: [],
source: "methodContext\x0a\x09self isBlockContext ifFalse: [ ^ self ].\x0a\x09\x0a\x09^ self outerContext ifNotNil: [ :outer |\x0a\x09\x09outer methodContext ]",
messageSends: ["ifFalse:", "isBlockContext", "ifNotNil:", "outerContext", "methodContext"],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "outerContext",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.outerContext || self.homeContext;
return self}, function($ctx1) {$ctx1.fill(self,"outerContext",{},globals.MethodContext)})},
args: [],
source: "outerContext\x0a\x09<return self.outerContext || self.homeContext>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
globals.MethodContext.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
_st(aStream)._nextPutAll_("(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aStream)._nextPutAll_(self._asString());
$ctx1.sendIdx["nextPutAll:"]=2;
$1=_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.MethodContext)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09aStream \x0a\x09\x09nextPutAll: '(';\x0a\x09\x09nextPutAll: self asString;\x0a\x09\x09nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "asString"],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.receiver;
return self}, function($ctx1) {$ctx1.fill(self,"receiver",{},globals.MethodContext)})},
args: [],
source: "receiver\x0a\x09<return self.receiver>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if(self.selector) {
			return smalltalk.convertSelector(self.selector);
		} else {
			return nil;
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"selector",{},globals.MethodContext)})},
args: [],
source: "selector\x0a\x09<\x0a\x09\x09if(self.selector) {\x0a\x09\x09\x09return smalltalk.convertSelector(self.selector);\x0a\x09\x09} else {\x0a\x09\x09\x09return nil;\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "sendIndexAt:",
protocol: 'accessing',
fn: function (aSelector){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.sendIdx[aSelector] || 0;
return self}, function($ctx1) {$ctx1.fill(self,"sendIndexAt:",{aSelector:aSelector},globals.MethodContext)})},
args: ["aSelector"],
source: "sendIndexAt: aSelector\x0a\x09<return self.sendIdx[aSelector] || 0>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "sendIndexes",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.sendIdx;
return self}, function($ctx1) {$ctx1.fill(self,"sendIndexes",{},globals.MethodContext)})},
args: [],
source: "sendIndexes\x0a\x09<return self.sendIdx>",
messageSends: [],
referencedClasses: []
}),
globals.MethodContext);

smalltalk.addMethod(
smalltalk.method({
selector: "temps",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=self._locals();
return $1;
}, function($ctx1) {$ctx1.fill(self,"temps",{},globals.MethodContext)})},
args: [],
source: "temps\x0a\x09self deprecatedAPI.\x0a\x09\x0a\x09^ self locals",
messageSends: ["deprecatedAPI", "locals"],
referencedClasses: []
}),
globals.MethodContext);



smalltalk.addClass('NativeFunction', globals.Object, [], 'Kernel-Methods');
globals.NativeFunction.comment="I am a wrapper around native functions, such as `WebSocket`.\x0aFor 'normal' functions (whose constructor is the JavaScript `Function` object), use `BlockClosure`.\x0a\x0a## API\x0a\x0aSee the class-side `instance creation` methods for instance creation.\x0a\x0aCreated instances will most probably be instance of `JSObjectProxy`.\x0a\x0a## Usage example:\x0a\x0a\x09| ws |\x0a\x09ws := NativeFunction constructor: 'WebSocket' value: 'ws://localhost'.\x0a\x09ws at: 'onopen' put: [ ws send: 'hey there from Amber' ]";

smalltalk.addMethod(
smalltalk.method({
selector: "constructor:",
protocol: 'instance creation',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var native=eval(aString);
		return new native();
	;
return self}, function($ctx1) {$ctx1.fill(self,"constructor:",{aString:aString},globals.NativeFunction.klass)})},
args: ["aString"],
source: "constructor: aString\x0a\x09<\x0a\x09\x09var native=eval(aString);\x0a\x09\x09return new native();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.NativeFunction.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "constructor:value:",
protocol: 'instance creation',
fn: function (aString,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var native=eval(aString);
		return new native(anObject);
	;
return self}, function($ctx1) {$ctx1.fill(self,"constructor:value:",{aString:aString,anObject:anObject},globals.NativeFunction.klass)})},
args: ["aString", "anObject"],
source: "constructor: aString value:anObject\x0a\x09<\x0a\x09\x09var native=eval(aString);\x0a\x09\x09return new native(anObject);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.NativeFunction.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "constructor:value:value:",
protocol: 'instance creation',
fn: function (aString,anObject,anObject2){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var native=eval(aString);
		return new native(anObject,anObject2);
	;
return self}, function($ctx1) {$ctx1.fill(self,"constructor:value:value:",{aString:aString,anObject:anObject,anObject2:anObject2},globals.NativeFunction.klass)})},
args: ["aString", "anObject", "anObject2"],
source: "constructor: aString value:anObject value: anObject2\x0a\x09<\x0a\x09\x09var native=eval(aString);\x0a\x09\x09return new native(anObject,anObject2);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.NativeFunction.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "constructor:value:value:value:",
protocol: 'instance creation',
fn: function (aString,anObject,anObject2,anObject3){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var native=eval(aString);
		return new native(anObject,anObject2, anObject3);
	;
return self}, function($ctx1) {$ctx1.fill(self,"constructor:value:value:value:",{aString:aString,anObject:anObject,anObject2:anObject2,anObject3:anObject3},globals.NativeFunction.klass)})},
args: ["aString", "anObject", "anObject2", "anObject3"],
source: "constructor: aString value:anObject value: anObject2 value:anObject3\x0a\x09<\x0a\x09\x09var native=eval(aString);\x0a\x09\x09return new native(anObject,anObject2, anObject3);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.NativeFunction.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "exists:",
protocol: 'testing',
fn: function (aString){
var self=this;
function $PlatformInterface(){return globals.PlatformInterface||(typeof PlatformInterface=="undefined"?nil:PlatformInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($PlatformInterface())._existsGlobal_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"exists:",{aString:aString},globals.NativeFunction.klass)})},
args: ["aString"],
source: "exists: aString\x0a\x09^ PlatformInterface existsGlobal: aString",
messageSends: ["existsGlobal:"],
referencedClasses: ["PlatformInterface"]
}),
globals.NativeFunction.klass);


smalltalk.addClass('Timeout', globals.Object, ['rawTimeout'], 'Kernel-Methods');
globals.Timeout.comment="I am wrapping the returns from `set{Timeout,Interval}`.\x0a\x0a## Motivation\x0a\x0aNumber suffices in browsers, but node.js returns an object.";
smalltalk.addMethod(
smalltalk.method({
selector: "clearInterval",
protocol: 'timeout/interval',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var interval = self["@rawTimeout"];
		clearInterval(interval);
	;
return self}, function($ctx1) {$ctx1.fill(self,"clearInterval",{},globals.Timeout)})},
args: [],
source: "clearInterval\x0a\x09<\x0a\x09\x09var interval = self[\x22@rawTimeout\x22];\x0a\x09\x09clearInterval(interval);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Timeout);

smalltalk.addMethod(
smalltalk.method({
selector: "clearTimeout",
protocol: 'timeout/interval',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var timeout = self["@rawTimeout"];
		clearTimeout(timeout);
	;
return self}, function($ctx1) {$ctx1.fill(self,"clearTimeout",{},globals.Timeout)})},
args: [],
source: "clearTimeout\x0a\x09<\x0a\x09\x09var timeout = self[\x22@rawTimeout\x22];\x0a\x09\x09clearTimeout(timeout);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Timeout);

smalltalk.addMethod(
smalltalk.method({
selector: "rawTimeout:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@rawTimeout"]=anObject;
return self},
args: ["anObject"],
source: "rawTimeout: anObject\x0a\x09rawTimeout := anObject",
messageSends: [],
referencedClasses: []
}),
globals.Timeout);


smalltalk.addMethod(
smalltalk.method({
selector: "on:",
protocol: 'instance creation',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._rawTimeout_(anObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:",{anObject:anObject},globals.Timeout.klass)})},
args: ["anObject"],
source: "on: anObject\x0a\x09^ self new rawTimeout: anObject; yourself",
messageSends: ["rawTimeout:", "new", "yourself"],
referencedClasses: []
}),
globals.Timeout.klass);

});

define("amber_core/Kernel-Collections", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Collections');
smalltalk.packages["Kernel-Collections"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('Association', globals.Object, ['key', 'value'], 'Kernel-Collections');
globals.Association.comment="I represent a pair of associated objects, a key and a value. My instances can serve as entries in a dictionary.\x0a\x0aInstances can be created with the class-side method `#key:value:`";
smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (anAssociation){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$5,$4,$6,$1;
$3=self._class();
$ctx1.sendIdx["class"]=1;
$2=_st($3).__eq(_st(anAssociation)._class());
$ctx1.sendIdx["="]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$5=self._key();
$ctx2.sendIdx["key"]=1;
$4=_st($5).__eq(_st(anAssociation)._key());
$ctx2.sendIdx["="]=2;
return _st($4)._and_((function(){
return smalltalk.withContext(function($ctx3) {
$6=self._value();
$ctx3.sendIdx["value"]=1;
return _st($6).__eq(_st(anAssociation)._value());
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$ctx1.sendIdx["and:"]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"=",{anAssociation:anAssociation},globals.Association)})},
args: ["anAssociation"],
source: "= anAssociation\x0a\x09^ self class = anAssociation class and: [\x0a\x09\x09self key = anAssociation key and: [\x0a\x09\x09self value = anAssociation value ]]",
messageSends: ["and:", "=", "class", "key", "value"],
referencedClasses: []
}),
globals.Association);

smalltalk.addMethod(
smalltalk.method({
selector: "key",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@key"];
return $1;
},
args: [],
source: "key\x0a\x09^ key",
messageSends: [],
referencedClasses: []
}),
globals.Association);

smalltalk.addMethod(
smalltalk.method({
selector: "key:",
protocol: 'accessing',
fn: function (aKey){
var self=this;
self["@key"]=aKey;
return self},
args: ["aKey"],
source: "key: aKey\x0a\x09key := aKey",
messageSends: [],
referencedClasses: []
}),
globals.Association);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._key())._printOn_(aStream);
$ctx1.sendIdx["printOn:"]=1;
_st(aStream)._nextPutAll_(" -> ");
_st(self._value())._printOn_(aStream);
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Association)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09self key printOn: aStream.\x0a\x09aStream nextPutAll: ' -> '.\x0a\x09self value printOn: aStream",
messageSends: ["printOn:", "key", "nextPutAll:", "value"],
referencedClasses: []
}),
globals.Association);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@value"];
return $1;
},
args: [],
source: "value\x0a\x09^ value",
messageSends: [],
referencedClasses: []
}),
globals.Association);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'accessing',
fn: function (aValue){
var self=this;
self["@value"]=aValue;
return self},
args: ["aValue"],
source: "value: aValue\x0a\x09value := aValue",
messageSends: [],
referencedClasses: []
}),
globals.Association);


smalltalk.addMethod(
smalltalk.method({
selector: "key:value:",
protocol: 'instance creation',
fn: function (aKey,aValue){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._key_(aKey);
_st($2)._value_(aValue);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"key:value:",{aKey:aKey,aValue:aValue},globals.Association.klass)})},
args: ["aKey", "aValue"],
source: "key: aKey value: aValue\x0a\x09\x09^ self new\x0a\x09\x09key: aKey;\x0a\x09\x09value: aValue;\x0a\x09\x09yourself",
messageSends: ["key:", "new", "value:", "yourself"],
referencedClasses: []
}),
globals.Association.klass);


smalltalk.addClass('BucketStore', globals.Object, ['buckets', 'hashBlock'], 'Kernel-Collections');
globals.BucketStore.comment="I am an helper class for hash-based stores.\x0a\x0aI hold buckets which are selected by a hash, specified using `#hashBlock:`.\x0aThe hash can be any object, and\x0ait is used as a JS property (that is, in ES5\x0aits toString() value counts).\x0a\x0a## API\x0aI maintain a list of buckets. Client code can use this API:\x0a - `#bucketOfElement:` (to ask a bucket for element, I can return JS null if n/a)\x0a - `#do:` (to enumerate all elements of all buckets)\x0a - `#removeAll` (to remove all buckets)\x0a\x0aClient code itself should add/remove elements\x0ain a bucket. The `nil` object should not be put into any bucket.\x0a\x0aTypes of buckets are the responsibility of subclasses via `#newBucket`.";
smalltalk.addMethod(
smalltalk.method({
selector: "bucketOfElement:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var hash = self['@hashBlock'](anObject);
		if (!hash) return null;
		var buckets = self['@buckets'],
			bucket = buckets[hash];
		if (!bucket) { bucket = buckets[hash] = self._newBucket(); }
		return bucket;
	;
return self}, function($ctx1) {$ctx1.fill(self,"bucketOfElement:",{anObject:anObject},globals.BucketStore)})},
args: ["anObject"],
source: "bucketOfElement: anObject\x0a\x09<\x0a\x09\x09var hash = self['@hashBlock'](anObject);\x0a\x09\x09if (!hash) return null;\x0a\x09\x09var buckets = self['@buckets'],\x0a\x09\x09\x09bucket = buckets[hash];\x0a\x09\x09if (!bucket) { bucket = buckets[hash] = self._newBucket(); }\x0a\x09\x09return bucket;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BucketStore);

smalltalk.addMethod(
smalltalk.method({
selector: "do:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var buckets = self['@buckets'];
		var keys = Object.keys(buckets);
		for (var i = 0; i < keys.length; ++i) { buckets[keys[i]]._do_(aBlock); }
	;
return self}, function($ctx1) {$ctx1.fill(self,"do:",{aBlock:aBlock},globals.BucketStore)})},
args: ["aBlock"],
source: "do: aBlock\x0a\x09<\x0a\x09\x09var buckets = self['@buckets'];\x0a\x09\x09var keys = Object.keys(buckets);\x0a\x09\x09for (var i = 0; i < keys.length; ++i) { buckets[keys[i]]._do_(aBlock); }\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.BucketStore);

smalltalk.addMethod(
smalltalk.method({
selector: "hashBlock:",
protocol: 'accessing',
fn: function (aBlock){
var self=this;
self["@hashBlock"]=aBlock;
return self},
args: ["aBlock"],
source: "hashBlock: aBlock\x0a\x09hashBlock := aBlock",
messageSends: [],
referencedClasses: []
}),
globals.BucketStore);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.BucketStore.superclass.fn.prototype._initialize.apply(_st(self), []);
self._removeAll();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.BucketStore)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09self removeAll",
messageSends: ["initialize", "removeAll"],
referencedClasses: []
}),
globals.BucketStore);

smalltalk.addMethod(
smalltalk.method({
selector: "newBucket",
protocol: 'private',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"newBucket",{},globals.BucketStore)})},
args: [],
source: "newBucket\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.BucketStore);

smalltalk.addMethod(
smalltalk.method({
selector: "removeAll",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self['@buckets'] = Object.create(null);;
return self}, function($ctx1) {$ctx1.fill(self,"removeAll",{},globals.BucketStore)})},
args: [],
source: "removeAll\x0a\x09<self['@buckets'] = Object.create(null);>",
messageSends: [],
referencedClasses: []
}),
globals.BucketStore);


smalltalk.addMethod(
smalltalk.method({
selector: "hashBlock:",
protocol: 'instance creation',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._hashBlock_(aBlock);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"hashBlock:",{aBlock:aBlock},globals.BucketStore.klass)})},
args: ["aBlock"],
source: "hashBlock: aBlock\x0a\x09^ self new\x0a\x09\x09hashBlock: aBlock;\x0a\x09\x09yourself",
messageSends: ["hashBlock:", "new", "yourself"],
referencedClasses: []
}),
globals.BucketStore.klass);


smalltalk.addClass('ArrayBucketStore', globals.BucketStore, [], 'Kernel-Collections');
globals.ArrayBucketStore.comment="I am a concrete `BucketStore` with buckets being instance of `Array`.";
smalltalk.addMethod(
smalltalk.method({
selector: "newBucket",
protocol: 'private',
fn: function (){
var self=this;
var $1;
$1=[];
return $1;
},
args: [],
source: "newBucket\x0a\x09^ #()",
messageSends: [],
referencedClasses: []
}),
globals.ArrayBucketStore);



smalltalk.addClass('Collection', globals.Object, [], 'Kernel-Collections');
globals.Collection.comment="I am the abstract superclass of all classes that represent a group of elements.\x0a\x0aI provide a set of useful methods to the Collection hierarchy such as enumerating and converting methods.";
smalltalk.addMethod(
smalltalk.method({
selector: ",",
protocol: 'copying',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._copy();
_st($2)._addAll_(aCollection);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,",",{aCollection:aCollection},globals.Collection)})},
args: ["aCollection"],
source: ", aCollection\x0a\x09^ self copy\x0a\x09\x09addAll: aCollection;\x0a\x09\x09yourself",
messageSends: ["addAll:", "copy", "yourself"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "add:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"add:",{anObject:anObject},globals.Collection)})},
args: ["anObject"],
source: "add: anObject\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "addAll:",
protocol: 'adding/removing',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return aCollection;
}, function($ctx1) {$ctx1.fill(self,"addAll:",{aCollection:aCollection},globals.Collection)})},
args: ["aCollection"],
source: "addAll: aCollection\x0a\x09aCollection do: [ :each |\x0a\x09\x09self add: each ].\x0a\x09^ aCollection",
messageSends: ["do:", "add:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "allSatisfy:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
var $early={};
try {
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(each);
if(! smalltalk.assert($1)){
throw $early=[false];
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return true;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"allSatisfy:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "allSatisfy: aBlock\x0a\x09\x22Evaluate aBlock with the elements of the receiver.\x0a\x09If aBlock returns false for any element return false.\x0a\x09Otherwise return true.\x22\x0a\x0a\x09self do: [ :each | (aBlock value: each) ifFalse: [ ^ false ] ].\x0a\x09^ true",
messageSends: ["do:", "ifFalse:", "value:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "anyOne",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $early={};
try {
self._ifEmpty_((function(){
return smalltalk.withContext(function($ctx2) {
return self._error_("Collection is empty");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
self._do_((function(each){
throw $early=[each];
}));
return self}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"anyOne",{},globals.Collection)})},
args: [],
source: "anyOne\x0a\x09\x22Answer a representative sample of the receiver. This method can\x0a\x09be helpful when needing to preinfer the nature of the contents of \x0a\x09semi-homogeneous collections.\x22\x0a\x0a\x09self ifEmpty: [ self error: 'Collection is empty' ].\x0a\x09self do: [ :each | ^ each ]",
messageSends: ["ifEmpty:", "error:", "do:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "anySatisfy:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
var $early={};
try {
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(each);
if(smalltalk.assert($1)){
throw $early=[true];
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return false;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"anySatisfy:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "anySatisfy: aBlock\x0a\x09\x22Evaluate aBlock with the elements of the receiver.\x0a\x09If aBlock returns true for any element return true.\x0a\x09Otherwise return false.\x22\x0a\x0a\x09self do: [ :each | (aBlock value: each) ifTrue: [ ^ true ] ].\x0a\x09^ false",
messageSends: ["do:", "ifTrue:", "value:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "asArray",
protocol: 'converting',
fn: function (){
var self=this;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Array())._withAll_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"asArray",{},globals.Collection)})},
args: [],
source: "asArray\x0a\x09^ Array withAll: self",
messageSends: ["withAll:"],
referencedClasses: ["Array"]
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._asArray())._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._asJSON();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJSON",{},globals.Collection)})},
args: [],
source: "asJSON\x0a\x09^ self asArray collect: [ :each | each asJSON ]",
messageSends: ["collect:", "asArray", "asJSON"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "asOrderedCollection",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._asArray();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asOrderedCollection",{},globals.Collection)})},
args: [],
source: "asOrderedCollection\x0a\x09^ self asArray",
messageSends: ["asArray"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "asSet",
protocol: 'converting',
fn: function (){
var self=this;
function $Set(){return globals.Set||(typeof Set=="undefined"?nil:Set)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Set())._withAll_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"asSet",{},globals.Collection)})},
args: [],
source: "asSet\x0a\x09^ Set withAll: self",
messageSends: ["withAll:"],
referencedClasses: ["Set"]
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "collect:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var stream;
return smalltalk.withContext(function($ctx1) { 
var $1;
stream=_st(_st(self._class())._new())._writeStream();
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(stream)._nextPut_(_st(aBlock)._value_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=_st(stream)._contents();
return $1;
}, function($ctx1) {$ctx1.fill(self,"collect:",{aBlock:aBlock,stream:stream},globals.Collection)})},
args: ["aBlock"],
source: "collect: aBlock\x0a\x09| stream |\x0a\x09stream := self class new writeStream.\x0a\x09self do: [ :each |\x0a\x09\x09stream nextPut: (aBlock value: each) ].\x0a\x09^ stream contents",
messageSends: ["writeStream", "new", "class", "do:", "nextPut:", "value:", "contents"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "contains:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=self._anySatisfy_(aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"contains:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "contains: aBlock\x0a\x09self deprecatedAPI.\x0a\x0a\x09^ self anySatisfy: aBlock",
messageSends: ["deprecatedAPI", "anySatisfy:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "copyWith:",
protocol: 'copying',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._copy();
_st($2)._add_(anObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"copyWith:",{anObject:anObject},globals.Collection)})},
args: ["anObject"],
source: "copyWith: anObject\x0a\x09^ self copy add: anObject; yourself",
messageSends: ["add:", "copy", "yourself"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "copyWithAll:",
protocol: 'copying',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._copy();
_st($2)._addAll_(aCollection);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"copyWithAll:",{aCollection:aCollection},globals.Collection)})},
args: ["aCollection"],
source: "copyWithAll: aCollection\x0a\x09^ self copy addAll: aCollection; yourself",
messageSends: ["addAll:", "copy", "yourself"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "copyWithoutAll:",
protocol: 'copying',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._reject_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(aCollection)._includes_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"copyWithoutAll:",{aCollection:aCollection},globals.Collection)})},
args: ["aCollection"],
source: "copyWithoutAll: aCollection\x0a\x09\x22Answer a copy of the receiver that does not contain any elements\x0a\x09equal to those in aCollection.\x22\x0a\x0a\x09^ self reject: [ :each | aCollection includes: each ]",
messageSends: ["reject:", "includes:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "detect:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._detect_ifNone_(aBlock,(function(){
return smalltalk.withContext(function($ctx2) {
return self._errorNotFound();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"detect:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "detect: aBlock\x0a\x09^ self detect: aBlock ifNone: [ self errorNotFound ]",
messageSends: ["detect:ifNone:", "errorNotFound"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "detect:ifNone:",
protocol: 'enumerating',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"detect:ifNone:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Collection)})},
args: ["aBlock", "anotherBlock"],
source: "detect: aBlock ifNone: anotherBlock\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "do:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"do:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "do: aBlock\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "do:separatedBy:",
protocol: 'enumerating',
fn: function (aBlock,anotherBlock){
var self=this;
var actionBeforeElement;
return smalltalk.withContext(function($ctx1) { 
actionBeforeElement=(function(){
actionBeforeElement=anotherBlock;
return actionBeforeElement;
});
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
_st(actionBeforeElement)._value();
return _st(aBlock)._value_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"do:separatedBy:",{aBlock:aBlock,anotherBlock:anotherBlock,actionBeforeElement:actionBeforeElement},globals.Collection)})},
args: ["aBlock", "anotherBlock"],
source: "do: aBlock separatedBy: anotherBlock\x0a\x09| actionBeforeElement |\x0a\x09actionBeforeElement := [ actionBeforeElement := anotherBlock ].\x0a\x09self do: [ :each |\x0a\x09\x09actionBeforeElement value.\x0a\x09\x09aBlock value: each ]",
messageSends: ["do:", "value", "value:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "errorNotFound",
protocol: 'error handling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("Object is not in the collection");
return self}, function($ctx1) {$ctx1.fill(self,"errorNotFound",{},globals.Collection)})},
args: [],
source: "errorNotFound\x0a\x09self error: 'Object is not in the collection'",
messageSends: ["error:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "ifEmpty:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._isEmpty();
$1=_st($2)._ifTrue_ifFalse_(aBlock,(function(){
return self;
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifEmpty:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "ifEmpty: aBlock\x0a\x09\x22Evaluate the given block with the receiver as argument, answering its value if the receiver is empty, otherwise answer the receiver. \x0a\x09Note that the fact that this method returns its argument in case the receiver is not empty allows one to write expressions like the following ones: \x0a\x09\x09self classifyMethodAs:\x0a\x09\x09\x09(myProtocol ifEmpty: ['As yet unclassified'])\x22\x0a\x09^ self isEmpty\x0a\x09\x09ifTrue: aBlock\x0a\x09\x09ifFalse: [ self ]",
messageSends: ["ifTrue:ifFalse:", "isEmpty"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "ifEmpty:ifNotEmpty:",
protocol: 'testing',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._isEmpty();
$1=_st($2)._ifTrue_ifFalse_(aBlock,anotherBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifEmpty:ifNotEmpty:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Collection)})},
args: ["aBlock", "anotherBlock"],
source: "ifEmpty: aBlock ifNotEmpty: anotherBlock\x0a\x09^ self isEmpty\x0a\x09\x09ifTrue: aBlock\x0a\x09\x09ifFalse: anotherBlock",
messageSends: ["ifTrue:ifFalse:", "isEmpty"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNotEmpty:",
protocol: 'testing',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._notEmpty();
$1=_st($2)._ifTrue_ifFalse_(aBlock,(function(){
return self;
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNotEmpty:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "ifNotEmpty: aBlock\x0a\x09^ self notEmpty\x0a\x09\x09ifTrue: aBlock\x0a\x09\x09ifFalse: [ self ]",
messageSends: ["ifTrue:ifFalse:", "notEmpty"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNotEmpty:ifEmpty:",
protocol: 'testing',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._notEmpty();
$1=_st($2)._ifTrue_ifFalse_(aBlock,anotherBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifNotEmpty:ifEmpty:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Collection)})},
args: ["aBlock", "anotherBlock"],
source: "ifNotEmpty: aBlock ifEmpty: anotherBlock\x0a\x09^ self notEmpty\x0a\x09\x09ifTrue: aBlock\x0a\x09\x09ifFalse: anotherBlock",
messageSends: ["ifTrue:ifFalse:", "notEmpty"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "includes:",
protocol: 'testing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._anySatisfy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each).__eq(anObject);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"includes:",{anObject:anObject},globals.Collection)})},
args: ["anObject"],
source: "includes: anObject\x0a\x09^ self anySatisfy: [ :each | each = anObject ]",
messageSends: ["anySatisfy:", "="],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "inject:into:",
protocol: 'enumerating',
fn: function (anObject,aBlock){
var self=this;
var result;
return smalltalk.withContext(function($ctx1) { 
var $1;
result=anObject;
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
result=_st(aBlock)._value_value_(result,each);
return result;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=result;
return $1;
}, function($ctx1) {$ctx1.fill(self,"inject:into:",{anObject:anObject,aBlock:aBlock,result:result},globals.Collection)})},
args: ["anObject", "aBlock"],
source: "inject: anObject into: aBlock\x0a\x09| result |\x0a\x09result := anObject.\x0a\x09self do: [ :each |\x0a\x09\x09result := aBlock value: result value: each ].\x0a\x09^ result",
messageSends: ["do:", "value:value:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "intersection:",
protocol: 'enumerating',
fn: function (aCollection){
var self=this;
var set,outputSet;
function $Set(){return globals.Set||(typeof Set=="undefined"?nil:Set)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3;
set=self._asSet();
outputSet=_st($Set())._new();
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$2=_st(set)._includes_(each);
$ctx2.sendIdx["includes:"]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx3) {
return _st(_st(outputSet)._includes_(each))._not();
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
if(smalltalk.assert($1)){
return _st(outputSet)._add_(each);
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$3=_st(self._class())._withAll_(_st(outputSet)._asArray());
return $3;
}, function($ctx1) {$ctx1.fill(self,"intersection:",{aCollection:aCollection,set:set,outputSet:outputSet},globals.Collection)})},
args: ["aCollection"],
source: "intersection: aCollection\x0a\x09\x22Answer the set theoretic intersection of two collections.\x22\x0a\x0a\x09| set outputSet |\x0a\x09\x0a\x09set := self asSet.\x0a\x09outputSet := Set new.\x0a\x09\x0a\x09aCollection do: [ :each |\x0a\x09\x09((set includes: each) and: [ (outputSet includes: each) not ])\x0a\x09\x09\x09ifTrue: [\x0a\x09\x09\x09\x09outputSet add: each ]].\x0a\x09\x09\x0a\x09^ self class withAll: outputSet asArray",
messageSends: ["asSet", "new", "do:", "ifTrue:", "and:", "includes:", "not", "add:", "withAll:", "class", "asArray"],
referencedClasses: ["Set"]
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "isEmpty",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._size()).__eq((0));
return $1;
}, function($ctx1) {$ctx1.fill(self,"isEmpty",{},globals.Collection)})},
args: [],
source: "isEmpty\x0a\x09^ self size = 0",
messageSends: ["=", "size"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "noneSatisfy:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
var $early={};
try {
self._do_((function(item){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(item);
if(smalltalk.assert($1)){
throw $early=[false];
};
}, function($ctx2) {$ctx2.fillBlock({item:item},$ctx1,1)})}));
return true;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"noneSatisfy:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "noneSatisfy: aBlock\x0a\x09\x22Evaluate aBlock with the elements of the receiver.\x0a\x09If aBlock returns false for all elements return true.\x0a\x09Otherwise return false\x22\x0a\x0a\x09self do: [ :item | (aBlock value: item) ifTrue: [ ^ false ] ].\x0a\x09^ true",
messageSends: ["do:", "ifTrue:", "value:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "notEmpty",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._isEmpty())._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"notEmpty",{},globals.Collection)})},
args: [],
source: "notEmpty\x0a\x09^ self isEmpty not",
messageSends: ["not", "isEmpty"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "occurrencesOf:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
var tally;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
tally=(0);
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(anObject).__eq(each);
if(smalltalk.assert($1)){
tally=_st(tally).__plus((1));
return tally;
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=tally;
return $2;
}, function($ctx1) {$ctx1.fill(self,"occurrencesOf:",{anObject:anObject,tally:tally},globals.Collection)})},
args: ["anObject"],
source: "occurrencesOf: anObject\x0a\x09\x22Answer how many of the receiver's elements are equal to anObject.\x22\x0a\x0a\x09| tally |\x0a\x09tally := 0.\x0a\x09self do: [ :each | anObject = each ifTrue: [ tally := tally + 1 ]].\x0a\x09^ tally",
messageSends: ["do:", "ifTrue:", "=", "+"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "putOn:",
protocol: 'streaming',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._putOn_(aStream);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"putOn:",{aStream:aStream},globals.Collection)})},
args: ["aStream"],
source: "putOn: aStream\x0a\x09self do: [ :each | each putOn: aStream ]",
messageSends: ["do:", "putOn:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "reject:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._select_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(aBlock)._value_(each)).__eq(false);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"reject:",{aBlock:aBlock},globals.Collection)})},
args: ["aBlock"],
source: "reject: aBlock\x0a\x09^ self select: [ :each | (aBlock value: each) = false ]",
messageSends: ["select:", "=", "value:"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._remove_ifAbsent_(anObject,(function(){
return smalltalk.withContext(function($ctx2) {
return self._errorNotFound();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"remove:",{anObject:anObject},globals.Collection)})},
args: ["anObject"],
source: "remove: anObject\x0a\x09^ self remove: anObject ifAbsent: [ self errorNotFound ]",
messageSends: ["remove:ifAbsent:", "errorNotFound"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:ifAbsent:",
protocol: 'adding/removing',
fn: function (anObject,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"remove:ifAbsent:",{anObject:anObject,aBlock:aBlock},globals.Collection)})},
args: ["anObject", "aBlock"],
source: "remove: anObject ifAbsent: aBlock\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "removeAll",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"removeAll",{},globals.Collection)})},
args: [],
source: "removeAll\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "select:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var stream;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
stream=_st(_st(self._class())._new())._writeStream();
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(each);
if(smalltalk.assert($1)){
return _st(stream)._nextPut_(each);
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=_st(stream)._contents();
return $2;
}, function($ctx1) {$ctx1.fill(self,"select:",{aBlock:aBlock,stream:stream},globals.Collection)})},
args: ["aBlock"],
source: "select: aBlock\x0a\x09| stream |\x0a\x09stream := self class new writeStream.\x0a\x09self do: [ :each |\x0a\x09\x09(aBlock value: each) ifTrue: [\x0a\x09\x09stream nextPut: each ] ].\x0a\x09^ stream contents",
messageSends: ["writeStream", "new", "class", "do:", "ifTrue:", "value:", "nextPut:", "contents"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "select:thenCollect:",
protocol: 'enumerating',
fn: function (selectBlock,collectBlock){
var self=this;
var stream;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
stream=_st(_st(self._class())._new())._writeStream();
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(selectBlock)._value_(each);
$ctx2.sendIdx["value:"]=1;
if(smalltalk.assert($1)){
return _st(stream)._nextPut_(_st(collectBlock)._value_(each));
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=_st(stream)._contents();
return $2;
}, function($ctx1) {$ctx1.fill(self,"select:thenCollect:",{selectBlock:selectBlock,collectBlock:collectBlock,stream:stream},globals.Collection)})},
args: ["selectBlock", "collectBlock"],
source: "select: selectBlock thenCollect: collectBlock\x0a\x09| stream |\x0a\x09stream := self class new writeStream.\x0a\x09self do: [ :each |\x0a\x09\x09(selectBlock value: each) ifTrue: [\x0a\x09\x09stream nextPut: (collectBlock value: each) ] ].\x0a\x09^ stream contents",
messageSends: ["writeStream", "new", "class", "do:", "ifTrue:", "value:", "nextPut:", "contents"],
referencedClasses: []
}),
globals.Collection);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"size",{},globals.Collection)})},
args: [],
source: "size\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.Collection);


smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "collection";
},
args: [],
source: "heliosClass\x0a\x09^ 'collection'",
messageSends: [],
referencedClasses: []
}),
globals.Collection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "new:",
protocol: 'instance creation',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._new();
return $1;
}, function($ctx1) {$ctx1.fill(self,"new:",{anInteger:anInteger},globals.Collection.klass)})},
args: ["anInteger"],
source: "new: anInteger\x0a\x09^ self new",
messageSends: ["new"],
referencedClasses: []
}),
globals.Collection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "with:",
protocol: 'instance creation',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._add_(anObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"with:",{anObject:anObject},globals.Collection.klass)})},
args: ["anObject"],
source: "with: anObject\x0a\x09\x09^ self new\x0a\x09\x09add: anObject;\x0a\x09\x09yourself",
messageSends: ["add:", "new", "yourself"],
referencedClasses: []
}),
globals.Collection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "with:with:",
protocol: 'instance creation',
fn: function (anObject,anotherObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._add_(anObject);
$ctx1.sendIdx["add:"]=1;
_st($2)._add_(anotherObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"with:with:",{anObject:anObject,anotherObject:anotherObject},globals.Collection.klass)})},
args: ["anObject", "anotherObject"],
source: "with: anObject with: anotherObject\x0a\x09\x09^ self new\x0a\x09\x09add: anObject;\x0a\x09\x09add: anotherObject;\x0a\x09\x09yourself",
messageSends: ["add:", "new", "yourself"],
referencedClasses: []
}),
globals.Collection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "with:with:with:",
protocol: 'instance creation',
fn: function (firstObject,secondObject,thirdObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._add_(firstObject);
$ctx1.sendIdx["add:"]=1;
_st($2)._add_(secondObject);
$ctx1.sendIdx["add:"]=2;
_st($2)._add_(thirdObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"with:with:with:",{firstObject:firstObject,secondObject:secondObject,thirdObject:thirdObject},globals.Collection.klass)})},
args: ["firstObject", "secondObject", "thirdObject"],
source: "with: firstObject with: secondObject with: thirdObject\x0a\x09\x09^ self new\x0a\x09\x09add: firstObject;\x0a\x09\x09add: secondObject;\x0a\x09\x09add: thirdObject;\x0a\x09\x09yourself",
messageSends: ["add:", "new", "yourself"],
referencedClasses: []
}),
globals.Collection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "withAll:",
protocol: 'instance creation',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._addAll_(aCollection);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"withAll:",{aCollection:aCollection},globals.Collection.klass)})},
args: ["aCollection"],
source: "withAll: aCollection\x0a\x09\x09^ self new\x0a\x09\x09addAll: aCollection;\x0a\x09\x09yourself",
messageSends: ["addAll:", "new", "yourself"],
referencedClasses: []
}),
globals.Collection.klass);


smalltalk.addClass('IndexableCollection', globals.Collection, [], 'Kernel-Collections');
globals.IndexableCollection.comment="I am a key-value store collection, that is,\x0aI store values under indexes.\x0a\x0aAs a rule of thumb, if a collection has `#at:` and `#at:put:`,\x0ait is an IndexableCollection.";
smalltalk.addMethod(
smalltalk.method({
selector: "at:",
protocol: 'accessing',
fn: function (anIndex){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_ifAbsent_(anIndex,(function(){
return smalltalk.withContext(function($ctx2) {
return self._errorNotFound();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:",{anIndex:anIndex},globals.IndexableCollection)})},
args: ["anIndex"],
source: "at: anIndex\x0a\x09\x22Lookup the given index in the receiver.\x0a\x09If it is present, answer the value stored at anIndex.\x0a\x09Otherwise, raise an error.\x22\x0a\x0a\x09^ self at: anIndex ifAbsent: [ self errorNotFound ]",
messageSends: ["at:ifAbsent:", "errorNotFound"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (anIndex,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{anIndex:anIndex,aBlock:aBlock},globals.IndexableCollection)})},
args: ["anIndex", "aBlock"],
source: "at: anIndex ifAbsent: aBlock\x0a\x09\x22Lookup the given index in the receiver.\x0a\x09If it is present, answer the value stored at anIndex.\x0a\x09Otherwise, answer the value of aBlock.\x22\x0a\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsentPut:",
protocol: 'accessing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_ifAbsent_(aKey,(function(){
return smalltalk.withContext(function($ctx2) {
return self._at_put_(aKey,_st(aBlock)._value());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:ifAbsentPut:",{aKey:aKey,aBlock:aBlock},globals.IndexableCollection)})},
args: ["aKey", "aBlock"],
source: "at: aKey ifAbsentPut: aBlock\x0a\x09^ self at: aKey ifAbsent: [\x0a\x09\x09self at: aKey put: aBlock value ]",
messageSends: ["at:ifAbsent:", "at:put:", "value"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:",
protocol: 'accessing',
fn: function (anIndex,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_ifPresent_ifAbsent_(anIndex,aBlock,(function(){
return nil;
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:",{anIndex:anIndex,aBlock:aBlock},globals.IndexableCollection)})},
args: ["anIndex", "aBlock"],
source: "at: anIndex ifPresent: aBlock\x0a\x09\x22Lookup the given index in the receiver.\x0a\x09If it is present, answer the value of evaluating aBlock with the value stored at anIndex.\x0a\x09Otherwise, answer nil.\x22\x0a\x0a\x09^ self at: anIndex ifPresent: aBlock ifAbsent: [ nil ]",
messageSends: ["at:ifPresent:ifAbsent:"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:ifAbsent:",
protocol: 'accessing',
fn: function (anIndex,aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:ifAbsent:",{anIndex:anIndex,aBlock:aBlock,anotherBlock:anotherBlock},globals.IndexableCollection)})},
args: ["anIndex", "aBlock", "anotherBlock"],
source: "at: anIndex ifPresent: aBlock ifAbsent: anotherBlock\x0a\x09\x22Lookup the given index in the receiver.\x0a\x09If it is present, answer the value of evaluating aBlock with the value stored at anIndex.\x0a\x09Otherwise, answer the value of anotherBlock.\x22\x0a\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (anIndex,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"at:put:",{anIndex:anIndex,anObject:anObject},globals.IndexableCollection)})},
args: ["anIndex", "anObject"],
source: "at: anIndex put: anObject\x0a\x09\x22Store anObject under the given index in the receiver.\x22\x0a\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._indexOf_ifAbsent_(anObject,(function(){
return smalltalk.withContext(function($ctx2) {
return self._errorNotFound();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"indexOf:",{anObject:anObject},globals.IndexableCollection)})},
args: ["anObject"],
source: "indexOf: anObject\x0a\x09\x22Lookup index at which anObject is stored in the receiver.\x0a\x09If not present, raise an error.\x22\x0a\x0a\x09^ self indexOf: anObject ifAbsent: [ self errorNotFound ]",
messageSends: ["indexOf:ifAbsent:", "errorNotFound"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:ifAbsent:",
protocol: 'accessing',
fn: function (anObject,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"indexOf:ifAbsent:",{anObject:anObject,aBlock:aBlock},globals.IndexableCollection)})},
args: ["anObject", "aBlock"],
source: "indexOf: anObject ifAbsent: aBlock\x0a\x09\x22Lookup index at which anObject is stored in the receiver.\x0a\x09If not present, return value of executing aBlock.\x22\x0a\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "with:do:",
protocol: 'enumarating',
fn: function (anotherCollection,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._withIndexDo_((function(each,index){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_value_(each,_st(anotherCollection)._at_(index));
}, function($ctx2) {$ctx2.fillBlock({each:each,index:index},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"with:do:",{anotherCollection:anotherCollection,aBlock:aBlock},globals.IndexableCollection)})},
args: ["anotherCollection", "aBlock"],
source: "with: anotherCollection do: aBlock\x0a\x09\x22Calls aBlock with every value from self\x0a\x09and with indetically-indexed value from anotherCollection\x22\x0a\x0a\x09self withIndexDo: [ :each :index |\x0a\x09\x09aBlock value: each value: (anotherCollection at: index) ]",
messageSends: ["withIndexDo:", "value:value:", "at:"],
referencedClasses: []
}),
globals.IndexableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "withIndexDo:",
protocol: 'enumarating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"withIndexDo:",{aBlock:aBlock},globals.IndexableCollection)})},
args: ["aBlock"],
source: "withIndexDo: aBlock\x0a\x09\x22Calls aBlock with every value from self\x0a\x09and with its index as the second argument\x22\x0a\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.IndexableCollection);



smalltalk.addClass('AssociativeCollection', globals.IndexableCollection, [], 'Kernel-Collections');
globals.AssociativeCollection.comment="I am a base class for object-indexed collections (Dictionary et.al.).";
smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (anAssocitativeCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$4,$3,$6,$5;
$2=self._class();
$ctx1.sendIdx["class"]=1;
$1=_st($2).__eq(_st(anAssocitativeCollection)._class());
$ctx1.sendIdx["="]=1;
if(! smalltalk.assert($1)){
return false;
};
$4=self._size();
$ctx1.sendIdx["size"]=1;
$3=_st($4).__eq(_st(anAssocitativeCollection)._size());
$ctx1.sendIdx["="]=2;
if(! smalltalk.assert($3)){
return false;
};
$6=self._associations();
$ctx1.sendIdx["associations"]=1;
$5=_st($6).__eq(_st(anAssocitativeCollection)._associations());
return $5;
}, function($ctx1) {$ctx1.fill(self,"=",{anAssocitativeCollection:anAssocitativeCollection},globals.AssociativeCollection)})},
args: ["anAssocitativeCollection"],
source: "= anAssocitativeCollection\x0a\x09self class = anAssocitativeCollection class ifFalse: [ ^ false ].\x0a\x09self size = anAssocitativeCollection size ifFalse: [ ^ false ].\x0a\x09^ self associations = anAssocitativeCollection associations",
messageSends: ["ifFalse:", "=", "class", "size", "associations"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "add:",
protocol: 'adding/removing',
fn: function (anAssociation){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._at_put_(_st(anAssociation)._key(),_st(anAssociation)._value());
return self}, function($ctx1) {$ctx1.fill(self,"add:",{anAssociation:anAssociation},globals.AssociativeCollection)})},
args: ["anAssociation"],
source: "add: anAssociation\x0a\x09self at: anAssociation key put: anAssociation value",
messageSends: ["at:put:", "key", "value"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "addAll:",
protocol: 'adding/removing',
fn: function (anAssociativeCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.AssociativeCollection.superclass.fn.prototype._addAll_.apply(_st(self), [_st(anAssociativeCollection)._associations()]);
return anAssociativeCollection;
}, function($ctx1) {$ctx1.fill(self,"addAll:",{anAssociativeCollection:anAssociativeCollection},globals.AssociativeCollection)})},
args: ["anAssociativeCollection"],
source: "addAll: anAssociativeCollection\x0a\x09super addAll: anAssociativeCollection associations.\x0a\x09^ anAssociativeCollection",
messageSends: ["addAll:", "associations"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "asDictionary",
protocol: 'converting',
fn: function (){
var self=this;
function $Dictionary(){return globals.Dictionary||(typeof Dictionary=="undefined"?nil:Dictionary)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Dictionary())._from_(self._associations());
return $1;
}, function($ctx1) {$ctx1.fill(self,"asDictionary",{},globals.AssociativeCollection)})},
args: [],
source: "asDictionary\x0a\x09^ Dictionary from: self associations",
messageSends: ["from:", "associations"],
referencedClasses: ["Dictionary"]
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "asHashedCollection",
protocol: 'converting',
fn: function (){
var self=this;
function $HashedCollection(){return globals.HashedCollection||(typeof HashedCollection=="undefined"?nil:HashedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($HashedCollection())._from_(self._associations());
return $1;
}, function($ctx1) {$ctx1.fill(self,"asHashedCollection",{},globals.AssociativeCollection)})},
args: [],
source: "asHashedCollection\x0a\x09^ HashedCollection from: self associations",
messageSends: ["from:", "associations"],
referencedClasses: ["HashedCollection"]
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
var hash;
function $HashedCollection(){return globals.HashedCollection||(typeof HashedCollection=="undefined"?nil:HashedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $1;
hash=_st($HashedCollection())._new();
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
return _st(hash)._at_put_(key,_st(value)._asJSON());
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
$1=hash;
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJSON",{hash:hash},globals.AssociativeCollection)})},
args: [],
source: "asJSON\x0a\x09| hash |\x0a\x09hash := HashedCollection new.\x0a\x09self keysAndValuesDo: [ :key :value |\x0a\x09\x09hash at: key put: value asJSON ].\x0a\x09^ hash",
messageSends: ["new", "keysAndValuesDo:", "at:put:", "asJSON"],
referencedClasses: ["HashedCollection"]
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "associations",
protocol: 'accessing',
fn: function (){
var self=this;
var associations;
return smalltalk.withContext(function($ctx1) { 
var $1;
associations=[];
self._associationsDo_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(associations)._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=associations;
return $1;
}, function($ctx1) {$ctx1.fill(self,"associations",{associations:associations},globals.AssociativeCollection)})},
args: [],
source: "associations\x0a\x09| associations |\x0a\x09associations := #().\x0a\x09self associationsDo: [ :each | associations add: each ].\x0a\x09^ associations",
messageSends: ["associationsDo:", "add:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "associationsDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
function $Association(){return globals.Association||(typeof Association=="undefined"?nil:Association)}
return smalltalk.withContext(function($ctx1) { 
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_(_st($Association())._key_value_(key,value));
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"associationsDo:",{aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "associationsDo: aBlock\x0a\x09self keysAndValuesDo: [ :key :value |\x0a\x09\x09aBlock value: (Association key: key value: value) ]",
messageSends: ["keysAndValuesDo:", "value:", "key:value:"],
referencedClasses: ["Association"]
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:ifAbsent:",
protocol: 'accessing',
fn: function (aKey,aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._includesKey_(aKey);
if(smalltalk.assert($2)){
$1=_st(aBlock)._value_(self._at_(aKey));
} else {
$1=_st(anotherBlock)._value();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:ifAbsent:",{aKey:aKey,aBlock:aBlock,anotherBlock:anotherBlock},globals.AssociativeCollection)})},
args: ["aKey", "aBlock", "anotherBlock"],
source: "at: aKey ifPresent: aBlock ifAbsent: anotherBlock\x0a\x09\x22Lookup the given key in the receiver.\x0a\x09If it is present, answer the value of evaluating the oneArgBlock \x0a\x09with the value associated with the key, otherwise answer the value \x0a\x09of absentBlock.\x22\x0a\x09\x0a\x09^ (self includesKey: aKey)\x0a\x09\x09ifTrue: [ aBlock value: (self at: aKey) ]\x0a\x09\x09ifFalse: [ anotherBlock value ]",
messageSends: ["ifTrue:ifFalse:", "includesKey:", "value:", "at:", "value"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "collect:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var newDict;
return smalltalk.withContext(function($ctx1) { 
var $1;
newDict=_st(self._class())._new();
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
return _st(newDict)._at_put_(key,_st(aBlock)._value_(value));
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
$1=newDict;
return $1;
}, function($ctx1) {$ctx1.fill(self,"collect:",{aBlock:aBlock,newDict:newDict},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "collect: aBlock\x0a\x09| newDict |\x0a\x09newDict := self class new.\x0a\x09self keysAndValuesDo: [ :key :value |\x0a\x09\x09newDict at: key put: (aBlock value: value) ].\x0a\x09^ newDict",
messageSends: ["new", "class", "keysAndValuesDo:", "at:put:", "value:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
var copy;
return smalltalk.withContext(function($ctx1) { 
var $1;
copy=_st(self._class())._new();
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
return _st(copy)._at_put_(key,_st(value)._deepCopy());
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
$1=copy;
return $1;
}, function($ctx1) {$ctx1.fill(self,"deepCopy",{copy:copy},globals.AssociativeCollection)})},
args: [],
source: "deepCopy\x0a\x09| copy |\x0a\x09copy := self class new.\x0a\x09self keysAndValuesDo: [ :key :value |\x0a\x09\x09copy at: key put: value deepCopy ].\x0a\x09^ copy",
messageSends: ["new", "class", "keysAndValuesDo:", "at:put:", "deepCopy"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "detect:ifNone:",
protocol: 'enumerating',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._values())._detect_ifNone_(aBlock,anotherBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"detect:ifNone:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.AssociativeCollection)})},
args: ["aBlock", "anotherBlock"],
source: "detect: aBlock ifNone: anotherBlock\x0a\x09^ self values detect: aBlock ifNone: anotherBlock",
messageSends: ["detect:ifNone:", "values"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "do:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._valuesDo_(aBlock);
return self}, function($ctx1) {$ctx1.fill(self,"do:",{aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "do: aBlock\x0a\x09self valuesDo: aBlock",
messageSends: ["valuesDo:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "includes:",
protocol: 'enumerating',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._values())._includes_(anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"includes:",{anObject:anObject},globals.AssociativeCollection)})},
args: ["anObject"],
source: "includes: anObject\x0a\x09^ self values includes: anObject",
messageSends: ["includes:", "values"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "includesKey:",
protocol: 'testing',
fn: function (aKey){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"includesKey:",{aKey:aKey},globals.AssociativeCollection)})},
args: ["aKey"],
source: "includesKey: aKey\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:ifAbsent:",
protocol: 'accessing',
fn: function (anObject,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._keys())._detect_ifNone_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self._at_(each)).__eq(anObject);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"indexOf:ifAbsent:",{anObject:anObject,aBlock:aBlock},globals.AssociativeCollection)})},
args: ["anObject", "aBlock"],
source: "indexOf: anObject ifAbsent: aBlock\x0a\x09^ self keys \x0a\x09\x09detect: [ :each | (self at: each) = anObject ] \x0a\x09\x09ifNone: aBlock",
messageSends: ["detect:ifNone:", "keys", "=", "at:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keyAtValue:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._keyAtValue_ifAbsent_(anObject,(function(){
return smalltalk.withContext(function($ctx2) {
return self._errorNotFound();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"keyAtValue:",{anObject:anObject},globals.AssociativeCollection)})},
args: ["anObject"],
source: "keyAtValue: anObject\x0a\x09^ self keyAtValue: anObject ifAbsent: [ self errorNotFound ]",
messageSends: ["keyAtValue:ifAbsent:", "errorNotFound"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keyAtValue:ifAbsent:",
protocol: 'accessing',
fn: function (anObject,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._indexOf_ifAbsent_(anObject,aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"keyAtValue:ifAbsent:",{anObject:anObject,aBlock:aBlock},globals.AssociativeCollection)})},
args: ["anObject", "aBlock"],
source: "keyAtValue: anObject ifAbsent: aBlock\x0a\x09^ self indexOf: anObject ifAbsent: aBlock",
messageSends: ["indexOf:ifAbsent:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keys",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"keys",{},globals.AssociativeCollection)})},
args: [],
source: "keys\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keysAndValuesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._keysDo_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_value_(each,self._at_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"keysAndValuesDo:",{aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "keysAndValuesDo: aBlock\x0a\x09self keysDo: [ :each |\x0a\x09\x09aBlock value: each value: (self at: each) ]",
messageSends: ["keysDo:", "value:value:", "at:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keysDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"keysDo:",{aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "keysDo: aBlock\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.AssociativeCollection.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
$ctx1.sendIdx["printOn:"]=1;
_st(aStream)._nextPutAll_(" (");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(self._associations())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._printOn_(aStream);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(aStream)._nextPutAll_(" , ");
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.AssociativeCollection)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09\x0a\x09aStream nextPutAll: ' ('.\x0a\x09self associations\x0a\x09\x09do: [ :each | each printOn: aStream ]\x0a\x09\x09separatedBy: [ aStream nextPutAll: ' , ' ].\x0a\x09aStream nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "do:separatedBy:", "associations"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:ifAbsent:",
protocol: 'adding/removing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._removeKey_ifAbsent_(aKey,aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"remove:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aKey", "aBlock"],
source: "remove: aKey ifAbsent: aBlock\x0a\x09^ self removeKey: aKey ifAbsent: aBlock",
messageSends: ["removeKey:ifAbsent:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "removeAll",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._keys())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._removeKey_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"removeAll",{},globals.AssociativeCollection)})},
args: [],
source: "removeAll\x0a\x09^ self keys do: [ :each | self removeKey: each ]",
messageSends: ["do:", "keys", "removeKey:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "removeKey:",
protocol: 'adding/removing',
fn: function (aKey){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._remove_(aKey);
return $1;
}, function($ctx1) {$ctx1.fill(self,"removeKey:",{aKey:aKey},globals.AssociativeCollection)})},
args: ["aKey"],
source: "removeKey: aKey\x0a\x09^ self remove: aKey",
messageSends: ["remove:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "removeKey:ifAbsent:",
protocol: 'adding/removing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"removeKey:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aKey", "aBlock"],
source: "removeKey: aKey ifAbsent: aBlock\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "select:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var newDict;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
newDict=_st(self._class())._new();
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(value);
if(smalltalk.assert($1)){
return _st(newDict)._at_put_(key,value);
};
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
$2=newDict;
return $2;
}, function($ctx1) {$ctx1.fill(self,"select:",{aBlock:aBlock,newDict:newDict},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "select: aBlock\x0a\x09| newDict |\x0a\x09newDict := self class new.\x0a\x09self keysAndValuesDo: [ :key :value |\x0a\x09\x09(aBlock value: value) ifTrue: [ newDict at: key put: value ]].\x0a\x09^ newDict",
messageSends: ["new", "class", "keysAndValuesDo:", "ifTrue:", "value:", "at:put:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "shallowCopy",
protocol: 'copying',
fn: function (){
var self=this;
var copy;
return smalltalk.withContext(function($ctx1) { 
var $1;
copy=_st(self._class())._new();
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
return _st(copy)._at_put_(key,value);
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
$1=copy;
return $1;
}, function($ctx1) {$ctx1.fill(self,"shallowCopy",{copy:copy},globals.AssociativeCollection)})},
args: [],
source: "shallowCopy\x0a\x09| copy |\x0a\x09copy := self class new.\x0a\x09self keysAndValuesDo: [ :key :value |\x0a\x09\x09copy at: key put: value ].\x0a\x09^ copy",
messageSends: ["new", "class", "keysAndValuesDo:", "at:put:"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._keys())._size();
return $1;
}, function($ctx1) {$ctx1.fill(self,"size",{},globals.AssociativeCollection)})},
args: [],
source: "size\x0a\x09^ self keys size",
messageSends: ["size", "keys"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "values",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"values",{},globals.AssociativeCollection)})},
args: [],
source: "values\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "valuesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"valuesDo:",{aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "valuesDo: aBlock\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AssociativeCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "withIndexDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._keysAndValuesDo_((function(key,value){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_value_(value,key);
}, function($ctx2) {$ctx2.fillBlock({key:key,value:value},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"withIndexDo:",{aBlock:aBlock},globals.AssociativeCollection)})},
args: ["aBlock"],
source: "withIndexDo: aBlock\x0a\x09self keysAndValuesDo: [ :key :value | aBlock value: value value: key ]",
messageSends: ["keysAndValuesDo:", "value:value:"],
referencedClasses: []
}),
globals.AssociativeCollection);


smalltalk.addMethod(
smalltalk.method({
selector: "from:",
protocol: 'instance creation',
fn: function (aCollection){
var self=this;
var newCollection;
return smalltalk.withContext(function($ctx1) { 
var $1;
newCollection=self._new();
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(newCollection)._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=newCollection;
return $1;
}, function($ctx1) {$ctx1.fill(self,"from:",{aCollection:aCollection,newCollection:newCollection},globals.AssociativeCollection.klass)})},
args: ["aCollection"],
source: "from: aCollection\x0a\x09| newCollection |\x0a\x09newCollection := self new.\x0a\x09aCollection do: [ :each | newCollection add: each ].\x0a\x09^ newCollection",
messageSends: ["new", "do:", "add:"],
referencedClasses: []
}),
globals.AssociativeCollection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "fromPairs:",
protocol: 'instance creation',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._from_(aCollection);
return $1;
}, function($ctx1) {$ctx1.fill(self,"fromPairs:",{aCollection:aCollection},globals.AssociativeCollection.klass)})},
args: ["aCollection"],
source: "fromPairs: aCollection\x0a\x09\x22This message is poorly named and has been replaced by #from:\x22\x0a\x09^ self from: aCollection",
messageSends: ["from:"],
referencedClasses: []
}),
globals.AssociativeCollection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "newFromPairs:",
protocol: 'instance creation',
fn: function (aCollection){
var self=this;
var newCollection;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3,$4,$5;
$2=_st(aCollection)._size();
$ctx1.sendIdx["size"]=1;
$1=_st($2)._even();
if(! smalltalk.assert($1)){
self._error_("#newFromPairs only accepts arrays of an even length");
};
newCollection=self._new();
_st((1)._to_by_(_st(aCollection)._size(),(2)))._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$3=newCollection;
$4=_st(aCollection)._at_(each);
$ctx2.sendIdx["at:"]=1;
return _st($3)._at_put_($4,_st(aCollection)._at_(_st(each).__plus((1))));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
$5=newCollection;
return $5;
}, function($ctx1) {$ctx1.fill(self,"newFromPairs:",{aCollection:aCollection,newCollection:newCollection},globals.AssociativeCollection.klass)})},
args: ["aCollection"],
source: "newFromPairs: aCollection\x0a\x09\x22Accept an array of elements where every two elements form an \x0a\x09association - the odd element being the key, and the even element the value.\x22\x0a\x09\x0a\x09| newCollection |\x0a\x09\x0a\x09aCollection size even ifFalse: [ \x0a\x09\x09self error: '#newFromPairs only accepts arrays of an even length' ].\x0a\x09\x09\x0a\x09newCollection := self new.\x0a\x09( 1 to: aCollection size by: 2 ) do: [ :each | \x0a\x09\x09newCollection at: (aCollection at: each) put: (aCollection at: each + 1) ].\x0a\x09\x09\x0a\x09^ newCollection",
messageSends: ["ifFalse:", "even", "size", "error:", "new", "do:", "to:by:", "at:put:", "at:", "+"],
referencedClasses: []
}),
globals.AssociativeCollection.klass);


smalltalk.addClass('Dictionary', globals.AssociativeCollection, ['keys', 'values'], 'Kernel-Collections');
globals.Dictionary.comment="I represent a set of elements that can be viewed from one of two perspectives: a set of associations,\x0aor a container of values that are externally named where the name can be any object that responds to `=`.\x0a\x0aThe external name is referred to as the key.";
smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var index = self._positionOfKey_(aKey);
		return index >=0 ? self['@values'][index] : aBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.Dictionary)})},
args: ["aKey", "aBlock"],
source: "at: aKey ifAbsent: aBlock\x0a\x09<\x0a\x09\x09var index = self._positionOfKey_(aKey);\x0a\x09\x09return index >>=0 ? self['@values'][index] : aBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (aKey,aValue){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var index = self._positionOfKey_(aKey);
		if(index === -1) {
			var keys = self['@keys'];
			index = keys.length;
			keys.push(aKey);
		}

		return self['@values'][index] = aValue;
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:put:",{aKey:aKey,aValue:aValue},globals.Dictionary)})},
args: ["aKey", "aValue"],
source: "at: aKey put: aValue\x0a\x09<\x0a\x09\x09var index = self._positionOfKey_(aKey);\x0a\x09\x09if(index === -1) {\x0a\x09\x09\x09var keys = self['@keys'];\x0a\x09\x09\x09index = keys.length;\x0a\x09\x09\x09keys.push(aKey);\x0a\x09\x09}\x0a\x0a\x09\x09return self['@values'][index] = aValue;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "includesKey:",
protocol: 'testing',
fn: function (aKey){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 return self._positionOfKey_(aKey) >= 0; ;
return self}, function($ctx1) {$ctx1.fill(self,"includesKey:",{aKey:aKey},globals.Dictionary)})},
args: ["aKey"],
source: "includesKey: aKey\x0a\x09< return self._positionOfKey_(aKey) >>= 0; >",
messageSends: [],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:ifAbsent:",
protocol: 'accessing',
fn: function (anObject,aBlock){
var self=this;
var index;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
index=_st(self["@values"])._indexOf_ifAbsent_(anObject,(function(){
return (0);
}));
$2=_st(index).__eq((0));
if(smalltalk.assert($2)){
$1=_st(aBlock)._value();
} else {
$1=_st(self["@keys"])._at_(index);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"indexOf:ifAbsent:",{anObject:anObject,aBlock:aBlock,index:index},globals.Dictionary)})},
args: ["anObject", "aBlock"],
source: "indexOf: anObject ifAbsent: aBlock\x0a\x09| index |\x0a\x09index := values \x0a\x09\x09indexOf: anObject \x0a\x09\x09ifAbsent: [ 0 ].\x0a\x09^ index = 0 \x0a\x09\x09ifTrue: [ aBlock value ] \x0a\x09\x09ifFalse: [ keys at: index ]",
messageSends: ["indexOf:ifAbsent:", "ifTrue:ifFalse:", "=", "value", "at:"],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.Dictionary.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@keys"]=[];
self["@values"]=[];
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.Dictionary)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09keys := #().\x0a\x09values := #()",
messageSends: ["initialize"],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "keys",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@keys"])._copy();
return $1;
}, function($ctx1) {$ctx1.fill(self,"keys",{},globals.Dictionary)})},
args: [],
source: "keys\x0a\x09^ keys copy",
messageSends: ["copy"],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "keysAndValuesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@keys"])._with_do_(self["@values"],aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"keysAndValuesDo:",{aBlock:aBlock},globals.Dictionary)})},
args: ["aBlock"],
source: "keysAndValuesDo: aBlock\x0a\x09^ keys with: values do: aBlock",
messageSends: ["with:do:"],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "keysDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@keys"])._do_(aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"keysDo:",{aBlock:aBlock},globals.Dictionary)})},
args: ["aBlock"],
source: "keysDo: aBlock\x0a\x09^ keys do: aBlock",
messageSends: ["do:"],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "positionOfKey:",
protocol: 'private',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var keys = self['@keys'];
		for(var i=0;i<keys.length;i++){
			if(keys[i].__eq(anObject)) { return i;}
		}
		return -1;
	;
return self}, function($ctx1) {$ctx1.fill(self,"positionOfKey:",{anObject:anObject},globals.Dictionary)})},
args: ["anObject"],
source: "positionOfKey: anObject\x0a\x09<\x0a\x09\x09var keys = self['@keys'];\x0a\x09\x09for(var i=0;i<keys.length;i++){\x0a\x09\x09\x09if(keys[i].__eq(anObject)) { return i;}\x0a\x09\x09}\x0a\x09\x09return -1;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "removeAll",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@keys"])._removeAll();
$ctx1.sendIdx["removeAll"]=1;
_st(self["@values"])._removeAll();
return self}, function($ctx1) {$ctx1.fill(self,"removeAll",{},globals.Dictionary)})},
args: [],
source: "removeAll\x0a\x09keys removeAll.\x0a\x09values removeAll",
messageSends: ["removeAll"],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "removeKey:ifAbsent:",
protocol: 'adding/removing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var index = self._positionOfKey_(aKey);
		if(index === -1) {
			return aBlock._value()
		} else {
			var keys = self['@keys'], values = self['@values'];
			var value = values[index], l = keys.length;
			keys[index] = keys[l-1];
			keys.pop();
			values[index] = values[l-1];
			values.pop();
			return value;
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"removeKey:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.Dictionary)})},
args: ["aKey", "aBlock"],
source: "removeKey: aKey ifAbsent: aBlock\x0a\x09<\x0a\x09\x09var index = self._positionOfKey_(aKey);\x0a\x09\x09if(index === -1) {\x0a\x09\x09\x09return aBlock._value()\x0a\x09\x09} else {\x0a\x09\x09\x09var keys = self['@keys'], values = self['@values'];\x0a\x09\x09\x09var value = values[index], l = keys.length;\x0a\x09\x09\x09keys[index] = keys[l-1];\x0a\x09\x09\x09keys.pop();\x0a\x09\x09\x09values[index] = values[l-1];\x0a\x09\x09\x09values.pop();\x0a\x09\x09\x09return value;\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "values",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@values"];
return $1;
},
args: [],
source: "values\x0a\x09^ values",
messageSends: [],
referencedClasses: []
}),
globals.Dictionary);

smalltalk.addMethod(
smalltalk.method({
selector: "valuesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@values"])._do_(aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"valuesDo:",{aBlock:aBlock},globals.Dictionary)})},
args: ["aBlock"],
source: "valuesDo: aBlock\x0a\x09^ values do: aBlock",
messageSends: ["do:"],
referencedClasses: []
}),
globals.Dictionary);



smalltalk.addClass('HashedCollection', globals.AssociativeCollection, [], 'Kernel-Collections');
globals.HashedCollection.comment="I am a traditional JavaScript object, or a Smalltalk `Dictionary`.\x0a\x0aUnlike a `Dictionary`, I can only have strings as keys.";
smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._includesKey_(aKey);
if(smalltalk.assert($2)){
$1=self._basicAt_(aKey);
} else {
$1=_st(aBlock)._value();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.HashedCollection)})},
args: ["aKey", "aBlock"],
source: "at: aKey ifAbsent: aBlock\x0a\x09^ (self includesKey: aKey)\x0a\x09\x09ifTrue: [ self basicAt: aKey ]\x0a\x09\x09ifFalse: [ aBlock value ]",
messageSends: ["ifTrue:ifFalse:", "includesKey:", "basicAt:", "value"],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (aKey,aValue){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_put_(aKey,aValue);
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:put:",{aKey:aKey,aValue:aValue},globals.HashedCollection)})},
args: ["aKey", "aValue"],
source: "at: aKey put: aValue\x0a\x09^ self basicAt: aKey put: aValue",
messageSends: ["basicAt:put:"],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "includesKey:",
protocol: 'testing',
fn: function (aKey){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.hasOwnProperty(aKey);
return self}, function($ctx1) {$ctx1.fill(self,"includesKey:",{aKey:aKey},globals.HashedCollection)})},
args: ["aKey"],
source: "includesKey: aKey\x0a\x09<return self.hasOwnProperty(aKey)>",
messageSends: [],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keys",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Object.keys(self);
return self}, function($ctx1) {$ctx1.fill(self,"keys",{},globals.HashedCollection)})},
args: [],
source: "keys\x0a\x09<return Object.keys(self)>",
messageSends: [],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "keysDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._keys())._do_(aBlock);
return self}, function($ctx1) {$ctx1.fill(self,"keysDo:",{aBlock:aBlock},globals.HashedCollection)})},
args: ["aBlock"],
source: "keysDo: aBlock\x0a\x09self keys do: aBlock",
messageSends: ["do:", "keys"],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "removeKey:ifAbsent:",
protocol: 'adding/removing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_ifPresent_ifAbsent_(aKey,(function(removed){
return smalltalk.withContext(function($ctx2) {
self._basicDelete_(aKey);
return removed;
}, function($ctx2) {$ctx2.fillBlock({removed:removed},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"removeKey:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.HashedCollection)})},
args: ["aKey", "aBlock"],
source: "removeKey: aKey ifAbsent: aBlock\x0a\x09^ self\x0a\x09\x09at: aKey\x0a\x09\x09ifPresent: [ :removed | self basicDelete: aKey. removed ]\x0a\x09\x09ifAbsent: [ aBlock value ]",
messageSends: ["at:ifPresent:ifAbsent:", "basicDelete:", "value"],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "values",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return self._keys().map(function(key){
			return self._at_(key);
		});
	;
return self}, function($ctx1) {$ctx1.fill(self,"values",{},globals.HashedCollection)})},
args: [],
source: "values\x0a\x09<\x0a\x09\x09return self._keys().map(function(key){\x0a\x09\x09\x09return self._at_(key);\x0a\x09\x09});\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.HashedCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "valuesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._values())._do_(aBlock);
return self}, function($ctx1) {$ctx1.fill(self,"valuesDo:",{aBlock:aBlock},globals.HashedCollection)})},
args: ["aBlock"],
source: "valuesDo: aBlock\x0a\x09self values do: aBlock",
messageSends: ["do:", "values"],
referencedClasses: []
}),
globals.HashedCollection);



smalltalk.addClass('SequenceableCollection', globals.IndexableCollection, [], 'Kernel-Collections');
globals.SequenceableCollection.comment="I am an IndexableCollection\x0awith numeric indexes starting with 1.";
smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$4,$1,$5;
var $early={};
try {
$3=self._class();
$ctx1.sendIdx["class"]=1;
$2=_st($3).__eq(_st(aCollection)._class());
$ctx1.sendIdx["="]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=self._size();
$ctx2.sendIdx["size"]=1;
return _st($4).__eq(_st(aCollection)._size());
$ctx2.sendIdx["="]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
if(! smalltalk.assert($1)){
return false;
};
self._withIndexDo_((function(each,i){
return smalltalk.withContext(function($ctx2) {
$5=_st(_st(aCollection)._at_(i)).__eq(each);
if(! smalltalk.assert($5)){
throw $early=[false];
};
}, function($ctx2) {$ctx2.fillBlock({each:each,i:i},$ctx1,3)})}));
return true;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"=",{aCollection:aCollection},globals.SequenceableCollection)})},
args: ["aCollection"],
source: "= aCollection\x0a\x09(self class = aCollection class and: [\x0a\x09\x09self size = aCollection size ]) ifFalse: [ ^ false ].\x0a\x09self withIndexDo: [ :each :i |\x0a\x09\x09\x09\x09(aCollection at: i) = each ifFalse: [ ^ false ]].\x0a\x09^ true",
messageSends: ["ifFalse:", "and:", "=", "class", "size", "withIndexDo:", "at:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "addLast:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._add_(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"addLast:",{anObject:anObject},globals.SequenceableCollection)})},
args: ["anObject"],
source: "addLast: anObject\x0a\x09self add: anObject",
messageSends: ["add:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "allButFirst",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._copyFrom_to_((2),self._size());
return $1;
}, function($ctx1) {$ctx1.fill(self,"allButFirst",{},globals.SequenceableCollection)})},
args: [],
source: "allButFirst\x0a\x09^ self copyFrom: 2 to: self size",
messageSends: ["copyFrom:to:", "size"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "allButLast",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._copyFrom_to_((1),_st(self._size()).__minus((1)));
return $1;
}, function($ctx1) {$ctx1.fill(self,"allButLast",{},globals.SequenceableCollection)})},
args: [],
source: "allButLast\x0a\x09^ self copyFrom: 1 to: self size - 1",
messageSends: ["copyFrom:to:", "-", "size"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "atRandom",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_(_st(self._size())._atRandom());
return $1;
}, function($ctx1) {$ctx1.fill(self,"atRandom",{},globals.SequenceableCollection)})},
args: [],
source: "atRandom\x0a\x09^ self at: self size atRandom",
messageSends: ["at:", "atRandom", "size"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "beginsWith:",
protocol: 'testing',
fn: function (prefix){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1,$4;
$2=self._size();
$ctx1.sendIdx["size"]=1;
$3=_st(prefix)._size();
$ctx1.sendIdx["size"]=2;
$1=_st($2).__lt($3);
if(smalltalk.assert($1)){
return false;
};
$4=_st(self._first_(_st(prefix)._size())).__eq(prefix);
return $4;
}, function($ctx1) {$ctx1.fill(self,"beginsWith:",{prefix:prefix},globals.SequenceableCollection)})},
args: ["prefix"],
source: "beginsWith: prefix\x0a\x09self size < prefix size ifTrue: [ ^ false ].\x0a\x09^ (self first: prefix size) = prefix",
messageSends: ["ifTrue:", "<", "size", "=", "first:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "copyFrom:to:",
protocol: 'copying',
fn: function (anIndex,anotherIndex){
var self=this;
var range,newCollection;
return smalltalk.withContext(function($ctx1) { 
var $1;
range=_st(anIndex)._to_(anotherIndex);
newCollection=_st(self._class())._new_(_st(range)._size());
_st(range)._withIndexDo_((function(each,i){
return smalltalk.withContext(function($ctx2) {
return _st(newCollection)._at_put_(i,self._at_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each,i:i},$ctx1,1)})}));
$1=newCollection;
return $1;
}, function($ctx1) {$ctx1.fill(self,"copyFrom:to:",{anIndex:anIndex,anotherIndex:anotherIndex,range:range,newCollection:newCollection},globals.SequenceableCollection)})},
args: ["anIndex", "anotherIndex"],
source: "copyFrom: anIndex to: anotherIndex\x0a\x09| range newCollection |\x0a\x09range := anIndex to: anotherIndex.\x0a\x09newCollection := self class new: range size.\x0a\x09range withIndexDo: [ :each :i |\x0a\x09\x09newCollection at: i put: (self at: each) ].\x0a\x09^ newCollection",
messageSends: ["to:", "new:", "class", "size", "withIndexDo:", "at:put:", "at:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
var newCollection;
return smalltalk.withContext(function($ctx1) { 
var $1;
newCollection=_st(self._class())._new_(self._size());
self._withIndexDo_((function(each,index){
return smalltalk.withContext(function($ctx2) {
return _st(newCollection)._at_put_(index,_st(each)._deepCopy());
}, function($ctx2) {$ctx2.fillBlock({each:each,index:index},$ctx1,1)})}));
$1=newCollection;
return $1;
}, function($ctx1) {$ctx1.fill(self,"deepCopy",{newCollection:newCollection},globals.SequenceableCollection)})},
args: [],
source: "deepCopy\x0a\x09| newCollection |\x0a\x09newCollection := self class new: self size.\x0a\x09self withIndexDo: [ :each :index |\x0a\x09\x09newCollection at: index put: each deepCopy ].\x0a\x09^ newCollection",
messageSends: ["new:", "class", "size", "withIndexDo:", "at:put:", "deepCopy"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "detect:ifNone:",
protocol: 'enumerating',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self = self._numericallyIndexable();
		for(var i = 0; i < self.length; i++)
			if(aBlock._value_(self[i]))
				return self[i];
		return anotherBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"detect:ifNone:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.SequenceableCollection)})},
args: ["aBlock", "anotherBlock"],
source: "detect: aBlock ifNone: anotherBlock\x0a\x09<\x0a\x09\x09self = self._numericallyIndexable();\x0a\x09\x09for(var i = 0; i < self.length; i++)\x0a\x09\x09\x09if(aBlock._value_(self[i]))\x0a\x09\x09\x09\x09return self[i];\x0a\x09\x09return anotherBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "do:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self = self._numericallyIndexable();
		for(var i=0; i < self.length; i++) {
			aBlock._value_(self[i]);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"do:",{aBlock:aBlock},globals.SequenceableCollection)})},
args: ["aBlock"],
source: "do: aBlock\x0a\x09<\x0a\x09\x09self = self._numericallyIndexable();\x0a\x09\x09for(var i=0; i < self.length; i++) {\x0a\x09\x09\x09aBlock._value_(self[i]);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "endsWith:",
protocol: 'testing',
fn: function (suffix){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1,$4;
$2=self._size();
$ctx1.sendIdx["size"]=1;
$3=_st(suffix)._size();
$ctx1.sendIdx["size"]=2;
$1=_st($2).__lt($3);
if(smalltalk.assert($1)){
return false;
};
$4=_st(self._last_(_st(suffix)._size())).__eq(suffix);
return $4;
}, function($ctx1) {$ctx1.fill(self,"endsWith:",{suffix:suffix},globals.SequenceableCollection)})},
args: ["suffix"],
source: "endsWith: suffix\x0a\x09self size < suffix size ifTrue: [ ^ false ].\x0a\x09^ (self last: suffix size) = suffix",
messageSends: ["ifTrue:", "<", "size", "=", "last:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "first",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_((1));
return $1;
}, function($ctx1) {$ctx1.fill(self,"first",{},globals.SequenceableCollection)})},
args: [],
source: "first\x0a\x09^ self at: 1",
messageSends: ["at:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "first:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st(self._size()).__lt(aNumber);
if(smalltalk.assert($1)){
self._error_("Invalid number of elements");
};
$2=self._copyFrom_to_((1),aNumber);
return $2;
}, function($ctx1) {$ctx1.fill(self,"first:",{aNumber:aNumber},globals.SequenceableCollection)})},
args: ["aNumber"],
source: "first: aNumber\x0a\x09\x22Answer the first `aNumber` elements of the receiver.\x0a\x09Raise an error if there are not enough elements in the receiver.\x22\x0a\x0a\x09self size < aNumber ifTrue: [ self error: 'Invalid number of elements' ].\x0a\x0a\x09^ self copyFrom: 1 to: aNumber",
messageSends: ["ifTrue:", "<", "size", "error:", "copyFrom:to:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "fourth",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_((4));
return $1;
}, function($ctx1) {$ctx1.fill(self,"fourth",{},globals.SequenceableCollection)})},
args: [],
source: "fourth\x0a\x09^ self at: 4",
messageSends: ["at:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "includes:",
protocol: 'testing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._indexOf_ifAbsent_(anObject,(function(){
return nil;
})))._notNil();
return $1;
}, function($ctx1) {$ctx1.fill(self,"includes:",{anObject:anObject},globals.SequenceableCollection)})},
args: ["anObject"],
source: "includes: anObject\x0a\x09^ (self indexOf: anObject ifAbsent: [ nil ]) notNil",
messageSends: ["notNil", "indexOf:ifAbsent:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:ifAbsent:",
protocol: 'accessing',
fn: function (anObject,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self = self._numericallyIndexable();
		for(var i=0; i < self.length; i++) {
			if(_st(self[i]).__eq(anObject)) {return i+1}
		};
		return aBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"indexOf:ifAbsent:",{anObject:anObject,aBlock:aBlock},globals.SequenceableCollection)})},
args: ["anObject", "aBlock"],
source: "indexOf: anObject ifAbsent: aBlock\x0a\x09<\x0a\x09\x09self = self._numericallyIndexable();\x0a\x09\x09for(var i=0; i < self.length; i++) {\x0a\x09\x09\x09if(_st(self[i]).__eq(anObject)) {return i+1}\x0a\x09\x09};\x0a\x09\x09return aBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:startingAt:",
protocol: 'accessing',
fn: function (anObject,start){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._indexOf_startingAt_ifAbsent_(anObject,start,(function(){
return (0);
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"indexOf:startingAt:",{anObject:anObject,start:start},globals.SequenceableCollection)})},
args: ["anObject", "start"],
source: "indexOf: anObject startingAt: start\x0a\x09\x22Answer the index of the first occurence of anElement after start\x0a\x09within the receiver. If the receiver does not contain anElement,\x0a\x09answer 0.\x22\x0a\x09^ self indexOf: anObject startingAt: start ifAbsent: [ 0 ]",
messageSends: ["indexOf:startingAt:ifAbsent:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "indexOf:startingAt:ifAbsent:",
protocol: 'accessing',
fn: function (anObject,start,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self = self._numericallyIndexable();
		for(var i=start - 1; i < self.length; i++){
			if(_st(self[i]).__eq(anObject)) {return i+1}
		}
		return aBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"indexOf:startingAt:ifAbsent:",{anObject:anObject,start:start,aBlock:aBlock},globals.SequenceableCollection)})},
args: ["anObject", "start", "aBlock"],
source: "indexOf: anObject startingAt: start ifAbsent: aBlock\x0a\x09<\x0a\x09\x09self = self._numericallyIndexable();\x0a\x09\x09for(var i=start - 1; i < self.length; i++){\x0a\x09\x09\x09if(_st(self[i]).__eq(anObject)) {return i+1}\x0a\x09\x09}\x0a\x09\x09return aBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "last",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_(self._size());
return $1;
}, function($ctx1) {$ctx1.fill(self,"last",{},globals.SequenceableCollection)})},
args: [],
source: "last\x0a\x09^ self at: self size",
messageSends: ["at:", "size"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "last:",
protocol: 'accessing',
fn: function (aNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$6,$5,$4,$3;
$2=self._size();
$ctx1.sendIdx["size"]=1;
$1=_st($2).__lt(aNumber);
if(smalltalk.assert($1)){
self._error_("Invalid number of elements");
};
$6=self._size();
$ctx1.sendIdx["size"]=2;
$5=_st($6).__minus(aNumber);
$4=_st($5).__plus((1));
$3=self._copyFrom_to_($4,self._size());
return $3;
}, function($ctx1) {$ctx1.fill(self,"last:",{aNumber:aNumber},globals.SequenceableCollection)})},
args: ["aNumber"],
source: "last: aNumber\x0a\x09\x22Answer the last aNumber elements of the receiver.\x0a\x09Raise an error if there are not enough elements in the receiver.\x22\x0a\x0a\x09self size < aNumber ifTrue: [ self error: 'Invalid number of elements' ].\x0a\x0a\x09^ self copyFrom: self size - aNumber + 1 to: self size",
messageSends: ["ifTrue:", "<", "size", "error:", "copyFrom:to:", "+", "-"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "newStream",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._streamClass())._on_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"newStream",{},globals.SequenceableCollection)})},
args: [],
source: "newStream\x0a\x09^ self streamClass on: self",
messageSends: ["on:", "streamClass"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "numericallyIndexable",
protocol: 'private',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"numericallyIndexable",{},globals.SequenceableCollection)})},
args: [],
source: "numericallyIndexable\x0a\x09\x22This is an internal converting message.\x0a\x09It answeres a representation of the receiver\x0a\x09that can use foo[i] in JavaScript code.\x0a\x09\x0a\x09It fixes IE8, where boxed String is unable\x0a\x09to numerically index its characters,\x0a\x09but primitive string can.\x22\x0a\x09\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "readStream",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._stream();
return $1;
}, function($ctx1) {$ctx1.fill(self,"readStream",{},globals.SequenceableCollection)})},
args: [],
source: "readStream\x0a\x09\x22For Pharo compatibility\x22\x0a\x09\x0a\x09^ self stream",
messageSends: ["stream"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "removeLast",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._remove_(self._last());
return $1;
}, function($ctx1) {$ctx1.fill(self,"removeLast",{},globals.SequenceableCollection)})},
args: [],
source: "removeLast\x0a\x09^ self remove: self last",
messageSends: ["remove:", "last"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "reversed",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"reversed",{},globals.SequenceableCollection)})},
args: [],
source: "reversed\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "second",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_((2));
return $1;
}, function($ctx1) {$ctx1.fill(self,"second",{},globals.SequenceableCollection)})},
args: [],
source: "second\x0a\x09^ self at: 2",
messageSends: ["at:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "shallowCopy",
protocol: 'copying',
fn: function (){
var self=this;
var newCollection;
return smalltalk.withContext(function($ctx1) { 
var $1;
newCollection=_st(self._class())._new_(self._size());
self._withIndexDo_((function(each,index){
return smalltalk.withContext(function($ctx2) {
return _st(newCollection)._at_put_(index,each);
}, function($ctx2) {$ctx2.fillBlock({each:each,index:index},$ctx1,1)})}));
$1=newCollection;
return $1;
}, function($ctx1) {$ctx1.fill(self,"shallowCopy",{newCollection:newCollection},globals.SequenceableCollection)})},
args: [],
source: "shallowCopy\x0a\x09| newCollection |\x0a\x09newCollection := self class new: self size.\x0a\x09self withIndexDo: [ :each :index |\x0a\x09\x09newCollection at: index put: each ].\x0a\x09^ newCollection",
messageSends: ["new:", "class", "size", "withIndexDo:", "at:put:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "stream",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._newStream();
return $1;
}, function($ctx1) {$ctx1.fill(self,"stream",{},globals.SequenceableCollection)})},
args: [],
source: "stream\x0a\x09^ self newStream",
messageSends: ["newStream"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "streamClass",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._streamClass();
return $1;
}, function($ctx1) {$ctx1.fill(self,"streamClass",{},globals.SequenceableCollection)})},
args: [],
source: "streamClass\x0a\x09^ self class streamClass",
messageSends: ["streamClass", "class"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "third",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._at_((3));
return $1;
}, function($ctx1) {$ctx1.fill(self,"third",{},globals.SequenceableCollection)})},
args: [],
source: "third\x0a\x09^ self at: 3",
messageSends: ["at:"],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "with:do:",
protocol: 'enumerating',
fn: function (anotherCollection,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self = self._numericallyIndexable();
		anotherCollection = anotherCollection._numericallyIndexable();
		for(var i=0; i<self.length; i++) {
			aBlock._value_value_(self[i], anotherCollection[i]);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"with:do:",{anotherCollection:anotherCollection,aBlock:aBlock},globals.SequenceableCollection)})},
args: ["anotherCollection", "aBlock"],
source: "with: anotherCollection do: aBlock\x0a\x09<\x0a\x09\x09self = self._numericallyIndexable();\x0a\x09\x09anotherCollection = anotherCollection._numericallyIndexable();\x0a\x09\x09for(var i=0; i<self.length; i++) {\x0a\x09\x09\x09aBlock._value_value_(self[i], anotherCollection[i]);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "withIndexDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self = self._numericallyIndexable();
		for(var i=0; i < self.length; i++) {
			aBlock._value_value_(self[i], i+1);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"withIndexDo:",{aBlock:aBlock},globals.SequenceableCollection)})},
args: ["aBlock"],
source: "withIndexDo: aBlock\x0a\x09<\x0a\x09\x09self = self._numericallyIndexable();\x0a\x09\x09for(var i=0; i < self.length; i++) {\x0a\x09\x09\x09aBlock._value_value_(self[i], i+1);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "writeStream",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._stream();
return $1;
}, function($ctx1) {$ctx1.fill(self,"writeStream",{},globals.SequenceableCollection)})},
args: [],
source: "writeStream\x0a\x09\x22For Pharo compatibility\x22\x0a\x09\x0a\x09^ self stream",
messageSends: ["stream"],
referencedClasses: []
}),
globals.SequenceableCollection);


smalltalk.addMethod(
smalltalk.method({
selector: "streamClass",
protocol: 'accessing',
fn: function (){
var self=this;
function $Stream(){return globals.Stream||(typeof Stream=="undefined"?nil:Stream)}
return $Stream();
},
args: [],
source: "streamClass\x0a\x09\x09^ Stream",
messageSends: [],
referencedClasses: ["Stream"]
}),
globals.SequenceableCollection.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "streamContents:",
protocol: 'streaming',
fn: function (aBlock){
var self=this;
var stream;
return smalltalk.withContext(function($ctx1) { 
var $1;
stream=_st(self._streamClass())._on_(self._new());
_st(aBlock)._value_(stream);
$1=_st(stream)._contents();
return $1;
}, function($ctx1) {$ctx1.fill(self,"streamContents:",{aBlock:aBlock,stream:stream},globals.SequenceableCollection.klass)})},
args: ["aBlock"],
source: "streamContents: aBlock\x0a\x09| stream |\x0a\x09stream := (self streamClass on: self new).\x0a\x09aBlock value: stream.\x0a\x09^ stream contents",
messageSends: ["on:", "streamClass", "new", "value:", "contents"],
referencedClasses: []
}),
globals.SequenceableCollection.klass);


smalltalk.addClass('Array', globals.SequenceableCollection, [], 'Kernel-Collections');
globals.Array.comment="I represent a collection of objects ordered by the collector. The size of arrays is dynamic.\x0a\x0aI am directly mapped to JavaScript Number.\x0a\x0a*Note* In Amber, `OrderedCollection` is an alias for `Array`.";
smalltalk.addMethod(
smalltalk.method({
selector: "add:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.push(anObject); return anObject;;
return self}, function($ctx1) {$ctx1.fill(self,"add:",{anObject:anObject},globals.Array)})},
args: ["anObject"],
source: "add: anObject\x0a\x09<self.push(anObject); return anObject;>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "addFirst:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.unshift(anObject); return anObject;;
return self}, function($ctx1) {$ctx1.fill(self,"addFirst:",{anObject:anObject},globals.Array)})},
args: ["anObject"],
source: "addFirst: anObject\x0a\x09<self.unshift(anObject); return anObject;>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "asJavascript",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st("[".__comma(_st(self._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._asJavascript();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})})))._join_(", "))).__comma("]");
$ctx1.sendIdx[","]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJavascript",{},globals.Array)})},
args: [],
source: "asJavascript\x0a\x09^ '[', ((self collect: [:each | each asJavascript ]) join: ', '), ']'",
messageSends: [",", "join:", "collect:", "asJavascript"],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (anIndex,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return anIndex >= 1 && anIndex <= self.length
			? self[anIndex - 1]
			: aBlock._value()
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{anIndex:anIndex,aBlock:aBlock},globals.Array)})},
args: ["anIndex", "aBlock"],
source: "at: anIndex ifAbsent: aBlock\x0a\x09<\x0a\x09\x09return anIndex >>= 1 && anIndex <= self.length\x0a\x09\x09\x09? self[anIndex - 1]\x0a\x09\x09\x09: aBlock._value()\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:ifAbsent:",
protocol: 'accessing',
fn: function (anIndex,aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return anIndex >= 1 && anIndex <= self.length
			? aBlock._value_(self[anIndex - 1])
			: anotherBlock._value()
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:ifAbsent:",{anIndex:anIndex,aBlock:aBlock,anotherBlock:anotherBlock},globals.Array)})},
args: ["anIndex", "aBlock", "anotherBlock"],
source: "at: anIndex ifPresent: aBlock ifAbsent: anotherBlock\x0a\x09<\x0a\x09\x09return anIndex >>= 1 && anIndex <= self.length\x0a\x09\x09\x09? aBlock._value_(self[anIndex - 1])\x0a\x09\x09\x09: anotherBlock._value()\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (anIndex,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self[anIndex - 1] = anObject;
return self}, function($ctx1) {$ctx1.fill(self,"at:put:",{anIndex:anIndex,anObject:anObject},globals.Array)})},
args: ["anIndex", "anObject"],
source: "at: anIndex put: anObject\x0a\x09<return self[anIndex - 1] = anObject>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "collect:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.map(function(each) {return aBlock._value_(each)});
return self}, function($ctx1) {$ctx1.fill(self,"collect:",{aBlock:aBlock},globals.Array)})},
args: ["aBlock"],
source: "collect: aBlock\x0a\x09\x22Optimized version\x22\x0a\x09\x0a\x09<return self.map(function(each) {return aBlock._value_(each)})>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "join:",
protocol: 'enumerating',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.join(aString);
return self}, function($ctx1) {$ctx1.fill(self,"join:",{aString:aString},globals.Array)})},
args: ["aString"],
source: "join: aString\x0a\x09<return self.join(aString)>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "numericallyIndexable",
protocol: 'private',
fn: function (){
var self=this;
return self;
},
args: [],
source: "numericallyIndexable\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.Array.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
$ctx1.sendIdx["printOn:"]=1;
_st(aStream)._nextPutAll_(" (");
$ctx1.sendIdx["nextPutAll:"]=1;
self._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._printOn_(aStream);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(aStream)._nextPutAll_(" ");
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Array)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09\x0a\x09aStream nextPutAll: ' ('.\x0a\x09self \x0a\x09\x09do: [ :each | each printOn: aStream ]\x0a\x09\x09separatedBy: [ aStream nextPutAll: ' ' ].\x0a\x09aStream nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "do:separatedBy:"],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:ifAbsent:",
protocol: 'adding/removing',
fn: function (anObject,aBlock){
var self=this;
var index;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
index=self._indexOf_ifAbsent_(anObject,(function(){
return (0);
}));
$2=_st(index).__eq((0));
if(smalltalk.assert($2)){
$1=_st(aBlock)._value();
} else {
self._removeIndex_(index);
$1=anObject;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"remove:ifAbsent:",{anObject:anObject,aBlock:aBlock,index:index},globals.Array)})},
args: ["anObject", "aBlock"],
source: "remove: anObject ifAbsent: aBlock\x0a\x09| index |\x0a\x09index := self indexOf: anObject ifAbsent: [ 0 ].\x0a\x09^ index = 0\x0a\x09\x09ifFalse: [ self removeIndex: index. anObject ]\x0a\x09\x09ifTrue: [ aBlock value ]",
messageSends: ["indexOf:ifAbsent:", "ifFalse:ifTrue:", "=", "removeIndex:", "value"],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "removeAll",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.length = 0;
return self}, function($ctx1) {$ctx1.fill(self,"removeAll",{},globals.Array)})},
args: [],
source: "removeAll\x0a\x09<self.length = 0>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "removeFrom:to:",
protocol: 'adding/removing',
fn: function (aNumber,anotherNumber){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.splice(aNumber -1, anotherNumber - aNumber + 1);
return self}, function($ctx1) {$ctx1.fill(self,"removeFrom:to:",{aNumber:aNumber,anotherNumber:anotherNumber},globals.Array)})},
args: ["aNumber", "anotherNumber"],
source: "removeFrom: aNumber to: anotherNumber\x0a\x09<self.splice(aNumber -1, anotherNumber - aNumber + 1)>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "removeIndex:",
protocol: 'adding/removing',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.splice(anInteger - 1, 1);
return self}, function($ctx1) {$ctx1.fill(self,"removeIndex:",{anInteger:anInteger},globals.Array)})},
args: ["anInteger"],
source: "removeIndex: anInteger\x0a\x09<self.splice(anInteger - 1, 1)>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "removeLast",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.pop();;
return self}, function($ctx1) {$ctx1.fill(self,"removeLast",{},globals.Array)})},
args: [],
source: "removeLast\x0a\x09<return self.pop();>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "reversed",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self._copy().reverse();
return self}, function($ctx1) {$ctx1.fill(self,"reversed",{},globals.Array)})},
args: [],
source: "reversed\x0a\x09<return self._copy().reverse()>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "select:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var result = self.klass._new();
		for(var i=0; i<self.length; i++) {
			if(aBlock._value_(self[i])) {
				result.push(self[i]);
			}
		}
		return result;
	;
return self}, function($ctx1) {$ctx1.fill(self,"select:",{aBlock:aBlock},globals.Array)})},
args: ["aBlock"],
source: "select: aBlock\x0a\x09\x22Optimized version\x22\x0a\x09\x0a\x09<\x0a\x09\x09var result = self.klass._new();\x0a\x09\x09for(var i=0; i<self.length; i++) {\x0a\x09\x09\x09if(aBlock._value_(self[i])) {\x0a\x09\x09\x09\x09result.push(self[i]);\x0a\x09\x09\x09}\x0a\x09\x09}\x0a\x09\x09return result;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.length;
return self}, function($ctx1) {$ctx1.fill(self,"size",{},globals.Array)})},
args: [],
source: "size\x0a\x09<return self.length>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "sort",
protocol: 'enumerating',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._sort_((function(a,b){
return smalltalk.withContext(function($ctx2) {
return _st(a).__lt(b);
}, function($ctx2) {$ctx2.fillBlock({a:a,b:b},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"sort",{},globals.Array)})},
args: [],
source: "sort\x0a\x09^ self sort: [ :a :b | a < b ]",
messageSends: ["sort:", "<"],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "sort:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return self.sort(function(a, b) {
			if(aBlock._value_value_(a,b)) {return -1} else {return 1}
		})
	;
return self}, function($ctx1) {$ctx1.fill(self,"sort:",{aBlock:aBlock},globals.Array)})},
args: ["aBlock"],
source: "sort: aBlock\x0a\x09<\x0a\x09\x09return self.sort(function(a, b) {\x0a\x09\x09\x09if(aBlock._value_value_(a,b)) {return -1} else {return 1}\x0a\x09\x09})\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "sorted",
protocol: 'enumerating',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._copy())._sort();
return $1;
}, function($ctx1) {$ctx1.fill(self,"sorted",{},globals.Array)})},
args: [],
source: "sorted\x0a\x09^ self copy sort",
messageSends: ["sort", "copy"],
referencedClasses: []
}),
globals.Array);

smalltalk.addMethod(
smalltalk.method({
selector: "sorted:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._copy())._sort_(aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"sorted:",{aBlock:aBlock},globals.Array)})},
args: ["aBlock"],
source: "sorted: aBlock\x0a\x09^ self copy sort: aBlock",
messageSends: ["sort:", "copy"],
referencedClasses: []
}),
globals.Array);


smalltalk.addMethod(
smalltalk.method({
selector: "new:",
protocol: 'instance creation',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return new Array(anInteger);
return self}, function($ctx1) {$ctx1.fill(self,"new:",{anInteger:anInteger},globals.Array.klass)})},
args: ["anInteger"],
source: "new: anInteger\x0a\x09<return new Array(anInteger)>",
messageSends: [],
referencedClasses: []
}),
globals.Array.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "with:",
protocol: 'instance creation',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new_((1));
_st($2)._at_put_((1),anObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"with:",{anObject:anObject},globals.Array.klass)})},
args: ["anObject"],
source: "with: anObject\x0a\x09\x09^ (self new: 1)\x0a\x09\x09at: 1 put: anObject;\x0a\x09\x09yourself",
messageSends: ["at:put:", "new:", "yourself"],
referencedClasses: []
}),
globals.Array.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "with:with:",
protocol: 'instance creation',
fn: function (anObject,anObject2){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new_((2));
_st($2)._at_put_((1),anObject);
$ctx1.sendIdx["at:put:"]=1;
_st($2)._at_put_((2),anObject2);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"with:with:",{anObject:anObject,anObject2:anObject2},globals.Array.klass)})},
args: ["anObject", "anObject2"],
source: "with: anObject with: anObject2\x0a\x09\x09^ (self new: 2)\x0a\x09\x09at: 1 put: anObject;\x0a\x09\x09at: 2 put: anObject2;\x0a\x09\x09yourself",
messageSends: ["at:put:", "new:", "yourself"],
referencedClasses: []
}),
globals.Array.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "with:with:with:",
protocol: 'instance creation',
fn: function (anObject,anObject2,anObject3){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new_((3));
_st($2)._at_put_((1),anObject);
$ctx1.sendIdx["at:put:"]=1;
_st($2)._at_put_((2),anObject2);
$ctx1.sendIdx["at:put:"]=2;
_st($2)._at_put_((3),anObject3);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"with:with:with:",{anObject:anObject,anObject2:anObject2,anObject3:anObject3},globals.Array.klass)})},
args: ["anObject", "anObject2", "anObject3"],
source: "with: anObject with: anObject2 with: anObject3\x0a\x09\x09^ (self new: 3)\x0a\x09\x09at: 1 put: anObject;\x0a\x09\x09at: 2 put: anObject2;\x0a\x09\x09at: 3 put: anObject3;\x0a\x09\x09yourself",
messageSends: ["at:put:", "new:", "yourself"],
referencedClasses: []
}),
globals.Array.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "withAll:",
protocol: 'instance creation',
fn: function (aCollection){
var self=this;
var instance,index;
return smalltalk.withContext(function($ctx1) { 
var $1;
index=(1);
instance=self._new_(_st(aCollection)._size());
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
_st(instance)._at_put_(index,each);
index=_st(index).__plus((1));
return index;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=instance;
return $1;
}, function($ctx1) {$ctx1.fill(self,"withAll:",{aCollection:aCollection,instance:instance,index:index},globals.Array.klass)})},
args: ["aCollection"],
source: "withAll: aCollection\x0a\x09| instance index |\x0a\x09index := 1.\x0a\x09instance := self new: aCollection size.\x0a\x09aCollection do: [ :each |\x0a\x09\x09instance at: index put: each.\x0a\x09\x09index := index + 1 ].\x0a\x09^ instance",
messageSends: ["new:", "size", "do:", "at:put:", "+"],
referencedClasses: []
}),
globals.Array.klass);


smalltalk.addClass('CharacterArray', globals.SequenceableCollection, [], 'Kernel-Collections');
globals.CharacterArray.comment="I am the abstract superclass of string-like collections.";
smalltalk.addMethod(
smalltalk.method({
selector: ",",
protocol: 'copying',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._asString();
$ctx1.sendIdx["asString"]=1;
$1=_st($2).__comma(_st(aString)._asString());
return $1;
}, function($ctx1) {$ctx1.fill(self,",",{aString:aString},globals.CharacterArray)})},
args: ["aString"],
source: ", aString\x0a\x09^ self asString, aString asString",
messageSends: [",", "asString"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "add:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._errorReadOnly();
return self}, function($ctx1) {$ctx1.fill(self,"add:",{anObject:anObject},globals.CharacterArray)})},
args: ["anObject"],
source: "add: anObject\x0a\x09self errorReadOnly",
messageSends: ["errorReadOnly"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "asLowercase",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._fromString_(_st(self._asString())._asLowercase());
return $1;
}, function($ctx1) {$ctx1.fill(self,"asLowercase",{},globals.CharacterArray)})},
args: [],
source: "asLowercase\x0a\x09^ self class fromString: self asString asLowercase",
messageSends: ["fromString:", "class", "asLowercase", "asString"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "asNumber",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._asString())._asNumber();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asNumber",{},globals.CharacterArray)})},
args: [],
source: "asNumber\x0a\x09^ self asString asNumber",
messageSends: ["asNumber", "asString"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._subclassResponsibility();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asString",{},globals.CharacterArray)})},
args: [],
source: "asString\x0a\x09^ self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "asSymbol",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._asString();
return $1;
}, function($ctx1) {$ctx1.fill(self,"asSymbol",{},globals.CharacterArray)})},
args: [],
source: "asSymbol\x0a\x09^ self asString",
messageSends: ["asString"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "asUppercase",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._fromString_(_st(self._asString())._asUppercase());
return $1;
}, function($ctx1) {$ctx1.fill(self,"asUppercase",{},globals.CharacterArray)})},
args: [],
source: "asUppercase\x0a\x09^ self class fromString: self asString asUppercase",
messageSends: ["fromString:", "class", "asUppercase", "asString"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (anIndex,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._errorReadOnly();
return self}, function($ctx1) {$ctx1.fill(self,"at:put:",{anIndex:anIndex,anObject:anObject},globals.CharacterArray)})},
args: ["anIndex", "anObject"],
source: "at: anIndex put: anObject\x0a\x09self errorReadOnly",
messageSends: ["errorReadOnly"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "errorReadOnly",
protocol: 'error handling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._error_("Object is read-only");
return self}, function($ctx1) {$ctx1.fill(self,"errorReadOnly",{},globals.CharacterArray)})},
args: [],
source: "errorReadOnly\x0a\x09self error: 'Object is read-only'",
messageSends: ["error:"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._asString())._printOn_(aStream);
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.CharacterArray)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09self asString printOn: aStream",
messageSends: ["printOn:", "asString"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "putOn:",
protocol: 'streaming',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutString_(self);
return self}, function($ctx1) {$ctx1.fill(self,"putOn:",{aStream:aStream},globals.CharacterArray)})},
args: ["aStream"],
source: "putOn: aStream\x0a\x09aStream nextPutString: self",
messageSends: ["nextPutString:"],
referencedClasses: []
}),
globals.CharacterArray);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._errorReadOnly();
return self}, function($ctx1) {$ctx1.fill(self,"remove:",{anObject:anObject},globals.CharacterArray)})},
args: ["anObject"],
source: "remove: anObject\x0a\x09self errorReadOnly",
messageSends: ["errorReadOnly"],
referencedClasses: []
}),
globals.CharacterArray);


smalltalk.addMethod(
smalltalk.method({
selector: "fromString:",
protocol: 'instance creation',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"fromString:",{aString:aString},globals.CharacterArray.klass)})},
args: ["aString"],
source: "fromString: aString\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.CharacterArray.klass);


smalltalk.addClass('String', globals.CharacterArray, [], 'Kernel-Collections');
globals.String.comment="I am an indexed collection of Characters. Unlike most Smalltalk dialects, Amber doesn't provide the Character class. Instead, elements of a String are single character strings.\x0a\x0aString inherits many useful methods from its hierarchy, such as\x0a\x09`Collection >> #,`";
smalltalk.addMethod(
smalltalk.method({
selector: ",",
protocol: 'copying',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self) + aString;
return self}, function($ctx1) {$ctx1.fill(self,",",{aString:aString},globals.String)})},
args: ["aString"],
source: ", aString\x0a\x09<return String(self) + aString>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "<",
protocol: 'comparing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self) < aString._asString();
return self}, function($ctx1) {$ctx1.fill(self,"<",{aString:aString},globals.String)})},
args: ["aString"],
source: "< aString\x0a\x09<return String(self) < aString._asString()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "<=",
protocol: 'comparing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self) <= aString._asString();
return self}, function($ctx1) {$ctx1.fill(self,"<=",{aString:aString},globals.String)})},
args: ["aString"],
source: "<= aString\x0a\x09<return String(self) <= aString._asString()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return aString != null &&
			typeof aString._isString === "function" &&
			aString._isString() &&
			String(self) === String(aString)
	;
return self}, function($ctx1) {$ctx1.fill(self,"=",{aString:aString},globals.String)})},
args: ["aString"],
source: "= aString\x0a\x09<\x0a\x09\x09return aString != null &&\x0a\x09\x09\x09typeof aString._isString === \x22function\x22 &&\x0a\x09\x09\x09aString._isString() &&\x0a\x09\x09\x09String(self) === String(aString)\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "==",
protocol: 'comparing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__eq(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"==",{aString:aString},globals.String)})},
args: ["aString"],
source: "== aString\x0a\x09^ self = aString",
messageSends: ["="],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: ">",
protocol: 'comparing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self) > aString._asString();
return self}, function($ctx1) {$ctx1.fill(self,">",{aString:aString},globals.String)})},
args: ["aString"],
source: "> aString\x0a\x09<return String(self) >> aString._asString()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: ">=",
protocol: 'comparing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self) >= aString._asString();
return self}, function($ctx1) {$ctx1.fill(self,">=",{aString:aString},globals.String)})},
args: ["aString"],
source: ">= aString\x0a\x09<return String(self) >>= aString._asString()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asJSON\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asJavascript",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if(self.search(/^[a-zA-Z0-9_:.$ ]*$/) == -1)
			return "\"" + self.replace(/[\x00-\x1f"\\\x7f-\x9f]/g, function(ch){var c=ch.charCodeAt(0);return "\\x"+("0"+c.toString(16)).slice(-2)}) + "\"";
		else
			return "\"" + self + "\"";
	;
return self}, function($ctx1) {$ctx1.fill(self,"asJavascript",{},globals.String)})},
args: [],
source: "asJavascript\x0a\x09<\x0a\x09\x09if(self.search(/^[a-zA-Z0-9_:.$ ]*$/) == -1)\x0a\x09\x09\x09return \x22\x5c\x22\x22 + self.replace(/[\x5cx00-\x5cx1f\x22\x5c\x5c\x5cx7f-\x5cx9f]/g, function(ch){var c=ch.charCodeAt(0);return \x22\x5c\x5cx\x22+(\x220\x22+c.toString(16)).slice(-2)}) + \x22\x5c\x22\x22;\x0a\x09\x09else\x0a\x09\x09\x09return \x22\x5c\x22\x22 + self + \x22\x5c\x22\x22;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asLowercase",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toLowerCase();
return self}, function($ctx1) {$ctx1.fill(self,"asLowercase",{},globals.String)})},
args: [],
source: "asLowercase\x0a\x09<return self.toLowerCase()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asMutator",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st(self._last()).__eq(":");
if(! smalltalk.assert($1)){
$2=self.__comma(":");
return $2;
};
return self;
}, function($ctx1) {$ctx1.fill(self,"asMutator",{},globals.String)})},
args: [],
source: "asMutator\x0a\x09\x22Answer a setter selector. For example,\x0a\x09#name asMutator returns #name:\x22\x0a\x0a\x09self last = ':' ifFalse: [  ^ self, ':' ].\x0a\x09^ self",
messageSends: ["ifFalse:", "=", "last", ","],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asNumber",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return Number(self);
return self}, function($ctx1) {$ctx1.fill(self,"asNumber",{},globals.String)})},
args: [],
source: "asNumber\x0a\x09<return Number(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asRegexp",
protocol: 'converting',
fn: function (){
var self=this;
function $RegularExpression(){return globals.RegularExpression||(typeof RegularExpression=="undefined"?nil:RegularExpression)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($RegularExpression())._fromString_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"asRegexp",{},globals.String)})},
args: [],
source: "asRegexp\x0a\x09^ RegularExpression fromString: self",
messageSends: ["fromString:"],
referencedClasses: ["RegularExpression"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asSelector",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.selector(self);
return self}, function($ctx1) {$ctx1.fill(self,"asSelector",{},globals.String)})},
args: [],
source: "asSelector\x0a\x09<return smalltalk.selector(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asString",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asString\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asSymbol",
protocol: 'converting',
fn: function (){
var self=this;
return self;
},
args: [],
source: "asSymbol\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asUppercase",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.toUpperCase();
return self}, function($ctx1) {$ctx1.fill(self,"asUppercase",{},globals.String)})},
args: [],
source: "asUppercase\x0a\x09<return self.toUpperCase()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asciiValue",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.charCodeAt(0);;
return self}, function($ctx1) {$ctx1.fill(self,"asciiValue",{},globals.String)})},
args: [],
source: "asciiValue\x0a\x09<return self.charCodeAt(0);>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (anIndex,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self)[anIndex - 1] || aBlock._value();
return self}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{anIndex:anIndex,aBlock:aBlock},globals.String)})},
args: ["anIndex", "aBlock"],
source: "at: anIndex ifAbsent: aBlock\x0a\x09<return String(self)[anIndex - 1] || aBlock._value()>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:ifAbsent:",
protocol: 'accessing',
fn: function (anIndex,aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var result = String(self)[anIndex - 1];
		return result ? aBlock._value_(result) : anotherBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:ifAbsent:",{anIndex:anIndex,aBlock:aBlock,anotherBlock:anotherBlock},globals.String)})},
args: ["anIndex", "aBlock", "anotherBlock"],
source: "at: anIndex ifPresent: aBlock ifAbsent: anotherBlock\x0a\x09<\x0a\x09\x09var result = String(self)[anIndex - 1];\x0a\x09\x09return result ? aBlock._value_(result) : anotherBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "capitalized",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._isEmpty();
if(smalltalk.assert($2)){
$1=self;
} else {
$1=_st(_st(self._first())._asUppercase()).__comma(self._allButFirst());
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"capitalized",{},globals.String)})},
args: [],
source: "capitalized\x0a\x09^ self isEmpty\x0a\x09\x09ifTrue: [ self ]\x0a\x09\x09ifFalse: [ self first asUppercase, self allButFirst ]",
messageSends: ["ifTrue:ifFalse:", "isEmpty", ",", "asUppercase", "first", "allButFirst"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "charCodeAt:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.charCodeAt(anInteger - 1);
return self}, function($ctx1) {$ctx1.fill(self,"charCodeAt:",{anInteger:anInteger},globals.String)})},
args: ["anInteger"],
source: "charCodeAt: anInteger\x0a\x09<return self.charCodeAt(anInteger - 1)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "copyFrom:to:",
protocol: 'copying',
fn: function (anIndex,anotherIndex){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.substring(anIndex - 1, anotherIndex);
return self}, function($ctx1) {$ctx1.fill(self,"copyFrom:to:",{anIndex:anIndex,anotherIndex:anotherIndex},globals.String)})},
args: ["anIndex", "anotherIndex"],
source: "copyFrom: anIndex to: anotherIndex\x0a\x09<return self.substring(anIndex - 1, anotherIndex)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "crlfSanitized",
protocol: 'converting',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._lines())._join_(_st($String())._lf());
return $1;
}, function($ctx1) {$ctx1.fill(self,"crlfSanitized",{},globals.String)})},
args: [],
source: "crlfSanitized\x0a\x09^ self lines join: String lf",
messageSends: ["join:", "lines", "lf"],
referencedClasses: ["String"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "deepCopy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._shallowCopy();
return $1;
}, function($ctx1) {$ctx1.fill(self,"deepCopy",{},globals.String)})},
args: [],
source: "deepCopy\x0a\x09^ self shallowCopy",
messageSends: ["shallowCopy"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "escaped",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return escape(self);
return self}, function($ctx1) {$ctx1.fill(self,"escaped",{},globals.String)})},
args: [],
source: "escaped\x0a\x09<return escape(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "identityHash",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self.__comma("s");
return $1;
}, function($ctx1) {$ctx1.fill(self,"identityHash",{},globals.String)})},
args: [],
source: "identityHash\x0a\x09^ self, 's'",
messageSends: [","],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "includesSubString:",
protocol: 'testing',
fn: function (subString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.indexOf(subString) != -1;
return self}, function($ctx1) {$ctx1.fill(self,"includesSubString:",{subString:subString},globals.String)})},
args: ["subString"],
source: "includesSubString: subString\x0a\x09<return self.indexOf(subString) != -1>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "isCapitalized",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1;
$3=self._first();
$ctx1.sendIdx["first"]=1;
$2=_st($3)._asUppercase();
$1=_st($2).__eq_eq(self._first());
return $1;
}, function($ctx1) {$ctx1.fill(self,"isCapitalized",{},globals.String)})},
args: [],
source: "isCapitalized\x0a\x09^ self first asUppercase == self first",
messageSends: ["==", "asUppercase", "first"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isImmutable\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "isString",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isString\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "isVowel",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._size()).__eq((1)))._and_((function(){
return smalltalk.withContext(function($ctx2) {
return "aeiou"._includes_(self._asLowercase());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"isVowel",{},globals.String)})},
args: [],
source: "isVowel\x0a\x09\x22Answer true if the receiver is a one character string containing a voyel\x22\x0a\x09\x0a\x09^ self size = 1 and: [ 'aeiou' includes: self asLowercase ]",
messageSends: ["and:", "=", "size", "includes:", "asLowercase"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "join:",
protocol: 'split join',
fn: function (aCollection){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($String())._streamContents_((function(stream){
return smalltalk.withContext(function($ctx2) {
return _st(aCollection)._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx3) {
return _st(stream)._nextPutAll_(_st(each)._asString());
$ctx3.sendIdx["nextPutAll:"]=1;
}, function($ctx3) {$ctx3.fillBlock({each:each},$ctx2,2)})}),(function(){
return smalltalk.withContext(function($ctx3) {
return _st(stream)._nextPutAll_(self);
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,3)})}));
}, function($ctx2) {$ctx2.fillBlock({stream:stream},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"join:",{aCollection:aCollection},globals.String)})},
args: ["aCollection"],
source: "join: aCollection\x0a\x09^ String\x0a\x09\x09streamContents: [ :stream | aCollection\x0a\x09\x09\x09\x09do: [ :each | stream nextPutAll: each asString ]\x0a\x09\x09\x09\x09separatedBy: [ stream nextPutAll: self ]]",
messageSends: ["streamContents:", "do:separatedBy:", "nextPutAll:", "asString"],
referencedClasses: ["String"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "lineIndicesDo:",
protocol: 'split join',
fn: function (aBlock){
var self=this;
var cr,lf,start,sz,nextLF,nextCR;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$4,$5,$3,$6,$7,$9,$8,$10,$11;
var $early={};
try {
start=(1);
sz=self._size();
cr=_st($String())._cr();
nextCR=self._indexOf_startingAt_(cr,(1));
$ctx1.sendIdx["indexOf:startingAt:"]=1;
lf=_st($String())._lf();
nextLF=self._indexOf_startingAt_(lf,(1));
$ctx1.sendIdx["indexOf:startingAt:"]=2;
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(start).__lt_eq(sz);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileTrue_((function(){
return smalltalk.withContext(function($ctx2) {
$2=_st(nextLF).__eq((0));
$ctx2.sendIdx["="]=1;
$1=_st($2)._and_((function(){
return smalltalk.withContext(function($ctx3) {
return _st(nextCR).__eq((0));
$ctx3.sendIdx["="]=2;
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,3)})}));
$ctx2.sendIdx["and:"]=1;
if(smalltalk.assert($1)){
_st(aBlock)._value_value_value_(start,sz,sz);
$ctx2.sendIdx["value:value:value:"]=1;
throw $early=[self];
};
$4=_st(nextCR).__eq((0));
$ctx2.sendIdx["="]=3;
$3=_st($4)._or_((function(){
return smalltalk.withContext(function($ctx3) {
$5=(0).__lt(nextLF);
$ctx3.sendIdx["<"]=1;
return _st($5)._and_((function(){
return smalltalk.withContext(function($ctx4) {
return _st(nextLF).__lt(nextCR);
}, function($ctx4) {$ctx4.fillBlock({},$ctx3,6)})}));
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,5)})}));
if(smalltalk.assert($3)){
$6=start;
$7=_st(nextLF).__minus((1));
$ctx2.sendIdx["-"]=1;
_st(aBlock)._value_value_value_($6,$7,nextLF);
$ctx2.sendIdx["value:value:value:"]=2;
start=(1).__plus(nextLF);
$ctx2.sendIdx["+"]=1;
start;
nextLF=self._indexOf_startingAt_(lf,start);
$ctx2.sendIdx["indexOf:startingAt:"]=3;
return nextLF;
} else {
$9=(1).__plus(nextCR);
$ctx2.sendIdx["+"]=2;
$8=_st($9).__eq(nextLF);
if(smalltalk.assert($8)){
$10=start;
$11=_st(nextCR).__minus((1));
$ctx2.sendIdx["-"]=2;
_st(aBlock)._value_value_value_($10,$11,nextLF);
$ctx2.sendIdx["value:value:value:"]=3;
start=(1).__plus(nextLF);
$ctx2.sendIdx["+"]=3;
start;
nextCR=self._indexOf_startingAt_(cr,start);
$ctx2.sendIdx["indexOf:startingAt:"]=4;
nextCR;
nextLF=self._indexOf_startingAt_(lf,start);
$ctx2.sendIdx["indexOf:startingAt:"]=5;
return nextLF;
} else {
_st(aBlock)._value_value_value_(start,_st(nextCR).__minus((1)),nextCR);
start=(1).__plus(nextCR);
start;
nextCR=self._indexOf_startingAt_(cr,start);
return nextCR;
};
};
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"lineIndicesDo:",{aBlock:aBlock,cr:cr,lf:lf,start:start,sz:sz,nextLF:nextLF,nextCR:nextCR},globals.String)})},
args: ["aBlock"],
source: "lineIndicesDo: aBlock\x0a\x09\x22execute aBlock with 3 arguments for each line:\x0a\x09- start index of line\x0a\x09- end index of line without line delimiter\x0a\x09- end index of line including line delimiter(s) CR, LF or CRLF\x22\x0a\x09\x0a\x09| cr lf start sz nextLF nextCR |\x0a\x09start := 1.\x0a\x09sz := self size.\x0a\x09cr := String cr.\x0a\x09nextCR := self indexOf: cr startingAt: 1.\x0a\x09lf := String lf.\x0a\x09nextLF := self indexOf: lf startingAt: 1.\x0a\x09[ start <= sz ] whileTrue: [ \x0a\x09\x09(nextLF = 0 and: [ nextCR = 0 ])\x0a\x09\x09\x09ifTrue: [ \x22No more CR, nor LF, the string is over\x22\x0a\x09\x09\x09\x09\x09aBlock value: start value: sz value: sz.\x0a\x09\x09\x09\x09\x09^ self ].\x0a\x09\x09(nextCR = 0 or: [ 0 < nextLF and: [ nextLF < nextCR ] ])\x0a\x09\x09\x09ifTrue: [ \x22Found a LF\x22\x0a\x09\x09\x09\x09\x09aBlock value: start value: nextLF - 1 value: nextLF.\x0a\x09\x09\x09\x09\x09start := 1 + nextLF.\x0a\x09\x09\x09\x09\x09nextLF := self indexOf: lf startingAt: start ]\x0a\x09\x09\x09ifFalse: [ 1 + nextCR = nextLF\x0a\x09\x09\x09\x09ifTrue: [ \x22Found a CR-LF pair\x22\x0a\x09\x09\x09\x09\x09aBlock value: start value: nextCR - 1 value: nextLF.\x0a\x09\x09\x09\x09\x09start := 1 + nextLF.\x0a\x09\x09\x09\x09\x09nextCR := self indexOf: cr startingAt: start.\x0a\x09\x09\x09\x09\x09nextLF := self indexOf: lf startingAt: start ]\x0a\x09\x09\x09\x09ifFalse: [ \x22Found a CR\x22\x0a\x09\x09\x09\x09\x09aBlock value: start value: nextCR - 1 value: nextCR.\x0a\x09\x09\x09\x09\x09start := 1 + nextCR.\x0a\x09\x09\x09\x09\x09nextCR := self indexOf: cr startingAt: start ] ]]",
messageSends: ["size", "cr", "indexOf:startingAt:", "lf", "whileTrue:", "<=", "ifTrue:", "and:", "=", "value:value:value:", "ifTrue:ifFalse:", "or:", "<", "-", "+"],
referencedClasses: ["String"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "lineNumber:",
protocol: 'split join',
fn: function (anIndex){
var self=this;
var lineCount;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3;
var $early={};
try {
lineCount=(0);
self._lineIndicesDo_((function(start,endWithoutDelimiters,end){
return smalltalk.withContext(function($ctx2) {
lineCount=_st(lineCount).__plus((1));
$2=lineCount;
$1=_st($2).__eq(anIndex);
if(smalltalk.assert($1)){
$3=self._copyFrom_to_(start,endWithoutDelimiters);
throw $early=[$3];
};
}, function($ctx2) {$ctx2.fillBlock({start:start,endWithoutDelimiters:endWithoutDelimiters,end:end},$ctx1,1)})}));
return nil;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"lineNumber:",{anIndex:anIndex,lineCount:lineCount},globals.String)})},
args: ["anIndex"],
source: "lineNumber: anIndex\x0a\x09\x22Answer a string containing the characters in the given line number.\x22\x0a\x0a\x09| lineCount |\x0a\x09lineCount := 0.\x0a\x09self lineIndicesDo: [ :start :endWithoutDelimiters :end |\x0a\x09\x09(lineCount := lineCount + 1) = anIndex ifTrue: [ ^ self copyFrom: start to: endWithoutDelimiters ]].\x0a\x09^ nil",
messageSends: ["lineIndicesDo:", "ifTrue:", "=", "+", "copyFrom:to:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "lines",
protocol: 'split join',
fn: function (){
var self=this;
var lines;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1;
lines=_st($Array())._new();
self._linesDo_((function(aLine){
return smalltalk.withContext(function($ctx2) {
return _st(lines)._add_(aLine);
}, function($ctx2) {$ctx2.fillBlock({aLine:aLine},$ctx1,1)})}));
$1=lines;
return $1;
}, function($ctx1) {$ctx1.fill(self,"lines",{lines:lines},globals.String)})},
args: [],
source: "lines\x0a\x09\x22Answer an array of lines composing this receiver without the line ending delimiters.\x22\x0a\x0a\x09| lines |\x0a\x09lines := Array new.\x0a\x09self linesDo: [ :aLine | lines add: aLine ].\x0a\x09^ lines",
messageSends: ["new", "linesDo:", "add:"],
referencedClasses: ["Array"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "linesDo:",
protocol: 'split join',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._lineIndicesDo_((function(start,endWithoutDelimiters,end){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_(self._copyFrom_to_(start,endWithoutDelimiters));
}, function($ctx2) {$ctx2.fillBlock({start:start,endWithoutDelimiters:endWithoutDelimiters,end:end},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"linesDo:",{aBlock:aBlock},globals.String)})},
args: ["aBlock"],
source: "linesDo: aBlock\x0a\x09\x22Execute aBlock with each line in this string. The terminating line\x0a\x09delimiters CR, LF or CRLF pairs are not included in what is passed to aBlock\x22\x0a\x0a\x09self lineIndicesDo: [ :start :endWithoutDelimiters :end |\x0a\x09\x09aBlock value: (self copyFrom: start to: endWithoutDelimiters) ]",
messageSends: ["lineIndicesDo:", "value:", "copyFrom:to:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "match:",
protocol: 'regular expressions',
fn: function (aRegexp){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.search(aRegexp) != -1;
return self}, function($ctx1) {$ctx1.fill(self,"match:",{aRegexp:aRegexp},globals.String)})},
args: ["aRegexp"],
source: "match: aRegexp\x0a\x09<return self.search(aRegexp) != -1>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "matchesOf:",
protocol: 'regular expressions',
fn: function (aRegularExpression){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.match(aRegularExpression);
return self}, function($ctx1) {$ctx1.fill(self,"matchesOf:",{aRegularExpression:aRegularExpression},globals.String)})},
args: ["aRegularExpression"],
source: "matchesOf: aRegularExpression\x0a\x09<return self.match(aRegularExpression)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "numericallyIndexable",
protocol: 'private',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(self);
return self}, function($ctx1) {$ctx1.fill(self,"numericallyIndexable",{},globals.String)})},
args: [],
source: "numericallyIndexable\x0a\x09<return String(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "printNl",
protocol: 'printing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
console.log(self);
return self}, function($ctx1) {$ctx1.fill(self,"printNl",{},globals.String)})},
args: [],
source: "printNl\x0a\x09<console.log(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(aStream)._nextPutAll_("'");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aStream)._nextPutAll_(self);
$ctx1.sendIdx["nextPutAll:"]=2;
$1=_st(aStream)._nextPutAll_("'");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.String)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream \x0a\x09\x09nextPutAll: '''';\x0a\x09\x09nextPutAll: self;\x0a\x09\x09nextPutAll: ''''",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "replace:with:",
protocol: 'regular expressions',
fn: function (aString,anotherString){
var self=this;
function $RegularExpression(){return globals.RegularExpression||(typeof RegularExpression=="undefined"?nil:RegularExpression)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._replaceRegexp_with_(_st($RegularExpression())._fromString_flag_(aString,"g"),anotherString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"replace:with:",{aString:aString,anotherString:anotherString},globals.String)})},
args: ["aString", "anotherString"],
source: "replace: aString with: anotherString\x0a\x09^ self replaceRegexp: (RegularExpression fromString: aString flag: 'g') with: anotherString",
messageSends: ["replaceRegexp:with:", "fromString:flag:"],
referencedClasses: ["RegularExpression"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "replaceRegexp:with:",
protocol: 'regular expressions',
fn: function (aRegexp,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.replace(aRegexp, aString);
return self}, function($ctx1) {$ctx1.fill(self,"replaceRegexp:with:",{aRegexp:aRegexp,aString:aString},globals.String)})},
args: ["aRegexp", "aString"],
source: "replaceRegexp: aRegexp with: aString\x0a\x09<return self.replace(aRegexp, aString)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "reversed",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.split("").reverse().join("");
return self}, function($ctx1) {$ctx1.fill(self,"reversed",{},globals.String)})},
args: [],
source: "reversed\x0a\x09<return self.split(\x22\x22).reverse().join(\x22\x22)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "shallowCopy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._fromString_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"shallowCopy",{},globals.String)})},
args: [],
source: "shallowCopy\x0a\x09^ self class fromString: self",
messageSends: ["fromString:", "class"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.length;
return self}, function($ctx1) {$ctx1.fill(self,"size",{},globals.String)})},
args: [],
source: "size\x0a\x09<return self.length>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "subStrings:",
protocol: 'split join',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._tokenize_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"subStrings:",{aString:aString},globals.String)})},
args: ["aString"],
source: "subStrings: aString\x0a\x09^ self tokenize: aString",
messageSends: ["tokenize:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "tokenize:",
protocol: 'split join',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.split(aString);
return self}, function($ctx1) {$ctx1.fill(self,"tokenize:",{aString:aString},globals.String)})},
args: ["aString"],
source: "tokenize: aString\x0a\x09<return self.split(aString)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "trimBoth",
protocol: 'regular expressions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._trimBoth_("\x5cs");
return $1;
}, function($ctx1) {$ctx1.fill(self,"trimBoth",{},globals.String)})},
args: [],
source: "trimBoth\x0a\x09^ self trimBoth: '\x5cs'",
messageSends: ["trimBoth:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "trimBoth:",
protocol: 'regular expressions',
fn: function (separators){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._trimLeft_(separators))._trimRight_(separators);
return $1;
}, function($ctx1) {$ctx1.fill(self,"trimBoth:",{separators:separators},globals.String)})},
args: ["separators"],
source: "trimBoth: separators\x0a\x09^ (self trimLeft: separators) trimRight: separators",
messageSends: ["trimRight:", "trimLeft:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "trimLeft",
protocol: 'regular expressions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._trimLeft_("\x5cs");
return $1;
}, function($ctx1) {$ctx1.fill(self,"trimLeft",{},globals.String)})},
args: [],
source: "trimLeft\x0a\x09^ self trimLeft: '\x5cs'",
messageSends: ["trimLeft:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "trimLeft:",
protocol: 'regular expressions',
fn: function (separators){
var self=this;
function $RegularExpression(){return globals.RegularExpression||(typeof RegularExpression=="undefined"?nil:RegularExpression)}
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1;
$3=_st("^[".__comma(separators)).__comma("]+");
$ctx1.sendIdx[","]=1;
$2=_st($RegularExpression())._fromString_flag_($3,"g");
$1=self._replaceRegexp_with_($2,"");
return $1;
}, function($ctx1) {$ctx1.fill(self,"trimLeft:",{separators:separators},globals.String)})},
args: ["separators"],
source: "trimLeft: separators\x0a\x09^ self replaceRegexp: (RegularExpression fromString: '^[', separators, ']+' flag: 'g') with: ''",
messageSends: ["replaceRegexp:with:", "fromString:flag:", ","],
referencedClasses: ["RegularExpression"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "trimRight",
protocol: 'regular expressions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._trimRight_("\x5cs");
return $1;
}, function($ctx1) {$ctx1.fill(self,"trimRight",{},globals.String)})},
args: [],
source: "trimRight\x0a\x09^ self trimRight: '\x5cs'",
messageSends: ["trimRight:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "trimRight:",
protocol: 'regular expressions',
fn: function (separators){
var self=this;
function $RegularExpression(){return globals.RegularExpression||(typeof RegularExpression=="undefined"?nil:RegularExpression)}
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1;
$3=_st("[".__comma(separators)).__comma("]+$");
$ctx1.sendIdx[","]=1;
$2=_st($RegularExpression())._fromString_flag_($3,"g");
$1=self._replaceRegexp_with_($2,"");
return $1;
}, function($ctx1) {$ctx1.fill(self,"trimRight:",{separators:separators},globals.String)})},
args: ["separators"],
source: "trimRight: separators\x0a\x09^ self replaceRegexp: (RegularExpression fromString: '[', separators, ']+$' flag: 'g') with: ''",
messageSends: ["replaceRegexp:with:", "fromString:flag:", ","],
referencedClasses: ["RegularExpression"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "unescaped",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return unescape(self);
return self}, function($ctx1) {$ctx1.fill(self,"unescaped",{},globals.String)})},
args: [],
source: "unescaped\x0a\x09<return unescape(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "uriComponentDecoded",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return decodeURIComponent(self);
return self}, function($ctx1) {$ctx1.fill(self,"uriComponentDecoded",{},globals.String)})},
args: [],
source: "uriComponentDecoded\x0a\x09<return decodeURIComponent(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "uriComponentEncoded",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return encodeURIComponent(self);
return self}, function($ctx1) {$ctx1.fill(self,"uriComponentEncoded",{},globals.String)})},
args: [],
source: "uriComponentEncoded\x0a\x09<return encodeURIComponent(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "uriDecoded",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return decodeURI(self);
return self}, function($ctx1) {$ctx1.fill(self,"uriDecoded",{},globals.String)})},
args: [],
source: "uriDecoded\x0a\x09<return decodeURI(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "uriEncoded",
protocol: 'converting',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return encodeURI(self);
return self}, function($ctx1) {$ctx1.fill(self,"uriEncoded",{},globals.String)})},
args: [],
source: "uriEncoded\x0a\x09<return encodeURI(self)>",
messageSends: [],
referencedClasses: []
}),
globals.String);


smalltalk.addMethod(
smalltalk.method({
selector: "cr",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return '\r';
return self}, function($ctx1) {$ctx1.fill(self,"cr",{},globals.String.klass)})},
args: [],
source: "cr\x0a\x09<return '\x5cr'>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "crlf",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return '\r\n';
return self}, function($ctx1) {$ctx1.fill(self,"crlf",{},globals.String.klass)})},
args: [],
source: "crlf\x0a\x09<return '\x5cr\x5cn'>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "esc",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._fromCharCode_((27));
return $1;
}, function($ctx1) {$ctx1.fill(self,"esc",{},globals.String.klass)})},
args: [],
source: "esc\x0a\x09^ self fromCharCode: 27",
messageSends: ["fromCharCode:"],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "fromCharCode:",
protocol: 'instance creation',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String.fromCharCode(anInteger);
return self}, function($ctx1) {$ctx1.fill(self,"fromCharCode:",{anInteger:anInteger},globals.String.klass)})},
args: ["anInteger"],
source: "fromCharCode: anInteger\x0a\x09<return String.fromCharCode(anInteger)>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "fromString:",
protocol: 'instance creation',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String(aString);
return self}, function($ctx1) {$ctx1.fill(self,"fromString:",{aString:aString},globals.String.klass)})},
args: ["aString"],
source: "fromString: aString\x0a\x09\x09<return String(aString)>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "lf",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return '\n';
return self}, function($ctx1) {$ctx1.fill(self,"lf",{},globals.String.klass)})},
args: [],
source: "lf\x0a\x09<return '\x5cn'>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "random",
protocol: 'random',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return (Math.random()*(22/32)+(10/32)).toString(32).slice(2);;
return self}, function($ctx1) {$ctx1.fill(self,"random",{},globals.String.klass)})},
args: [],
source: "random\x0a\x09\x22Returns random alphanumeric string beginning with letter\x22\x0a\x09<return (Math.random()*(22/32)+(10/32)).toString(32).slice(2);>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "randomNotIn:",
protocol: 'random',
fn: function (aString){
var self=this;
var result;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st((function(){
return smalltalk.withContext(function($ctx2) {
result=self._random();
result;
return _st(aString)._includesSubString_(result);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileTrue();
$1=result;
return $1;
}, function($ctx1) {$ctx1.fill(self,"randomNotIn:",{aString:aString,result:result},globals.String.klass)})},
args: ["aString"],
source: "randomNotIn: aString\x0a\x09| result |\x0a\x09[ result := self random. aString includesSubString: result ] whileTrue.\x0a\x09^ result",
messageSends: ["whileTrue", "random", "includesSubString:"],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "space",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return ' ';
return self}, function($ctx1) {$ctx1.fill(self,"space",{},globals.String.klass)})},
args: [],
source: "space\x0a\x09<return ' '>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "streamClass",
protocol: 'accessing',
fn: function (){
var self=this;
function $StringStream(){return globals.StringStream||(typeof StringStream=="undefined"?nil:StringStream)}
return $StringStream();
},
args: [],
source: "streamClass\x0a\x09\x09^ StringStream",
messageSends: [],
referencedClasses: ["StringStream"]
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "tab",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return '\t';
return self}, function($ctx1) {$ctx1.fill(self,"tab",{},globals.String.klass)})},
args: [],
source: "tab\x0a\x09<return '\x5ct'>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'instance creation',
fn: function (aUTFCharCode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return String.fromCharCode(aUTFCharCode);;
return self}, function($ctx1) {$ctx1.fill(self,"value:",{aUTFCharCode:aUTFCharCode},globals.String.klass)})},
args: ["aUTFCharCode"],
source: "value: aUTFCharCode\x0a\x0a\x09<return String.fromCharCode(aUTFCharCode);>",
messageSends: [],
referencedClasses: []
}),
globals.String.klass);


smalltalk.addClass('Set', globals.Collection, ['defaultBucket', 'slowBucketStores', 'fastBuckets', 'size'], 'Kernel-Collections');
globals.Set.comment="I represent an unordered set of objects without duplicates.\x0a\x0a## Implementation notes\x0a\x0aI put elements into different stores based on their type.\x0aThe goal is to store some elements into native JS object property names to be fast.\x0a\x0aIf an unboxed element has typeof 'string', 'boolean' or 'number', or an element is nil, null or undefined,\x0aI store it as a property name in an empty (== Object.create(null)) JS object, different for each type\x0a(for simplicity, nil/null/undefined is treated as one and included with the two booleans).\x0a\x0aIf element happen to be an object, I try to store them in `ArrayBucketStore`. I have two of them by default,\x0aone hashed using the Smalltalk class name, the other one using the JS constructor name. It is possible to have more or less\x0ainstances of `ArrayBucketStores`, see `#initializeSlowBucketStores`.\x0a\x0aAs a last resort, if none of the `ArrayBucketStore` instances can find a suitable bucket, the `defaultBucket` is used,\x0awhich is an `Array`.";
smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$4,$3,$5;
var $early={};
try {
$2=self._class();
$ctx1.sendIdx["class"]=1;
$1=_st($2).__eq(_st(aCollection)._class());
$ctx1.sendIdx["="]=1;
if(! smalltalk.assert($1)){
return false;
};
$4=self._size();
$ctx1.sendIdx["size"]=1;
$3=_st($4).__eq(_st(aCollection)._size());
if(! smalltalk.assert($3)){
return false;
};
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$5=_st(aCollection)._includes_(each);
if(! smalltalk.assert($5)){
throw $early=[false];
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,3)})}));
return true;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"=",{aCollection:aCollection},globals.Set)})},
args: ["aCollection"],
source: "= aCollection\x0a\x09self class = aCollection class ifFalse: [ ^ false ].\x0a\x09self size = aCollection size ifFalse: [ ^ false ].\x0a\x09self do: [ :each | (aCollection includes: each) ifFalse: [ ^ false ] ].\x0a\x09^ true",
messageSends: ["ifFalse:", "=", "class", "size", "do:", "includes:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "add:",
protocol: 'adding/removing',
fn: function (anObject){
var self=this;
var bucket;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
bucket=self._bucketsOfElement_(anObject);
$2=_st(bucket)._second();
if(($receiver = $2) == nil || $receiver == null){
var object,slowBucket;
object=_st(bucket)._first();
$ctx1.sendIdx["first"]=1;
object;
slowBucket=_st(bucket)._third();
slowBucket;
_st(slowBucket)._indexOf_ifAbsent_(object,(function(){
return smalltalk.withContext(function($ctx2) {
_st(slowBucket)._add_(object);
self["@size"]=_st(self["@size"]).__plus((1));
return self["@size"];
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
$1=object;
} else {
var primitiveBucket;
primitiveBucket=$receiver;
$1=self._add_in_(_st(bucket)._first(),primitiveBucket);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"add:",{anObject:anObject,bucket:bucket},globals.Set)})},
args: ["anObject"],
source: "add: anObject\x0a\x09| bucket |\x0a\x09bucket := self bucketsOfElement: anObject.\x0a\x09^ bucket second\x0a\x09\x09ifNil: [\x0a\x09\x09\x09| object slowBucket |\x0a\x09\x09\x09object := bucket first.\x0a\x09\x09\x09slowBucket := bucket third.\x0a\x09\x09\x09slowBucket \x0a\x09\x09\x09\x09indexOf: object \x0a\x09\x09\x09\x09ifAbsent: [ \x0a\x09\x09\x09\x09\x09slowBucket add: object. \x0a\x09\x09\x09\x09\x09size := size + 1 ].\x0a\x09\x09\x09object ]\x0a\x09\x09ifNotNil: [ :primitiveBucket | \x0a\x09\x09\x09self \x0a\x09\x09\x09\x09add: bucket first \x0a\x09\x09\x09\x09in: primitiveBucket ]",
messageSends: ["bucketsOfElement:", "ifNil:ifNotNil:", "second", "first", "third", "indexOf:ifAbsent:", "add:", "+", "add:in:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "add:in:",
protocol: 'private',
fn: function (anObject,anotherObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		if (anObject in anotherObject.store) { return false; }
		self['@size']++;
		return anotherObject.store[anObject] = true;
	;
return self}, function($ctx1) {$ctx1.fill(self,"add:in:",{anObject:anObject,anotherObject:anotherObject},globals.Set)})},
args: ["anObject", "anotherObject"],
source: "add: anObject in: anotherObject\x0a\x09<\x0a\x09\x09if (anObject in anotherObject.store) { return false; }\x0a\x09\x09self['@size']++;\x0a\x09\x09return anotherObject.store[anObject] = true;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "bucketsOfElement:",
protocol: 'private',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var type, bucket, prim = anObject == null ? (anObject = nil) : anObject.valueOf();
		if ((type = typeof prim) === "object") {
			if (anObject !== nil) {
				bucket = null;
				self['@slowBucketStores'].some(function (store) {
					return bucket = store._bucketOfElement_(anObject);
				});
				return [ anObject, null, bucket || self['@defaultBucket'] ];
			}
			
			// include nil to well-known objects under 'boolean' fastBucket
			prim = null;
			type = 'boolean';
		}
		return [ prim, self['@fastBuckets'][type] ];
	;
return self}, function($ctx1) {$ctx1.fill(self,"bucketsOfElement:",{anObject:anObject},globals.Set)})},
args: ["anObject"],
source: "bucketsOfElement: anObject\x0a\x09\x22Find the appropriate bucket for `anObject`.\x0a\x09For optimization purposes, directly answer an array with: \x0a\x09- the object to be store\x0a\x09- the primitive bucket\x0a\x09- the slow bucket\x22\x0a\x09\x0a\x09<\x0a\x09\x09var type, bucket, prim = anObject == null ? (anObject = nil) : anObject.valueOf();\x0a\x09\x09if ((type = typeof prim) === \x22object\x22) {\x0a\x09\x09\x09if (anObject !== nil) {\x0a\x09\x09\x09\x09bucket = null;\x0a\x09\x09\x09\x09self['@slowBucketStores'].some(function (store) {\x0a\x09\x09\x09\x09\x09return bucket = store._bucketOfElement_(anObject);\x0a\x09\x09\x09\x09});\x0a\x09\x09\x09\x09return [ anObject, null, bucket || self['@defaultBucket'] ];\x0a\x09\x09\x09}\x0a\x09\x09\x09\x0a\x09\x09\x09// include nil to well-known objects under 'boolean' fastBucket\x0a\x09\x09\x09prim = null;\x0a\x09\x09\x09type = 'boolean';\x0a\x09\x09}\x0a\x09\x09return [ prim, self['@fastBuckets'][type] ];\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "classNameOf:",
protocol: 'private',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return anObject.klass && anObject.klass.className;
return self}, function($ctx1) {$ctx1.fill(self,"classNameOf:",{anObject:anObject},globals.Set)})},
args: ["anObject"],
source: "classNameOf: anObject\x0a\x09\x22Answer the class name of `anObject`, or `undefined` \x0a\x09if `anObject` is not an Smalltalk object\x22\x0a\x09\x0a\x09<return anObject.klass && anObject.klass.className>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "collect:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var collection;
return smalltalk.withContext(function($ctx1) { 
var $1;
collection=_st(self._class())._new();
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(collection)._add_(_st(aBlock)._value_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=collection;
return $1;
}, function($ctx1) {$ctx1.fill(self,"collect:",{aBlock:aBlock,collection:collection},globals.Set)})},
args: ["aBlock"],
source: "collect: aBlock\x0a\x09| collection |\x0a\x09collection := self class new.\x0a\x09self do: [ :each | collection add: (aBlock value: each) ].\x0a\x09^ collection",
messageSends: ["new", "class", "do:", "add:", "value:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "detect:ifNone:",
protocol: 'enumerating',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
var $early={};
try {
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(each);
if(smalltalk.assert($1)){
throw $early=[each];
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=_st(anotherBlock)._value();
return $2;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"detect:ifNone:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.Set)})},
args: ["aBlock", "anotherBlock"],
source: "detect: aBlock ifNone: anotherBlock\x0a\x09self do: [ :each | (aBlock value: each) ifTrue: [ ^each ] ].\x0a\x09^ anotherBlock value",
messageSends: ["do:", "ifTrue:", "value:", "value"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "do:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var el, keys, i;
		el = self['@fastBuckets'];
		keys = Object.keys(el);
		for (i = 0; i < keys.length; ++i) {
			var fastBucket = el[keys[i]], fn = fastBucket.fn, store = Object.keys(fastBucket.store);
			if (fn) { for (var j = 0; j < store.length; ++j) { aBlock._value_(fn(store[j])); } }
			else { store._do_(aBlock); }
		}
		el = self['@slowBucketStores'];
		for (i = 0; i < el.length; ++i) { el[i]._do_(aBlock); }
		self['@defaultBucket']._do_(aBlock);
	;
return self}, function($ctx1) {$ctx1.fill(self,"do:",{aBlock:aBlock},globals.Set)})},
args: ["aBlock"],
source: "do: aBlock\x0a\x09<\x0a\x09\x09var el, keys, i;\x0a\x09\x09el = self['@fastBuckets'];\x0a\x09\x09keys = Object.keys(el);\x0a\x09\x09for (i = 0; i < keys.length; ++i) {\x0a\x09\x09\x09var fastBucket = el[keys[i]], fn = fastBucket.fn, store = Object.keys(fastBucket.store);\x0a\x09\x09\x09if (fn) { for (var j = 0; j < store.length; ++j) { aBlock._value_(fn(store[j])); } }\x0a\x09\x09\x09else { store._do_(aBlock); }\x0a\x09\x09}\x0a\x09\x09el = self['@slowBucketStores'];\x0a\x09\x09for (i = 0; i < el.length; ++i) { el[i]._do_(aBlock); }\x0a\x09\x09self['@defaultBucket']._do_(aBlock);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "includes:",
protocol: 'testing',
fn: function (anObject){
var self=this;
var bucket;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$4,$1;
bucket=self._bucketsOfElement_(anObject);
$2=_st(bucket)._second();
if(($receiver = $2) == nil || $receiver == null){
$3=_st(bucket)._third();
$4=_st(bucket)._first();
$ctx1.sendIdx["first"]=1;
$1=_st($3)._includes_($4);
} else {
var primitiveBucket;
primitiveBucket=$receiver;
$1=self._includes_in_(_st(bucket)._first(),primitiveBucket);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"includes:",{anObject:anObject,bucket:bucket},globals.Set)})},
args: ["anObject"],
source: "includes: anObject\x0a\x09| bucket |\x0a\x09bucket := self bucketsOfElement: anObject.\x0a\x09^ bucket second\x0a\x09\x09ifNil: [ bucket third includes: bucket first ]\x0a\x09\x09ifNotNil: [ :primitiveBucket | self includes: bucket first in: primitiveBucket ]",
messageSends: ["bucketsOfElement:", "ifNil:ifNotNil:", "second", "includes:", "third", "first", "includes:in:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "includes:in:",
protocol: 'private',
fn: function (anObject,anotherObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return anObject in anotherObject.store;
return self}, function($ctx1) {$ctx1.fill(self,"includes:in:",{anObject:anObject,anotherObject:anotherObject},globals.Set)})},
args: ["anObject", "anotherObject"],
source: "includes: anObject in: anotherObject\x0a\x09<return anObject in anotherObject.store>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
globals.Set.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@defaultBucket"]=[];
self._initializeSlowBucketStores();
$1=self._removeAll();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.Set)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09\x0a\x09defaultBucket := #().\x0a\x09self\x0a\x09\x09initializeSlowBucketStores;\x0a\x09\x09removeAll",
messageSends: ["initialize", "initializeSlowBucketStores", "removeAll"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "initializeSlowBucketStores",
protocol: 'initialization',
fn: function (){
var self=this;
function $ArrayBucketStore(){return globals.ArrayBucketStore||(typeof ArrayBucketStore=="undefined"?nil:ArrayBucketStore)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($ArrayBucketStore())._hashBlock_((function(x){
return smalltalk.withContext(function($ctx2) {
return self._classNameOf_(x);
}, function($ctx2) {$ctx2.fillBlock({x:x},$ctx1,1)})}));
$ctx1.sendIdx["hashBlock:"]=1;
self["@slowBucketStores"]=[$1,_st($ArrayBucketStore())._hashBlock_((function(x){
return smalltalk.withContext(function($ctx2) {
return self._jsConstructorNameOf_(x);
}, function($ctx2) {$ctx2.fillBlock({x:x},$ctx1,2)})}))];
return self}, function($ctx1) {$ctx1.fill(self,"initializeSlowBucketStores",{},globals.Set)})},
args: [],
source: "initializeSlowBucketStores\x0a\x09slowBucketStores := {\x0a\x09\x09ArrayBucketStore hashBlock: [ :x | self classNameOf: x ].\x0a\x09\x09ArrayBucketStore hashBlock: [ :x | self jsConstructorNameOf: x ]\x0a\x09}",
messageSends: ["hashBlock:", "classNameOf:", "jsConstructorNameOf:"],
referencedClasses: ["ArrayBucketStore"]
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "jsConstructorNameOf:",
protocol: 'private',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return anObject.constructor && anObject.constructor.name;
return self}, function($ctx1) {$ctx1.fill(self,"jsConstructorNameOf:",{anObject:anObject},globals.Set)})},
args: ["anObject"],
source: "jsConstructorNameOf: anObject\x0a\x09<return anObject.constructor && anObject.constructor.name>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.Set.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
$ctx1.sendIdx["printOn:"]=1;
_st(aStream)._nextPutAll_(" (");
$ctx1.sendIdx["nextPutAll:"]=1;
self._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._printOn_(aStream);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(aStream)._nextPutAll_(" ");
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Set)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09\x0a\x09aStream nextPutAll: ' ('.\x0a\x09self \x0a\x09\x09do: [ :each | each printOn: aStream ]\x0a\x09\x09separatedBy: [ aStream nextPutAll: ' ' ].\x0a\x09aStream nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "do:separatedBy:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:ifAbsent:",
protocol: 'adding/removing',
fn: function (anObject,aBlock){
var self=this;
var bucket;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$4,$5,$1;
var $early={};
try {
bucket=self._bucketsOfElement_(anObject);
$2=_st(bucket)._second();
if(($receiver = $2) == nil || $receiver == null){
$3=_st(bucket)._third();
$4=_st(bucket)._first();
$ctx1.sendIdx["first"]=1;
_st($3)._remove_ifAbsent_($4,(function(){
return smalltalk.withContext(function($ctx2) {
$5=_st(aBlock)._value();
throw $early=[$5];
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
self["@size"]=_st(self["@size"]).__minus((1));
$1=self["@size"];
} else {
var primitiveBucket;
primitiveBucket=$receiver;
$1=self._remove_in_(_st(bucket)._first(),primitiveBucket);
};
return $1;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"remove:ifAbsent:",{anObject:anObject,aBlock:aBlock,bucket:bucket},globals.Set)})},
args: ["anObject", "aBlock"],
source: "remove: anObject ifAbsent: aBlock\x0a\x09| bucket |\x0a\x09bucket := self bucketsOfElement: anObject.\x0a\x09^ bucket second\x0a\x09\x09ifNil: [ bucket third remove: bucket first ifAbsent: [ ^aBlock value ]. size := size - 1 ]\x0a\x09\x09ifNotNil: [ :primitiveBucket | self remove: bucket first in: primitiveBucket ]",
messageSends: ["bucketsOfElement:", "ifNil:ifNotNil:", "second", "remove:ifAbsent:", "third", "first", "value", "-", "remove:in:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:in:",
protocol: 'private',
fn: function (anObject,anotherObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
if (delete anotherObject.store[anObject]) self['@size']--;
return self}, function($ctx1) {$ctx1.fill(self,"remove:in:",{anObject:anObject,anotherObject:anotherObject},globals.Set)})},
args: ["anObject", "anotherObject"],
source: "remove: anObject in: anotherObject\x0a\x09<if (delete anotherObject.store[anObject]) self['@size']-->",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "removeAll",
protocol: 'adding/removing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self['@fastBuckets'] = {
			'boolean': { store: Object.create(null), fn: function (x) { return {'true': true, 'false': false, 'null': null}[x]; } },
			'number': { store: Object.create(null), fn: Number },
			'string': { store: Object.create(null) }
		};
		self['@slowBucketStores'].forEach(function (x) { x._removeAll(); });
		self['@defaultBucket']._removeAll();
		self['@size'] = 0;
	;
return self}, function($ctx1) {$ctx1.fill(self,"removeAll",{},globals.Set)})},
args: [],
source: "removeAll\x0a\x09<\x0a\x09\x09self['@fastBuckets'] = {\x0a\x09\x09\x09'boolean': { store: Object.create(null), fn: function (x) { return {'true': true, 'false': false, 'null': null}[x]; } },\x0a\x09\x09\x09'number': { store: Object.create(null), fn: Number },\x0a\x09\x09\x09'string': { store: Object.create(null) }\x0a\x09\x09};\x0a\x09\x09self['@slowBucketStores'].forEach(function (x) { x._removeAll(); });\x0a\x09\x09self['@defaultBucket']._removeAll();\x0a\x09\x09self['@size'] = 0;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "select:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
var collection;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
collection=_st(self._class())._new();
self._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value_(each);
if(smalltalk.assert($1)){
return _st(collection)._add_(each);
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=collection;
return $2;
}, function($ctx1) {$ctx1.fill(self,"select:",{aBlock:aBlock,collection:collection},globals.Set)})},
args: ["aBlock"],
source: "select: aBlock\x0a\x09| collection |\x0a\x09collection := self class new.\x0a\x09self do: [ :each |\x0a\x09\x09(aBlock value: each) ifTrue: [\x0a\x09\x09\x09collection add: each ] ].\x0a\x09^ collection",
messageSends: ["new", "class", "do:", "ifTrue:", "value:", "add:"],
referencedClasses: []
}),
globals.Set);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@size"];
return $1;
},
args: [],
source: "size\x0a\x09^ size",
messageSends: [],
referencedClasses: []
}),
globals.Set);



smalltalk.addClass('Queue', globals.Object, ['read', 'readIndex', 'write'], 'Kernel-Collections');
globals.Queue.comment="I am a one-sided queue.\x0a\x0a## Usage\x0a\x0aUse `#nextPut:` to add items to the queue.\x0aUse `#next` or `#nextIfAbsent:` to get (and remove) the next item in the queue.\x0a\x0a## Implementation notes\x0a\x0aA Queue uses two OrderedCollections inside,\x0a`read` is at the front, is not modified and only read using `readIndex`.\x0a`write` is at the back and is appended new items.\x0aWhen `read` is exhausted, `write` is promoted to `read` and new `write` is created.\x0a\x0aAs a consequence, no data moving is done by me, write appending may do data moving\x0awhen growing `write`, but this is left to engine to implement as good as it chooses to.";
smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
globals.Queue.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@read"]=_st($OrderedCollection())._new();
$ctx1.sendIdx["new"]=1;
self["@write"]=_st($OrderedCollection())._new();
self["@readIndex"]=(1);
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.Queue)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09read := OrderedCollection new.\x0a\x09write := OrderedCollection new.\x0a\x09readIndex := 1",
messageSends: ["initialize", "new"],
referencedClasses: ["OrderedCollection"]
}),
globals.Queue);

smalltalk.addMethod(
smalltalk.method({
selector: "next",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._nextIfAbsent_((function(){
return smalltalk.withContext(function($ctx2) {
return self._error_("Cannot read from empty Queue.");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"next",{},globals.Queue)})},
args: [],
source: "next\x0a\x09^ self nextIfAbsent: [ self error: 'Cannot read from empty Queue.' ]",
messageSends: ["nextIfAbsent:", "error:"],
referencedClasses: []
}),
globals.Queue);

smalltalk.addMethod(
smalltalk.method({
selector: "nextIfAbsent:",
protocol: 'accessing',
fn: function (aBlock){
var self=this;
var result;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4;
var $early={};
try {
result=_st(self["@read"])._at_ifAbsent_(self["@readIndex"],(function(){
return smalltalk.withContext(function($ctx2) {
$1=_st(self["@write"])._isEmpty();
if(smalltalk.assert($1)){
$2=_st(self["@readIndex"]).__gt((1));
if(smalltalk.assert($2)){
self["@read"]=[];
self["@read"];
self["@readIndex"]=(1);
self["@readIndex"];
};
$3=_st(aBlock)._value();
throw $early=[$3];
};
self["@read"]=self["@write"];
self["@read"];
self["@readIndex"]=(1);
self["@readIndex"];
self["@write"]=_st($OrderedCollection())._new();
self["@write"];
return _st(self["@read"])._first();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
_st(self["@read"])._at_put_(self["@readIndex"],nil);
self["@readIndex"]=_st(self["@readIndex"]).__plus((1));
$4=result;
return $4;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"nextIfAbsent:",{aBlock:aBlock,result:result},globals.Queue)})},
args: ["aBlock"],
source: "nextIfAbsent: aBlock\x0a\x09| result |\x0a\x09result := read at: readIndex ifAbsent: [\x0a\x09\x09write isEmpty ifTrue: [\x0a\x09\x09\x09readIndex > 1 ifTrue: [ read := #(). readIndex := 1 ].\x0a\x09\x09\x09^ aBlock value ].\x0a\x09\x09read := write.\x0a\x09\x09readIndex := 1.\x0a\x09\x09write := OrderedCollection new.\x0a\x09\x09read first ].\x0a\x09read at: readIndex put: nil.\x0a\x09readIndex := readIndex + 1.\x0a\x09^ result",
messageSends: ["at:ifAbsent:", "ifTrue:", "isEmpty", ">", "value", "new", "first", "at:put:", "+"],
referencedClasses: ["OrderedCollection"]
}),
globals.Queue);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPut:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@write"])._add_(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"nextPut:",{anObject:anObject},globals.Queue)})},
args: ["anObject"],
source: "nextPut: anObject\x0a\x09write add: anObject",
messageSends: ["add:"],
referencedClasses: []
}),
globals.Queue);



smalltalk.addClass('RegularExpression', globals.Object, [], 'Kernel-Collections');
globals.RegularExpression.comment="I represent a regular expression object. My instances are JavaScript `RegExp` object.";
smalltalk.addMethod(
smalltalk.method({
selector: "compile:",
protocol: 'evaluating',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.compile(aString);
return self}, function($ctx1) {$ctx1.fill(self,"compile:",{aString:aString},globals.RegularExpression)})},
args: ["aString"],
source: "compile: aString\x0a\x09<return self.compile(aString)>",
messageSends: [],
referencedClasses: []
}),
globals.RegularExpression);

smalltalk.addMethod(
smalltalk.method({
selector: "exec:",
protocol: 'evaluating',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.exec(aString) || nil;
return self}, function($ctx1) {$ctx1.fill(self,"exec:",{aString:aString},globals.RegularExpression)})},
args: ["aString"],
source: "exec: aString\x0a\x09<return self.exec(aString) || nil>",
messageSends: [],
referencedClasses: []
}),
globals.RegularExpression);

smalltalk.addMethod(
smalltalk.method({
selector: "test:",
protocol: 'evaluating',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.test(aString);
return self}, function($ctx1) {$ctx1.fill(self,"test:",{aString:aString},globals.RegularExpression)})},
args: ["aString"],
source: "test: aString\x0a\x09<return self.test(aString)>",
messageSends: [],
referencedClasses: []
}),
globals.RegularExpression);


smalltalk.addMethod(
smalltalk.method({
selector: "fromString:",
protocol: 'instance creation',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._fromString_flag_(aString,"");
return $1;
}, function($ctx1) {$ctx1.fill(self,"fromString:",{aString:aString},globals.RegularExpression.klass)})},
args: ["aString"],
source: "fromString: aString\x0a\x09\x09^ self fromString: aString flag: ''",
messageSends: ["fromString:flag:"],
referencedClasses: []
}),
globals.RegularExpression.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "fromString:flag:",
protocol: 'instance creation',
fn: function (aString,anotherString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return new RegExp(aString, anotherString);
return self}, function($ctx1) {$ctx1.fill(self,"fromString:flag:",{aString:aString,anotherString:anotherString},globals.RegularExpression.klass)})},
args: ["aString", "anotherString"],
source: "fromString: aString flag: anotherString\x0a\x09<return new RegExp(aString, anotherString)>",
messageSends: [],
referencedClasses: []
}),
globals.RegularExpression.klass);


smalltalk.addClass('Stream', globals.Object, ['collection', 'position', 'streamSize'], 'Kernel-Collections');
globals.Stream.comment="I represent an accessor for a sequence of objects. This sequence is referred to as my \x22contents\x22.\x0aMy instances are read/write streams to the contents sequence collection.";
smalltalk.addMethod(
smalltalk.method({
selector: "<<",
protocol: 'writing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._write_(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"<<",{anObject:anObject},globals.Stream)})},
args: ["anObject"],
source: "<< anObject\x0a\x09self write: anObject",
messageSends: ["write:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "atEnd",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._position()).__eq(self._size());
return $1;
}, function($ctx1) {$ctx1.fill(self,"atEnd",{},globals.Stream)})},
args: [],
source: "atEnd\x0a\x09^ self position = self size",
messageSends: ["=", "position", "size"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "atStart",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._position()).__eq((0));
return $1;
}, function($ctx1) {$ctx1.fill(self,"atStart",{},globals.Stream)})},
args: [],
source: "atStart\x0a\x09^ self position = 0",
messageSends: ["=", "position"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "close",
protocol: 'actions',
fn: function (){
var self=this;
return self},
args: [],
source: "close",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "collection",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@collection"];
return $1;
},
args: [],
source: "collection\x0a\x09^ collection",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "contents",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._collection())._copyFrom_to_((1),self._streamSize());
return $1;
}, function($ctx1) {$ctx1.fill(self,"contents",{},globals.Stream)})},
args: [],
source: "contents\x0a\x09^ self collection\x0a\x09\x09copyFrom: 1\x0a\x09\x09to: self streamSize",
messageSends: ["copyFrom:to:", "collection", "streamSize"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "do:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st((function(){
return smalltalk.withContext(function($ctx2) {
return self._atEnd();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._whileFalse_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(aBlock)._value_(self._next());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"do:",{aBlock:aBlock},globals.Stream)})},
args: ["aBlock"],
source: "do: aBlock\x0a\x09[ self atEnd ] whileFalse: [ aBlock value: self next ]",
messageSends: ["whileFalse:", "atEnd", "value:", "next"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "flush",
protocol: 'actions',
fn: function (){
var self=this;
return self},
args: [],
source: "flush",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "isEmpty",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._size()).__eq((0));
return $1;
}, function($ctx1) {$ctx1.fill(self,"isEmpty",{},globals.Stream)})},
args: [],
source: "isEmpty\x0a\x09^ self size = 0",
messageSends: ["=", "size"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "next",
protocol: 'reading',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$4,$3,$1;
$2=self._atEnd();
if(smalltalk.assert($2)){
$1=nil;
} else {
$4=self._position();
$ctx1.sendIdx["position"]=1;
$3=_st($4).__plus((1));
self._position_($3);
$1=_st(self["@collection"])._at_(self._position());
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"next",{},globals.Stream)})},
args: [],
source: "next\x0a\x09^ self atEnd\x0a\x09\x09ifTrue: [ nil ]\x0a\x09\x09ifFalse: [\x0a\x09\x09\x09self position: self position + 1.\x0a\x09\x09\x09collection at: self position ]",
messageSends: ["ifTrue:ifFalse:", "atEnd", "position:", "+", "position", "at:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "next:",
protocol: 'reading',
fn: function (anInteger){
var self=this;
var tempCollection;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
tempCollection=_st(_st(self._collection())._class())._new();
_st(anInteger)._timesRepeat_((function(){
return smalltalk.withContext(function($ctx2) {
$1=self._atEnd();
if(! smalltalk.assert($1)){
return _st(tempCollection)._add_(self._next());
};
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$2=tempCollection;
return $2;
}, function($ctx1) {$ctx1.fill(self,"next:",{anInteger:anInteger,tempCollection:tempCollection},globals.Stream)})},
args: ["anInteger"],
source: "next: anInteger\x0a\x09| tempCollection |\x0a\x09tempCollection := self collection class new.\x0a\x09anInteger timesRepeat: [\x0a\x09\x09self atEnd ifFalse: [\x0a\x09\x09tempCollection add: self next ]].\x0a\x09^ tempCollection",
messageSends: ["new", "class", "collection", "timesRepeat:", "ifFalse:", "atEnd", "add:", "next"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPut:",
protocol: 'writing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3,$4;
$2=self._position();
$ctx1.sendIdx["position"]=1;
$1=_st($2).__plus((1));
self._position_($1);
$3=self._collection();
$4=self._position();
$ctx1.sendIdx["position"]=2;
_st($3)._at_put_($4,anObject);
self._setStreamSize_(_st(self._streamSize())._max_(self._position()));
return self}, function($ctx1) {$ctx1.fill(self,"nextPut:",{anObject:anObject},globals.Stream)})},
args: ["anObject"],
source: "nextPut: anObject\x0a\x09self position: self position + 1.\x0a\x09self collection at: self position put: anObject.\x0a\x09self setStreamSize: (self streamSize max: self position)",
messageSends: ["position:", "+", "position", "at:put:", "collection", "setStreamSize:", "max:", "streamSize"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutAll:",
protocol: 'writing',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._nextPut_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"nextPutAll:",{aCollection:aCollection},globals.Stream)})},
args: ["aCollection"],
source: "nextPutAll: aCollection\x0a\x09aCollection do: [ :each |\x0a\x09\x09self nextPut: each ]",
messageSends: ["do:", "nextPut:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutString:",
protocol: 'writing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._nextPut_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"nextPutString:",{aString:aString},globals.Stream)})},
args: ["aString"],
source: "nextPutString: aString\x0a\x09self nextPut: aString",
messageSends: ["nextPut:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "peek",
protocol: 'reading',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._atEnd();
if(! smalltalk.assert($2)){
$1=_st(self._collection())._at_(_st(self._position()).__plus((1)));
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"peek",{},globals.Stream)})},
args: [],
source: "peek\x0a\x09^ self atEnd ifFalse: [\x0a\x09\x09self collection at: self position + 1 ]",
messageSends: ["ifFalse:", "atEnd", "at:", "collection", "+", "position"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "position",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@position"];
if(($receiver = $2) == nil || $receiver == null){
self["@position"]=(0);
$1=self["@position"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"position",{},globals.Stream)})},
args: [],
source: "position\x0a\x09^ position ifNil: [ position := 0 ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "position:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
self["@position"]=anInteger;
return self},
args: ["anInteger"],
source: "position: anInteger\x0a\x09position := anInteger",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "reset",
protocol: 'actions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._position_((0));
return self}, function($ctx1) {$ctx1.fill(self,"reset",{},globals.Stream)})},
args: [],
source: "reset\x0a\x09self position: 0",
messageSends: ["position:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "resetContents",
protocol: 'actions',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._reset();
self._setStreamSize_((0));
return self}, function($ctx1) {$ctx1.fill(self,"resetContents",{},globals.Stream)})},
args: [],
source: "resetContents\x0a\x09self reset.\x0a\x09self setStreamSize: 0",
messageSends: ["reset", "setStreamSize:"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "setCollection:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@collection"]=aCollection;
return self},
args: ["aCollection"],
source: "setCollection: aCollection\x0a\x09collection := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "setStreamSize:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
self["@streamSize"]=anInteger;
return self},
args: ["anInteger"],
source: "setStreamSize: anInteger\x0a\x09streamSize := anInteger",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "setToEnd",
protocol: 'positioning',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._position_(self._size());
return self}, function($ctx1) {$ctx1.fill(self,"setToEnd",{},globals.Stream)})},
args: [],
source: "setToEnd\x0a\x09self position: self size",
messageSends: ["position:", "size"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._streamSize();
return $1;
}, function($ctx1) {$ctx1.fill(self,"size",{},globals.Stream)})},
args: [],
source: "size\x0a\x09^ self streamSize",
messageSends: ["streamSize"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "skip:",
protocol: 'positioning',
fn: function (anInteger){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._position_(_st(_st(self._position()).__plus(anInteger))._min_max_(self._size(),(0)));
return self}, function($ctx1) {$ctx1.fill(self,"skip:",{anInteger:anInteger},globals.Stream)})},
args: ["anInteger"],
source: "skip: anInteger\x0a\x09self position: ((self position + anInteger) min: self size max: 0)",
messageSends: ["position:", "min:max:", "+", "position", "size"],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "streamSize",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@streamSize"];
return $1;
},
args: [],
source: "streamSize\x0a\x09^ streamSize",
messageSends: [],
referencedClasses: []
}),
globals.Stream);

smalltalk.addMethod(
smalltalk.method({
selector: "write:",
protocol: 'writing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(anObject)._putOn_(self);
return self}, function($ctx1) {$ctx1.fill(self,"write:",{anObject:anObject},globals.Stream)})},
args: ["anObject"],
source: "write: anObject\x0a\x09anObject putOn: self",
messageSends: ["putOn:"],
referencedClasses: []
}),
globals.Stream);


smalltalk.addMethod(
smalltalk.method({
selector: "on:",
protocol: 'instance creation',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._setCollection_(aCollection);
_st($2)._setStreamSize_(_st(aCollection)._size());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:",{aCollection:aCollection},globals.Stream.klass)})},
args: ["aCollection"],
source: "on: aCollection\x0a\x09\x09^ self new\x0a\x09\x09setCollection: aCollection;\x0a\x09\x09setStreamSize: aCollection size;\x0a\x09\x09yourself",
messageSends: ["setCollection:", "new", "setStreamSize:", "size", "yourself"],
referencedClasses: []
}),
globals.Stream.klass);


smalltalk.addClass('StringStream', globals.Stream, [], 'Kernel-Collections');
globals.StringStream.comment="I am a Stream specific to `String` objects.";
smalltalk.addMethod(
smalltalk.method({
selector: "cr",
protocol: 'writing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._nextPutAll_(_st($String())._cr());
return $1;
}, function($ctx1) {$ctx1.fill(self,"cr",{},globals.StringStream)})},
args: [],
source: "cr\x0a\x09^ self nextPutAll: String cr",
messageSends: ["nextPutAll:", "cr"],
referencedClasses: ["String"]
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "crlf",
protocol: 'writing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._nextPutAll_(_st($String())._crlf());
return $1;
}, function($ctx1) {$ctx1.fill(self,"crlf",{},globals.StringStream)})},
args: [],
source: "crlf\x0a\x09^ self nextPutAll: String crlf",
messageSends: ["nextPutAll:", "crlf"],
referencedClasses: ["String"]
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "lf",
protocol: 'writing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._nextPutAll_(_st($String())._lf());
return $1;
}, function($ctx1) {$ctx1.fill(self,"lf",{},globals.StringStream)})},
args: [],
source: "lf\x0a\x09^ self nextPutAll: String lf",
messageSends: ["nextPutAll:", "lf"],
referencedClasses: ["String"]
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "next:",
protocol: 'reading',
fn: function (anInteger){
var self=this;
var tempCollection;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
tempCollection=_st(_st(self._collection())._class())._new();
_st(anInteger)._timesRepeat_((function(){
return smalltalk.withContext(function($ctx2) {
$1=self._atEnd();
if(! smalltalk.assert($1)){
tempCollection=_st(tempCollection).__comma(self._next());
return tempCollection;
};
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$2=tempCollection;
return $2;
}, function($ctx1) {$ctx1.fill(self,"next:",{anInteger:anInteger,tempCollection:tempCollection},globals.StringStream)})},
args: ["anInteger"],
source: "next: anInteger\x0a\x09| tempCollection |\x0a\x09tempCollection := self collection class new.\x0a\x09anInteger timesRepeat: [\x0a\x09\x09self atEnd ifFalse: [\x0a\x09\x09tempCollection := tempCollection, self next ]].\x0a\x09^ tempCollection",
messageSends: ["new", "class", "collection", "timesRepeat:", "ifFalse:", "atEnd", ",", "next"],
referencedClasses: []
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPut:",
protocol: 'writing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._nextPutAll_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"nextPut:",{aString:aString},globals.StringStream)})},
args: ["aString"],
source: "nextPut: aString\x0a\x09self nextPutAll: aString",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutAll:",
protocol: 'writing',
fn: function (aString){
var self=this;
var pre,post;
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$2,$4,$5,$6,$9,$8,$10,$7,$11,$12,$14,$13;
$1=self._atEnd();
if(smalltalk.assert($1)){
$3=self._collection();
$ctx1.sendIdx["collection"]=1;
$2=_st($3).__comma(aString);
$ctx1.sendIdx[","]=1;
self._setCollection_($2);
$ctx1.sendIdx["setCollection:"]=1;
} else {
$4=self._collection();
$ctx1.sendIdx["collection"]=2;
$5=self._position();
$ctx1.sendIdx["position"]=1;
pre=_st($4)._copyFrom_to_((1),$5);
$ctx1.sendIdx["copyFrom:to:"]=1;
pre;
$6=self._collection();
$ctx1.sendIdx["collection"]=3;
$9=self._position();
$ctx1.sendIdx["position"]=2;
$8=_st($9).__plus((1));
$ctx1.sendIdx["+"]=2;
$10=_st(aString)._size();
$ctx1.sendIdx["size"]=1;
$7=_st($8).__plus($10);
$ctx1.sendIdx["+"]=1;
$11=_st(self._collection())._size();
$ctx1.sendIdx["size"]=2;
post=_st($6)._copyFrom_to_($7,$11);
post;
$12=_st(_st(pre).__comma(aString)).__comma(post);
$ctx1.sendIdx[","]=2;
self._setCollection_($12);
};
$14=self._position();
$ctx1.sendIdx["position"]=3;
$13=_st($14).__plus(_st(aString)._size());
self._position_($13);
self._setStreamSize_(_st(self._streamSize())._max_(self._position()));
return self}, function($ctx1) {$ctx1.fill(self,"nextPutAll:",{aString:aString,pre:pre,post:post},globals.StringStream)})},
args: ["aString"],
source: "nextPutAll: aString\x0a\x09| pre post |\x0a\x09self atEnd ifTrue: [ self setCollection: self collection, aString ] ifFalse: [\x0a\x09\x09pre := self collection copyFrom: 1 to: self position.\x0a\x09\x09post := self collection copyFrom: (self position + 1 + aString size) to: self collection size.\x0a\x09\x09self setCollection: pre, aString, post\x0a\x09].\x0a\x09self position: self position + aString size.\x0a\x09self setStreamSize: (self streamSize max: self position)",
messageSends: ["ifTrue:ifFalse:", "atEnd", "setCollection:", ",", "collection", "copyFrom:to:", "position", "+", "size", "position:", "setStreamSize:", "max:", "streamSize"],
referencedClasses: []
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutString:",
protocol: 'writing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._nextPutAll_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"nextPutString:",{aString:aString},globals.StringStream)})},
args: ["aString"],
source: "nextPutString: aString\x0a\x09self nextPutAll: aString",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "space",
protocol: 'writing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._nextPut_(" ");
return self}, function($ctx1) {$ctx1.fill(self,"space",{},globals.StringStream)})},
args: [],
source: "space\x0a\x09self nextPut: ' '",
messageSends: ["nextPut:"],
referencedClasses: []
}),
globals.StringStream);

smalltalk.addMethod(
smalltalk.method({
selector: "tab",
protocol: 'writing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._nextPutAll_(_st($String())._tab());
return $1;
}, function($ctx1) {$ctx1.fill(self,"tab",{},globals.StringStream)})},
args: [],
source: "tab\x0a\x09^ self nextPutAll: String tab",
messageSends: ["nextPutAll:", "tab"],
referencedClasses: ["String"]
}),
globals.StringStream);


});

define("amber_core/Kernel-Infrastructure", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects", "amber_core/Kernel-Collections"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Infrastructure');
smalltalk.packages["Kernel-Infrastructure"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('ConsoleErrorHandler', globals.Object, [], 'Kernel-Infrastructure');
globals.ConsoleErrorHandler.comment="I am manage Smalltalk errors, displaying the stack in the console.";
smalltalk.addMethod(
smalltalk.method({
selector: "handleError:",
protocol: 'error handling',
fn: function (anError){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(anError)._context();
$ctx1.sendIdx["context"]=1;
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
self._logErrorContext_(_st(anError)._context());
};
self._logError_(anError);
return self}, function($ctx1) {$ctx1.fill(self,"handleError:",{anError:anError},globals.ConsoleErrorHandler)})},
args: ["anError"],
source: "handleError: anError\x0a\x09anError context ifNotNil: [ self logErrorContext: anError context ].\x0a\x09self logError: anError",
messageSends: ["ifNotNil:", "context", "logErrorContext:", "logError:"],
referencedClasses: []
}),
globals.ConsoleErrorHandler);

smalltalk.addMethod(
smalltalk.method({
selector: "log:",
protocol: 'private',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(console)._log_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"log:",{aString:aString},globals.ConsoleErrorHandler)})},
args: ["aString"],
source: "log: aString\x0a\x09console log: aString",
messageSends: ["log:"],
referencedClasses: []
}),
globals.ConsoleErrorHandler);

smalltalk.addMethod(
smalltalk.method({
selector: "logContext:",
protocol: 'private',
fn: function (aContext){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aContext)._home();
$ctx1.sendIdx["home"]=1;
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
self._logContext_(_st(aContext)._home());
};
self._log_(_st(aContext)._asString());
return self}, function($ctx1) {$ctx1.fill(self,"logContext:",{aContext:aContext},globals.ConsoleErrorHandler)})},
args: ["aContext"],
source: "logContext: aContext\x0a\x09aContext home ifNotNil: [\x0a\x09\x09self logContext: aContext home ].\x0a\x09self log: aContext asString",
messageSends: ["ifNotNil:", "home", "logContext:", "log:", "asString"],
referencedClasses: []
}),
globals.ConsoleErrorHandler);

smalltalk.addMethod(
smalltalk.method({
selector: "logError:",
protocol: 'private',
fn: function (anError){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._log_(_st(anError)._messageText());
return self}, function($ctx1) {$ctx1.fill(self,"logError:",{anError:anError},globals.ConsoleErrorHandler)})},
args: ["anError"],
source: "logError: anError\x0a\x09self log: anError messageText",
messageSends: ["log:", "messageText"],
referencedClasses: []
}),
globals.ConsoleErrorHandler);

smalltalk.addMethod(
smalltalk.method({
selector: "logErrorContext:",
protocol: 'private',
fn: function (aContext){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
if(($receiver = aContext) == nil || $receiver == null){
aContext;
} else {
$1=_st(aContext)._home();
$ctx1.sendIdx["home"]=1;
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
self._logContext_(_st(aContext)._home());
};
};
return self}, function($ctx1) {$ctx1.fill(self,"logErrorContext:",{aContext:aContext},globals.ConsoleErrorHandler)})},
args: ["aContext"],
source: "logErrorContext: aContext\x0a\x09aContext ifNotNil: [\x0a\x09\x09aContext home ifNotNil: [\x0a\x09\x09\x09self logContext: aContext home ]]",
messageSends: ["ifNotNil:", "home", "logContext:"],
referencedClasses: []
}),
globals.ConsoleErrorHandler);


globals.ConsoleErrorHandler.klass.iVarNames = ['current'];
smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $ErrorHandler(){return globals.ErrorHandler||(typeof ErrorHandler=="undefined"?nil:ErrorHandler)}
return smalltalk.withContext(function($ctx1) { 
_st($ErrorHandler())._registerIfNone_(self._new());
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.ConsoleErrorHandler.klass)})},
args: [],
source: "initialize\x0a\x09ErrorHandler registerIfNone: self new",
messageSends: ["registerIfNone:", "new"],
referencedClasses: ["ErrorHandler"]
}),
globals.ConsoleErrorHandler.klass);


smalltalk.addClass('InterfacingObject', globals.Object, [], 'Kernel-Infrastructure');
globals.InterfacingObject.comment="I am superclass of all object that interface with user or environment. `Widget` and a few other classes are subclasses of me. I delegate all of the above APIs to `PlatformInterface`.\x0a\x0a## API\x0a\x0a    self alert: 'Hey, there is a problem'.\x0a    self confirm: 'Affirmative?'.\x0a    self prompt: 'Your name:'.\x0a\x0a    self ajax: #{\x0a        'url' -> '/patch.js'. 'type' -> 'GET'. dataType->'script'\x0a    }.";
smalltalk.addMethod(
smalltalk.method({
selector: "ajax:",
protocol: 'actions',
fn: function (anObject){
var self=this;
function $PlatformInterface(){return globals.PlatformInterface||(typeof PlatformInterface=="undefined"?nil:PlatformInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($PlatformInterface())._ajax_(anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ajax:",{anObject:anObject},globals.InterfacingObject)})},
args: ["anObject"],
source: "ajax: anObject\x0a\x09^ PlatformInterface ajax: anObject",
messageSends: ["ajax:"],
referencedClasses: ["PlatformInterface"]
}),
globals.InterfacingObject);

smalltalk.addMethod(
smalltalk.method({
selector: "alert:",
protocol: 'actions',
fn: function (aString){
var self=this;
function $PlatformInterface(){return globals.PlatformInterface||(typeof PlatformInterface=="undefined"?nil:PlatformInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($PlatformInterface())._alert_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"alert:",{aString:aString},globals.InterfacingObject)})},
args: ["aString"],
source: "alert: aString\x0a\x09^ PlatformInterface alert: aString",
messageSends: ["alert:"],
referencedClasses: ["PlatformInterface"]
}),
globals.InterfacingObject);

smalltalk.addMethod(
smalltalk.method({
selector: "confirm:",
protocol: 'actions',
fn: function (aString){
var self=this;
function $PlatformInterface(){return globals.PlatformInterface||(typeof PlatformInterface=="undefined"?nil:PlatformInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($PlatformInterface())._confirm_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"confirm:",{aString:aString},globals.InterfacingObject)})},
args: ["aString"],
source: "confirm: aString\x0a\x09^ PlatformInterface confirm: aString",
messageSends: ["confirm:"],
referencedClasses: ["PlatformInterface"]
}),
globals.InterfacingObject);

smalltalk.addMethod(
smalltalk.method({
selector: "prompt:",
protocol: 'actions',
fn: function (aString){
var self=this;
function $PlatformInterface(){return globals.PlatformInterface||(typeof PlatformInterface=="undefined"?nil:PlatformInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($PlatformInterface())._prompt_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"prompt:",{aString:aString},globals.InterfacingObject)})},
args: ["aString"],
source: "prompt: aString\x0a\x09^ PlatformInterface prompt: aString",
messageSends: ["prompt:"],
referencedClasses: ["PlatformInterface"]
}),
globals.InterfacingObject);



smalltalk.addClass('Environment', globals.InterfacingObject, [], 'Kernel-Infrastructure');
globals.Environment.comment="I provide an unified entry point to manipulate Amber packages, classes and methods.\x0a\x0aTypical use cases include IDEs, remote access and restricting browsing.";
smalltalk.addMethod(
smalltalk.method({
selector: "addInstVarNamed:to:",
protocol: 'compiling',
fn: function (aString,aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$5;
$1=self._classBuilder();
$2=_st(aClass)._superclass();
$3=_st(aClass)._name();
$ctx1.sendIdx["name"]=1;
$4=_st(_st(aClass)._instanceVariableNames())._copy();
_st($4)._add_(aString);
$5=_st($4)._yourself();
_st($1)._addSubclassOf_named_instanceVariableNames_package_($2,$3,$5,_st(_st(aClass)._package())._name());
return self}, function($ctx1) {$ctx1.fill(self,"addInstVarNamed:to:",{aString:aString,aClass:aClass},globals.Environment)})},
args: ["aString", "aClass"],
source: "addInstVarNamed: aString to: aClass\x0a\x09self classBuilder\x0a\x09\x09addSubclassOf: aClass superclass \x0a\x09\x09named: aClass name \x0a\x09\x09instanceVariableNames: (aClass instanceVariableNames copy add: aString; yourself)\x0a\x09\x09package: aClass package name",
messageSends: ["addSubclassOf:named:instanceVariableNames:package:", "classBuilder", "superclass", "name", "add:", "copy", "instanceVariableNames", "yourself", "package"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "allSelectors",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Smalltalk())._vm())._allSelectors();
return $1;
}, function($ctx1) {$ctx1.fill(self,"allSelectors",{},globals.Environment)})},
args: [],
source: "allSelectors\x0a\x09^ Smalltalk vm allSelectors",
messageSends: ["allSelectors", "vm"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "availableClassNames",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Smalltalk())._classes())._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._name();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"availableClassNames",{},globals.Environment)})},
args: [],
source: "availableClassNames\x0a\x09^ Smalltalk classes \x0a\x09\x09collect: [ :each | each name ]",
messageSends: ["collect:", "classes", "name"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "availablePackageNames",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Smalltalk())._packages())._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._name();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"availablePackageNames",{},globals.Environment)})},
args: [],
source: "availablePackageNames\x0a\x09^ Smalltalk packages \x0a\x09\x09collect: [ :each | each name ]",
messageSends: ["collect:", "packages", "name"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "availableProtocolsFor:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
var protocols;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
protocols=_st(aClass)._protocols();
$1=_st(aClass)._superclass();
$ctx1.sendIdx["superclass"]=1;
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
_st(protocols)._addAll_(self._availableProtocolsFor_(_st(aClass)._superclass()));
};
$2=_st(_st(_st(protocols)._asSet())._asArray())._sort();
return $2;
}, function($ctx1) {$ctx1.fill(self,"availableProtocolsFor:",{aClass:aClass,protocols:protocols},globals.Environment)})},
args: ["aClass"],
source: "availableProtocolsFor: aClass\x0a\x09| protocols |\x0a\x09\x0a\x09protocols := aClass protocols.\x0a\x09aClass superclass ifNotNil: [ protocols addAll: (self availableProtocolsFor: aClass superclass) ].\x0a\x09^ protocols asSet asArray sort",
messageSends: ["protocols", "ifNotNil:", "superclass", "addAll:", "availableProtocolsFor:", "sort", "asArray", "asSet"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "classBuilder",
protocol: 'accessing',
fn: function (){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($ClassBuilder())._new();
return $1;
}, function($ctx1) {$ctx1.fill(self,"classBuilder",{},globals.Environment)})},
args: [],
source: "classBuilder\x0a\x09^ ClassBuilder new",
messageSends: ["new"],
referencedClasses: ["ClassBuilder"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "classNamed:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(_st($Smalltalk())._globals())._at_(_st(aString)._asSymbol());
if(($receiver = $2) == nil || $receiver == null){
$1=self._error_("Invalid class name");
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"classNamed:",{aString:aString},globals.Environment)})},
args: ["aString"],
source: "classNamed: aString\x0a\x09^ (Smalltalk globals at: aString asSymbol)\x0a\x09\x09ifNil: [ self error: 'Invalid class name' ]",
messageSends: ["ifNil:", "at:", "globals", "asSymbol", "error:"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "classes",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Smalltalk())._classes();
return $1;
}, function($ctx1) {$ctx1.fill(self,"classes",{},globals.Environment)})},
args: [],
source: "classes\x0a\x09^ Smalltalk classes",
messageSends: ["classes"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "commitPackage:onSuccess:onError:",
protocol: 'actions',
fn: function (aPackage,aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(_st(aPackage)._transport())._commitOnSuccess_onError_(aBlock,anotherBlock);
return self}, function($ctx1) {$ctx1.fill(self,"commitPackage:onSuccess:onError:",{aPackage:aPackage,aBlock:aBlock,anotherBlock:anotherBlock},globals.Environment)})},
args: ["aPackage", "aBlock", "anotherBlock"],
source: "commitPackage: aPackage onSuccess: aBlock onError: anotherBlock\x0a\x09aPackage transport\x0a\x09\x09commitOnSuccess: aBlock\x0a\x09\x09onError: anotherBlock",
messageSends: ["commitOnSuccess:onError:", "transport"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "compileClassComment:for:",
protocol: 'compiling',
fn: function (aString,aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aClass)._comment_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"compileClassComment:for:",{aString:aString,aClass:aClass},globals.Environment)})},
args: ["aString", "aClass"],
source: "compileClassComment: aString for: aClass\x0a\x09aClass comment: aString",
messageSends: ["comment:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "compileClassDefinition:",
protocol: 'compiling',
fn: function (aString){
var self=this;
function $DoIt(){return globals.DoIt||(typeof DoIt=="undefined"?nil:DoIt)}
function $Error(){return globals.Error||(typeof Error=="undefined"?nil:Error)}
return smalltalk.withContext(function($ctx1) { 
_st((function(){
return smalltalk.withContext(function($ctx2) {
return self._eval_on_(aString,_st($DoIt())._new());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._on_do_($Error(),(function(error){
return smalltalk.withContext(function($ctx2) {
return self._alert_(_st(error)._messageText());
}, function($ctx2) {$ctx2.fillBlock({error:error},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"compileClassDefinition:",{aString:aString},globals.Environment)})},
args: ["aString"],
source: "compileClassDefinition: aString\x0a\x09[ self eval: aString on: DoIt new ]\x0a\x09\x09on: Error\x0a\x09\x09do: [ :error | self alert: error messageText ]",
messageSends: ["on:do:", "eval:on:", "new", "alert:", "messageText"],
referencedClasses: ["DoIt", "Error"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "compileMethod:for:protocol:",
protocol: 'compiling',
fn: function (sourceCode,class_,protocol){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(class_)._compile_protocol_(sourceCode,protocol);
return $1;
}, function($ctx1) {$ctx1.fill(self,"compileMethod:for:protocol:",{sourceCode:sourceCode,class_:class_,protocol:protocol},globals.Environment)})},
args: ["sourceCode", "class", "protocol"],
source: "compileMethod: sourceCode for: class protocol: protocol\x0a\x09^ class\x0a\x09\x09compile: sourceCode\x0a\x09\x09protocol: protocol",
messageSends: ["compile:protocol:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "copyClass:to:",
protocol: 'actions',
fn: function (aClass,aClassName){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st(_st($Smalltalk())._globals())._at_(aClassName);
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
$2=_st("A class named ".__comma(aClassName)).__comma(" already exists");
$ctx1.sendIdx[","]=1;
self._error_($2);
};
_st(_st($ClassBuilder())._new())._copyClass_named_(aClass,aClassName);
return self}, function($ctx1) {$ctx1.fill(self,"copyClass:to:",{aClass:aClass,aClassName:aClassName},globals.Environment)})},
args: ["aClass", "aClassName"],
source: "copyClass: aClass to: aClassName\x0a\x09(Smalltalk globals at: aClassName)\x0a\x09\x09ifNotNil: [ self error: 'A class named ', aClassName, ' already exists' ].\x0a\x09\x09\x0a\x09ClassBuilder new copyClass: aClass named: aClassName",
messageSends: ["ifNotNil:", "at:", "globals", "error:", ",", "copyClass:named:", "new"],
referencedClasses: ["Smalltalk", "ClassBuilder"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "doItReceiver",
protocol: 'accessing',
fn: function (){
var self=this;
function $DoIt(){return globals.DoIt||(typeof DoIt=="undefined"?nil:DoIt)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($DoIt())._new();
return $1;
}, function($ctx1) {$ctx1.fill(self,"doItReceiver",{},globals.Environment)})},
args: [],
source: "doItReceiver\x0a\x09^ DoIt new",
messageSends: ["new"],
referencedClasses: ["DoIt"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "eval:on:",
protocol: 'evaluating',
fn: function (aString,aReceiver){
var self=this;
var compiler;
function $Compiler(){return globals.Compiler||(typeof Compiler=="undefined"?nil:Compiler)}
function $Error(){return globals.Error||(typeof Error=="undefined"?nil:Error)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
var $early={};
try {
compiler=_st($Compiler())._new();
_st((function(){
return smalltalk.withContext(function($ctx2) {
return _st(compiler)._parseExpression_(aString);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._on_do_($Error(),(function(ex){
return smalltalk.withContext(function($ctx2) {
$1=self._alert_(_st(ex)._messageText());
throw $early=[$1];
}, function($ctx2) {$ctx2.fillBlock({ex:ex},$ctx1,2)})}));
$2=_st(compiler)._evaluateExpression_on_(aString,aReceiver);
return $2;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"eval:on:",{aString:aString,aReceiver:aReceiver,compiler:compiler},globals.Environment)})},
args: ["aString", "aReceiver"],
source: "eval: aString on: aReceiver\x0a\x09| compiler |\x0a\x09compiler := Compiler new.\x0a\x09[ compiler parseExpression: aString ] on: Error do: [ :ex |\x0a\x09\x09^ self alert: ex messageText ].\x0a\x09^ compiler evaluateExpression: aString on: aReceiver",
messageSends: ["new", "on:do:", "parseExpression:", "alert:", "messageText", "evaluateExpression:on:"],
referencedClasses: ["Compiler", "Error"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "evaluate:on:do:",
protocol: 'error handling',
fn: function (aBlock,anErrorClass,exceptionBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(aBlock)._tryCatch_((function(exception){
return smalltalk.withContext(function($ctx2) {
$1=_st(exception)._isKindOf_(self._classNamed_(_st(anErrorClass)._name()));
if(smalltalk.assert($1)){
return _st(exceptionBlock)._value_(exception);
} else {
return _st(exception)._signal();
};
}, function($ctx2) {$ctx2.fillBlock({exception:exception},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"evaluate:on:do:",{aBlock:aBlock,anErrorClass:anErrorClass,exceptionBlock:exceptionBlock},globals.Environment)})},
args: ["aBlock", "anErrorClass", "exceptionBlock"],
source: "evaluate: aBlock on: anErrorClass do: exceptionBlock\x0a\x09\x22Evaluate a block and catch exceptions happening on the environment stack\x22\x0a\x09\x0a\x09aBlock tryCatch: [ :exception | \x0a\x09\x09(exception isKindOf: (self classNamed: anErrorClass name))\x0a\x09\x09\x09ifTrue: [ exceptionBlock value: exception ]\x0a \x09\x09\x09ifFalse: [ exception signal ] ]",
messageSends: ["tryCatch:", "ifTrue:ifFalse:", "isKindOf:", "classNamed:", "name", "value:", "signal"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "inspect:",
protocol: 'actions',
fn: function (anObject){
var self=this;
function $Inspector(){return globals.Inspector||(typeof Inspector=="undefined"?nil:Inspector)}
return smalltalk.withContext(function($ctx1) { 
_st($Inspector())._inspect_(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"inspect:",{anObject:anObject},globals.Environment)})},
args: ["anObject"],
source: "inspect: anObject\x0a\x09Inspector inspect: anObject",
messageSends: ["inspect:"],
referencedClasses: ["Inspector"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "interpret:inContext:",
protocol: 'evaluating',
fn: function (aString,anAIContext){
var self=this;
var compiler,ast;
function $Compiler(){return globals.Compiler||(typeof Compiler=="undefined"?nil:Compiler)}
function $Error(){return globals.Error||(typeof Error=="undefined"?nil:Error)}
function $SemanticAnalyzer(){return globals.SemanticAnalyzer||(typeof SemanticAnalyzer=="undefined"?nil:SemanticAnalyzer)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
var $early={};
try {
compiler=_st($Compiler())._new();
_st((function(){
return smalltalk.withContext(function($ctx2) {
ast=_st(compiler)._parseExpression_(aString);
return ast;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._on_do_($Error(),(function(ex){
return smalltalk.withContext(function($ctx2) {
$1=self._alert_(_st(ex)._messageText());
throw $early=[$1];
}, function($ctx2) {$ctx2.fillBlock({ex:ex},$ctx1,2)})}));
_st(_st($SemanticAnalyzer())._on_(_st(_st(anAIContext)._receiver())._class()))._visit_(ast);
$2=_st(anAIContext)._evaluateNode_(ast);
return $2;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"interpret:inContext:",{aString:aString,anAIContext:anAIContext,compiler:compiler,ast:ast},globals.Environment)})},
args: ["aString", "anAIContext"],
source: "interpret: aString inContext: anAIContext\x0a\x09\x22Similar to #eval:on:, with the following differences:\x0a\x09- instead of compiling and running `aString`, `aString` is interpreted using an `ASTInterpreter`\x0a\x09- instead of evaluating against a receiver, evaluate in the context of `anAIContext`\x22\x0a\x0a\x09| compiler ast |\x0a\x09compiler := Compiler new.\x0a\x09[ ast := compiler parseExpression: aString ] on: Error do: [ :ex |\x0a\x09\x09^ self alert: ex messageText ].\x0a\x09(SemanticAnalyzer on: anAIContext receiver class)\x0a\x09\x09visit: ast.\x0a\x09^ anAIContext evaluateNode: ast",
messageSends: ["new", "on:do:", "parseExpression:", "alert:", "messageText", "visit:", "on:", "class", "receiver", "evaluateNode:"],
referencedClasses: ["Compiler", "Error", "SemanticAnalyzer"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "moveClass:toPackage:",
protocol: 'actions',
fn: function (aClass,aPackageName){
var self=this;
var package_;
function $Package(){return globals.Package||(typeof Package=="undefined"?nil:Package)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
package_=_st($Package())._named_(aPackageName);
$1=package_;
if(($receiver = $1) == nil || $receiver == null){
self._error_("Invalid package name");
} else {
$1;
};
$2=_st(package_).__eq_eq(_st(aClass)._package());
if(smalltalk.assert($2)){
return self;
};
_st(aClass)._package_(package_);
return self}, function($ctx1) {$ctx1.fill(self,"moveClass:toPackage:",{aClass:aClass,aPackageName:aPackageName,package_:package_},globals.Environment)})},
args: ["aClass", "aPackageName"],
source: "moveClass: aClass toPackage: aPackageName\x0a\x09| package |\x0a\x09\x0a\x09package := Package named: aPackageName.\x0a\x09package ifNil: [ self error: 'Invalid package name' ].\x0a\x09package == aClass package ifTrue: [ ^ self ].\x0a\x09\x0a\x09aClass package: package",
messageSends: ["named:", "ifNil:", "error:", "ifTrue:", "==", "package", "package:"],
referencedClasses: ["Package"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "moveMethod:toClass:",
protocol: 'actions',
fn: function (aMethod,aClassName){
var self=this;
var destinationClass;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
destinationClass=self._classNamed_(aClassName);
$2=destinationClass;
$3=_st(aMethod)._methodClass();
$ctx1.sendIdx["methodClass"]=1;
$1=_st($2).__eq_eq($3);
if(smalltalk.assert($1)){
return self;
};
_st(destinationClass)._compile_protocol_(_st(aMethod)._source(),_st(aMethod)._protocol());
_st(_st(aMethod)._methodClass())._removeCompiledMethod_(aMethod);
return self}, function($ctx1) {$ctx1.fill(self,"moveMethod:toClass:",{aMethod:aMethod,aClassName:aClassName,destinationClass:destinationClass},globals.Environment)})},
args: ["aMethod", "aClassName"],
source: "moveMethod: aMethod toClass: aClassName\x0a\x09| destinationClass |\x0a\x09\x0a\x09destinationClass := self classNamed: aClassName.\x0a\x09destinationClass == aMethod methodClass ifTrue: [ ^ self ].\x0a\x09\x0a\x09destinationClass \x0a\x09\x09compile: aMethod source\x0a\x09\x09protocol: aMethod protocol.\x0a\x09aMethod methodClass \x0a\x09\x09removeCompiledMethod: aMethod",
messageSends: ["classNamed:", "ifTrue:", "==", "methodClass", "compile:protocol:", "source", "protocol", "removeCompiledMethod:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "moveMethod:toProtocol:",
protocol: 'actions',
fn: function (aMethod,aProtocol){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aMethod)._protocol_(aProtocol);
return self}, function($ctx1) {$ctx1.fill(self,"moveMethod:toProtocol:",{aMethod:aMethod,aProtocol:aProtocol},globals.Environment)})},
args: ["aMethod", "aProtocol"],
source: "moveMethod: aMethod toProtocol: aProtocol\x0a\x09aMethod protocol: aProtocol",
messageSends: ["protocol:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "packages",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Smalltalk())._packages();
return $1;
}, function($ctx1) {$ctx1.fill(self,"packages",{},globals.Environment)})},
args: [],
source: "packages\x0a\x09^ Smalltalk packages",
messageSends: ["packages"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "registerErrorHandler:",
protocol: 'services',
fn: function (anErrorHandler){
var self=this;
function $ErrorHandler(){return globals.ErrorHandler||(typeof ErrorHandler=="undefined"?nil:ErrorHandler)}
return smalltalk.withContext(function($ctx1) { 
_st($ErrorHandler())._register_(anErrorHandler);
return self}, function($ctx1) {$ctx1.fill(self,"registerErrorHandler:",{anErrorHandler:anErrorHandler},globals.Environment)})},
args: ["anErrorHandler"],
source: "registerErrorHandler: anErrorHandler\x0a\x09ErrorHandler register: anErrorHandler",
messageSends: ["register:"],
referencedClasses: ["ErrorHandler"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "registerFinder:",
protocol: 'services',
fn: function (aFinder){
var self=this;
function $Finder(){return globals.Finder||(typeof Finder=="undefined"?nil:Finder)}
return smalltalk.withContext(function($ctx1) { 
_st($Finder())._register_(aFinder);
return self}, function($ctx1) {$ctx1.fill(self,"registerFinder:",{aFinder:aFinder},globals.Environment)})},
args: ["aFinder"],
source: "registerFinder: aFinder\x0a\x09Finder register: aFinder",
messageSends: ["register:"],
referencedClasses: ["Finder"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "registerInspector:",
protocol: 'services',
fn: function (anInspector){
var self=this;
function $Inspector(){return globals.Inspector||(typeof Inspector=="undefined"?nil:Inspector)}
return smalltalk.withContext(function($ctx1) { 
_st($Inspector())._register_(anInspector);
return self}, function($ctx1) {$ctx1.fill(self,"registerInspector:",{anInspector:anInspector},globals.Environment)})},
args: ["anInspector"],
source: "registerInspector: anInspector\x0a\x09Inspector register: anInspector",
messageSends: ["register:"],
referencedClasses: ["Inspector"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "registerProgressHandler:",
protocol: 'services',
fn: function (aProgressHandler){
var self=this;
function $ProgressHandler(){return globals.ProgressHandler||(typeof ProgressHandler=="undefined"?nil:ProgressHandler)}
return smalltalk.withContext(function($ctx1) { 
_st($ProgressHandler())._register_(aProgressHandler);
return self}, function($ctx1) {$ctx1.fill(self,"registerProgressHandler:",{aProgressHandler:aProgressHandler},globals.Environment)})},
args: ["aProgressHandler"],
source: "registerProgressHandler: aProgressHandler\x0a\x09ProgressHandler register: aProgressHandler",
messageSends: ["register:"],
referencedClasses: ["ProgressHandler"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "registerTranscript:",
protocol: 'services',
fn: function (aTranscript){
var self=this;
function $Transcript(){return globals.Transcript||(typeof Transcript=="undefined"?nil:Transcript)}
return smalltalk.withContext(function($ctx1) { 
_st($Transcript())._register_(aTranscript);
return self}, function($ctx1) {$ctx1.fill(self,"registerTranscript:",{aTranscript:aTranscript},globals.Environment)})},
args: ["aTranscript"],
source: "registerTranscript: aTranscript\x0a\x09Transcript register: aTranscript",
messageSends: ["register:"],
referencedClasses: ["Transcript"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "removeClass:",
protocol: 'actions',
fn: function (aClass){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
_st($Smalltalk())._removeClass_(aClass);
return self}, function($ctx1) {$ctx1.fill(self,"removeClass:",{aClass:aClass},globals.Environment)})},
args: ["aClass"],
source: "removeClass: aClass\x0a\x09Smalltalk removeClass: aClass",
messageSends: ["removeClass:"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "removeMethod:",
protocol: 'actions',
fn: function (aMethod){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(_st(aMethod)._methodClass())._removeCompiledMethod_(aMethod);
return self}, function($ctx1) {$ctx1.fill(self,"removeMethod:",{aMethod:aMethod},globals.Environment)})},
args: ["aMethod"],
source: "removeMethod: aMethod\x0a\x09aMethod methodClass removeCompiledMethod: aMethod",
messageSends: ["removeCompiledMethod:", "methodClass"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "removeProtocol:from:",
protocol: 'actions',
fn: function (aString,aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(_st(aClass)._methodsInProtocol_(aString))._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(aClass)._removeCompiledMethod_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"removeProtocol:from:",{aString:aString,aClass:aClass},globals.Environment)})},
args: ["aString", "aClass"],
source: "removeProtocol: aString from: aClass\x0a\x09(aClass methodsInProtocol: aString)\x0a\x09\x09do: [ :each | aClass removeCompiledMethod: each ]",
messageSends: ["do:", "methodsInProtocol:", "removeCompiledMethod:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "renameClass:to:",
protocol: 'actions',
fn: function (aClass,aClassName){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st(_st($Smalltalk())._globals())._at_(aClassName);
if(($receiver = $1) == nil || $receiver == null){
$1;
} else {
$2=_st("A class named ".__comma(aClassName)).__comma(" already exists");
$ctx1.sendIdx[","]=1;
self._error_($2);
};
_st(_st($ClassBuilder())._new())._renameClass_to_(aClass,aClassName);
return self}, function($ctx1) {$ctx1.fill(self,"renameClass:to:",{aClass:aClass,aClassName:aClassName},globals.Environment)})},
args: ["aClass", "aClassName"],
source: "renameClass: aClass to: aClassName\x0a\x09(Smalltalk globals at: aClassName)\x0a\x09\x09ifNotNil: [ self error: 'A class named ', aClassName, ' already exists' ].\x0a\x09\x09\x0a\x09ClassBuilder new renameClass: aClass to: aClassName",
messageSends: ["ifNotNil:", "at:", "globals", "error:", ",", "renameClass:to:", "new"],
referencedClasses: ["Smalltalk", "ClassBuilder"]
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "renameProtocol:to:in:",
protocol: 'actions',
fn: function (aString,anotherString,aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(_st(aClass)._methodsInProtocol_(aString))._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._protocol_(anotherString);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"renameProtocol:to:in:",{aString:aString,anotherString:anotherString,aClass:aClass},globals.Environment)})},
args: ["aString", "anotherString", "aClass"],
source: "renameProtocol: aString to: anotherString in: aClass\x0a\x09(aClass methodsInProtocol: aString)\x0a\x09\x09do: [ :each | each protocol: anotherString ]",
messageSends: ["do:", "methodsInProtocol:", "protocol:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "setClassCommentOf:to:",
protocol: 'actions',
fn: function (aClass,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aClass)._comment_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"setClassCommentOf:to:",{aClass:aClass,aString:aString},globals.Environment)})},
args: ["aClass", "aString"],
source: "setClassCommentOf: aClass to: aString\x0a\x09aClass comment: aString",
messageSends: ["comment:"],
referencedClasses: []
}),
globals.Environment);

smalltalk.addMethod(
smalltalk.method({
selector: "systemAnnouncer",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(_st($Smalltalk())._globals())._at_("SystemAnnouncer"))._current();
return $1;
}, function($ctx1) {$ctx1.fill(self,"systemAnnouncer",{},globals.Environment)})},
args: [],
source: "systemAnnouncer\x0a\x09^ (Smalltalk globals at: #SystemAnnouncer) current",
messageSends: ["current", "at:", "globals"],
referencedClasses: ["Smalltalk"]
}),
globals.Environment);



smalltalk.addClass('JSObjectProxy', globals.ProtoObject, ['jsObject'], 'Kernel-Infrastructure');
globals.JSObjectProxy.comment="I handle sending messages to JavaScript objects, making  JavaScript object accessing from Amber fully transparent.\x0aMy instances make intensive use of `#doesNotUnderstand:`.\x0a\x0aMy instances are automatically created by Amber whenever a message is sent to a JavaScript object.\x0a\x0a## Usage examples\x0a\x0aJSObjectProxy objects are instanciated by Amber when a Smalltalk message is sent to a JavaScript object.\x0a\x0a\x09window alert: 'hello world'.\x0a\x09window inspect.\x0a\x09(window jQuery: 'body') append: 'hello world'\x0a\x0aAmber messages sends are converted to JavaScript function calls or object property access _(in this order)_. If n one of them match, a `MessageNotUnderstood` error will be thrown.\x0a\x0a## Message conversion rules\x0a\x0a- `someUser name` becomes `someUser.name`\x0a- `someUser name: 'John'` becomes `someUser name = \x22John\x22`\x0a- `console log: 'hello world'` becomes `console.log('hello world')`\x0a- `(window jQuery: 'foo') css: 'background' color: 'red'` becomes `window.jQuery('foo').css('background', 'red')`\x0a\x0a__Note:__ For keyword-based messages, only the first keyword is kept: `window foo: 1 bar: 2` is equivalent to `window foo: 1 baz: 2`.";
smalltalk.addMethod(
smalltalk.method({
selector: "=",
protocol: 'comparing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3;
$2=_st(anObject)._class();
$ctx1.sendIdx["class"]=1;
$1=_st($2).__eq_eq(self._class());
if(! smalltalk.assert($1)){
return false;
};
$3=self._compareJSObjectWith_(_st(anObject)._jsObject());
return $3;
}, function($ctx1) {$ctx1.fill(self,"=",{anObject:anObject},globals.JSObjectProxy)})},
args: ["anObject"],
source: "= anObject\x0a\x09anObject class == self class ifFalse: [ ^ false ].\x0a\x09^ self compareJSObjectWith: anObject jsObject",
messageSends: ["ifFalse:", "==", "class", "compareJSObjectWith:", "jsObject"],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "addObjectVariablesTo:",
protocol: 'proxy',
fn: function (aDictionary){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		for(var i in self['@jsObject']) {
			aDictionary._at_put_(i, self['@jsObject'][i]);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"addObjectVariablesTo:",{aDictionary:aDictionary},globals.JSObjectProxy)})},
args: ["aDictionary"],
source: "addObjectVariablesTo: aDictionary\x0a\x09<\x0a\x09\x09for(var i in self['@jsObject']) {\x0a\x09\x09\x09aDictionary._at_put_(i, self['@jsObject'][i]);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "asJSON",
protocol: 'enumerating',
fn: function (){
var self=this;
var $1;
$1=self["@jsObject"];
return $1;
},
args: [],
source: "asJSON\x0a\x09\x22Answers the receiver in a stringyfy-friendly fashion\x22\x0a\x0a\x09^ jsObject",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "at:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self['@jsObject'][aString];
return self}, function($ctx1) {$ctx1.fill(self,"at:",{aString:aString},globals.JSObjectProxy)})},
args: ["aString"],
source: "at: aString\x0a\x09<return self['@jsObject'][aString]>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (aString,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var obj = self['@jsObject'];
		return aString in obj ? obj[aString] : aBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{aString:aString,aBlock:aBlock},globals.JSObjectProxy)})},
args: ["aString", "aBlock"],
source: "at: aString ifAbsent: aBlock\x0a\x09\x22return the aString property or evaluate aBlock if the property is not defined on the object\x22\x0a\x09<\x0a\x09\x09var obj = self['@jsObject'];\x0a\x09\x09return aString in obj ? obj[aString] : aBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:",
protocol: 'accessing',
fn: function (aString,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var obj = self['@jsObject'];
		return aString in obj ? aBlock._value_(obj[aString]) : nil;
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:",{aString:aString,aBlock:aBlock},globals.JSObjectProxy)})},
args: ["aString", "aBlock"],
source: "at: aString ifPresent: aBlock\x0a\x09\x22return the evaluation of aBlock with the value if the property is defined or return nil\x22\x0a\x09<\x0a\x09\x09var obj = self['@jsObject'];\x0a\x09\x09return aString in obj ? aBlock._value_(obj[aString]) : nil;\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifPresent:ifAbsent:",
protocol: 'accessing',
fn: function (aString,aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var obj = self['@jsObject'];
		return aString in obj ? aBlock._value_(obj[aString]) : anotherBlock._value();
	;
return self}, function($ctx1) {$ctx1.fill(self,"at:ifPresent:ifAbsent:",{aString:aString,aBlock:aBlock,anotherBlock:anotherBlock},globals.JSObjectProxy)})},
args: ["aString", "aBlock", "anotherBlock"],
source: "at: aString ifPresent: aBlock ifAbsent: anotherBlock\x0a\x09\x22return the evaluation of aBlock with the value if the property is defined\x0a\x09or return value of anotherBlock\x22\x0a\x09<\x0a\x09\x09var obj = self['@jsObject'];\x0a\x09\x09return aString in obj ? aBlock._value_(obj[aString]) : anotherBlock._value();\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (aString,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self['@jsObject'][aString] = anObject;
return self}, function($ctx1) {$ctx1.fill(self,"at:put:",{aString:aString,anObject:anObject},globals.JSObjectProxy)})},
args: ["aString", "anObject"],
source: "at: aString put: anObject\x0a\x09<return self['@jsObject'][aString] = anObject>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "compareJSObjectWith:",
protocol: 'private',
fn: function (aJSObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self["@jsObject"] === aJSObject;
return self}, function($ctx1) {$ctx1.fill(self,"compareJSObjectWith:",{aJSObject:aJSObject},globals.JSObjectProxy)})},
args: ["aJSObject"],
source: "compareJSObjectWith: aJSObject\x0a \x09<return self[\x22@jsObject\x22] === aJSObject>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "doesNotUnderstand:",
protocol: 'proxy',
fn: function (aMessage){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._lookupProperty_(_st(_st(aMessage)._selector())._asJavaScriptSelector());
if(($receiver = $2) == nil || $receiver == null){
$1=globals.JSObjectProxy.superclass.fn.prototype._doesNotUnderstand_.apply(_st(self), [aMessage]);
} else {
var jsSelector;
jsSelector=$receiver;
$1=self._forwardMessage_withArguments_(jsSelector,_st(aMessage)._arguments());
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"doesNotUnderstand:",{aMessage:aMessage},globals.JSObjectProxy)})},
args: ["aMessage"],
source: "doesNotUnderstand: aMessage\x0a\x09^ (self lookupProperty: aMessage selector asJavaScriptSelector)\x0a\x09\x09ifNil: [ super doesNotUnderstand: aMessage ]\x0a\x09\x09ifNotNil: [ :jsSelector | \x0a\x09\x09\x09self \x0a\x09\x09\x09\x09forwardMessage: jsSelector \x0a\x09\x09\x09\x09withArguments: aMessage arguments ]",
messageSends: ["ifNil:ifNotNil:", "lookupProperty:", "asJavaScriptSelector", "selector", "doesNotUnderstand:", "forwardMessage:withArguments:", "arguments"],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "forwardMessage:withArguments:",
protocol: 'proxy',
fn: function (aString,anArray){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return smalltalk.send(self._jsObject(), aString, anArray);
	;
return self}, function($ctx1) {$ctx1.fill(self,"forwardMessage:withArguments:",{aString:aString,anArray:anArray},globals.JSObjectProxy)})},
args: ["aString", "anArray"],
source: "forwardMessage: aString withArguments: anArray\x0a\x09<\x0a\x09\x09return smalltalk.send(self._jsObject(), aString, anArray);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "inspectOn:",
protocol: 'proxy',
fn: function (anInspector){
var self=this;
var variables;
function $Dictionary(){return globals.Dictionary||(typeof Dictionary=="undefined"?nil:Dictionary)}
return smalltalk.withContext(function($ctx1) { 
variables=_st($Dictionary())._new();
_st(variables)._at_put_("#self",self._jsObject());
_st(anInspector)._setLabel_(self._printString());
self._addObjectVariablesTo_(variables);
_st(anInspector)._setVariables_(variables);
return self}, function($ctx1) {$ctx1.fill(self,"inspectOn:",{anInspector:anInspector,variables:variables},globals.JSObjectProxy)})},
args: ["anInspector"],
source: "inspectOn: anInspector\x0a\x09| variables |\x0a\x09variables := Dictionary new.\x0a\x09variables at: '#self' put: self jsObject.\x0a\x09anInspector setLabel: self printString.\x0a\x09self addObjectVariablesTo: variables.\x0a\x09anInspector setVariables: variables",
messageSends: ["new", "at:put:", "jsObject", "setLabel:", "printString", "addObjectVariablesTo:", "setVariables:"],
referencedClasses: ["Dictionary"]
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "jsObject",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@jsObject"];
return $1;
},
args: [],
source: "jsObject\x0a\x09^ jsObject",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "jsObject:",
protocol: 'accessing',
fn: function (aJSObject){
var self=this;
self["@jsObject"]=aJSObject;
return self},
args: ["aJSObject"],
source: "jsObject: aJSObject\x0a\x09jsObject := aJSObject",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "keysAndValuesDo:",
protocol: 'enumerating',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var o = self['@jsObject'];
		for(var i in o) {
			aBlock._value_value_(i, o[i]);
		}
	;
return self}, function($ctx1) {$ctx1.fill(self,"keysAndValuesDo:",{aBlock:aBlock},globals.JSObjectProxy)})},
args: ["aBlock"],
source: "keysAndValuesDo: aBlock\x0a\x09<\x0a\x09\x09var o = self['@jsObject'];\x0a\x09\x09for(var i in o) {\x0a\x09\x09\x09aBlock._value_value_(i, o[i]);\x0a\x09\x09}\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "lookupProperty:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return aString in self._jsObject() ? aString : nil;
return self}, function($ctx1) {$ctx1.fill(self,"lookupProperty:",{aString:aString},globals.JSObjectProxy)})},
args: ["aString"],
source: "lookupProperty: aString\x0a\x09\x22Looks up a property in JS object.\x0a\x09Answer the property if it is present, or nil if it is not present.\x22\x0a\x09\x0a\x09<return aString in self._jsObject() ? aString : nil>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aStream)._nextPutAll_(self._printString());
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.JSObjectProxy)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09aStream nextPutAll: self printString",
messageSends: ["nextPutAll:", "printString"],
referencedClasses: []
}),
globals.JSObjectProxy);

smalltalk.addMethod(
smalltalk.method({
selector: "printString",
protocol: 'printing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		var js = self['@jsObject'];
		return js.toString
			? js.toString()
			: Object.prototype.toString.call(js)
	;
return self}, function($ctx1) {$ctx1.fill(self,"printString",{},globals.JSObjectProxy)})},
args: [],
source: "printString\x0a\x09<\x0a\x09\x09var js = self['@jsObject'];\x0a\x09\x09return js.toString\x0a\x09\x09\x09? js.toString()\x0a\x09\x09\x09: Object.prototype.toString.call(js)\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.JSObjectProxy);


smalltalk.addMethod(
smalltalk.method({
selector: "on:",
protocol: 'instance creation',
fn: function (aJSObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._jsObject_(aJSObject);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:",{aJSObject:aJSObject},globals.JSObjectProxy.klass)})},
args: ["aJSObject"],
source: "on: aJSObject\x0a\x09^ self new\x0a\x09\x09jsObject: aJSObject;\x0a\x09\x09yourself",
messageSends: ["jsObject:", "new", "yourself"],
referencedClasses: []
}),
globals.JSObjectProxy.klass);


smalltalk.addClass('NullProgressHandler', globals.Object, [], 'Kernel-Infrastructure');
globals.NullProgressHandler.comment="I am the default progress handler. I do not display any progress, and simply iterate over the collection.";
smalltalk.addMethod(
smalltalk.method({
selector: "do:on:displaying:",
protocol: 'progress handling',
fn: function (aBlock,aCollection,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aCollection)._do_(aBlock);
return self}, function($ctx1) {$ctx1.fill(self,"do:on:displaying:",{aBlock:aBlock,aCollection:aCollection,aString:aString},globals.NullProgressHandler)})},
args: ["aBlock", "aCollection", "aString"],
source: "do: aBlock on: aCollection displaying: aString\x0a\x09aCollection do: aBlock",
messageSends: ["do:"],
referencedClasses: []
}),
globals.NullProgressHandler);


globals.NullProgressHandler.klass.iVarNames = ['current'];
smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $ProgressHandler(){return globals.ProgressHandler||(typeof ProgressHandler=="undefined"?nil:ProgressHandler)}
return smalltalk.withContext(function($ctx1) { 
_st($ProgressHandler())._registerIfNone_(self._new());
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.NullProgressHandler.klass)})},
args: [],
source: "initialize\x0a\x09ProgressHandler registerIfNone: self new",
messageSends: ["registerIfNone:", "new"],
referencedClasses: ["ProgressHandler"]
}),
globals.NullProgressHandler.klass);


smalltalk.addClass('Organizer', globals.Object, [], 'Kernel-Infrastructure');
globals.Organizer.comment="I represent categorization information. \x0a\x0a## API\x0a\x0aUse `#addElement:` and `#removeElement:` to manipulate instances.";
smalltalk.addMethod(
smalltalk.method({
selector: "addElement:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.elements.addElement(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"addElement:",{anObject:anObject},globals.Organizer)})},
args: ["anObject"],
source: "addElement: anObject\x0a\x09<self.elements.addElement(anObject)>",
messageSends: [],
referencedClasses: []
}),
globals.Organizer);

smalltalk.addMethod(
smalltalk.method({
selector: "elements",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._basicAt_("elements"))._copy();
return $1;
}, function($ctx1) {$ctx1.fill(self,"elements",{},globals.Organizer)})},
args: [],
source: "elements\x0a\x09^ (self basicAt: 'elements') copy",
messageSends: ["copy", "basicAt:"],
referencedClasses: []
}),
globals.Organizer);

smalltalk.addMethod(
smalltalk.method({
selector: "removeElement:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.elements.removeElement(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"removeElement:",{anObject:anObject},globals.Organizer)})},
args: ["anObject"],
source: "removeElement: anObject\x0a\x09<self.elements.removeElement(anObject)>",
messageSends: [],
referencedClasses: []
}),
globals.Organizer);



smalltalk.addClass('ClassOrganizer', globals.Organizer, [], 'Kernel-Infrastructure');
globals.ClassOrganizer.comment="I am an organizer specific to classes. I hold method categorization information for classes.";
smalltalk.addMethod(
smalltalk.method({
selector: "addElement:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ProtocolAdded(){return globals.ProtocolAdded||(typeof ProtocolAdded=="undefined"?nil:ProtocolAdded)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
globals.ClassOrganizer.superclass.fn.prototype._addElement_.apply(_st(self), [aString]);
$1=_st($ProtocolAdded())._new();
_st($1)._protocol_(aString);
_st($1)._theClass_(self._theClass());
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
return self}, function($ctx1) {$ctx1.fill(self,"addElement:",{aString:aString},globals.ClassOrganizer)})},
args: ["aString"],
source: "addElement: aString\x0a\x09super addElement: aString.\x0a\x0a\x09SystemAnnouncer current announce: (ProtocolAdded new\x0a\x09\x09protocol: aString;\x0a\x09\x09theClass: self theClass;\x0a\x09\x09yourself)",
messageSends: ["addElement:", "announce:", "current", "protocol:", "new", "theClass:", "theClass", "yourself"],
referencedClasses: ["SystemAnnouncer", "ProtocolAdded"]
}),
globals.ClassOrganizer);

smalltalk.addMethod(
smalltalk.method({
selector: "removeElement:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ProtocolRemoved(){return globals.ProtocolRemoved||(typeof ProtocolRemoved=="undefined"?nil:ProtocolRemoved)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
globals.ClassOrganizer.superclass.fn.prototype._removeElement_.apply(_st(self), [aString]);
$1=_st($ProtocolRemoved())._new();
_st($1)._protocol_(aString);
_st($1)._theClass_(self._theClass());
$2=_st($1)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($2);
return self}, function($ctx1) {$ctx1.fill(self,"removeElement:",{aString:aString},globals.ClassOrganizer)})},
args: ["aString"],
source: "removeElement: aString\x0a\x09super removeElement: aString.\x0a\x0a\x09SystemAnnouncer current announce: (ProtocolRemoved new\x0a\x09\x09protocol: aString;\x0a\x09\x09theClass: self theClass;\x0a\x09\x09yourself)",
messageSends: ["removeElement:", "announce:", "current", "protocol:", "new", "theClass:", "theClass", "yourself"],
referencedClasses: ["SystemAnnouncer", "ProtocolRemoved"]
}),
globals.ClassOrganizer);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
 return self.theClass ;
return self}, function($ctx1) {$ctx1.fill(self,"theClass",{},globals.ClassOrganizer)})},
args: [],
source: "theClass\x0a\x09< return self.theClass >",
messageSends: [],
referencedClasses: []
}),
globals.ClassOrganizer);



smalltalk.addClass('PackageOrganizer', globals.Organizer, [], 'Kernel-Infrastructure');
globals.PackageOrganizer.comment="I am an organizer specific to packages. I hold classes categorization information.";


smalltalk.addClass('Package', globals.Object, ['transport'], 'Kernel-Infrastructure');
globals.Package.comment="I am similar to a \x22class category\x22 typically found in other Smalltalks like Pharo or Squeak. Amber does not have class categories anymore, it had in the beginning but now each class in the system knows which package it belongs to.\x0a\x0aEach package has a name and can be queried for its classes, but it will then resort to a reverse scan of all classes to find them.\x0a\x0a## API\x0a\x0aPackages are manipulated through \x22Smalltalk current\x22, like for example finding one based on a name or with `Package class >> #name` directly:\x0a\x0a    Smalltalk current packageAt: 'Kernel'\x0a    Package named: 'Kernel'\x0a\x0aA package differs slightly from a Monticello package which can span multiple class categories using a naming convention based on hyphenation. But just as in Monticello a package supports \x22class extensions\x22 so a package can define behaviors in foreign classes using a naming convention for method categories where the category starts with an asterisk and then the name of the owning package follows.\x0a\x0aYou can fetch a package from the server:\x0a\x0a\x09Package load: 'Additional-Examples'";
smalltalk.addMethod(
smalltalk.method({
selector: "basicTransport",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.transport;
return self}, function($ctx1) {$ctx1.fill(self,"basicTransport",{},globals.Package)})},
args: [],
source: "basicTransport\x0a\x09\x22Answer the transport literal JavaScript object as setup in the JavaScript file, if any\x22\x0a\x09\x0a\x09<return self.transport>",
messageSends: [],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "classTemplate",
protocol: 'accessing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $3,$4,$2,$5,$6,$7,$1;
$1=_st($String())._streamContents_((function(stream){
return smalltalk.withContext(function($ctx2) {
_st(stream)._nextPutAll_("Object");
$ctx2.sendIdx["nextPutAll:"]=1;
_st(stream)._nextPutAll_(" subclass: #NameOfSubclass");
$ctx2.sendIdx["nextPutAll:"]=2;
$3=_st($String())._lf();
$ctx2.sendIdx["lf"]=1;
$4=_st($String())._tab();
$ctx2.sendIdx["tab"]=1;
$2=_st($3).__comma($4);
$ctx2.sendIdx[","]=1;
_st(stream)._nextPutAll_($2);
$ctx2.sendIdx["nextPutAll:"]=3;
$5=_st(stream)._nextPutAll_("instanceVariableNames: ''");
$ctx2.sendIdx["nextPutAll:"]=4;
$5;
$6=_st("'".__comma(_st($String())._lf())).__comma(_st($String())._tab());
$ctx2.sendIdx[","]=2;
_st(stream)._nextPutAll_($6);
$ctx2.sendIdx["nextPutAll:"]=5;
_st(stream)._nextPutAll_("package: '");
$ctx2.sendIdx["nextPutAll:"]=6;
_st(stream)._nextPutAll_(self._name());
$ctx2.sendIdx["nextPutAll:"]=7;
$7=_st(stream)._nextPutAll_("'");
return $7;
}, function($ctx2) {$ctx2.fillBlock({stream:stream},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"classTemplate",{},globals.Package)})},
args: [],
source: "classTemplate\x0a\x09^ String streamContents: [ :stream |\x0a\x09\x09stream\x0a\x09\x09\x09nextPutAll: 'Object';\x0a\x09\x09\x09nextPutAll: ' subclass: #NameOfSubclass';\x0a\x09\x09\x09nextPutAll: String lf, String tab;\x0a\x09\x09\x09nextPutAll: 'instanceVariableNames: '''''.\x0a\x09\x09stream\x0a\x09\x09\x09nextPutAll: '''', String lf, String tab;\x0a\x09\x09\x09nextPutAll: 'package: ''';\x0a\x09\x09\x09nextPutAll: self name;\x0a\x09\x09\x09nextPutAll: '''' ]",
messageSends: ["streamContents:", "nextPutAll:", ",", "lf", "tab", "name"],
referencedClasses: ["String"]
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "classes",
protocol: 'classes',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._organization())._elements();
return $1;
}, function($ctx1) {$ctx1.fill(self,"classes",{},globals.Package)})},
args: [],
source: "classes\x0a\x09^ self organization elements",
messageSends: ["elements", "organization"],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "definition",
protocol: 'accessing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $2,$4,$5,$3,$7,$6,$8,$9,$1;
$1=_st($String())._streamContents_((function(stream){
return smalltalk.withContext(function($ctx2) {
$2=_st(self._class())._name();
$ctx2.sendIdx["name"]=1;
_st(stream)._nextPutAll_($2);
$ctx2.sendIdx["nextPutAll:"]=1;
$4=_st($String())._lf();
$ctx2.sendIdx["lf"]=1;
$5=_st($String())._tab();
$ctx2.sendIdx["tab"]=1;
$3=_st($4).__comma($5);
$ctx2.sendIdx[","]=1;
_st(stream)._nextPutAll_($3);
$ctx2.sendIdx["nextPutAll:"]=2;
_st(stream)._nextPutAll_(" named: ");
$ctx2.sendIdx["nextPutAll:"]=3;
$7="'".__comma(self._name());
$ctx2.sendIdx[","]=3;
$6=_st($7).__comma("'");
$ctx2.sendIdx[","]=2;
_st(stream)._nextPutAll_($6);
$ctx2.sendIdx["nextPutAll:"]=4;
$8=_st(_st($String())._lf()).__comma(_st($String())._tab());
$ctx2.sendIdx[","]=4;
_st(stream)._nextPutAll_($8);
$ctx2.sendIdx["nextPutAll:"]=5;
_st(stream)._nextPutAll_(" transport: (");
$ctx2.sendIdx["nextPutAll:"]=6;
$9=_st(stream)._nextPutAll_(_st(_st(self._transport())._definition()).__comma(")"));
return $9;
}, function($ctx2) {$ctx2.fillBlock({stream:stream},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"definition",{},globals.Package)})},
args: [],
source: "definition\x0a\x09^ String streamContents: [ :stream |\x0a\x09\x09stream \x0a\x09\x09\x09nextPutAll: self class name;\x0a\x09\x09\x09nextPutAll: String lf, String tab;\x0a\x09\x09\x09nextPutAll: ' named: ';\x0a\x09\x09\x09nextPutAll: '''', self name, '''';\x0a\x09\x09\x09nextPutAll: String lf, String tab;\x0a\x09\x09\x09nextPutAll:  ' transport: (';\x0a\x09\x09\x09nextPutAll: self transport definition, ')' ]",
messageSends: ["streamContents:", "nextPutAll:", "name", "class", ",", "lf", "tab", "definition", "transport"],
referencedClasses: ["String"]
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "isPackage",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isPackage\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "loadDependencies",
protocol: 'dependencies',
fn: function (){
var self=this;
var classes,packages;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
classes=self._loadDependencyClasses();
$2=_st(_st(classes)._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._package();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})})))._asSet();
_st($2)._remove_ifAbsent_(self,(function(){
}));
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"loadDependencies",{classes:classes,packages:packages},globals.Package)})},
args: [],
source: "loadDependencies\x0a\x09\x22Returns list of packages that need to be loaded\x0a\x09before loading this package.\x22\x0a\x09\x0a\x09| classes packages |\x0a\x09classes := self loadDependencyClasses.\x0a\x09^ (classes collect: [ :each | each package ]) asSet\x0a\x09\x09remove: self ifAbsent: [];\x0a\x09\x09yourself",
messageSends: ["loadDependencyClasses", "remove:ifAbsent:", "asSet", "collect:", "package", "yourself"],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "loadDependencyClasses",
protocol: 'dependencies',
fn: function (){
var self=this;
var starCategoryName;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $4,$3,$2,$6,$5,$7,$1;
starCategoryName="*".__comma(self._name());
$ctx1.sendIdx[","]=1;
$4=self._classes();
$ctx1.sendIdx["classes"]=1;
$3=_st($4)._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._superclass();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=_st($3)._asSet();
_st($2)._remove_ifAbsent_(nil,(function(){
}));
_st($2)._addAll_(_st(_st($Smalltalk())._classes())._select_((function(each){
return smalltalk.withContext(function($ctx2) {
$6=_st(each)._protocols();
$ctx2.sendIdx["protocols"]=1;
$5=_st($6).__comma(_st(_st(each)._class())._protocols());
return _st($5)._includes_(starCategoryName);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,3)})})));
$7=_st($2)._yourself();
$1=$7;
return $1;
}, function($ctx1) {$ctx1.fill(self,"loadDependencyClasses",{starCategoryName:starCategoryName},globals.Package)})},
args: [],
source: "loadDependencyClasses\x0a\x09\x22Returns classes needed at the time of loading a package.\x0a\x09These are all that are used to subclass\x0a\x09and to define an extension method\x22\x0a\x09\x0a\x09| starCategoryName |\x0a\x09starCategoryName := '*', self name.\x0a\x09^ (self classes collect: [ :each | each superclass ]) asSet\x0a\x09\x09remove: nil ifAbsent: [];\x0a\x09\x09addAll: (Smalltalk classes select: [ :each | each protocols, each class protocols includes: starCategoryName ]);\x0a\x09\x09yourself",
messageSends: [",", "name", "remove:ifAbsent:", "asSet", "collect:", "classes", "superclass", "addAll:", "select:", "includes:", "protocols", "class", "yourself"],
referencedClasses: ["Smalltalk"]
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "name",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.pkgName;
return self}, function($ctx1) {$ctx1.fill(self,"name",{},globals.Package)})},
args: [],
source: "name\x0a\x09<return self.pkgName>",
messageSends: [],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "name:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.pkgName = aString;
return self}, function($ctx1) {$ctx1.fill(self,"name:",{aString:aString},globals.Package)})},
args: ["aString"],
source: "name: aString\x0a\x09<self.pkgName = aString>",
messageSends: [],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "organization",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._basicAt_("organization");
return $1;
}, function($ctx1) {$ctx1.fill(self,"organization",{},globals.Package)})},
args: [],
source: "organization\x0a\x09^ self basicAt: 'organization'",
messageSends: ["basicAt:"],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "printOn:",
protocol: 'printing',
fn: function (aStream){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
globals.Package.superclass.fn.prototype._printOn_.apply(_st(self), [aStream]);
_st(aStream)._nextPutAll_(" (");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aStream)._nextPutAll_(self._name());
$ctx1.sendIdx["nextPutAll:"]=2;
$1=_st(aStream)._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"printOn:",{aStream:aStream},globals.Package)})},
args: ["aStream"],
source: "printOn: aStream\x0a\x09super printOn: aStream.\x0a\x09aStream \x0a\x09\x09nextPutAll: ' (';\x0a\x09\x09nextPutAll: self name;\x0a\x09\x09nextPutAll: ')'",
messageSends: ["printOn:", "nextPutAll:", "name"],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "setupClasses",
protocol: 'classes',
fn: function (){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self._classes();
_st($1)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st($ClassBuilder())._new())._setupClass_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$ctx1.sendIdx["do:"]=1;
$2=_st($1)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._initialize();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"setupClasses",{},globals.Package)})},
args: [],
source: "setupClasses\x0a\x09self classes\x0a\x09\x09do: [ :each | ClassBuilder new setupClass: each ];\x0a\x09\x09do: [ :each | each initialize ]",
messageSends: ["do:", "classes", "setupClass:", "new", "initialize"],
referencedClasses: ["ClassBuilder"]
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "sortedClasses",
protocol: 'classes',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._class())._sortedClasses_(self._classes());
return $1;
}, function($ctx1) {$ctx1.fill(self,"sortedClasses",{},globals.Package)})},
args: [],
source: "sortedClasses\x0a\x09\x22Answer all classes in the receiver, sorted by superclass/subclasses and by class name for common subclasses (Issue #143).\x22\x0a\x0a\x09^ self class sortedClasses: self classes",
messageSends: ["sortedClasses:", "class", "classes"],
referencedClasses: []
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "transport",
protocol: 'accessing',
fn: function (){
var self=this;
function $PackageTransport(){return globals.PackageTransport||(typeof PackageTransport=="undefined"?nil:PackageTransport)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$4,$1;
$2=self["@transport"];
if(($receiver = $2) == nil || $receiver == null){
$3=_st($PackageTransport())._fromJson_(self._basicTransport());
_st($3)._package_(self);
$4=_st($3)._yourself();
self["@transport"]=$4;
$1=self["@transport"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"transport",{},globals.Package)})},
args: [],
source: "transport\x0a\x09^ transport ifNil: [ \x0a\x09\x09transport := (PackageTransport fromJson: self basicTransport)\x0a\x09\x09\x09package: self;\x0a\x09\x09\x09yourself ]",
messageSends: ["ifNil:", "package:", "fromJson:", "basicTransport", "yourself"],
referencedClasses: ["PackageTransport"]
}),
globals.Package);

smalltalk.addMethod(
smalltalk.method({
selector: "transport:",
protocol: 'accessing',
fn: function (aPackageTransport){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self["@transport"]=aPackageTransport;
_st(aPackageTransport)._package_(self);
return self}, function($ctx1) {$ctx1.fill(self,"transport:",{aPackageTransport:aPackageTransport},globals.Package)})},
args: ["aPackageTransport"],
source: "transport: aPackageTransport\x0a\x09transport := aPackageTransport.\x0a\x09aPackageTransport package: self",
messageSends: ["package:"],
referencedClasses: []
}),
globals.Package);


globals.Package.klass.iVarNames = ['defaultCommitPathJs','defaultCommitPathSt'];
smalltalk.addMethod(
smalltalk.method({
selector: "named:",
protocol: 'accessing',
fn: function (aPackageName){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Smalltalk())._packageAt_ifAbsent_(aPackageName,(function(){
return smalltalk.withContext(function($ctx2) {
return _st($Smalltalk())._createPackage_(aPackageName);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"named:",{aPackageName:aPackageName},globals.Package.klass)})},
args: ["aPackageName"],
source: "named: aPackageName\x0a\x09^ Smalltalk \x0a\x09\x09packageAt: aPackageName\x0a\x09\x09ifAbsent: [ \x0a\x09\x09\x09Smalltalk createPackage: aPackageName ]",
messageSends: ["packageAt:ifAbsent:", "createPackage:"],
referencedClasses: ["Smalltalk"]
}),
globals.Package.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "named:ifAbsent:",
protocol: 'accessing',
fn: function (aPackageName,aBlock){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Smalltalk())._packageAt_ifAbsent_(aPackageName,aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"named:ifAbsent:",{aPackageName:aPackageName,aBlock:aBlock},globals.Package.klass)})},
args: ["aPackageName", "aBlock"],
source: "named: aPackageName ifAbsent: aBlock\x0a\x09^ Smalltalk packageAt: aPackageName ifAbsent: aBlock",
messageSends: ["packageAt:ifAbsent:"],
referencedClasses: ["Smalltalk"]
}),
globals.Package.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "named:transport:",
protocol: 'accessing',
fn: function (aPackageName,aTransport){
var self=this;
var package_;
return smalltalk.withContext(function($ctx1) { 
var $1;
package_=self._named_(aPackageName);
_st(package_)._transport_(aTransport);
$1=package_;
return $1;
}, function($ctx1) {$ctx1.fill(self,"named:transport:",{aPackageName:aPackageName,aTransport:aTransport,package_:package_},globals.Package.klass)})},
args: ["aPackageName", "aTransport"],
source: "named: aPackageName transport: aTransport\x0a\x09| package |\x0a\x09\x0a\x09package := self named: aPackageName.\x0a\x09package transport: aTransport.\x0a\x09\x0a\x09^ package",
messageSends: ["named:", "transport:"],
referencedClasses: []
}),
globals.Package.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "sortedClasses:",
protocol: 'sorting',
fn: function (classes){
var self=this;
var children,others,nodes,expandedClasses;
function $ClassSorterNode(){return globals.ClassSorterNode||(typeof ClassSorterNode=="undefined"?nil:ClassSorterNode)}
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$2,$4;
children=[];
others=[];
_st(classes)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$1=_st(classes)._includes_(_st(each)._superclass());
if(smalltalk.assert($1)){
return _st(others)._add_(each);
} else {
return _st(children)._add_(each);
$ctx2.sendIdx["add:"]=1;
};
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$ctx1.sendIdx["do:"]=1;
nodes=_st(children)._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st($ClassSorterNode())._on_classes_level_(each,others,(0));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,4)})}));
nodes=_st(nodes)._sorted_((function(a,b){
return smalltalk.withContext(function($ctx2) {
$3=_st(a)._theClass();
$ctx2.sendIdx["theClass"]=1;
$2=_st($3)._name();
$ctx2.sendIdx["name"]=1;
return _st($2).__lt_eq(_st(_st(b)._theClass())._name());
}, function($ctx2) {$ctx2.fillBlock({a:a,b:b},$ctx1,5)})}));
expandedClasses=_st($Array())._new();
_st(nodes)._do_((function(aNode){
return smalltalk.withContext(function($ctx2) {
return _st(aNode)._traverseClassesWith_(expandedClasses);
}, function($ctx2) {$ctx2.fillBlock({aNode:aNode},$ctx1,6)})}));
$4=expandedClasses;
return $4;
}, function($ctx1) {$ctx1.fill(self,"sortedClasses:",{classes:classes,children:children,others:others,nodes:nodes,expandedClasses:expandedClasses},globals.Package.klass)})},
args: ["classes"],
source: "sortedClasses: classes\x0a\x09\x22Answer classes, sorted by superclass/subclasses and by class name for common subclasses (Issue #143)\x22\x0a\x0a\x09| children others nodes expandedClasses |\x0a\x09children := #().\x0a\x09others := #().\x0a\x09classes do: [ :each |\x0a\x09\x09(classes includes: each superclass)\x0a\x09\x09\x09ifFalse: [ children add: each ]\x0a\x09\x09\x09ifTrue: [ others add: each ]].\x0a\x09nodes := children collect: [ :each |\x0a\x09\x09ClassSorterNode on: each classes: others level: 0 ].\x0a\x09nodes := nodes sorted: [ :a :b | a theClass name <= b theClass name ].\x0a\x09expandedClasses := Array new.\x0a\x09nodes do: [ :aNode |\x0a\x09\x09aNode traverseClassesWith: expandedClasses ].\x0a\x09^ expandedClasses",
messageSends: ["do:", "ifFalse:ifTrue:", "includes:", "superclass", "add:", "collect:", "on:classes:level:", "sorted:", "<=", "name", "theClass", "new", "traverseClassesWith:"],
referencedClasses: ["ClassSorterNode", "Array"]
}),
globals.Package.klass);


smalltalk.addClass('PlatformInterface', globals.Object, [], 'Kernel-Infrastructure');
globals.PlatformInterface.comment="I am single entry point to UI and environment interface.\x0aMy `initialize` tries several options (for now, browser environment only) to set myself up.\x0a\x0a## API\x0a\x0a    PlatformInterface alert: 'Hey, there is a problem'.\x0a    PlatformInterface confirm: 'Affirmative?'.\x0a    PlatformInterface prompt: 'Your name:'.\x0a\x0a    PlatformInterface ajax: #{\x0a        'url' -> '/patch.js'. 'type' -> 'GET'. dataType->'script'\x0a    }.";

globals.PlatformInterface.klass.iVarNames = ['worker'];
smalltalk.addMethod(
smalltalk.method({
selector: "ajax:",
protocol: 'actions',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@worker"];
if(($receiver = $2) == nil || $receiver == null){
$1=self._error_("ajax: not available");
} else {
$1=_st(self["@worker"])._ajax_(anObject);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"ajax:",{anObject:anObject},globals.PlatformInterface.klass)})},
args: ["anObject"],
source: "ajax: anObject\x0a\x09^ worker\x0a\x09\x09ifNotNil: [ worker ajax: anObject ]\x0a\x09\x09ifNil: [ self error: 'ajax: not available' ]",
messageSends: ["ifNotNil:ifNil:", "ajax:", "error:"],
referencedClasses: []
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "alert:",
protocol: 'actions',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@worker"];
if(($receiver = $2) == nil || $receiver == null){
$1=self._error_("alert: not available");
} else {
$1=_st(self["@worker"])._alert_(aString);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"alert:",{aString:aString},globals.PlatformInterface.klass)})},
args: ["aString"],
source: "alert: aString\x0a\x09^ worker\x0a\x09\x09ifNotNil: [ worker alert: aString ]\x0a\x09\x09ifNil: [ self error: 'alert: not available' ]",
messageSends: ["ifNotNil:ifNil:", "alert:", "error:"],
referencedClasses: []
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "confirm:",
protocol: 'actions',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@worker"];
if(($receiver = $2) == nil || $receiver == null){
$1=self._error_("confirm: not available");
} else {
$1=_st(self["@worker"])._confirm_(aString);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"confirm:",{aString:aString},globals.PlatformInterface.klass)})},
args: ["aString"],
source: "confirm: aString\x0a\x09^ worker\x0a\x09\x09ifNotNil: [ worker confirm: aString ]\x0a\x09\x09ifNil: [ self error: 'confirm: not available' ]",
messageSends: ["ifNotNil:ifNil:", "confirm:", "error:"],
referencedClasses: []
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "existsGlobal:",
protocol: 'actions',
fn: function (aString){
var self=this;
function $PlatformInterface(){return globals.PlatformInterface||(typeof PlatformInterface=="undefined"?nil:PlatformInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($PlatformInterface())._globals())._at_ifPresent_ifAbsent_(aString,(function(){
return true;
}),(function(){
return false;
}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"existsGlobal:",{aString:aString},globals.PlatformInterface.klass)})},
args: ["aString"],
source: "existsGlobal: aString\x0a\x09^ PlatformInterface globals \x0a\x09\x09at: aString \x0a\x09\x09ifPresent: [ true ] \x0a\x09\x09ifAbsent: [ false ]",
messageSends: ["at:ifPresent:ifAbsent:", "globals"],
referencedClasses: ["PlatformInterface"]
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "globals",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return (new Function('return this'))();;
return self}, function($ctx1) {$ctx1.fill(self,"globals",{},globals.PlatformInterface.klass)})},
args: [],
source: "globals\x0a\x09<return (new Function('return this'))();>",
messageSends: [],
referencedClasses: []
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
var candidate;
function $BrowserInterface(){return globals.BrowserInterface||(typeof BrowserInterface=="undefined"?nil:BrowserInterface)}
return smalltalk.withContext(function($ctx1) { 
var $1;
globals.PlatformInterface.klass.superclass.fn.prototype._initialize.apply(_st(self), []);
if(($receiver = $BrowserInterface()) == nil || $receiver == null){
$BrowserInterface();
} else {
candidate=_st($BrowserInterface())._new();
candidate;
$1=_st(candidate)._isAvailable();
if(smalltalk.assert($1)){
self._setWorker_(candidate);
return self;
};
};
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{candidate:candidate},globals.PlatformInterface.klass)})},
args: [],
source: "initialize\x0a\x09| candidate |\x0a\x09\x0a\x09super initialize.\x0a\x09\x0a\x09BrowserInterface ifNotNil: [\x0a\x09\x09candidate := BrowserInterface new.\x0a\x09\x09candidate isAvailable ifTrue: [ self setWorker: candidate. ^ self ]\x0a\x09]",
messageSends: ["initialize", "ifNotNil:", "new", "ifTrue:", "isAvailable", "setWorker:"],
referencedClasses: ["BrowserInterface"]
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "prompt:",
protocol: 'actions',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@worker"];
if(($receiver = $2) == nil || $receiver == null){
$1=self._error_("prompt: not available");
} else {
$1=_st(self["@worker"])._prompt_(aString);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"prompt:",{aString:aString},globals.PlatformInterface.klass)})},
args: ["aString"],
source: "prompt: aString\x0a\x09^ worker\x0a\x09\x09ifNotNil: [ worker prompt: aString ]\x0a\x09\x09ifNil: [ self error: 'prompt: not available' ]",
messageSends: ["ifNotNil:ifNil:", "prompt:", "error:"],
referencedClasses: []
}),
globals.PlatformInterface.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "setWorker:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@worker"]=anObject;
return self},
args: ["anObject"],
source: "setWorker: anObject\x0a\x09worker := anObject",
messageSends: [],
referencedClasses: []
}),
globals.PlatformInterface.klass);


smalltalk.addClass('Service', globals.Object, [], 'Kernel-Infrastructure');
globals.Service.comment="I implement the basic behavior for class registration to a service.\x0a\x0aSee the `Transcript` class for a concrete service.\x0a\x0a## API\x0a\x0aUse class-side methods `#register:` and `#registerIfNone:` to register classes to a specific service.";

globals.Service.klass.iVarNames = ['current'];
smalltalk.addMethod(
smalltalk.method({
selector: "current",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@current"];
return $1;
},
args: [],
source: "current\x0a\x09^ current",
messageSends: [],
referencedClasses: []
}),
globals.Service.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._shouldNotImplement();
return self}, function($ctx1) {$ctx1.fill(self,"new",{},globals.Service.klass)})},
args: [],
source: "new\x0a\x09self shouldNotImplement",
messageSends: ["shouldNotImplement"],
referencedClasses: []
}),
globals.Service.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "register:",
protocol: 'registration',
fn: function (anObject){
var self=this;
self["@current"]=anObject;
return self},
args: ["anObject"],
source: "register: anObject\x0a\x09current := anObject",
messageSends: [],
referencedClasses: []
}),
globals.Service.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "registerIfNone:",
protocol: 'registration',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._current();
if(($receiver = $1) == nil || $receiver == null){
self._register_(anObject);
} else {
$1;
};
return self}, function($ctx1) {$ctx1.fill(self,"registerIfNone:",{anObject:anObject},globals.Service.klass)})},
args: ["anObject"],
source: "registerIfNone: anObject\x0a\x09self current ifNil: [ self register: anObject ]",
messageSends: ["ifNil:", "current", "register:"],
referencedClasses: []
}),
globals.Service.klass);


smalltalk.addClass('ErrorHandler', globals.Service, [], 'Kernel-Infrastructure');
globals.ErrorHandler.comment="I am the service used to handle Smalltalk errors.\x0aSee `boot.js` `handleError()` function.\x0a\x0aRegistered service instances must implement `#handleError:` to perform an action on the thrown exception.";

smalltalk.addMethod(
smalltalk.method({
selector: "handleError:",
protocol: 'error handling',
fn: function (anError){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._handleUnhandledError_(anError);
return self}, function($ctx1) {$ctx1.fill(self,"handleError:",{anError:anError},globals.ErrorHandler.klass)})},
args: ["anError"],
source: "handleError: anError\x0a\x09self handleUnhandledError: anError",
messageSends: ["handleUnhandledError:"],
referencedClasses: []
}),
globals.ErrorHandler.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "handleUnhandledError:",
protocol: 'error handling',
fn: function (anError){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st(anError)._wasHandled();
if(smalltalk.assert($1)){
return self;
};
$2=_st(self._current())._handleError_(anError);
return $2;
}, function($ctx1) {$ctx1.fill(self,"handleUnhandledError:",{anError:anError},globals.ErrorHandler.klass)})},
args: ["anError"],
source: "handleUnhandledError: anError\x0a\x09anError wasHandled ifTrue: [ ^ self ].\x0a\x09\x0a\x09^ self current handleError: anError",
messageSends: ["ifTrue:", "wasHandled", "handleError:", "current"],
referencedClasses: []
}),
globals.ErrorHandler.klass);


smalltalk.addClass('Finder', globals.Service, [], 'Kernel-Infrastructure');
globals.Finder.comment="I am the service responsible for finding classes/methods.\x0a__There is no default finder.__\x0a\x0a## API\x0a\x0aUse `#browse` on an object to find it.";

smalltalk.addMethod(
smalltalk.method({
selector: "findClass:",
protocol: 'finding',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._current())._findClass_(aClass);
return $1;
}, function($ctx1) {$ctx1.fill(self,"findClass:",{aClass:aClass},globals.Finder.klass)})},
args: ["aClass"],
source: "findClass: aClass\x0a\x09^ self current findClass: aClass",
messageSends: ["findClass:", "current"],
referencedClasses: []
}),
globals.Finder.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "findMethod:",
protocol: 'finding',
fn: function (aCompiledMethod){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._current())._findMethod_(aCompiledMethod);
return $1;
}, function($ctx1) {$ctx1.fill(self,"findMethod:",{aCompiledMethod:aCompiledMethod},globals.Finder.klass)})},
args: ["aCompiledMethod"],
source: "findMethod: aCompiledMethod\x0a\x09^ self current findMethod: aCompiledMethod",
messageSends: ["findMethod:", "current"],
referencedClasses: []
}),
globals.Finder.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "findString:",
protocol: 'finding',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._current())._findString_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"findString:",{aString:aString},globals.Finder.klass)})},
args: ["aString"],
source: "findString: aString\x0a\x09^ self current findString: aString",
messageSends: ["findString:", "current"],
referencedClasses: []
}),
globals.Finder.klass);


smalltalk.addClass('Inspector', globals.Service, [], 'Kernel-Infrastructure');
globals.Inspector.comment="I am the service responsible for inspecting objects.\x0a\x0aThe default inspector object is the transcript.";

smalltalk.addMethod(
smalltalk.method({
selector: "inspect:",
protocol: 'inspecting',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._current())._inspect_(anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"inspect:",{anObject:anObject},globals.Inspector.klass)})},
args: ["anObject"],
source: "inspect: anObject\x0a\x09^ self current inspect: anObject",
messageSends: ["inspect:", "current"],
referencedClasses: []
}),
globals.Inspector.klass);


smalltalk.addClass('ProgressHandler', globals.Service, [], 'Kernel-Infrastructure');
globals.ProgressHandler.comment="I am used to manage progress in collection iterations, see `SequenceableCollection >> #do:displayingProgress:`.\x0a\x0aRegistered instances must implement `#do:on:displaying:`.\x0a\x0aThe default behavior is to simply iterate over the collection, using `NullProgressHandler`.";

smalltalk.addMethod(
smalltalk.method({
selector: "do:on:displaying:",
protocol: 'progress handling',
fn: function (aBlock,aCollection,aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._current())._do_on_displaying_(aBlock,aCollection,aString);
return self}, function($ctx1) {$ctx1.fill(self,"do:on:displaying:",{aBlock:aBlock,aCollection:aCollection,aString:aString},globals.ProgressHandler.klass)})},
args: ["aBlock", "aCollection", "aString"],
source: "do: aBlock on: aCollection displaying: aString\x0a\x09self current do: aBlock on: aCollection displaying: aString",
messageSends: ["do:on:displaying:", "current"],
referencedClasses: []
}),
globals.ProgressHandler.klass);


smalltalk.addClass('Transcript', globals.Service, [], 'Kernel-Infrastructure');
globals.Transcript.comment="I am a facade for Transcript actions.\x0a\x0aI delegate actions to the currently registered transcript.\x0a\x0a## API\x0a\x0a    Transcript \x0a        show: 'hello world';\x0a        cr;\x0a        show: anObject.";

smalltalk.addMethod(
smalltalk.method({
selector: "clear",
protocol: 'printing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._current())._clear();
return self}, function($ctx1) {$ctx1.fill(self,"clear",{},globals.Transcript.klass)})},
args: [],
source: "clear\x0a\x09self current clear",
messageSends: ["clear", "current"],
referencedClasses: []
}),
globals.Transcript.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "cr",
protocol: 'printing',
fn: function (){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
_st(self._current())._show_(_st($String())._cr());
return self}, function($ctx1) {$ctx1.fill(self,"cr",{},globals.Transcript.klass)})},
args: [],
source: "cr\x0a\x09self current show: String cr",
messageSends: ["show:", "current", "cr"],
referencedClasses: ["String"]
}),
globals.Transcript.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "inspect:",
protocol: 'printing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._show_(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"inspect:",{anObject:anObject},globals.Transcript.klass)})},
args: ["anObject"],
source: "inspect: anObject\x0a\x09self show: anObject",
messageSends: ["show:"],
referencedClasses: []
}),
globals.Transcript.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "open",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._current())._open();
return self}, function($ctx1) {$ctx1.fill(self,"open",{},globals.Transcript.klass)})},
args: [],
source: "open\x0a\x09self current open",
messageSends: ["open", "current"],
referencedClasses: []
}),
globals.Transcript.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "show:",
protocol: 'printing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._current())._show_(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"show:",{anObject:anObject},globals.Transcript.klass)})},
args: ["anObject"],
source: "show: anObject\x0a\x09self current show: anObject",
messageSends: ["show:", "current"],
referencedClasses: []
}),
globals.Transcript.klass);


smalltalk.addClass('Setting', globals.Object, ['key', 'value', 'defaultValue'], 'Kernel-Infrastructure');
globals.Setting.comment="I represent a setting accessible via `Smalltalk settings`.\x0a\x0a## API\x0a\x0aA `Setting` value can be read using `value` and set using `value:`.\x0a\x0aSettings are accessed with `'key' asSetting` or `'key' asSettingIfAbsent: 'defaultValue'`.";
smalltalk.addMethod(
smalltalk.method({
selector: "defaultValue",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@defaultValue"];
return $1;
},
args: [],
source: "defaultValue\x0a\x09^ defaultValue",
messageSends: [],
referencedClasses: []
}),
globals.Setting);

smalltalk.addMethod(
smalltalk.method({
selector: "defaultValue:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@defaultValue"]=anObject;
return self},
args: ["anObject"],
source: "defaultValue: anObject\x0a\x09defaultValue := anObject",
messageSends: [],
referencedClasses: []
}),
globals.Setting);

smalltalk.addMethod(
smalltalk.method({
selector: "key",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@key"];
return $1;
},
args: [],
source: "key\x0a\x09^ key",
messageSends: [],
referencedClasses: []
}),
globals.Setting);

smalltalk.addMethod(
smalltalk.method({
selector: "key:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@key"]=anObject;
return self},
args: ["anObject"],
source: "key: anObject\x0a\x09key := anObject",
messageSends: [],
referencedClasses: []
}),
globals.Setting);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Smalltalk())._settings())._at_ifAbsent_(self._key(),(function(){
return smalltalk.withContext(function($ctx2) {
return self._defaultValue();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"value",{},globals.Setting)})},
args: [],
source: "value\x0a\x09^ Smalltalk settings at: self key ifAbsent: [ self defaultValue ]",
messageSends: ["at:ifAbsent:", "settings", "key", "defaultValue"],
referencedClasses: ["Smalltalk"]
}),
globals.Setting);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'accessing',
fn: function (aString){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($Smalltalk())._settings())._at_put_(self._key(),aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"value:",{aString:aString},globals.Setting)})},
args: ["aString"],
source: "value: aString\x0a\x09^ Smalltalk settings at: self key put: aString",
messageSends: ["at:put:", "settings", "key"],
referencedClasses: ["Smalltalk"]
}),
globals.Setting);


smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'instance creation',
fn: function (aString,anotherString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=globals.Setting.klass.superclass.fn.prototype._new.apply(_st(self), []);
_st($2)._key_(aString);
_st($2)._defaultValue_(anotherString);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{aString:aString,anotherString:anotherString},globals.Setting.klass)})},
args: ["aString", "anotherString"],
source: "at: aString ifAbsent: anotherString\x0a\x09^ super new\x0a\x09\x09key: aString;\x0a\x09\x09defaultValue: anotherString;\x0a\x09\x09yourself",
messageSends: ["key:", "new", "defaultValue:", "yourself"],
referencedClasses: []
}),
globals.Setting.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._shouldNotImplement();
return self}, function($ctx1) {$ctx1.fill(self,"new",{},globals.Setting.klass)})},
args: [],
source: "new\x0a\x09self shouldNotImplement",
messageSends: ["shouldNotImplement"],
referencedClasses: []
}),
globals.Setting.klass);


smalltalk.addClass('SmalltalkImage', globals.Object, [], 'Kernel-Infrastructure');
globals.SmalltalkImage.comment="I represent the Smalltalk system, wrapping\x0aoperations of variable `smalltalk` declared in `support/boot.js`.\x0a\x0a## API\x0a\x0aI have only one instance, accessed with global variable `Smalltalk`.\x0a\x0aThe `smalltalk` object holds all class and packages defined in the system.\x0a\x0a## Classes\x0a\x0aClasses can be accessed using the following methods:\x0a\x0a- `#classes` answers the full list of Smalltalk classes in the system\x0a- `#at:` answers a specific class or `nil`\x0a\x0a## Packages\x0a\x0aPackages can be accessed using the following methods:\x0a\x0a- `#packages` answers the full list of packages\x0a- `#packageAt:` answers a specific package or `nil`\x0a\x0a## Parsing\x0a\x0aThe `#parse:` method is used to parse Amber source code.\x0aIt requires the `Compiler` package and the `support/parser.js` parser file in order to work.";
smalltalk.addMethod(
smalltalk.method({
selector: "addGlobalJsVariable:",
protocol: 'globals',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._globalJsVariables())._add_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"addGlobalJsVariable:",{aString:aString},globals.SmalltalkImage)})},
args: ["aString"],
source: "addGlobalJsVariable: aString\x0a\x09self globalJsVariables add: aString",
messageSends: ["add:", "globalJsVariables"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "amdRequire",
protocol: 'accessing amd',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._vm())._at_("amdRequire");
return $1;
}, function($ctx1) {$ctx1.fill(self,"amdRequire",{},globals.SmalltalkImage)})},
args: [],
source: "amdRequire\x0a\x09^ self vm at: 'amdRequire'",
messageSends: ["at:", "vm"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "asSmalltalkException:",
protocol: 'error handling',
fn: function (anObject){
var self=this;
function $Error(){return globals.Error||(typeof Error=="undefined"?nil:Error)}
function $JavaScriptException(){return globals.JavaScriptException||(typeof JavaScriptException=="undefined"?nil:JavaScriptException)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(self._isSmalltalkObject_(anObject))._and_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(anObject)._isKindOf_($Error());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
if(smalltalk.assert($2)){
$1=anObject;
} else {
$1=_st($JavaScriptException())._on_(anObject);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"asSmalltalkException:",{anObject:anObject},globals.SmalltalkImage)})},
args: ["anObject"],
source: "asSmalltalkException: anObject\x0a\x09\x22A JavaScript exception may be thrown.\x0a\x09We then need to convert it back to a Smalltalk object\x22\x0a\x09\x0a\x09^ ((self isSmalltalkObject: anObject) and: [ anObject isKindOf: Error ])\x0a\x09\x09ifTrue: [ anObject ]\x0a\x09\x09ifFalse: [ JavaScriptException on: anObject ]",
messageSends: ["ifTrue:ifFalse:", "and:", "isSmalltalkObject:", "isKindOf:", "on:"],
referencedClasses: ["Error", "JavaScriptException"]
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "at:",
protocol: 'accessing',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=_st(self._globals())._at_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:",{aString:aString},globals.SmalltalkImage)})},
args: ["aString"],
source: "at: aString\x0a\x09self deprecatedAPI.\x0a\x09^ self globals at: aString",
messageSends: ["deprecatedAPI", "at:", "globals"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "at:ifAbsent:",
protocol: 'accessing',
fn: function (aKey,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._includesKey_(aKey);
if(smalltalk.assert($2)){
$1=self._at_(aKey);
} else {
$1=_st(aBlock)._value();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:ifAbsent:",{aKey:aKey,aBlock:aBlock},globals.SmalltalkImage)})},
args: ["aKey", "aBlock"],
source: "at: aKey ifAbsent: aBlock\x0a\x09^ (self includesKey: aKey)\x0a\x09\x09ifTrue: [ self at: aKey ]\x0a\x09\x09ifFalse: [ aBlock value ]",
messageSends: ["ifTrue:ifFalse:", "includesKey:", "at:", "value"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "at:put:",
protocol: 'accessing',
fn: function (aString,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=_st(self._globals())._at_put_(aString,anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"at:put:",{aString:aString,anObject:anObject},globals.SmalltalkImage)})},
args: ["aString", "anObject"],
source: "at: aString put: anObject\x0a\x09self deprecatedAPI.\x0a\x09^ self globals at: aString put: anObject",
messageSends: ["deprecatedAPI", "at:put:", "globals"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "basicCreatePackage:",
protocol: 'private',
fn: function (packageName){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.addPackage(packageName);
return self}, function($ctx1) {$ctx1.fill(self,"basicCreatePackage:",{packageName:packageName},globals.SmalltalkImage)})},
args: ["packageName"],
source: "basicCreatePackage: packageName\x0a\x09\x22Create and bind a new bare package with given name and return it.\x22\x0a\x09<return smalltalk.addPackage(packageName)>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "basicParse:",
protocol: 'private',
fn: function (aString){
var self=this;
function $SmalltalkParser(){return globals.SmalltalkParser||(typeof SmalltalkParser=="undefined"?nil:SmalltalkParser)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($SmalltalkParser())._parse_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"basicParse:",{aString:aString},globals.SmalltalkImage)})},
args: ["aString"],
source: "basicParse: aString\x0a\x09^ SmalltalkParser parse: aString",
messageSends: ["parse:"],
referencedClasses: ["SmalltalkParser"]
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "classes",
protocol: 'classes',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.classes();
return self}, function($ctx1) {$ctx1.fill(self,"classes",{},globals.SmalltalkImage)})},
args: [],
source: "classes\x0a\x09<return smalltalk.classes()>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "createPackage:",
protocol: 'packages',
fn: function (packageName){
var self=this;
var package_,announcement;
function $PackageAdded(){return globals.PackageAdded||(typeof PackageAdded=="undefined"?nil:PackageAdded)}
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
package_=self._basicCreatePackage_(packageName);
$1=_st($PackageAdded())._new();
_st($1)._package_(package_);
$2=_st($1)._yourself();
announcement=$2;
_st(_st($SystemAnnouncer())._current())._announce_(announcement);
$3=package_;
return $3;
}, function($ctx1) {$ctx1.fill(self,"createPackage:",{packageName:packageName,package_:package_,announcement:announcement},globals.SmalltalkImage)})},
args: ["packageName"],
source: "createPackage: packageName\x0a\x09| package announcement |\x0a\x09\x0a\x09package := self basicCreatePackage: packageName.\x0a\x09announcement := PackageAdded new\x0a\x09\x09package: package;\x0a\x09\x09yourself.\x0a\x09\x09\x0a\x09SystemAnnouncer current announce: announcement.\x0a\x09\x0a\x09^ package",
messageSends: ["basicCreatePackage:", "package:", "new", "yourself", "announce:", "current"],
referencedClasses: ["PackageAdded", "SystemAnnouncer"]
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "createPackage:properties:",
protocol: 'private',
fn: function (packageName,aDict){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._deprecatedAPI();
$1=_st(aDict)._isEmpty();
if(! smalltalk.assert($1)){
self._error_("createPackage:properties: called with nonempty properties");
};
$2=self._createPackage_(packageName);
return $2;
}, function($ctx1) {$ctx1.fill(self,"createPackage:properties:",{packageName:packageName,aDict:aDict},globals.SmalltalkImage)})},
args: ["packageName", "aDict"],
source: "createPackage: packageName properties: aDict\x0a\x09\x22Needed to import .st files: they begin with this call.\x22\x0a\x09self deprecatedAPI.\x0a\x09\x0a\x09aDict isEmpty ifFalse: [ self error: 'createPackage:properties: called with nonempty properties' ].\x0a\x09^ self createPackage: packageName",
messageSends: ["deprecatedAPI", "ifFalse:", "isEmpty", "error:", "createPackage:"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "current",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._deprecatedAPI();
return self;
}, function($ctx1) {$ctx1.fill(self,"current",{},globals.SmalltalkImage)})},
args: [],
source: "current\x0a\x09\x22Backward compatibility for Smalltalk current ...\x22\x0a\x09self deprecatedAPI.\x0a\x09^ self",
messageSends: ["deprecatedAPI"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "defaultAmdNamespace",
protocol: 'accessing amd',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1="transport.defaultAmdNamespace"._settingValue();
return $1;
}, function($ctx1) {$ctx1.fill(self,"defaultAmdNamespace",{},globals.SmalltalkImage)})},
args: [],
source: "defaultAmdNamespace\x0a\x09^ 'transport.defaultAmdNamespace' settingValue",
messageSends: ["settingValue"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "defaultAmdNamespace:",
protocol: 'accessing amd',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
"transport.defaultAmdNamespace"._settingValue_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"defaultAmdNamespace:",{aString:aString},globals.SmalltalkImage)})},
args: ["aString"],
source: "defaultAmdNamespace: aString\x0a\x09'transport.defaultAmdNamespace' settingValue: aString",
messageSends: ["settingValue:"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "deleteClass:",
protocol: 'private',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
smalltalk.removeClass(aClass);
return self}, function($ctx1) {$ctx1.fill(self,"deleteClass:",{aClass:aClass},globals.SmalltalkImage)})},
args: ["aClass"],
source: "deleteClass: aClass\x0a\x09\x22Deletes a class by deleting its binding only. Use #removeClass instead\x22\x0a\x09\x0a\x09<smalltalk.removeClass(aClass)>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "deleteGlobalJsVariable:",
protocol: 'globals',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._globalJsVariables())._remove_ifAbsent_(aString,(function(){
}));
return self}, function($ctx1) {$ctx1.fill(self,"deleteGlobalJsVariable:",{aString:aString},globals.SmalltalkImage)})},
args: ["aString"],
source: "deleteGlobalJsVariable: aString\x0a\x09self globalJsVariables remove: aString ifAbsent:[]",
messageSends: ["remove:ifAbsent:", "globalJsVariables"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "deletePackage:",
protocol: 'private',
fn: function (packageName){
var self=this;
return smalltalk.withContext(function($ctx1) { 
delete smalltalk.packages[packageName];
return self}, function($ctx1) {$ctx1.fill(self,"deletePackage:",{packageName:packageName},globals.SmalltalkImage)})},
args: ["packageName"],
source: "deletePackage: packageName\x0a\x09\x22Deletes a package by deleting its binding, but does not check if it contains classes etc.\x0a\x09To remove a package, use #removePackage instead.\x22\x0a\x0a\x09<delete smalltalk.packages[packageName]>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "globalJsVariables",
protocol: 'globals',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.globalJsVariables;
return self}, function($ctx1) {$ctx1.fill(self,"globalJsVariables",{},globals.SmalltalkImage)})},
args: [],
source: "globalJsVariables\x0a\x09\x22Array of global JavaScript variables\x22\x0a\x09<return smalltalk.globalJsVariables>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "globals",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return globals;
return self}, function($ctx1) {$ctx1.fill(self,"globals",{},globals.SmalltalkImage)})},
args: [],
source: "globals\x0a\x09\x22Future compatibility to be able to use Smalltalk globals at: ...\x22\x0a\x09<return globals>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "includesKey:",
protocol: 'accessing',
fn: function (aKey){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.hasOwnProperty(aKey);
return self}, function($ctx1) {$ctx1.fill(self,"includesKey:",{aKey:aKey},globals.SmalltalkImage)})},
args: ["aKey"],
source: "includesKey: aKey\x0a\x09<return smalltalk.hasOwnProperty(aKey)>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "isSmalltalkObject:",
protocol: 'testing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return typeof anObject.klass !== 'undefined';
return self}, function($ctx1) {$ctx1.fill(self,"isSmalltalkObject:",{anObject:anObject},globals.SmalltalkImage)})},
args: ["anObject"],
source: "isSmalltalkObject: anObject\x0a\x09\x22Consider anObject a Smalltalk object if it has a 'klass' property.\x0a\x09Note that this may be unaccurate\x22\x0a\x09\x0a\x09<return typeof anObject.klass !== 'undefined'>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "packageAt:",
protocol: 'packages',
fn: function (packageName){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.packages[packageName];
return self}, function($ctx1) {$ctx1.fill(self,"packageAt:",{packageName:packageName},globals.SmalltalkImage)})},
args: ["packageName"],
source: "packageAt: packageName\x0a\x09<return smalltalk.packages[packageName]>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "packageAt:ifAbsent:",
protocol: 'packages',
fn: function (packageName,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._packageAt_(packageName);
$1=_st($2)._ifNil_(aBlock);
return $1;
}, function($ctx1) {$ctx1.fill(self,"packageAt:ifAbsent:",{packageName:packageName,aBlock:aBlock},globals.SmalltalkImage)})},
args: ["packageName", "aBlock"],
source: "packageAt: packageName ifAbsent: aBlock\x0a\x09^ (self packageAt: packageName) ifNil: aBlock",
messageSends: ["ifNil:", "packageAt:"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "packages",
protocol: 'packages',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		return Object.keys(smalltalk.packages).map(function(k) {
			return smalltalk.packages[k];
		})
	;
return self}, function($ctx1) {$ctx1.fill(self,"packages",{},globals.SmalltalkImage)})},
args: [],
source: "packages\x0a\x09\x22Return all Package instances in the system.\x22\x0a\x0a\x09<\x0a\x09\x09return Object.keys(smalltalk.packages).map(function(k) {\x0a\x09\x09\x09return smalltalk.packages[k];\x0a\x09\x09})\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "parse:",
protocol: 'accessing',
fn: function (aString){
var self=this;
var result;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
_st((function(){
return smalltalk.withContext(function($ctx2) {
result=self._basicParse_(aString);
return result;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}))._tryCatch_((function(ex){
return smalltalk.withContext(function($ctx2) {
return _st(self._parseError_parsing_(ex,aString))._signal();
}, function($ctx2) {$ctx2.fillBlock({ex:ex},$ctx1,2)})}));
$2=result;
_st($2)._source_(aString);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"parse:",{aString:aString,result:result},globals.SmalltalkImage)})},
args: ["aString"],
source: "parse: aString\x0a\x09| result |\x0a\x09\x0a\x09[ result := self basicParse: aString ] \x0a\x09\x09tryCatch: [ :ex | (self parseError: ex parsing: aString) signal ].\x0a\x09\x09\x0a\x09^ result\x0a\x09\x09source: aString;\x0a\x09\x09yourself",
messageSends: ["tryCatch:", "basicParse:", "signal", "parseError:parsing:", "source:", "yourself"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "parseError:parsing:",
protocol: 'error handling',
fn: function (anException,aString){
var self=this;
function $ParseError(){return globals.ParseError||(typeof ParseError=="undefined"?nil:ParseError)}
return smalltalk.withContext(function($ctx1) { 
var $2,$8,$7,$6,$9,$5,$4,$3,$1;
$2=_st($ParseError())._new();
$8=_st(anException)._basicAt_("line");
$ctx1.sendIdx["basicAt:"]=1;
$7="Parse error on line ".__comma($8);
$6=_st($7).__comma(" column ");
$ctx1.sendIdx[","]=4;
$9=_st(anException)._basicAt_("column");
$ctx1.sendIdx["basicAt:"]=2;
$5=_st($6).__comma($9);
$ctx1.sendIdx[","]=3;
$4=_st($5).__comma(" : Unexpected character ");
$ctx1.sendIdx[","]=2;
$3=_st($4).__comma(_st(anException)._basicAt_("found"));
$ctx1.sendIdx[","]=1;
$1=_st($2)._messageText_($3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"parseError:parsing:",{anException:anException,aString:aString},globals.SmalltalkImage)})},
args: ["anException", "aString"],
source: "parseError: anException parsing: aString\x0a\x09^ ParseError new messageText: 'Parse error on line ', (anException basicAt: 'line') ,' column ' , (anException basicAt: 'column') ,' : Unexpected character ', (anException basicAt: 'found')",
messageSends: ["messageText:", "new", ",", "basicAt:"],
referencedClasses: ["ParseError"]
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "pseudoVariableNames",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=["self", "super", "nil", "true", "false", "thisContext"];
return $1;
},
args: [],
source: "pseudoVariableNames\x0a\x09^ #('self' 'super' 'nil' 'true' 'false' 'thisContext')",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "readJSObject:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.readJSObject(anObject);
return self}, function($ctx1) {$ctx1.fill(self,"readJSObject:",{anObject:anObject},globals.SmalltalkImage)})},
args: ["anObject"],
source: "readJSObject: anObject\x0a\x09<return smalltalk.readJSObject(anObject)>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "removeClass:",
protocol: 'classes',
fn: function (aClass){
var self=this;
function $SystemAnnouncer(){return globals.SystemAnnouncer||(typeof SystemAnnouncer=="undefined"?nil:SystemAnnouncer)}
function $ClassRemoved(){return globals.ClassRemoved||(typeof ClassRemoved=="undefined"?nil:ClassRemoved)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
$1=_st(aClass)._isMetaclass();
if(smalltalk.assert($1)){
self._error_(_st(_st(aClass)._asString()).__comma(" is a Metaclass and cannot be removed!"));
};
self._deleteClass_(aClass);
$2=_st($ClassRemoved())._new();
_st($2)._theClass_(aClass);
$3=_st($2)._yourself();
_st(_st($SystemAnnouncer())._current())._announce_($3);
return self}, function($ctx1) {$ctx1.fill(self,"removeClass:",{aClass:aClass},globals.SmalltalkImage)})},
args: ["aClass"],
source: "removeClass: aClass\x0a\x09aClass isMetaclass ifTrue: [ self error: aClass asString, ' is a Metaclass and cannot be removed!' ].\x0a\x09\x0a\x09self deleteClass: aClass.\x0a\x09\x0a\x09SystemAnnouncer current\x0a\x09\x09announce: (ClassRemoved new\x0a\x09\x09\x09theClass: aClass;\x0a\x09\x09\x09yourself)",
messageSends: ["ifTrue:", "isMetaclass", "error:", ",", "asString", "deleteClass:", "announce:", "current", "theClass:", "new", "yourself"],
referencedClasses: ["SystemAnnouncer", "ClassRemoved"]
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "removePackage:",
protocol: 'packages',
fn: function (packageName){
var self=this;
var pkg;
return smalltalk.withContext(function($ctx1) { 
pkg=self._packageAt_ifAbsent_(packageName,(function(){
return smalltalk.withContext(function($ctx2) {
return self._error_("Missing package: ".__comma(packageName));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
_st(_st(pkg)._classes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._removeClass_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
self._deletePackage_(packageName);
return self}, function($ctx1) {$ctx1.fill(self,"removePackage:",{packageName:packageName,pkg:pkg},globals.SmalltalkImage)})},
args: ["packageName"],
source: "removePackage: packageName\x0a\x09\x22Removes a package and all its classes.\x22\x0a\x0a\x09| pkg |\x0a\x09pkg := self packageAt: packageName ifAbsent: [ self error: 'Missing package: ', packageName ].\x0a\x09pkg classes do: [ :each |\x0a\x09\x09\x09self removeClass: each ].\x0a\x09self deletePackage: packageName",
messageSends: ["packageAt:ifAbsent:", "error:", ",", "do:", "classes", "removeClass:", "deletePackage:"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "renamePackage:to:",
protocol: 'packages',
fn: function (packageName,newName){
var self=this;
var pkg;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
pkg=self._packageAt_ifAbsent_(packageName,(function(){
return smalltalk.withContext(function($ctx2) {
$1="Missing package: ".__comma(packageName);
$ctx2.sendIdx[","]=1;
return self._error_($1);
$ctx2.sendIdx["error:"]=1;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$2=self._packageAt_(newName);
if(($receiver = $2) == nil || $receiver == null){
$2;
} else {
self._error_("Already exists a package called: ".__comma(newName));
};
_st(self._at_("packages"))._at_put_(newName,pkg);
_st(pkg)._name_(newName);
self._deletePackage_(packageName);
return self}, function($ctx1) {$ctx1.fill(self,"renamePackage:to:",{packageName:packageName,newName:newName,pkg:pkg},globals.SmalltalkImage)})},
args: ["packageName", "newName"],
source: "renamePackage: packageName to: newName\x0a\x09\x22Rename a package.\x22\x0a\x0a\x09| pkg |\x0a\x09pkg := self packageAt: packageName ifAbsent: [ self error: 'Missing package: ', packageName ].\x0a\x09(self packageAt: newName) ifNotNil: [ self error: 'Already exists a package called: ', newName ].\x0a\x09(self at: 'packages') at: newName put: pkg.\x0a\x09pkg name: newName.\x0a\x09self deletePackage: packageName.",
messageSends: ["packageAt:ifAbsent:", "error:", ",", "ifNotNil:", "packageAt:", "at:put:", "at:", "name:", "deletePackage:"],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "reservedWords",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk.reservedWords;
return self}, function($ctx1) {$ctx1.fill(self,"reservedWords",{},globals.SmalltalkImage)})},
args: [],
source: "reservedWords\x0a\x09\x22JavaScript reserved words\x22\x0a\x09<return smalltalk.reservedWords>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "settings",
protocol: 'accessing',
fn: function (){
var self=this;
function $SmalltalkSettings(){return globals.SmalltalkSettings||(typeof SmalltalkSettings=="undefined"?nil:SmalltalkSettings)}
return $SmalltalkSettings();
},
args: [],
source: "settings\x0a\x09^ SmalltalkSettings",
messageSends: [],
referencedClasses: ["SmalltalkSettings"]
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "version",
protocol: 'accessing',
fn: function (){
var self=this;
return "0.13.0-pre";
},
args: [],
source: "version\x0a\x09\x22Answer the version string of Amber\x22\x0a\x09\x0a\x09^ '0.13.0-pre'",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);

smalltalk.addMethod(
smalltalk.method({
selector: "vm",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return smalltalk;
return self}, function($ctx1) {$ctx1.fill(self,"vm",{},globals.SmalltalkImage)})},
args: [],
source: "vm\x0a\x09\x22Future compatibility to be able to use Smalltalk vm ...\x22\x0a\x09<return smalltalk>",
messageSends: [],
referencedClasses: []
}),
globals.SmalltalkImage);


globals.SmalltalkImage.klass.iVarNames = ['current'];
smalltalk.addMethod(
smalltalk.method({
selector: "current",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@current"];
if(($receiver = $2) == nil || $receiver == null){
self["@current"]=globals.SmalltalkImage.klass.superclass.fn.prototype._new.apply(_st(self), []);
$1=self["@current"];
} else {
self._deprecatedAPI();
$1=self["@current"];
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"current",{},globals.SmalltalkImage.klass)})},
args: [],
source: "current\x0a\x09^ current ifNil: [ current := super new ] ifNotNil: [ self deprecatedAPI. current ]",
messageSends: ["ifNil:ifNotNil:", "new", "deprecatedAPI"],
referencedClasses: []
}),
globals.SmalltalkImage.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(globals)._at_put_("Smalltalk",self._current());
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.SmalltalkImage.klass)})},
args: [],
source: "initialize\x0a\x09globals at: 'Smalltalk' put: self current",
messageSends: ["at:put:", "current"],
referencedClasses: []
}),
globals.SmalltalkImage.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._shouldNotImplement();
return self}, function($ctx1) {$ctx1.fill(self,"new",{},globals.SmalltalkImage.klass)})},
args: [],
source: "new\x0a\x09self shouldNotImplement",
messageSends: ["shouldNotImplement"],
referencedClasses: []
}),
globals.SmalltalkImage.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "do:displayingProgress:",
protocol: '*Kernel-Infrastructure',
fn: function (aBlock,aString){
var self=this;
function $ProgressHandler(){return globals.ProgressHandler||(typeof ProgressHandler=="undefined"?nil:ProgressHandler)}
return smalltalk.withContext(function($ctx1) { 
_st($ProgressHandler())._do_on_displaying_(aBlock,self,aString);
return self}, function($ctx1) {$ctx1.fill(self,"do:displayingProgress:",{aBlock:aBlock,aString:aString},globals.SequenceableCollection)})},
args: ["aBlock", "aString"],
source: "do: aBlock displayingProgress: aString\x0a\x09ProgressHandler \x0a\x09\x09do: aBlock \x0a\x09\x09on: self \x0a\x09\x09displaying: aString",
messageSends: ["do:on:displaying:"],
referencedClasses: ["ProgressHandler"]
}),
globals.SequenceableCollection);

smalltalk.addMethod(
smalltalk.method({
selector: "asJavaScriptSelector",
protocol: '*Kernel-Infrastructure',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._replace_with_("^([a-zA-Z0-9]*).*$","$1");
return $1;
}, function($ctx1) {$ctx1.fill(self,"asJavaScriptSelector",{},globals.String)})},
args: [],
source: "asJavaScriptSelector\x0a\x09\x22Return first keyword of the selector, without trailing colon.\x22\x0a\x09^ self replace: '^([a-zA-Z0-9]*).*$' with: '$1'",
messageSends: ["replace:with:"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asSetting",
protocol: '*Kernel-Infrastructure',
fn: function (){
var self=this;
function $Setting(){return globals.Setting||(typeof Setting=="undefined"?nil:Setting)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Setting())._at_ifAbsent_(self,nil);
return $1;
}, function($ctx1) {$ctx1.fill(self,"asSetting",{},globals.String)})},
args: [],
source: "asSetting\x0a\x09^ Setting at: self ifAbsent: nil",
messageSends: ["at:ifAbsent:"],
referencedClasses: ["Setting"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "asSettingIfAbsent:",
protocol: '*Kernel-Infrastructure',
fn: function (aString){
var self=this;
function $Setting(){return globals.Setting||(typeof Setting=="undefined"?nil:Setting)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Setting())._at_ifAbsent_(self,aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"asSettingIfAbsent:",{aString:aString},globals.String)})},
args: ["aString"],
source: "asSettingIfAbsent: aString\x0a\x09^ Setting at: self ifAbsent: aString",
messageSends: ["at:ifAbsent:"],
referencedClasses: ["Setting"]
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "settingValue",
protocol: '*Kernel-Infrastructure',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._asSetting())._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"settingValue",{},globals.String)})},
args: [],
source: "settingValue\x0a\x09^ self asSetting value",
messageSends: ["value", "asSetting"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "settingValue:",
protocol: '*Kernel-Infrastructure',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._asSetting())._value_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"settingValue:",{aString:aString},globals.String)})},
args: ["aString"],
source: "settingValue: aString\x0a\x09^ self asSetting value: aString",
messageSends: ["value:", "asSetting"],
referencedClasses: []
}),
globals.String);

smalltalk.addMethod(
smalltalk.method({
selector: "settingValueIfAbsent:",
protocol: '*Kernel-Infrastructure',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._asSettingIfAbsent_(aString))._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"settingValueIfAbsent:",{aString:aString},globals.String)})},
args: ["aString"],
source: "settingValueIfAbsent: aString\x0a\x09^ (self asSettingIfAbsent: aString) value",
messageSends: ["value", "asSettingIfAbsent:"],
referencedClasses: []
}),
globals.String);

});

define("amber_core/Kernel-Exceptions", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Exceptions');
smalltalk.packages["Kernel-Exceptions"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('Error', globals.Object, ['messageText'], 'Kernel-Exceptions');
globals.Error.comment="From the ANSI standard:\x0a\x0aThis protocol describes the behavior of instances of class `Error`.\x0aThese are used to represent error conditions that prevent the normal continuation of processing.\x0aActual error exceptions used by an application may be subclasses of this class.\x0aAs `Error` is explicitly specified to be subclassable, conforming implementations must implement its behavior in a non-fragile manner.";
smalltalk.addMethod(
smalltalk.method({
selector: "beHandled",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.amberHandled = true;
return self}, function($ctx1) {$ctx1.fill(self,"beHandled",{},globals.Error)})},
args: [],
source: "beHandled\x0a\x09<self.amberHandled = true>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "beUnhandled",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.amberHandled = false;
return self}, function($ctx1) {$ctx1.fill(self,"beUnhandled",{},globals.Error)})},
args: [],
source: "beUnhandled\x0a\x09<self.amberHandled = false>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "context",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.context;
return self}, function($ctx1) {$ctx1.fill(self,"context",{},globals.Error)})},
args: [],
source: "context\x0a\x09<return self.context>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._messageText_("Errorclass: ".__comma(_st(self._class())._name()));
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.Error)})},
args: [],
source: "initialize\x0a\x09self messageText: 'Errorclass: ', (self class name).",
messageSends: ["messageText:", ",", "name", "class"],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "isSmalltalkError",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.smalltalkError === true;
return self}, function($ctx1) {$ctx1.fill(self,"isSmalltalkError",{},globals.Error)})},
args: [],
source: "isSmalltalkError\x0a\x09<return self.smalltalkError === true>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "jsStack",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.stack;
return self}, function($ctx1) {$ctx1.fill(self,"jsStack",{},globals.Error)})},
args: [],
source: "jsStack\x0a\x09<return self.stack>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "messageText",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@messageText"];
return $1;
},
args: [],
source: "messageText\x0a\x09^ messageText",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "messageText:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@messageText"]=aString;
return self},
args: ["aString"],
source: "messageText: aString\x0a\x09messageText := aString",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "resignal",
protocol: 'signaling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self.amberHandled = false;
		throw(self);
	;
return self}, function($ctx1) {$ctx1.fill(self,"resignal",{},globals.Error)})},
args: [],
source: "resignal\x0a\x09\x22Resignal the receiver without changing its exception context\x22\x0a\x09\x0a\x09<\x0a\x09\x09self.amberHandled = false;\x0a\x09\x09throw(self);\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "signal",
protocol: 'signaling',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 

		self.amberHandled = false;
		self.context = smalltalk.getThisContext(); 
		self.smalltalkError = true; throw(self)
	;
return self}, function($ctx1) {$ctx1.fill(self,"signal",{},globals.Error)})},
args: [],
source: "signal\x0a\x09<\x0a\x09\x09self.amberHandled = false;\x0a\x09\x09self.context = smalltalk.getThisContext(); \x0a\x09\x09self.smalltalkError = true; throw(self)\x0a\x09>",
messageSends: [],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "signal:",
protocol: 'signaling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._messageText_(aString);
self._signal();
return self}, function($ctx1) {$ctx1.fill(self,"signal:",{aString:aString},globals.Error)})},
args: ["aString"],
source: "signal: aString\x0a\x09self messageText: aString.\x0a\x09self signal",
messageSends: ["messageText:", "signal"],
referencedClasses: []
}),
globals.Error);

smalltalk.addMethod(
smalltalk.method({
selector: "wasHandled",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return self.amberHandled || false;
return self}, function($ctx1) {$ctx1.fill(self,"wasHandled",{},globals.Error)})},
args: [],
source: "wasHandled\x0a\x09<return self.amberHandled || false>",
messageSends: [],
referencedClasses: []
}),
globals.Error);


smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "exception";
},
args: [],
source: "heliosClass\x0a\x09^ 'exception'",
messageSends: [],
referencedClasses: []
}),
globals.Error.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "signal",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._new())._signal();
return $1;
}, function($ctx1) {$ctx1.fill(self,"signal",{},globals.Error.klass)})},
args: [],
source: "signal\x0a\x09^ self new signal",
messageSends: ["signal", "new"],
referencedClasses: []
}),
globals.Error.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "signal:",
protocol: 'instance creation',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._new())._signal_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"signal:",{aString:aString},globals.Error.klass)})},
args: ["aString"],
source: "signal: aString\x0a\x09^ self new\x0a\x09\x09signal: aString",
messageSends: ["signal:", "new"],
referencedClasses: []
}),
globals.Error.klass);


smalltalk.addClass('JavaScriptException', globals.Error, ['exception'], 'Kernel-Exceptions');
globals.JavaScriptException.comment="A JavaScriptException is thrown when a non-Smalltalk exception occurs while in the Smalltalk stack.\x0aSee `boot.js` `inContext()` and `BlockClosure >> on:do:`";
smalltalk.addMethod(
smalltalk.method({
selector: "context:",
protocol: 'accessing',
fn: function (aMethodContext){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self.context = aMethodContext;
return self}, function($ctx1) {$ctx1.fill(self,"context:",{aMethodContext:aMethodContext},globals.JavaScriptException)})},
args: ["aMethodContext"],
source: "context: aMethodContext\x0a\x09\x22Set the context from the outside.\x0a\x09See boot.js `inContext()` exception handling\x22\x0a\x09\x0a\x09<self.context = aMethodContext>",
messageSends: [],
referencedClasses: []
}),
globals.JavaScriptException);

smalltalk.addMethod(
smalltalk.method({
selector: "exception",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@exception"];
return $1;
},
args: [],
source: "exception\x0a\x09^ exception",
messageSends: [],
referencedClasses: []
}),
globals.JavaScriptException);

smalltalk.addMethod(
smalltalk.method({
selector: "exception:",
protocol: 'accessing',
fn: function (anException){
var self=this;
self["@exception"]=anException;
return self},
args: ["anException"],
source: "exception: anException\x0a\x09exception := anException",
messageSends: [],
referencedClasses: []
}),
globals.JavaScriptException);

smalltalk.addMethod(
smalltalk.method({
selector: "messageText",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return 'JavaScript exception: ' + self["@exception"].toString();
return self}, function($ctx1) {$ctx1.fill(self,"messageText",{},globals.JavaScriptException)})},
args: [],
source: "messageText\x0a\x09<return 'JavaScript exception: ' + self[\x22@exception\x22].toString()>",
messageSends: [],
referencedClasses: []
}),
globals.JavaScriptException);


smalltalk.addMethod(
smalltalk.method({
selector: "on:",
protocol: 'instance creation',
fn: function (anException){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._exception_(anException);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:",{anException:anException},globals.JavaScriptException.klass)})},
args: ["anException"],
source: "on: anException\x0a\x09^ self new\x0a\x09\x09exception: anException;\x0a\x09\x09yourself",
messageSends: ["exception:", "new", "yourself"],
referencedClasses: []
}),
globals.JavaScriptException.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "on:context:",
protocol: 'instance creation',
fn: function (anException,aMethodContext){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._exception_(anException);
_st($2)._context_(aMethodContext);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:context:",{anException:anException,aMethodContext:aMethodContext},globals.JavaScriptException.klass)})},
args: ["anException", "aMethodContext"],
source: "on: anException context: aMethodContext\x0a\x09^ self new\x0a\x09\x09exception: anException;\x0a\x09\x09context: aMethodContext;\x0a\x09\x09yourself",
messageSends: ["exception:", "new", "context:", "yourself"],
referencedClasses: []
}),
globals.JavaScriptException.klass);


smalltalk.addClass('MessageNotUnderstood', globals.Error, ['message', 'receiver'], 'Kernel-Exceptions');
globals.MessageNotUnderstood.comment="This exception is provided to support `Object>>doesNotUnderstand:`.";
smalltalk.addMethod(
smalltalk.method({
selector: "message",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@message"];
return $1;
},
args: [],
source: "message\x0a\x09^ message",
messageSends: [],
referencedClasses: []
}),
globals.MessageNotUnderstood);

smalltalk.addMethod(
smalltalk.method({
selector: "message:",
protocol: 'accessing',
fn: function (aMessage){
var self=this;
self["@message"]=aMessage;
return self},
args: ["aMessage"],
source: "message: aMessage\x0a\x09message := aMessage",
messageSends: [],
referencedClasses: []
}),
globals.MessageNotUnderstood);

smalltalk.addMethod(
smalltalk.method({
selector: "messageText",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(_st(self._receiver())._asString()).__comma(" does not understand #")).__comma(_st(self._message())._selector());
$ctx1.sendIdx[","]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageText",{},globals.MessageNotUnderstood)})},
args: [],
source: "messageText\x0a\x09^ self receiver asString, ' does not understand #', self message selector",
messageSends: [",", "asString", "receiver", "selector", "message"],
referencedClasses: []
}),
globals.MessageNotUnderstood);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@receiver"];
return $1;
},
args: [],
source: "receiver\x0a\x09^ receiver",
messageSends: [],
referencedClasses: []
}),
globals.MessageNotUnderstood);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@receiver"]=anObject;
return self},
args: ["anObject"],
source: "receiver: anObject\x0a\x09receiver := anObject",
messageSends: [],
referencedClasses: []
}),
globals.MessageNotUnderstood);



smalltalk.addClass('NonBooleanReceiver', globals.Error, ['object'], 'Kernel-Exceptions');
globals.NonBooleanReceiver.comment="NonBooleanReceiver exceptions may be thrown when executing inlined methods such as `#ifTrue:` with a non boolean receiver.";
smalltalk.addMethod(
smalltalk.method({
selector: "object",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@object"];
return $1;
},
args: [],
source: "object\x0a\x09^ object",
messageSends: [],
referencedClasses: []
}),
globals.NonBooleanReceiver);

smalltalk.addMethod(
smalltalk.method({
selector: "object:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@object"]=anObject;
return self},
args: ["anObject"],
source: "object: anObject\x0a\x09object := anObject",
messageSends: [],
referencedClasses: []
}),
globals.NonBooleanReceiver);



smalltalk.addClass('PackageCommitError', globals.Error, [], 'Kernel-Exceptions');
globals.PackageCommitError.comment="I get signaled when an attempt to commit a package has failed.";

});

define("amber_core/Kernel-Transcript", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Transcript');
smalltalk.packages["Kernel-Transcript"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('ConsoleTranscript', globals.Object, ['textarea'], 'Kernel-Transcript');
globals.ConsoleTranscript.comment="I am a specific transcript emitting to the JavaScript console.\x0a\x0aIf no other transcript is registered, I am the default.";
smalltalk.addMethod(
smalltalk.method({
selector: "clear",
protocol: 'printing',
fn: function (){
var self=this;
return self},
args: [],
source: "clear\x0a\x09\x22no op\x22",
messageSends: [],
referencedClasses: []
}),
globals.ConsoleTranscript);

smalltalk.addMethod(
smalltalk.method({
selector: "cr",
protocol: 'printing',
fn: function (){
var self=this;
return self},
args: [],
source: "cr\x0a\x09\x22no op\x22",
messageSends: [],
referencedClasses: []
}),
globals.ConsoleTranscript);

smalltalk.addMethod(
smalltalk.method({
selector: "open",
protocol: 'actions',
fn: function (){
var self=this;
return self},
args: [],
source: "open",
messageSends: [],
referencedClasses: []
}),
globals.ConsoleTranscript);

smalltalk.addMethod(
smalltalk.method({
selector: "show:",
protocol: 'printing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
console.log(String(_st(anObject)._asString()));
return self}, function($ctx1) {$ctx1.fill(self,"show:",{anObject:anObject},globals.ConsoleTranscript)})},
args: ["anObject"],
source: "show: anObject\x0a\x22Smalltalk objects should have no trouble displaying themselves on the Transcript; Javascript objects don't know how, so must be wrapped in a JSObectProxy.\x22\x0a<console.log(String(_st(anObject)._asString()))>",
messageSends: [],
referencedClasses: []
}),
globals.ConsoleTranscript);


smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $Transcript(){return globals.Transcript||(typeof Transcript=="undefined"?nil:Transcript)}
return smalltalk.withContext(function($ctx1) { 
_st($Transcript())._registerIfNone_(self._new());
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.ConsoleTranscript.klass)})},
args: [],
source: "initialize\x0a\x09Transcript registerIfNone: self new",
messageSends: ["registerIfNone:", "new"],
referencedClasses: ["Transcript"]
}),
globals.ConsoleTranscript.klass);

});

define("amber_core/Kernel-Announcements", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Kernel-Announcements');
smalltalk.packages["Kernel-Announcements"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('AnnouncementSubscription', globals.Object, ['valuable', 'announcementClass'], 'Kernel-Announcements');
globals.AnnouncementSubscription.comment="I am a single entry in a subscription registry of an `Announcer`.\x0aSeveral subscriptions by the same object is possible.";
smalltalk.addMethod(
smalltalk.method({
selector: "announcementClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@announcementClass"];
return $1;
},
args: [],
source: "announcementClass\x0a\x09^ announcementClass",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "announcementClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@announcementClass"]=aClass;
return self},
args: ["aClass"],
source: "announcementClass: aClass\x0a\x09announcementClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "block",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._deprecatedAPI();
$1=self._valuable();
return $1;
}, function($ctx1) {$ctx1.fill(self,"block",{},globals.AnnouncementSubscription)})},
args: [],
source: "block\x0a\x09\x22Use #valuable instead\x22\x0a\x09\x0a\x09self deprecatedAPI.\x0a\x09^ self valuable",
messageSends: ["deprecatedAPI", "valuable"],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "block:",
protocol: 'accessing',
fn: function (aValuable){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._deprecatedAPI();
self._valuable_(aValuable);
return self}, function($ctx1) {$ctx1.fill(self,"block:",{aValuable:aValuable},globals.AnnouncementSubscription)})},
args: ["aValuable"],
source: "block: aValuable\x0a\x09\x22Use #valuable instead\x22\x0a\x09\x0a\x09self deprecatedAPI.\x0a\x09self valuable: aValuable",
messageSends: ["deprecatedAPI", "valuable:"],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "deliver:",
protocol: 'announcing',
fn: function (anAnnouncement){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._handlesAnnouncement_(anAnnouncement);
if(smalltalk.assert($1)){
_st(self._valuable())._value_(anAnnouncement);
};
return self}, function($ctx1) {$ctx1.fill(self,"deliver:",{anAnnouncement:anAnnouncement},globals.AnnouncementSubscription)})},
args: ["anAnnouncement"],
source: "deliver: anAnnouncement\x0a\x09(self handlesAnnouncement: anAnnouncement)\x0a\x09\x09ifTrue: [ self valuable value: anAnnouncement ]",
messageSends: ["ifTrue:", "handlesAnnouncement:", "value:", "valuable"],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "handlesAnnouncement:",
protocol: 'announcing',
fn: function (anAnnouncement){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $3,$4,$2,$1;
$3=_st($Smalltalk())._globals();
$ctx1.sendIdx["globals"]=1;
$4=_st(self._announcementClass())._name();
$ctx1.sendIdx["name"]=1;
$2=_st($3)._at_($4);
$ctx1.sendIdx["at:"]=1;
if(($receiver = $2) == nil || $receiver == null){
return false;
} else {
var class_;
class_=$receiver;
$1=_st(_st(_st($Smalltalk())._globals())._at_(_st(_st(_st(anAnnouncement)._class())._theNonMetaClass())._name()))._includesBehavior_(class_);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"handlesAnnouncement:",{anAnnouncement:anAnnouncement},globals.AnnouncementSubscription)})},
args: ["anAnnouncement"],
source: "handlesAnnouncement: anAnnouncement\x0a\x09\x22anAnnouncement might be announced from within another Amber environment\x22\x0a\x09\x0a\x09^ (Smalltalk globals at: self announcementClass name)\x0a\x09\x09ifNil: [ ^ false ]\x0a\x09\x09ifNotNil: [ :class |\x0a\x09\x09(Smalltalk globals at: anAnnouncement class theNonMetaClass name) includesBehavior: class ]",
messageSends: ["ifNil:ifNotNil:", "at:", "globals", "name", "announcementClass", "includesBehavior:", "theNonMetaClass", "class"],
referencedClasses: ["Smalltalk"]
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._valuable())._receiver();
return $1;
}, function($ctx1) {$ctx1.fill(self,"receiver",{},globals.AnnouncementSubscription)})},
args: [],
source: "receiver\x0a\x09^ self valuable receiver",
messageSends: ["receiver", "valuable"],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "valuable",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@valuable"];
return $1;
},
args: [],
source: "valuable\x0a\x09^ valuable",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementSubscription);

smalltalk.addMethod(
smalltalk.method({
selector: "valuable:",
protocol: 'accessing',
fn: function (aValuable){
var self=this;
self["@valuable"]=aValuable;
return self},
args: ["aValuable"],
source: "valuable: aValuable\x0a\x09valuable := aValuable",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementSubscription);



smalltalk.addClass('AnnouncementValuable', globals.Object, ['valuable', 'receiver'], 'Kernel-Announcements');
globals.AnnouncementValuable.comment="I wrap `valuable` objects (typically instances of `BlockClosure`) with a `receiver` to be able to unregister subscriptions based on a `receiver`.";
smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@receiver"];
return $1;
},
args: [],
source: "receiver\x0a\x09^ receiver",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementValuable);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@receiver"]=anObject;
return self},
args: ["anObject"],
source: "receiver: anObject\x0a\x09receiver := anObject",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementValuable);

smalltalk.addMethod(
smalltalk.method({
selector: "valuable",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@valuable"];
return $1;
},
args: [],
source: "valuable\x0a\x09^ valuable",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementValuable);

smalltalk.addMethod(
smalltalk.method({
selector: "valuable:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@valuable"]=anObject;
return self},
args: ["anObject"],
source: "valuable: anObject\x0a\x09valuable := anObject",
messageSends: [],
referencedClasses: []
}),
globals.AnnouncementValuable);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'evaluating',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._valuable())._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"value",{},globals.AnnouncementValuable)})},
args: [],
source: "value\x0a\x09^ self valuable value",
messageSends: ["value", "valuable"],
referencedClasses: []
}),
globals.AnnouncementValuable);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'evaluating',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._valuable())._value_(anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"value:",{anObject:anObject},globals.AnnouncementValuable)})},
args: ["anObject"],
source: "value: anObject\x0a\x09^ self valuable value: anObject",
messageSends: ["value:", "valuable"],
referencedClasses: []
}),
globals.AnnouncementValuable);



smalltalk.addClass('Announcer', globals.Object, ['registry', 'subscriptions'], 'Kernel-Announcements');
globals.Announcer.comment="I hold annoncement subscriptions (instances of `AnnouncementSubscription`) in a private registry.\x0aI announce (trigger) announces, which are then dispatched to all subscriptions.\x0a\x0aThe code is based on the announcements as [described by Vassili Bykov](http://www.cincomsmalltalk.com/userblogs/vbykov/blogView?searchCategory=Announcements%20Framework).\x0a\x0a## API\x0a\x0aUse `#announce:` to trigger an announcement.\x0a\x0aUse `#on:do:` or `#on:send:to:` to register subscriptions.\x0a\x0aWhen using `#on:send:to:`, unregistration can be done with `#unregister:`.\x0a\x0a## Usage example:\x0a\x0a    SystemAnnouncer current\x0a        on: ClassAdded\x0a        do: [ :ann | window alert: ann theClass name, ' added' ].";
smalltalk.addMethod(
smalltalk.method({
selector: "announce:",
protocol: 'announcing',
fn: function (anAnnouncement){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@subscriptions"])._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._deliver_(anAnnouncement);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"announce:",{anAnnouncement:anAnnouncement},globals.Announcer)})},
args: ["anAnnouncement"],
source: "announce: anAnnouncement\x0a\x09subscriptions do: [ :each |\x0a\x09\x09each deliver: anAnnouncement ]",
messageSends: ["do:", "deliver:"],
referencedClasses: []
}),
globals.Announcer);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
globals.Announcer.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@subscriptions"]=_st($OrderedCollection())._new();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.Announcer)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09subscriptions := OrderedCollection new",
messageSends: ["initialize", "new"],
referencedClasses: ["OrderedCollection"]
}),
globals.Announcer);

smalltalk.addMethod(
smalltalk.method({
selector: "on:do:",
protocol: 'subscribing',
fn: function (aClass,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._on_do_for_(aClass,aBlock,nil);
return self}, function($ctx1) {$ctx1.fill(self,"on:do:",{aClass:aClass,aBlock:aBlock},globals.Announcer)})},
args: ["aClass", "aBlock"],
source: "on: aClass do: aBlock\x0a\x09self on: aClass do: aBlock for: nil",
messageSends: ["on:do:for:"],
referencedClasses: []
}),
globals.Announcer);

smalltalk.addMethod(
smalltalk.method({
selector: "on:do:for:",
protocol: 'subscribing',
fn: function (aClass,aBlock,aReceiver){
var self=this;
function $AnnouncementSubscription(){return globals.AnnouncementSubscription||(typeof AnnouncementSubscription=="undefined"?nil:AnnouncementSubscription)}
function $AnnouncementValuable(){return globals.AnnouncementValuable||(typeof AnnouncementValuable=="undefined"?nil:AnnouncementValuable)}
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$4,$6,$7,$5,$8,$2;
$1=self["@subscriptions"];
$3=_st($AnnouncementSubscription())._new();
$ctx1.sendIdx["new"]=1;
$4=$3;
$6=_st($AnnouncementValuable())._new();
_st($6)._valuable_(aBlock);
_st($6)._receiver_(aReceiver);
$7=_st($6)._yourself();
$ctx1.sendIdx["yourself"]=1;
$5=$7;
_st($4)._valuable_($5);
$ctx1.sendIdx["valuable:"]=1;
_st($3)._announcementClass_(aClass);
$8=_st($3)._yourself();
$2=$8;
_st($1)._add_($2);
return self}, function($ctx1) {$ctx1.fill(self,"on:do:for:",{aClass:aClass,aBlock:aBlock,aReceiver:aReceiver},globals.Announcer)})},
args: ["aClass", "aBlock", "aReceiver"],
source: "on: aClass do: aBlock for: aReceiver\x0a\x09subscriptions add: (AnnouncementSubscription new\x0a\x09\x09valuable: (AnnouncementValuable new\x0a\x09\x09\x09valuable: aBlock;\x0a\x09\x09\x09receiver: aReceiver;\x0a\x09\x09\x09yourself);\x0a\x09\x09announcementClass: aClass;\x0a\x09\x09yourself)",
messageSends: ["add:", "valuable:", "new", "receiver:", "yourself", "announcementClass:"],
referencedClasses: ["AnnouncementSubscription", "AnnouncementValuable"]
}),
globals.Announcer);

smalltalk.addMethod(
smalltalk.method({
selector: "on:doOnce:",
protocol: 'subscribing',
fn: function (aClass,aBlock){
var self=this;
var subscription;
function $AnnouncementSubscription(){return globals.AnnouncementSubscription||(typeof AnnouncementSubscription=="undefined"?nil:AnnouncementSubscription)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st($AnnouncementSubscription())._new();
_st($1)._announcementClass_(aClass);
$2=_st($1)._yourself();
subscription=$2;
_st(subscription)._valuable_((function(ann){
return smalltalk.withContext(function($ctx2) {
_st(self["@subscriptions"])._remove_(subscription);
return _st(aBlock)._value_(ann);
}, function($ctx2) {$ctx2.fillBlock({ann:ann},$ctx1,1)})}));
_st(self["@subscriptions"])._add_(subscription);
return self}, function($ctx1) {$ctx1.fill(self,"on:doOnce:",{aClass:aClass,aBlock:aBlock,subscription:subscription},globals.Announcer)})},
args: ["aClass", "aBlock"],
source: "on: aClass doOnce: aBlock\x0a\x09| subscription |\x0a\x09\x0a\x09subscription := AnnouncementSubscription new\x0a\x09\x09announcementClass: aClass;\x0a\x09\x09yourself.\x0a\x09subscription valuable: [ :ann |\x0a\x09\x09subscriptions remove: subscription.\x0a\x09\x09aBlock value: ann ].\x0a\x0a\x09subscriptions add: subscription",
messageSends: ["announcementClass:", "new", "yourself", "valuable:", "remove:", "value:", "add:"],
referencedClasses: ["AnnouncementSubscription"]
}),
globals.Announcer);

smalltalk.addMethod(
smalltalk.method({
selector: "on:send:to:",
protocol: 'subscribing',
fn: function (aClass,aSelector,anObject){
var self=this;
function $AnnouncementSubscription(){return globals.AnnouncementSubscription||(typeof AnnouncementSubscription=="undefined"?nil:AnnouncementSubscription)}
function $MessageSend(){return globals.MessageSend||(typeof MessageSend=="undefined"?nil:MessageSend)}
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$4,$6,$7,$5,$8,$2;
$1=self["@subscriptions"];
$3=_st($AnnouncementSubscription())._new();
$ctx1.sendIdx["new"]=1;
$4=$3;
$6=_st($MessageSend())._new();
_st($6)._receiver_(anObject);
_st($6)._selector_(aSelector);
$7=_st($6)._yourself();
$ctx1.sendIdx["yourself"]=1;
$5=$7;
_st($4)._valuable_($5);
_st($3)._announcementClass_(aClass);
$8=_st($3)._yourself();
$2=$8;
_st($1)._add_($2);
return self}, function($ctx1) {$ctx1.fill(self,"on:send:to:",{aClass:aClass,aSelector:aSelector,anObject:anObject},globals.Announcer)})},
args: ["aClass", "aSelector", "anObject"],
source: "on: aClass send: aSelector to: anObject\x0a\x09subscriptions add: (AnnouncementSubscription new\x0a\x09\x09valuable: (MessageSend new\x0a\x09\x09\x09receiver: anObject;\x0a\x09\x09\x09selector: aSelector;\x0a\x09\x09\x09yourself);\x0a\x09\x09announcementClass: aClass;\x0a\x09\x09yourself)",
messageSends: ["add:", "valuable:", "new", "receiver:", "selector:", "yourself", "announcementClass:"],
referencedClasses: ["AnnouncementSubscription", "MessageSend"]
}),
globals.Announcer);

smalltalk.addMethod(
smalltalk.method({
selector: "unsubscribe:",
protocol: 'subscribing',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self["@subscriptions"]=_st(self["@subscriptions"])._reject_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(each)._receiver()).__eq(anObject);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"unsubscribe:",{anObject:anObject},globals.Announcer)})},
args: ["anObject"],
source: "unsubscribe: anObject\x0a\x09subscriptions := subscriptions reject: [ :each |\x0a\x09\x09each receiver = anObject ]",
messageSends: ["reject:", "=", "receiver"],
referencedClasses: []
}),
globals.Announcer);



smalltalk.addClass('SystemAnnouncer', globals.Announcer, [], 'Kernel-Announcements');
globals.SystemAnnouncer.comment="My unique instance is the global announcer handling all Amber system-related announces.\x0a\x0a## API\x0a\x0aAccess to the unique instance is done via `#current`";

globals.SystemAnnouncer.klass.iVarNames = ['current'];
smalltalk.addMethod(
smalltalk.method({
selector: "current",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@current"];
if(($receiver = $2) == nil || $receiver == null){
self["@current"]=globals.SystemAnnouncer.klass.superclass.fn.prototype._new.apply(_st(self), []);
$1=self["@current"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"current",{},globals.SystemAnnouncer.klass)})},
args: [],
source: "current\x0a\x09^ current ifNil: [ current := super new ]",
messageSends: ["ifNil:", "new"],
referencedClasses: []
}),
globals.SystemAnnouncer.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "new",
protocol: 'instance creation',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._shouldNotImplement();
return self}, function($ctx1) {$ctx1.fill(self,"new",{},globals.SystemAnnouncer.klass)})},
args: [],
source: "new\x0a\x09self shouldNotImplement",
messageSends: ["shouldNotImplement"],
referencedClasses: []
}),
globals.SystemAnnouncer.klass);


smalltalk.addClass('SystemAnnouncement', globals.Object, [], 'Kernel-Announcements');
globals.SystemAnnouncement.comment="I am the superclass of all system announcements";

smalltalk.addMethod(
smalltalk.method({
selector: "heliosClass",
protocol: 'helios',
fn: function (){
var self=this;
return "announcement";
},
args: [],
source: "heliosClass\x0a\x09^ 'announcement'",
messageSends: [],
referencedClasses: []
}),
globals.SystemAnnouncement.klass);


smalltalk.addClass('ClassAnnouncement', globals.SystemAnnouncement, ['theClass'], 'Kernel-Announcements');
globals.ClassAnnouncement.comment="I am the abstract superclass of class-related announcements.";
smalltalk.addMethod(
smalltalk.method({
selector: "theClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@theClass"];
return $1;
},
args: [],
source: "theClass\x0a\x09^ theClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassAnnouncement);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@theClass"]=aClass;
return self},
args: ["aClass"],
source: "theClass: aClass\x0a\x09theClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassAnnouncement);



smalltalk.addClass('ClassAdded', globals.ClassAnnouncement, [], 'Kernel-Announcements');
globals.ClassAdded.comment="I am emitted when a class is added to the system.\x0aSee ClassBuilder >> #addSubclassOf:... methods";


smalltalk.addClass('ClassCommentChanged', globals.ClassAnnouncement, [], 'Kernel-Announcements');
globals.ClassCommentChanged.comment="I am emitted when the comment of a class changes. (Behavior >> #comment)";


smalltalk.addClass('ClassDefinitionChanged', globals.ClassAnnouncement, [], 'Kernel-Announcements');
globals.ClassDefinitionChanged.comment="I am emitted when the definition of a class changes.\x0aSee ClassBuilder >> #class:instanceVariableNames:";


smalltalk.addClass('ClassMigrated', globals.ClassAnnouncement, ['oldClass'], 'Kernel-Announcements');
globals.ClassMigrated.comment="I am emitted when a class is migrated.";
smalltalk.addMethod(
smalltalk.method({
selector: "oldClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@oldClass"];
return $1;
},
args: [],
source: "oldClass\x0a\x09^ oldClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassMigrated);

smalltalk.addMethod(
smalltalk.method({
selector: "oldClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@oldClass"]=aClass;
return self},
args: ["aClass"],
source: "oldClass: aClass\x0a\x09oldClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.ClassMigrated);



smalltalk.addClass('ClassMoved', globals.ClassAnnouncement, ['oldPackage'], 'Kernel-Announcements');
globals.ClassMoved.comment="I am emitted when a class is moved from one package to another.";
smalltalk.addMethod(
smalltalk.method({
selector: "oldPackage",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@oldPackage"];
return $1;
},
args: [],
source: "oldPackage\x0a\x09^ oldPackage",
messageSends: [],
referencedClasses: []
}),
globals.ClassMoved);

smalltalk.addMethod(
smalltalk.method({
selector: "oldPackage:",
protocol: 'accessing',
fn: function (aPackage){
var self=this;
self["@oldPackage"]=aPackage;
return self},
args: ["aPackage"],
source: "oldPackage: aPackage\x0a\x09oldPackage := aPackage",
messageSends: [],
referencedClasses: []
}),
globals.ClassMoved);



smalltalk.addClass('ClassRemoved', globals.ClassAnnouncement, [], 'Kernel-Announcements');
globals.ClassRemoved.comment="I am emitted when a class is removed.\x0aSee Smalltalk >> #removeClass:";


smalltalk.addClass('ClassRenamed', globals.ClassAnnouncement, [], 'Kernel-Announcements');
globals.ClassRenamed.comment="I am emitted when a class is renamed.\x0aSee ClassBuilder >> #renameClass:to:";


smalltalk.addClass('MethodAnnouncement', globals.SystemAnnouncement, ['method'], 'Kernel-Announcements');
globals.MethodAnnouncement.comment="I am the abstract superclass of method-related announcements.";
smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@method"];
return $1;
},
args: [],
source: "method\x0a\x09^ method",
messageSends: [],
referencedClasses: []
}),
globals.MethodAnnouncement);

smalltalk.addMethod(
smalltalk.method({
selector: "method:",
protocol: 'accessing',
fn: function (aCompiledMethod){
var self=this;
self["@method"]=aCompiledMethod;
return self},
args: ["aCompiledMethod"],
source: "method: aCompiledMethod\x0a\x09method := aCompiledMethod",
messageSends: [],
referencedClasses: []
}),
globals.MethodAnnouncement);



smalltalk.addClass('MethodAdded', globals.MethodAnnouncement, [], 'Kernel-Announcements');
globals.MethodAdded.comment="I am emitted when a `CompiledMethod` is added to a class.";


smalltalk.addClass('MethodModified', globals.MethodAnnouncement, ['oldMethod'], 'Kernel-Announcements');
globals.MethodModified.comment="I am emitted when a `CompiledMethod` is modified (a new method is installed). I hold a reference to the old method being replaced.";
smalltalk.addMethod(
smalltalk.method({
selector: "oldMethod",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@oldMethod"];
return $1;
},
args: [],
source: "oldMethod\x0a\x09^ oldMethod",
messageSends: [],
referencedClasses: []
}),
globals.MethodModified);

smalltalk.addMethod(
smalltalk.method({
selector: "oldMethod:",
protocol: 'accessing',
fn: function (aMethod){
var self=this;
self["@oldMethod"]=aMethod;
return self},
args: ["aMethod"],
source: "oldMethod: aMethod\x0a\x09oldMethod := aMethod",
messageSends: [],
referencedClasses: []
}),
globals.MethodModified);



smalltalk.addClass('MethodMoved', globals.MethodAnnouncement, ['oldProtocol'], 'Kernel-Announcements');
globals.MethodMoved.comment="I am emitted when a `CompiledMethod` is moved to another protocol. I hold a refernce to the old protocol of the method.";
smalltalk.addMethod(
smalltalk.method({
selector: "oldProtocol",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@oldProtocol"];
return $1;
},
args: [],
source: "oldProtocol\x0a\x09^ oldProtocol",
messageSends: [],
referencedClasses: []
}),
globals.MethodMoved);

smalltalk.addMethod(
smalltalk.method({
selector: "oldProtocol:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@oldProtocol"]=aString;
return self},
args: ["aString"],
source: "oldProtocol: aString\x0a\x09oldProtocol := aString",
messageSends: [],
referencedClasses: []
}),
globals.MethodMoved);



smalltalk.addClass('MethodRemoved', globals.MethodAnnouncement, [], 'Kernel-Announcements');
globals.MethodRemoved.comment="I am emitted when a `CompiledMethod` is removed from a class.";


smalltalk.addClass('PackageAnnouncement', globals.SystemAnnouncement, ['package'], 'Kernel-Announcements');
globals.PackageAnnouncement.comment="I am the abstract superclass of package-related announcements.";
smalltalk.addMethod(
smalltalk.method({
selector: "package",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@package"];
return $1;
},
args: [],
source: "package\x0a\x09^ package",
messageSends: [],
referencedClasses: []
}),
globals.PackageAnnouncement);

smalltalk.addMethod(
smalltalk.method({
selector: "package:",
protocol: 'accessing',
fn: function (aPackage){
var self=this;
self["@package"]=aPackage;
return self},
args: ["aPackage"],
source: "package: aPackage\x0a\x09package := aPackage",
messageSends: [],
referencedClasses: []
}),
globals.PackageAnnouncement);



smalltalk.addClass('PackageAdded', globals.PackageAnnouncement, [], 'Kernel-Announcements');
globals.PackageAdded.comment="I am emitted when a `Package` is added to the system.";


smalltalk.addClass('PackageRemoved', globals.PackageAnnouncement, [], 'Kernel-Announcements');
globals.PackageRemoved.comment="I am emitted when a `Package` is removed from the system.";


smalltalk.addClass('ProtocolAnnouncement', globals.SystemAnnouncement, ['theClass', 'protocol'], 'Kernel-Announcements');
globals.ProtocolAnnouncement.comment="I am the abstract superclass of protocol-related announcements.";
smalltalk.addMethod(
smalltalk.method({
selector: "protocol",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@protocol"];
return $1;
},
args: [],
source: "protocol\x0a\x09^ protocol",
messageSends: [],
referencedClasses: []
}),
globals.ProtocolAnnouncement);

smalltalk.addMethod(
smalltalk.method({
selector: "protocol:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@protocol"]=aString;
return self},
args: ["aString"],
source: "protocol: aString\x0a\x09protocol := aString",
messageSends: [],
referencedClasses: []
}),
globals.ProtocolAnnouncement);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@theClass"];
return $1;
},
args: [],
source: "theClass\x0a\x09^ theClass",
messageSends: [],
referencedClasses: []
}),
globals.ProtocolAnnouncement);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@theClass"]=aClass;
return self},
args: ["aClass"],
source: "theClass: aClass\x0a\x09theClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.ProtocolAnnouncement);



smalltalk.addClass('ProtocolAdded', globals.ProtocolAnnouncement, [], 'Kernel-Announcements');
globals.ProtocolAdded.comment="I am emitted when a protocol is added to a class.";


smalltalk.addClass('ProtocolRemoved', globals.ProtocolAnnouncement, [], 'Kernel-Announcements');
globals.ProtocolRemoved.comment="I am emitted when a protocol is removed from a class.";

});

define("amber_core/Compiler-Exceptions", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Exceptions", "amber_core/Kernel-Objects"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Compiler-Exceptions');
smalltalk.packages["Compiler-Exceptions"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('CompilerError', globals.Error, [], 'Compiler-Exceptions');
globals.CompilerError.comment="I am the common superclass of all compiling errors.";


smalltalk.addClass('ParseError', globals.CompilerError, [], 'Compiler-Exceptions');
globals.ParseError.comment="Instance of ParseError are signaled on any parsing error.\x0aSee `Smalltalk >> #parse:`";


smalltalk.addClass('SemanticError', globals.CompilerError, [], 'Compiler-Exceptions');
globals.SemanticError.comment="I represent an abstract semantic error thrown by the SemanticAnalyzer.\x0aSemantic errors can be unknown variable errors, etc.\x0aSee my subclasses for concrete errors.\x0a\x0aThe IDE should catch instances of Semantic error to deal with them when compiling";


smalltalk.addClass('InliningError', globals.SemanticError, [], 'Compiler-Exceptions');
globals.InliningError.comment="Instances of InliningError are signaled when using an `InliningCodeGenerator`in a `Compiler`.";


smalltalk.addClass('InvalidAssignmentError', globals.SemanticError, ['variableName'], 'Compiler-Exceptions');
globals.InvalidAssignmentError.comment="I get signaled when a pseudo variable gets assigned.";
smalltalk.addMethod(
smalltalk.method({
selector: "messageText",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=" Invalid assignment to variable: ".__comma(self._variableName());
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageText",{},globals.InvalidAssignmentError)})},
args: [],
source: "messageText\x0a\x09^ ' Invalid assignment to variable: ', self variableName",
messageSends: [",", "variableName"],
referencedClasses: []
}),
globals.InvalidAssignmentError);

smalltalk.addMethod(
smalltalk.method({
selector: "variableName",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@variableName"];
return $1;
},
args: [],
source: "variableName\x0a\x09^ variableName",
messageSends: [],
referencedClasses: []
}),
globals.InvalidAssignmentError);

smalltalk.addMethod(
smalltalk.method({
selector: "variableName:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@variableName"]=aString;
return self},
args: ["aString"],
source: "variableName: aString\x0a\x09variableName := aString",
messageSends: [],
referencedClasses: []
}),
globals.InvalidAssignmentError);



smalltalk.addClass('ShadowingVariableError', globals.SemanticError, ['variableName'], 'Compiler-Exceptions');
globals.ShadowingVariableError.comment="I get signaled when a variable in a block or method scope shadows a variable of the same name in an outer scope.";
smalltalk.addMethod(
smalltalk.method({
selector: "messageText",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st("Variable shadowing error: ".__comma(self._variableName())).__comma(" is already defined");
$ctx1.sendIdx[","]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageText",{},globals.ShadowingVariableError)})},
args: [],
source: "messageText\x0a\x09^ 'Variable shadowing error: ', self variableName, ' is already defined'",
messageSends: [",", "variableName"],
referencedClasses: []
}),
globals.ShadowingVariableError);

smalltalk.addMethod(
smalltalk.method({
selector: "variableName",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@variableName"];
return $1;
},
args: [],
source: "variableName\x0a\x09^ variableName",
messageSends: [],
referencedClasses: []
}),
globals.ShadowingVariableError);

smalltalk.addMethod(
smalltalk.method({
selector: "variableName:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@variableName"]=aString;
return self},
args: ["aString"],
source: "variableName: aString\x0a\x09variableName := aString",
messageSends: [],
referencedClasses: []
}),
globals.ShadowingVariableError);



smalltalk.addClass('UnknownVariableError', globals.SemanticError, ['variableName'], 'Compiler-Exceptions');
globals.UnknownVariableError.comment="I get signaled when a variable is not defined.\x0aThe default behavior is to allow it, as this is how Amber currently is able to seamlessly send messages to JavaScript objects.";
smalltalk.addMethod(
smalltalk.method({
selector: "messageText",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st("Unknown Variable error: ".__comma(self._variableName())).__comma(" is not defined");
$ctx1.sendIdx[","]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageText",{},globals.UnknownVariableError)})},
args: [],
source: "messageText\x0a\x09^ 'Unknown Variable error: ', self variableName, ' is not defined'",
messageSends: [",", "variableName"],
referencedClasses: []
}),
globals.UnknownVariableError);

smalltalk.addMethod(
smalltalk.method({
selector: "variableName",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@variableName"];
return $1;
},
args: [],
source: "variableName\x0a\x09^ variableName",
messageSends: [],
referencedClasses: []
}),
globals.UnknownVariableError);

smalltalk.addMethod(
smalltalk.method({
selector: "variableName:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@variableName"]=aString;
return self},
args: ["aString"],
source: "variableName: aString\x0a\x09variableName := aString",
messageSends: [],
referencedClasses: []
}),
globals.UnknownVariableError);



smalltalk.addClass('RethrowErrorHandler', globals.Object, [], 'Compiler-Exceptions');
globals.RethrowErrorHandler.comment="This class is used in the commandline version of the compiler.\x0aIt uses the handleError: message of ErrorHandler for printing the stacktrace and throws the error again as JS exception.\x0aAs a result Smalltalk errors are not swallowd by the Amber runtime and compilation can be aborted.";
smalltalk.addMethod(
smalltalk.method({
selector: "basicSignal:",
protocol: 'error handling',
fn: function (anError){
var self=this;
return smalltalk.withContext(function($ctx1) { 
throw anError;
return self}, function($ctx1) {$ctx1.fill(self,"basicSignal:",{anError:anError},globals.RethrowErrorHandler)})},
args: ["anError"],
source: "basicSignal: anError\x0a        <throw anError>",
messageSends: [],
referencedClasses: []
}),
globals.RethrowErrorHandler);

smalltalk.addMethod(
smalltalk.method({
selector: "handleError:",
protocol: 'error handling',
fn: function (anError){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.RethrowErrorHandler.superclass.fn.prototype._handleError_.apply(_st(self), [anError]);
self._basicSignal_(anError);
return self}, function($ctx1) {$ctx1.fill(self,"handleError:",{anError:anError},globals.RethrowErrorHandler)})},
args: ["anError"],
source: "handleError: anError\x0a        super handleError: anError.\x0a        self basicSignal: anError",
messageSends: ["handleError:", "basicSignal:"],
referencedClasses: []
}),
globals.RethrowErrorHandler);


});

define("amber_core/Compiler-Core", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects", "amber_core/Kernel-Collections"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Compiler-Core');
smalltalk.packages["Compiler-Core"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('AbstractCodeGenerator', globals.Object, ['currentClass', 'source'], 'Compiler-Core');
globals.AbstractCodeGenerator.comment="I am the abstract super class of all code generators and provide their common API.";
smalltalk.addMethod(
smalltalk.method({
selector: "classNameFor:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$4,$1;
$2=_st(aClass)._isMetaclass();
if(smalltalk.assert($2)){
$3=_st(_st(aClass)._instanceClass())._name();
$ctx1.sendIdx["name"]=1;
$1=_st($3).__comma(".klass");
} else {
$4=_st(aClass)._isNil();
if(smalltalk.assert($4)){
$1="nil";
} else {
$1=_st(aClass)._name();
};
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"classNameFor:",{aClass:aClass},globals.AbstractCodeGenerator)})},
args: ["aClass"],
source: "classNameFor: aClass\x0a\x09^ aClass isMetaclass\x0a\x09\x09ifTrue: [ aClass instanceClass name, '.klass' ]\x0a\x09\x09ifFalse: [\x0a\x09\x09aClass isNil\x0a\x09\x09\x09ifTrue: [ 'nil' ]\x0a\x09\x09\x09ifFalse: [ aClass name ]]",
messageSends: ["ifTrue:ifFalse:", "isMetaclass", ",", "name", "instanceClass", "isNil"],
referencedClasses: []
}),
globals.AbstractCodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "compileNode:",
protocol: 'compiling',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._subclassResponsibility();
return self}, function($ctx1) {$ctx1.fill(self,"compileNode:",{aNode:aNode},globals.AbstractCodeGenerator)})},
args: ["aNode"],
source: "compileNode: aNode\x0a\x09self subclassResponsibility",
messageSends: ["subclassResponsibility"],
referencedClasses: []
}),
globals.AbstractCodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "currentClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@currentClass"];
return $1;
},
args: [],
source: "currentClass\x0a\x09^ currentClass",
messageSends: [],
referencedClasses: []
}),
globals.AbstractCodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "currentClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@currentClass"]=aClass;
return self},
args: ["aClass"],
source: "currentClass: aClass\x0a\x09currentClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.AbstractCodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "pseudoVariables",
protocol: 'accessing',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Smalltalk())._pseudoVariableNames();
return $1;
}, function($ctx1) {$ctx1.fill(self,"pseudoVariables",{},globals.AbstractCodeGenerator)})},
args: [],
source: "pseudoVariables\x0a\x09^ Smalltalk pseudoVariableNames",
messageSends: ["pseudoVariableNames"],
referencedClasses: ["Smalltalk"]
}),
globals.AbstractCodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@source"];
if(($receiver = $2) == nil || $receiver == null){
$1="";
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"source",{},globals.AbstractCodeGenerator)})},
args: [],
source: "source\x0a\x09^ source ifNil: [ '' ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.AbstractCodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.AbstractCodeGenerator);



smalltalk.addClass('CodeGenerator', globals.AbstractCodeGenerator, [], 'Compiler-Core');
globals.CodeGenerator.comment="I am a basic code generator. I generate a valid JavaScript output, but no not perform any inlining.\x0aSee `InliningCodeGenerator` for an optimized JavaScript code generation.";
smalltalk.addMethod(
smalltalk.method({
selector: "compileNode:",
protocol: 'compiling',
fn: function (aNode){
var self=this;
var ir,stream;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
_st(self._semanticAnalyzer())._visit_(aNode);
$ctx1.sendIdx["visit:"]=1;
ir=_st(self._translator())._visit_(aNode);
$ctx1.sendIdx["visit:"]=2;
$2=self._irTranslator();
_st($2)._currentClass_(self._currentClass());
_st($2)._visit_(ir);
$3=_st($2)._contents();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"compileNode:",{aNode:aNode,ir:ir,stream:stream},globals.CodeGenerator)})},
args: ["aNode"],
source: "compileNode: aNode\x0a\x09| ir stream |\x0a\x09self semanticAnalyzer visit: aNode.\x0a\x09ir := self translator visit: aNode.\x0a\x09^ self irTranslator\x0a\x09\x09currentClass: self currentClass;\x0a\x09\x09visit: ir;\x0a\x09\x09contents",
messageSends: ["visit:", "semanticAnalyzer", "translator", "currentClass:", "irTranslator", "currentClass", "contents"],
referencedClasses: []
}),
globals.CodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "irTranslator",
protocol: 'compiling',
fn: function (){
var self=this;
function $IRJSTranslator(){return globals.IRJSTranslator||(typeof IRJSTranslator=="undefined"?nil:IRJSTranslator)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($IRJSTranslator())._new();
return $1;
}, function($ctx1) {$ctx1.fill(self,"irTranslator",{},globals.CodeGenerator)})},
args: [],
source: "irTranslator\x0a\x09^ IRJSTranslator new",
messageSends: ["new"],
referencedClasses: ["IRJSTranslator"]
}),
globals.CodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "semanticAnalyzer",
protocol: 'compiling',
fn: function (){
var self=this;
function $SemanticAnalyzer(){return globals.SemanticAnalyzer||(typeof SemanticAnalyzer=="undefined"?nil:SemanticAnalyzer)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($SemanticAnalyzer())._on_(self._currentClass());
return $1;
}, function($ctx1) {$ctx1.fill(self,"semanticAnalyzer",{},globals.CodeGenerator)})},
args: [],
source: "semanticAnalyzer\x0a\x09^ SemanticAnalyzer on: self currentClass",
messageSends: ["on:", "currentClass"],
referencedClasses: ["SemanticAnalyzer"]
}),
globals.CodeGenerator);

smalltalk.addMethod(
smalltalk.method({
selector: "translator",
protocol: 'compiling',
fn: function (){
var self=this;
function $IRASTTranslator(){return globals.IRASTTranslator||(typeof IRASTTranslator=="undefined"?nil:IRASTTranslator)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRASTTranslator())._new();
_st($2)._source_(self._source());
_st($2)._theClass_(self._currentClass());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"translator",{},globals.CodeGenerator)})},
args: [],
source: "translator\x0a\x09^ IRASTTranslator new\x0a\x09\x09source: self source;\x0a\x09\x09theClass: self currentClass;\x0a\x09\x09yourself",
messageSends: ["source:", "new", "source", "theClass:", "currentClass", "yourself"],
referencedClasses: ["IRASTTranslator"]
}),
globals.CodeGenerator);



smalltalk.addClass('Compiler', globals.Object, ['currentClass', 'source', 'unknownVariables', 'codeGeneratorClass'], 'Compiler-Core');
globals.Compiler.comment="I provide the public interface for compiling Amber source code into JavaScript.\x0a\x0aThe code generator used to produce JavaScript can be plugged with `#codeGeneratorClass`.\x0aThe default code generator is an instance of `InlinedCodeGenerator`";
smalltalk.addMethod(
smalltalk.method({
selector: "codeGeneratorClass",
protocol: 'accessing',
fn: function (){
var self=this;
function $InliningCodeGenerator(){return globals.InliningCodeGenerator||(typeof InliningCodeGenerator=="undefined"?nil:InliningCodeGenerator)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@codeGeneratorClass"];
if(($receiver = $2) == nil || $receiver == null){
$1=$InliningCodeGenerator();
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"codeGeneratorClass",{},globals.Compiler)})},
args: [],
source: "codeGeneratorClass\x0a\x09^ codeGeneratorClass ifNil: [ InliningCodeGenerator ]",
messageSends: ["ifNil:"],
referencedClasses: ["InliningCodeGenerator"]
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "codeGeneratorClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@codeGeneratorClass"]=aClass;
return self},
args: ["aClass"],
source: "codeGeneratorClass: aClass\x0a\x09codeGeneratorClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "compile:",
protocol: 'compiling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._compileNode_(self._parse_(aString));
return $1;
}, function($ctx1) {$ctx1.fill(self,"compile:",{aString:aString},globals.Compiler)})},
args: ["aString"],
source: "compile: aString\x0a\x09^ self compileNode: (self parse: aString)",
messageSends: ["compileNode:", "parse:"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "compile:forClass:",
protocol: 'compiling',
fn: function (aString,aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._currentClass_(aClass);
self._source_(aString);
$1=self._compile_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"compile:forClass:",{aString:aString,aClass:aClass},globals.Compiler)})},
args: ["aString", "aClass"],
source: "compile: aString forClass: aClass\x0a\x09self currentClass: aClass.\x0a\x09self source: aString.\x0a\x09^ self compile: aString",
messageSends: ["currentClass:", "source:", "compile:"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "compileExpression:",
protocol: 'compiling',
fn: function (aString){
var self=this;
function $DoIt(){return globals.DoIt||(typeof DoIt=="undefined"?nil:DoIt)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._currentClass_($DoIt());
$1=_st("doIt ^ [ ".__comma(aString)).__comma(" ] value");
$ctx1.sendIdx[","]=1;
self._source_($1);
$2=self._compileNode_(self._parse_(self._source()));
return $2;
}, function($ctx1) {$ctx1.fill(self,"compileExpression:",{aString:aString},globals.Compiler)})},
args: ["aString"],
source: "compileExpression: aString\x0a\x09self currentClass: DoIt.\x0a\x09self source: 'doIt ^ [ ', aString, ' ] value'.\x0a\x09^ self compileNode: (self parse: self source)",
messageSends: ["currentClass:", "source:", ",", "compileNode:", "parse:", "source"],
referencedClasses: ["DoIt"]
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "compileExpression:on:",
protocol: 'compiling',
fn: function (aString,anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
self._currentClass_(_st(anObject)._class());
$1=_st("xxxDoIt ^ [ ".__comma(aString)).__comma(" ] value");
$ctx1.sendIdx[","]=1;
self._source_($1);
$2=self._compileNode_(self._parse_(self._source()));
return $2;
}, function($ctx1) {$ctx1.fill(self,"compileExpression:on:",{aString:aString,anObject:anObject},globals.Compiler)})},
args: ["aString", "anObject"],
source: "compileExpression: aString on: anObject\x0a\x09self currentClass: anObject class.\x0a\x09self source: 'xxxDoIt ^ [ ', aString, ' ] value'.\x0a\x09^ self compileNode: (self parse: self source)",
messageSends: ["currentClass:", "class", "source:", ",", "compileNode:", "parse:", "source"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "compileNode:",
protocol: 'compiling',
fn: function (aNode){
var self=this;
var generator,result;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
generator=_st(self._codeGeneratorClass())._new();
$1=generator;
_st($1)._source_(self._source());
$2=_st($1)._currentClass_(self._currentClass());
result=_st(generator)._compileNode_(aNode);
self._unknownVariables_([]);
$3=result;
return $3;
}, function($ctx1) {$ctx1.fill(self,"compileNode:",{aNode:aNode,generator:generator,result:result},globals.Compiler)})},
args: ["aNode"],
source: "compileNode: aNode\x0a\x09| generator result |\x0a\x09generator := self codeGeneratorClass new.\x0a\x09generator\x0a\x09\x09source: self source;\x0a\x09\x09currentClass: self currentClass.\x0a\x09result := generator compileNode: aNode.\x0a\x09self unknownVariables: #().\x0a\x09^ result",
messageSends: ["new", "codeGeneratorClass", "source:", "source", "currentClass:", "currentClass", "compileNode:", "unknownVariables:"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "currentClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@currentClass"];
return $1;
},
args: [],
source: "currentClass\x0a\x09^ currentClass",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "currentClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@currentClass"]=aClass;
return self},
args: ["aClass"],
source: "currentClass: aClass\x0a\x09currentClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "eval:",
protocol: 'compiling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
return eval(aString);
return self}, function($ctx1) {$ctx1.fill(self,"eval:",{aString:aString},globals.Compiler)})},
args: ["aString"],
source: "eval: aString\x0a\x09<return eval(aString)>",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "evaluateExpression:",
protocol: 'compiling',
fn: function (aString){
var self=this;
function $DoIt(){return globals.DoIt||(typeof DoIt=="undefined"?nil:DoIt)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._evaluateExpression_on_(aString,_st($DoIt())._new());
return $1;
}, function($ctx1) {$ctx1.fill(self,"evaluateExpression:",{aString:aString},globals.Compiler)})},
args: ["aString"],
source: "evaluateExpression: aString\x0a\x09\x22Unlike #eval: evaluate a Smalltalk expression and answer the returned object\x22\x0a\x09^ self evaluateExpression: aString on: DoIt new",
messageSends: ["evaluateExpression:on:", "new"],
referencedClasses: ["DoIt"]
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "evaluateExpression:on:",
protocol: 'compiling',
fn: function (aString,anObject){
var self=this;
var result,method;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
method=self._eval_(self._compileExpression_on_(aString,anObject));
_st(method)._protocol_("xxxDoIt");
$1=_st(anObject)._class();
$ctx1.sendIdx["class"]=1;
_st($1)._addCompiledMethod_(method);
result=_st(anObject)._xxxDoIt();
_st(_st(anObject)._class())._removeCompiledMethod_(method);
$2=result;
return $2;
}, function($ctx1) {$ctx1.fill(self,"evaluateExpression:on:",{aString:aString,anObject:anObject,result:result,method:method},globals.Compiler)})},
args: ["aString", "anObject"],
source: "evaluateExpression: aString on: anObject\x0a\x09\x22Unlike #eval: evaluate a Smalltalk expression with anObject as the receiver and answer the returned object\x22\x0a\x09| result method |\x0a\x09method := self eval: (self compileExpression: aString on: anObject).\x0a\x09method protocol: 'xxxDoIt'.\x0a\x09anObject class addCompiledMethod: method.\x0a\x09result := anObject xxxDoIt.\x0a\x09anObject class removeCompiledMethod: method.\x0a\x09^ result",
messageSends: ["eval:", "compileExpression:on:", "protocol:", "addCompiledMethod:", "class", "xxxDoIt", "removeCompiledMethod:"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "install:forClass:protocol:",
protocol: 'compiling',
fn: function (aString,aBehavior,anotherString){
var self=this;
function $ClassBuilder(){return globals.ClassBuilder||(typeof ClassBuilder=="undefined"?nil:ClassBuilder)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st($ClassBuilder())._new())._installMethod_forClass_protocol_(self._eval_(self._compile_forClass_(aString,aBehavior)),aBehavior,anotherString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"install:forClass:protocol:",{aString:aString,aBehavior:aBehavior,anotherString:anotherString},globals.Compiler)})},
args: ["aString", "aBehavior", "anotherString"],
source: "install: aString forClass: aBehavior protocol: anotherString\x0a\x09^ ClassBuilder new\x0a\x09\x09installMethod: (self eval: (self compile: aString forClass: aBehavior))\x0a\x09\x09forClass: aBehavior\x0a\x09\x09protocol: anotherString",
messageSends: ["installMethod:forClass:protocol:", "new", "eval:", "compile:forClass:"],
referencedClasses: ["ClassBuilder"]
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "parse:",
protocol: 'compiling',
fn: function (aString){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Smalltalk())._parse_(aString);
return $1;
}, function($ctx1) {$ctx1.fill(self,"parse:",{aString:aString},globals.Compiler)})},
args: ["aString"],
source: "parse: aString\x0a\x09^ Smalltalk parse: aString",
messageSends: ["parse:"],
referencedClasses: ["Smalltalk"]
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "parseExpression:",
protocol: 'compiling',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st("doIt ^ [ ".__comma(aString)).__comma(" ] value");
$ctx1.sendIdx[","]=1;
$1=self._parse_($2);
return $1;
}, function($ctx1) {$ctx1.fill(self,"parseExpression:",{aString:aString},globals.Compiler)})},
args: ["aString"],
source: "parseExpression: aString\x0a\x09^ self parse: 'doIt ^ [ ', aString, ' ] value'",
messageSends: ["parse:", ","],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "recompile:",
protocol: 'compiling',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(_st(_st(aClass)._methodDictionary())._values())._do_displayingProgress_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._install_forClass_protocol_(_st(each)._source(),aClass,_st(each)._protocol());
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),"Recompiling ".__comma(_st(aClass)._name()));
$1=_st(aClass)._isMetaclass();
if(! smalltalk.assert($1)){
self._recompile_(_st(aClass)._class());
};
return self}, function($ctx1) {$ctx1.fill(self,"recompile:",{aClass:aClass},globals.Compiler)})},
args: ["aClass"],
source: "recompile: aClass\x0a\x09aClass methodDictionary values\x0a\x09\x09do: [ :each | \x0a\x09\x09\x09self \x0a\x09\x09\x09\x09install: each source \x0a\x09\x09\x09\x09forClass: aClass \x0a\x09\x09\x09\x09protocol: each protocol ]\x0a\x09\x09displayingProgress: 'Recompiling ', aClass name.\x0a\x09aClass isMetaclass ifFalse: [ self recompile: aClass class ]",
messageSends: ["do:displayingProgress:", "values", "methodDictionary", "install:forClass:protocol:", "source", "protocol", ",", "name", "ifFalse:", "isMetaclass", "recompile:", "class"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "recompileAll",
protocol: 'compiling',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
_st(_st($Smalltalk())._classes())._do_displayingProgress_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._recompile_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),"Compiling all classes...");
return self}, function($ctx1) {$ctx1.fill(self,"recompileAll",{},globals.Compiler)})},
args: [],
source: "recompileAll\x0a\x09Smalltalk classes \x0a\x09\x09do: [ :each | self recompile: each ]\x0a\x09\x09displayingProgress: 'Compiling all classes...'",
messageSends: ["do:displayingProgress:", "classes", "recompile:"],
referencedClasses: ["Smalltalk"]
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@source"];
if(($receiver = $2) == nil || $receiver == null){
$1="";
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"source",{},globals.Compiler)})},
args: [],
source: "source\x0a\x09^ source ifNil: [ '' ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "unknownVariables",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@unknownVariables"];
return $1;
},
args: [],
source: "unknownVariables\x0a\x09^ unknownVariables",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);

smalltalk.addMethod(
smalltalk.method({
selector: "unknownVariables:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@unknownVariables"]=aCollection;
return self},
args: ["aCollection"],
source: "unknownVariables: aCollection\x0a\x09unknownVariables := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.Compiler);


smalltalk.addMethod(
smalltalk.method({
selector: "recompile:",
protocol: 'compiling',
fn: function (aClass){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._new())._recompile_(aClass);
return self}, function($ctx1) {$ctx1.fill(self,"recompile:",{aClass:aClass},globals.Compiler.klass)})},
args: ["aClass"],
source: "recompile: aClass\x0a\x09self new recompile: aClass",
messageSends: ["recompile:", "new"],
referencedClasses: []
}),
globals.Compiler.klass);

smalltalk.addMethod(
smalltalk.method({
selector: "recompileAll",
protocol: 'compiling',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
_st(_st($Smalltalk())._classes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._recompile_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"recompileAll",{},globals.Compiler.klass)})},
args: [],
source: "recompileAll\x0a\x09Smalltalk classes do: [ :each |\x0a\x09\x09self recompile: each ]",
messageSends: ["do:", "classes", "recompile:"],
referencedClasses: ["Smalltalk"]
}),
globals.Compiler.klass);


smalltalk.addClass('DoIt', globals.Object, [], 'Compiler-Core');
globals.DoIt.comment="`DoIt` is the class used to compile and evaluate expressions. See `Compiler >> evaluateExpression:`.";


smalltalk.addClass('NodeVisitor', globals.Object, [], 'Compiler-Core');
globals.NodeVisitor.comment="I am the abstract super class of all AST node visitors.";
smalltalk.addMethod(
smalltalk.method({
selector: "visit:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aNode)._accept_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visit:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visit: aNode\x0a\x09^ aNode accept: self",
messageSends: ["accept:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitAll:",
protocol: 'visiting',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aCollection)._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitAll:",{aCollection:aCollection},globals.NodeVisitor)})},
args: ["aCollection"],
source: "visitAll: aCollection\x0a\x09^ aCollection collect: [ :each | self visit: each ]",
messageSends: ["collect:", "visit:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitAssignmentNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitAssignmentNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitAssignmentNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitBlockNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitBlockNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitBlockNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitBlockSequenceNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitSequenceNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitBlockSequenceNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitBlockSequenceNode: aNode\x0a\x09^ self visitSequenceNode: aNode",
messageSends: ["visitSequenceNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitCascadeNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitCascadeNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitCascadeNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitDynamicArrayNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitDynamicArrayNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitDynamicArrayNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitDynamicDictionaryNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitDynamicDictionaryNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitDynamicDictionaryNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitJSStatementNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitJSStatementNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitJSStatementNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitMethodNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitMethodNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitMethodNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitAll_(_st(aNode)._nodes());
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitNode: aNode\x0a\x09^ self visitAll: aNode nodes",
messageSends: ["visitAll:", "nodes"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitReturnNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitReturnNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitReturnNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitSendNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitSendNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitSendNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitSequenceNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitSequenceNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitSequenceNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitValueNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitValueNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitValueNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitVariableNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitNode_(aNode);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitVariableNode:",{aNode:aNode},globals.NodeVisitor)})},
args: ["aNode"],
source: "visitVariableNode: aNode\x0a\x09^ self visitNode: aNode",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.NodeVisitor);


smalltalk.addMethod(
smalltalk.method({
selector: "asVariableName",
protocol: '*Compiler-Core',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(_st($Smalltalk())._reservedWords())._includes_(self);
if(smalltalk.assert($2)){
$1=self.__comma("_");
} else {
$1=self;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"asVariableName",{},globals.String)})},
args: [],
source: "asVariableName\x0a\x09^ (Smalltalk reservedWords includes: self)\x0a\x09\x09ifTrue: [ self, '_' ]\x0a\x09\x09ifFalse: [ self ]",
messageSends: ["ifTrue:ifFalse:", "includes:", "reservedWords", ","],
referencedClasses: ["Smalltalk"]
}),
globals.String);

});

define("amber_core/Compiler-AST", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Kernel-Objects", "amber_core/Kernel-Methods"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Compiler-AST');
smalltalk.packages["Compiler-AST"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('Node', globals.Object, ['parent', 'position', 'source', 'nodes', 'shouldBeInlined', 'shouldBeAliased'], 'Compiler-AST');
globals.Node.comment="I am the abstract root class of the abstract syntax tree.\x0a\x0aConcrete classes should implement `#accept:` to allow visiting.\x0a\x0a`position` holds a point containing line and column number of the symbol location in the original source file.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.Node)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitNode: self",
messageSends: ["visitNode:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "addNode:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._nodes())._add_(aNode);
_st(aNode)._parent_(self);
return self}, function($ctx1) {$ctx1.fill(self,"addNode:",{aNode:aNode},globals.Node)})},
args: ["aNode"],
source: "addNode: aNode\x0a\x09self nodes add: aNode.\x0a\x09aNode parent: self",
messageSends: ["add:", "nodes", "parent:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "allNodes",
protocol: 'accessing',
fn: function (){
var self=this;
var allNodes;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self._nodes();
$ctx1.sendIdx["nodes"]=1;
allNodes=_st($1)._asSet();
_st(self._nodes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(allNodes)._addAll_(_st(each)._allNodes());
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$2=allNodes;
return $2;
}, function($ctx1) {$ctx1.fill(self,"allNodes",{allNodes:allNodes},globals.Node)})},
args: [],
source: "allNodes\x0a\x09| allNodes |\x0a\x09\x0a\x09allNodes := self nodes asSet.\x0a\x09self nodes do: [ :each | \x0a\x09\x09allNodes addAll: each allNodes ].\x0a\x09\x0a\x09^ allNodes",
messageSends: ["asSet", "nodes", "do:", "addAll:", "allNodes"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "inPosition:",
protocol: 'testing',
fn: function (aPoint){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._positionStart()).__lt_eq(aPoint))._and_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._positionEnd()).__gt_eq(aPoint);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"inPosition:",{aPoint:aPoint},globals.Node)})},
args: ["aPoint"],
source: "inPosition: aPoint\x0a\x09^ (self positionStart <= aPoint and: [\x0a\x09\x09self positionEnd >= aPoint ])",
messageSends: ["and:", "<=", "positionStart", ">=", "positionEnd"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isAssignmentNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isAssignmentNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isBlockNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockSequenceNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isBlockSequenceNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isCascadeNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isCascadeNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isImmutable\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isJSStatementNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isJSStatementNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isLastChild",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(_st(self._parent())._nodes())._last()).__eq(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"isLastChild",{},globals.Node)})},
args: [],
source: "isLastChild\x0a\x09^ self parent nodes last = self",
messageSends: ["=", "last", "nodes", "parent"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isNavigationNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isNavigationNode\x0a\x09\x22Answer true if the node can be navigated to\x22\x0a\x09\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isReferenced",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $4,$3,$2,$1;
$4=self._parent();
$ctx1.sendIdx["parent"]=1;
$3=_st($4)._isSequenceNode();
$2=_st($3)._or_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._parent())._isAssignmentNode();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$1=_st($2)._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isReferenced",{},globals.Node)})},
args: [],
source: "isReferenced\x0a\x09\x22Answer true if the receiver is referenced by other nodes.\x0a\x09Do not take sequences or assignments into account\x22\x0a\x09\x0a\x09^ (self parent isSequenceNode or: [\x0a\x09\x09self parent isAssignmentNode ]) not",
messageSends: ["not", "or:", "isSequenceNode", "parent", "isAssignmentNode"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isReturnNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isReturnNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isSendNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isSendNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isSequenceNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isSequenceNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isValueNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isValueNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "isVariableNode",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isVariableNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._parent();
if(($receiver = $2) == nil || $receiver == null){
$1=$2;
} else {
var node;
node=$receiver;
$1=_st(node)._method();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"method",{},globals.Node)})},
args: [],
source: "method\x0a\x09^ self parent ifNotNil: [ :node | node method ]",
messageSends: ["ifNotNil:", "parent", "method"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "navigationNodeAt:ifAbsent:",
protocol: 'accessing',
fn: function (aPoint,aBlock){
var self=this;
var children;
return smalltalk.withContext(function($ctx1) { 
var $1,$4,$3,$2;
var $early={};
try {
children=_st(self._allNodes())._select_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(each)._isNavigationNode())._and_((function(){
return smalltalk.withContext(function($ctx3) {
return _st(each)._inPosition_(aPoint);
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
_st(children)._ifEmpty_((function(){
return smalltalk.withContext(function($ctx2) {
$1=_st(aBlock)._value();
throw $early=[$1];
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
$2=_st(_st(_st(children)._asArray())._sort_((function(a,b){
return smalltalk.withContext(function($ctx2) {
$4=_st(a)._positionStart();
$ctx2.sendIdx["positionStart"]=1;
$3=_st($4)._dist_(aPoint);
$ctx2.sendIdx["dist:"]=1;
return _st($3).__lt_eq(_st(_st(b)._positionStart())._dist_(aPoint));
}, function($ctx2) {$ctx2.fillBlock({a:a,b:b},$ctx1,4)})})))._first();
return $2;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"navigationNodeAt:ifAbsent:",{aPoint:aPoint,aBlock:aBlock,children:children},globals.Node)})},
args: ["aPoint", "aBlock"],
source: "navigationNodeAt: aPoint ifAbsent: aBlock\x0a\x09\x22Answer the navigation node in the receiver's tree at aPoint \x0a\x09or nil if no navigation node was found.\x0a\x09\x0a\x09See `node >> isNaviationNode`\x22\x0a\x09\x0a\x09| children |\x0a\x09\x0a\x09children := self allNodes select: [ :each | \x0a\x09\x09each isNavigationNode and: [ each inPosition: aPoint ] ].\x0a\x09\x0a\x09children ifEmpty: [ ^ aBlock value ].\x0a\x09\x0a\x09^ (children asArray sort: [ :a :b | \x0a\x09\x09(a positionStart dist: aPoint) <= \x0a\x09\x09(b positionStart dist: aPoint) ]) first",
messageSends: ["select:", "allNodes", "and:", "isNavigationNode", "inPosition:", "ifEmpty:", "value", "first", "sort:", "asArray", "<=", "dist:", "positionStart"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "nextChild",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1;
$3=self._nodes();
$ctx1.sendIdx["nodes"]=1;
$2=_st($3)._isEmpty();
if(smalltalk.assert($2)){
$1=self;
} else {
$1=_st(_st(self._nodes())._first())._nextChild();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"nextChild",{},globals.Node)})},
args: [],
source: "nextChild\x0a\x09\x22Answer the next node after aNode.\x0a\x09Recurse into the possible children of the receiver to answer the next node to be evaluated\x22\x0a\x09\x0a\x09^ self nodes isEmpty\x0a\x09\x09ifTrue: [ self ]\x0a\x09\x09ifFalse: [ self nodes first nextChild ]",
messageSends: ["ifTrue:ifFalse:", "isEmpty", "nodes", "nextChild", "first"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "nextNode",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._parent();
if(($receiver = $2) == nil || $receiver == null){
$1=$2;
} else {
var node;
node=$receiver;
$1=_st(node)._nextNode_(self);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"nextNode",{},globals.Node)})},
args: [],
source: "nextNode\x0a\x09^ self parent ifNotNil: [ :node |\x0a\x09\x09node nextNode: self ]",
messageSends: ["ifNotNil:", "parent", "nextNode:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "nextNode:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
var next;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
var $early={};
try {
$1=self._nodes();
$ctx1.sendIdx["nodes"]=1;
next=_st($1)._at_ifAbsent_(_st(_st(self._nodes())._indexOf_(aNode)).__plus((1)),(function(){
throw $early=[self];
}));
$2=_st(next)._nextChild();
return $2;
}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"nextNode:",{aNode:aNode,next:next},globals.Node)})},
args: ["aNode"],
source: "nextNode: aNode\x0a\x09\x22Answer the next node after aNode.\x0a\x09Recurse into the possible children of the next node to answer the next node to be evaluated\x22\x0a\x09\x0a\x09| next |\x0a\x09\x0a\x09next := self nodes \x0a\x09\x09at: (self nodes indexOf: aNode) + 1\x0a\x09\x09ifAbsent: [ ^ self ].\x0a\x09\x0a\x09^ next nextChild",
messageSends: ["at:ifAbsent:", "nodes", "+", "indexOf:", "nextChild"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "nodes",
protocol: 'accessing',
fn: function (){
var self=this;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@nodes"];
if(($receiver = $2) == nil || $receiver == null){
self["@nodes"]=_st($Array())._new();
$1=self["@nodes"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"nodes",{},globals.Node)})},
args: [],
source: "nodes\x0a\x09^ nodes ifNil: [ nodes := Array new ]",
messageSends: ["ifNil:", "new"],
referencedClasses: ["Array"]
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "nodes:",
protocol: 'building',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self["@nodes"]=aCollection;
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._parent_(self);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"nodes:",{aCollection:aCollection},globals.Node)})},
args: ["aCollection"],
source: "nodes: aCollection\x0a\x09nodes := aCollection.\x0a\x09aCollection do: [ :each | each parent: self ]",
messageSends: ["do:", "parent:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "parent",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@parent"];
return $1;
},
args: [],
source: "parent\x0a\x09^ parent",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "parent:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
self["@parent"]=aNode;
return self},
args: ["aNode"],
source: "parent: aNode\x0a\x09parent := aNode",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "position",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self["@position"];
if(($receiver = $2) == nil || $receiver == null){
$3=self._parent();
if(($receiver = $3) == nil || $receiver == null){
$1=$3;
} else {
var node;
node=$receiver;
$1=_st(node)._position();
};
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"position",{},globals.Node)})},
args: [],
source: "position\x0a\x09\x22answer the line and column of the receiver in the source code\x22\x0a\x09\x0a\x09^ position ifNil: [ \x0a\x09\x09self parent ifNotNil: [ :node | node position ] ]",
messageSends: ["ifNil:", "ifNotNil:", "parent", "position"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "position:",
protocol: 'accessing',
fn: function (aPosition){
var self=this;
self["@position"]=aPosition;
return self},
args: ["aPosition"],
source: "position: aPosition\x0a\x09position := aPosition",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "positionEnd",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$7,$6,$5,$4,$3,$1;
$2=self._positionStart();
$7=self._source();
$ctx1.sendIdx["source"]=1;
$6=_st($7)._lines();
$ctx1.sendIdx["lines"]=1;
$5=_st($6)._size();
$ctx1.sendIdx["size"]=1;
$4=_st($5).__minus((1));
$ctx1.sendIdx["-"]=1;
$3=_st($4).__at(_st(_st(_st(_st(self._source())._lines())._last())._size()).__minus((1)));
$1=_st($2).__plus($3);
return $1;
}, function($ctx1) {$ctx1.fill(self,"positionEnd",{},globals.Node)})},
args: [],
source: "positionEnd\x0a\x09^ self positionStart + ((self source lines size - 1) @ (self source lines last size - 1))",
messageSends: ["+", "positionStart", "@", "-", "size", "lines", "source", "last"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "positionStart",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._position();
return $1;
}, function($ctx1) {$ctx1.fill(self,"positionStart",{},globals.Node)})},
args: [],
source: "positionStart\x0a\x09^ self position",
messageSends: ["position"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "postCopy",
protocol: 'copying',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.Node.superclass.fn.prototype._postCopy.apply(_st(self), []);
_st(self._nodes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._parent_(self);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"postCopy",{},globals.Node)})},
args: [],
source: "postCopy\x0a\x09super postCopy.\x0a\x09self nodes do: [ :each | each parent: self ]",
messageSends: ["postCopy", "do:", "nodes", "parent:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "requiresSmalltalkContext",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._nodes())._detect_ifNone_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._requiresSmalltalkContext();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return nil;
})))._notNil();
return $1;
}, function($ctx1) {$ctx1.fill(self,"requiresSmalltalkContext",{},globals.Node)})},
args: [],
source: "requiresSmalltalkContext\x0a\x09\x22Answer true if the receiver requires a smalltalk context.\x0a\x09Only send nodes require a context.\x0a\x09\x0a\x09If no node requires a context, the method will be compiled without one.\x0a\x09See `IRJSTranslator` and `JSStream` for context creation\x22\x0a\x09\x0a\x09^ (self nodes \x0a\x09\x09detect: [ :each | each requiresSmalltalkContext ]\x0a\x09\x09ifNone: [ nil ]) notNil",
messageSends: ["notNil", "detect:ifNone:", "nodes", "requiresSmalltalkContext"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldBeAliased",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@shouldBeAliased"];
if(($receiver = $2) == nil || $receiver == null){
$1=false;
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldBeAliased",{},globals.Node)})},
args: [],
source: "shouldBeAliased\x0a\x09^ shouldBeAliased ifNil: [ false ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldBeAliased:",
protocol: 'accessing',
fn: function (aBoolean){
var self=this;
self["@shouldBeAliased"]=aBoolean;
return self},
args: ["aBoolean"],
source: "shouldBeAliased: aBoolean\x0a\x09shouldBeAliased := aBoolean",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldBeInlined",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@shouldBeInlined"];
if(($receiver = $2) == nil || $receiver == null){
$1=false;
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldBeInlined",{},globals.Node)})},
args: [],
source: "shouldBeInlined\x0a\x09^ shouldBeInlined ifNil: [ false ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldBeInlined:",
protocol: 'accessing',
fn: function (aBoolean){
var self=this;
self["@shouldBeInlined"]=aBoolean;
return self},
args: ["aBoolean"],
source: "shouldBeInlined: aBoolean\x0a\x09shouldBeInlined := aBoolean",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "size",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._source())._size();
return $1;
}, function($ctx1) {$ctx1.fill(self,"size",{},globals.Node)})},
args: [],
source: "size\x0a\x09^ self source size",
messageSends: ["size", "source"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@source"];
if(($receiver = $2) == nil || $receiver == null){
$1="";
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"source",{},globals.Node)})},
args: [],
source: "source\x0a\x09^ source ifNil: [ '' ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "stopOnStepping",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "stopOnStepping\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Node);

smalltalk.addMethod(
smalltalk.method({
selector: "subtreeNeedsAliasing",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._shouldBeAliased())._or_((function(){
return smalltalk.withContext(function($ctx2) {
return self._shouldBeInlined();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})})))._or_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._nodes())._anySatisfy_((function(each){
return smalltalk.withContext(function($ctx3) {
return _st(each)._subtreeNeedsAliasing();
}, function($ctx3) {$ctx3.fillBlock({each:each},$ctx2,3)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
$ctx1.sendIdx["or:"]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"subtreeNeedsAliasing",{},globals.Node)})},
args: [],
source: "subtreeNeedsAliasing\x0a\x09^ (self shouldBeAliased or: [ self shouldBeInlined ]) or: [\x0a\x09\x09self nodes anySatisfy: [ :each | each subtreeNeedsAliasing ] ]",
messageSends: ["or:", "shouldBeAliased", "shouldBeInlined", "anySatisfy:", "nodes", "subtreeNeedsAliasing"],
referencedClasses: []
}),
globals.Node);



smalltalk.addClass('AssignmentNode', globals.Node, ['left', 'right'], 'Compiler-AST');
globals.AssignmentNode.comment="I represent an assignment node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitAssignmentNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.AssignmentNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitAssignmentNode: self",
messageSends: ["visitAssignmentNode:"],
referencedClasses: []
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isAssignmentNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isAssignmentNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "left",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@left"];
return $1;
},
args: [],
source: "left\x0a\x09^ left",
messageSends: [],
referencedClasses: []
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "left:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self["@left"]=aNode;
_st(aNode)._parent_(self);
return self}, function($ctx1) {$ctx1.fill(self,"left:",{aNode:aNode},globals.AssignmentNode)})},
args: ["aNode"],
source: "left: aNode\x0a\x09left := aNode.\x0a\x09aNode parent: self",
messageSends: ["parent:"],
referencedClasses: []
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "nodes",
protocol: 'accessing',
fn: function (){
var self=this;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st($Array())._with_with_(self._left(),self._right());
return $1;
}, function($ctx1) {$ctx1.fill(self,"nodes",{},globals.AssignmentNode)})},
args: [],
source: "nodes\x0a\x09^ Array with: self left with: self right",
messageSends: ["with:with:", "left", "right"],
referencedClasses: ["Array"]
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "right",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@right"];
return $1;
},
args: [],
source: "right\x0a\x09^ right",
messageSends: [],
referencedClasses: []
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "right:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self["@right"]=aNode;
_st(aNode)._parent_(self);
return self}, function($ctx1) {$ctx1.fill(self,"right:",{aNode:aNode},globals.AssignmentNode)})},
args: ["aNode"],
source: "right: aNode\x0a\x09right := aNode.\x0a\x09aNode parent: self",
messageSends: ["parent:"],
referencedClasses: []
}),
globals.AssignmentNode);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldBeAliased",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(globals.AssignmentNode.superclass.fn.prototype._shouldBeAliased.apply(_st(self), []))._or_((function(){
return smalltalk.withContext(function($ctx2) {
return self._isReferenced();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldBeAliased",{},globals.AssignmentNode)})},
args: [],
source: "shouldBeAliased\x0a\x09^ super shouldBeAliased or: [ self isReferenced ]",
messageSends: ["or:", "shouldBeAliased", "isReferenced"],
referencedClasses: []
}),
globals.AssignmentNode);



smalltalk.addClass('BlockNode', globals.Node, ['parameters', 'scope'], 'Compiler-AST');
globals.BlockNode.comment="I represent an block closure node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitBlockNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.BlockNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitBlockNode: self",
messageSends: ["visitBlockNode:"],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isBlockNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "nextChild",
protocol: 'accessing',
fn: function (){
var self=this;
return self;
},
args: [],
source: "nextChild\x0a\x09\x22Answer the receiver as we want to avoid eager evaluation\x22\x0a\x09\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "nextNode:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
return self;
},
args: ["aNode"],
source: "nextNode: aNode\x0a\x09\x22Answer the receiver as we want to avoid eager evaluation\x22\x0a\x09\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "parameters",
protocol: 'accessing',
fn: function (){
var self=this;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@parameters"];
if(($receiver = $2) == nil || $receiver == null){
self["@parameters"]=_st($Array())._new();
$1=self["@parameters"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"parameters",{},globals.BlockNode)})},
args: [],
source: "parameters\x0a\x09^ parameters ifNil: [ parameters := Array new ]",
messageSends: ["ifNil:", "new"],
referencedClasses: ["Array"]
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "parameters:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@parameters"]=aCollection;
return self},
args: ["aCollection"],
source: "parameters: aCollection\x0a\x09parameters := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@scope"];
return $1;
},
args: [],
source: "scope\x0a\x09^ scope",
messageSends: [],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope:",
protocol: 'accessing',
fn: function (aLexicalScope){
var self=this;
self["@scope"]=aLexicalScope;
return self},
args: ["aLexicalScope"],
source: "scope: aLexicalScope\x0a\x09scope := aLexicalScope",
messageSends: [],
referencedClasses: []
}),
globals.BlockNode);

smalltalk.addMethod(
smalltalk.method({
selector: "subtreeNeedsAliasing",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._shouldBeAliased())._or_((function(){
return smalltalk.withContext(function($ctx2) {
return self._shouldBeInlined();
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"subtreeNeedsAliasing",{},globals.BlockNode)})},
args: [],
source: "subtreeNeedsAliasing\x0a\x09^ self shouldBeAliased or: [ self shouldBeInlined ]",
messageSends: ["or:", "shouldBeAliased", "shouldBeInlined"],
referencedClasses: []
}),
globals.BlockNode);



smalltalk.addClass('CascadeNode', globals.Node, ['receiver'], 'Compiler-AST');
globals.CascadeNode.comment="I represent an cascade node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitCascadeNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.CascadeNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitCascadeNode: self",
messageSends: ["visitCascadeNode:"],
referencedClasses: []
}),
globals.CascadeNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isCascadeNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isCascadeNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.CascadeNode);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@receiver"];
return $1;
},
args: [],
source: "receiver\x0a\x09^ receiver",
messageSends: [],
referencedClasses: []
}),
globals.CascadeNode);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
self["@receiver"]=aNode;
return self},
args: ["aNode"],
source: "receiver: aNode\x0a\x09receiver := aNode",
messageSends: [],
referencedClasses: []
}),
globals.CascadeNode);



smalltalk.addClass('DynamicArrayNode', globals.Node, [], 'Compiler-AST');
globals.DynamicArrayNode.comment="I represent an dynamic array node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitDynamicArrayNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.DynamicArrayNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitDynamicArrayNode: self",
messageSends: ["visitDynamicArrayNode:"],
referencedClasses: []
}),
globals.DynamicArrayNode);



smalltalk.addClass('DynamicDictionaryNode', globals.Node, [], 'Compiler-AST');
globals.DynamicDictionaryNode.comment="I represent an dynamic dictionary node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitDynamicDictionaryNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.DynamicDictionaryNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitDynamicDictionaryNode: self",
messageSends: ["visitDynamicDictionaryNode:"],
referencedClasses: []
}),
globals.DynamicDictionaryNode);



smalltalk.addClass('JSStatementNode', globals.Node, [], 'Compiler-AST');
globals.JSStatementNode.comment="I represent an JavaScript statement node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitJSStatementNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.JSStatementNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitJSStatementNode: self",
messageSends: ["visitJSStatementNode:"],
referencedClasses: []
}),
globals.JSStatementNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isJSStatementNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isJSStatementNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.JSStatementNode);

smalltalk.addMethod(
smalltalk.method({
selector: "requiresSmalltalkContext",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "requiresSmalltalkContext\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.JSStatementNode);



smalltalk.addClass('MethodNode', globals.Node, ['selector', 'arguments', 'source', 'scope', 'classReferences', 'sendIndexes', 'superSends'], 'Compiler-AST');
globals.MethodNode.comment="I represent an method node.\x0a\x0aA method node must be the root and only method node of a valid AST.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitMethodNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.MethodNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitMethodNode: self",
messageSends: ["visitMethodNode:"],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@arguments"];
if(($receiver = $2) == nil || $receiver == null){
$1=[];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"arguments",{},globals.MethodNode)})},
args: [],
source: "arguments\x0a\x09^ arguments ifNil: [ #() ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@arguments"]=aCollection;
return self},
args: ["aCollection"],
source: "arguments: aCollection\x0a\x09arguments := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "classReferences",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@classReferences"];
return $1;
},
args: [],
source: "classReferences\x0a\x09^ classReferences",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "classReferences:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@classReferences"]=aCollection;
return self},
args: ["aCollection"],
source: "classReferences: aCollection\x0a\x09classReferences := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "messageSends",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._sendIndexes())._keys();
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageSends",{},globals.MethodNode)})},
args: [],
source: "messageSends\x0a\x09^ self sendIndexes keys",
messageSends: ["keys", "sendIndexes"],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
return self;
},
args: [],
source: "method\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@scope"];
return $1;
},
args: [],
source: "scope\x0a\x09^ scope",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope:",
protocol: 'accessing',
fn: function (aMethodScope){
var self=this;
self["@scope"]=aMethodScope;
return self},
args: ["aMethodScope"],
source: "scope: aMethodScope\x0a\x09scope := aMethodScope",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@selector"];
return $1;
},
args: [],
source: "selector\x0a\x09^ selector",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@selector"]=aString;
return self},
args: ["aString"],
source: "selector: aString\x0a\x09selector := aString",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "sendIndexes",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@sendIndexes"];
return $1;
},
args: [],
source: "sendIndexes\x0a\x09^ sendIndexes",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "sendIndexes:",
protocol: 'accessing',
fn: function (aDictionary){
var self=this;
self["@sendIndexes"]=aDictionary;
return self},
args: ["aDictionary"],
source: "sendIndexes: aDictionary\x0a\x09sendIndexes := aDictionary",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@source"];
return $1;
},
args: [],
source: "source\x0a\x09^ source",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "superSends",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@superSends"];
return $1;
},
args: [],
source: "superSends\x0a\x09^ superSends",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);

smalltalk.addMethod(
smalltalk.method({
selector: "superSends:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@superSends"]=aCollection;
return self},
args: ["aCollection"],
source: "superSends: aCollection\x0a\x09superSends := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.MethodNode);



smalltalk.addClass('ReturnNode', globals.Node, ['scope'], 'Compiler-AST');
globals.ReturnNode.comment="I represent an return node. At the AST level, there is not difference between a local return or non-local return.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitReturnNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.ReturnNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitReturnNode: self",
messageSends: ["visitReturnNode:"],
referencedClasses: []
}),
globals.ReturnNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isReturnNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isReturnNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.ReturnNode);

smalltalk.addMethod(
smalltalk.method({
selector: "nonLocalReturn",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._scope())._isMethodScope())._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"nonLocalReturn",{},globals.ReturnNode)})},
args: [],
source: "nonLocalReturn\x0a\x09^ self scope isMethodScope not",
messageSends: ["not", "isMethodScope", "scope"],
referencedClasses: []
}),
globals.ReturnNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@scope"];
return $1;
},
args: [],
source: "scope\x0a\x09^ scope",
messageSends: [],
referencedClasses: []
}),
globals.ReturnNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope:",
protocol: 'accessing',
fn: function (aLexicalScope){
var self=this;
self["@scope"]=aLexicalScope;
return self},
args: ["aLexicalScope"],
source: "scope: aLexicalScope\x0a\x09scope := aLexicalScope",
messageSends: [],
referencedClasses: []
}),
globals.ReturnNode);



smalltalk.addClass('SendNode', globals.Node, ['selector', 'arguments', 'receiver', 'superSend', 'index'], 'Compiler-AST');
globals.SendNode.comment="I represent an message send node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitSendNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.SendNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitSendNode: self",
messageSends: ["visitSendNode:"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@arguments"];
if(($receiver = $2) == nil || $receiver == null){
self["@arguments"]=[];
$1=self["@arguments"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"arguments",{},globals.SendNode)})},
args: [],
source: "arguments\x0a\x09^ arguments ifNil: [ arguments := #() ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self["@arguments"]=aCollection;
_st(aCollection)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._parent_(self);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"arguments:",{aCollection:aCollection},globals.SendNode)})},
args: ["aCollection"],
source: "arguments: aCollection\x0a\x09arguments := aCollection.\x0a\x09aCollection do: [ :each | each parent: self ]",
messageSends: ["do:", "parent:"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "cascadeNodeWithMessages:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
var first;
function $SendNode(){return globals.SendNode||(typeof SendNode=="undefined"?nil:SendNode)}
function $CascadeNode(){return globals.CascadeNode||(typeof CascadeNode=="undefined"?nil:CascadeNode)}
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$5,$3;
$1=_st($SendNode())._new();
$ctx1.sendIdx["new"]=1;
_st($1)._selector_(self._selector());
_st($1)._arguments_(self._arguments());
$2=_st($1)._yourself();
$ctx1.sendIdx["yourself"]=1;
first=$2;
$4=_st($CascadeNode())._new();
_st($4)._receiver_(self._receiver());
_st($4)._nodes_(_st(_st($Array())._with_(first)).__comma(aCollection));
$5=_st($4)._yourself();
$3=$5;
return $3;
}, function($ctx1) {$ctx1.fill(self,"cascadeNodeWithMessages:",{aCollection:aCollection,first:first},globals.SendNode)})},
args: ["aCollection"],
source: "cascadeNodeWithMessages: aCollection\x0a\x09| first |\x0a\x09first := SendNode new\x0a\x09\x09selector: self selector;\x0a\x09\x09arguments: self arguments;\x0a\x09\x09yourself.\x0a\x09^ CascadeNode new\x0a\x09\x09receiver: self receiver;\x0a\x09\x09nodes: (Array with: first), aCollection;\x0a\x09\x09yourself",
messageSends: ["selector:", "new", "selector", "arguments:", "arguments", "yourself", "receiver:", "receiver", "nodes:", ",", "with:"],
referencedClasses: ["SendNode", "CascadeNode", "Array"]
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "index",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@index"];
return $1;
},
args: [],
source: "index\x0a\x09^ index",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "index:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
self["@index"]=anInteger;
return self},
args: ["anInteger"],
source: "index: anInteger\x0a\x09index := anInteger",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isCascadeSendNode",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._parent())._isCascadeNode();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isCascadeSendNode",{},globals.SendNode)})},
args: [],
source: "isCascadeSendNode\x0a\x09^ self parent isCascadeNode",
messageSends: ["isCascadeNode", "parent"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isNavigationNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isNavigationNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isSendNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isSendNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "navigationLink",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._selector();
return $1;
}, function($ctx1) {$ctx1.fill(self,"navigationLink",{},globals.SendNode)})},
args: [],
source: "navigationLink\x0a\x09^ self selector",
messageSends: ["selector"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "nodes",
protocol: 'accessing',
fn: function (){
var self=this;
function $Array(){return globals.Array||(typeof Array=="undefined"?nil:Array)}
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$2,$5,$6,$4;
$1=self._receiver();
$ctx1.sendIdx["receiver"]=1;
if(($receiver = $1) == nil || $receiver == null){
$3=self._arguments();
$ctx1.sendIdx["arguments"]=1;
$2=_st($3)._copy();
return $2;
} else {
$1;
};
$5=_st($Array())._with_(self._receiver());
_st($5)._addAll_(self._arguments());
$6=_st($5)._yourself();
$4=$6;
return $4;
}, function($ctx1) {$ctx1.fill(self,"nodes",{},globals.SendNode)})},
args: [],
source: "nodes\x0a\x09self receiver ifNil: [ ^ self arguments copy ].\x0a\x09\x0a\x09^ (Array with: self receiver)\x0a\x09\x09addAll: self arguments;\x0a\x09\x09yourself",
messageSends: ["ifNil:", "receiver", "copy", "arguments", "addAll:", "with:", "yourself"],
referencedClasses: ["Array"]
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@receiver"];
return $1;
},
args: [],
source: "receiver\x0a\x09^ receiver",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "receiver:",
protocol: 'accessing',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self["@receiver"]=aNode;
$1=_st(aNode)._isNode();
if(smalltalk.assert($1)){
_st(aNode)._parent_(self);
};
return self}, function($ctx1) {$ctx1.fill(self,"receiver:",{aNode:aNode},globals.SendNode)})},
args: ["aNode"],
source: "receiver: aNode\x0a\x09receiver := aNode.\x0a\x09aNode isNode ifTrue: [\x0a\x09\x09aNode parent: self ]",
messageSends: ["ifTrue:", "isNode", "parent:"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "requiresSmalltalkContext",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "requiresSmalltalkContext\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@selector"];
return $1;
},
args: [],
source: "selector\x0a\x09^ selector",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@selector"]=aString;
return self},
args: ["aString"],
source: "selector: aString\x0a\x09selector := aString",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldBeAliased",
protocol: 'testing',
fn: function (){
var self=this;
var sends;
return smalltalk.withContext(function($ctx1) { 
var $1;
sends=_st(_st(_st(self._method())._sendIndexes())._at_(self._selector()))._size();
$1=_st(globals.SendNode.superclass.fn.prototype._shouldBeAliased.apply(_st(self), []))._or_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(_st(_st(sends).__gt((1)))._and_((function(){
return smalltalk.withContext(function($ctx3) {
return _st(self._index()).__lt(sends);
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})})))._and_((function(){
return smalltalk.withContext(function($ctx3) {
return self._isReferenced();
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,3)})}));
$ctx2.sendIdx["and:"]=1;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldBeAliased",{sends:sends},globals.SendNode)})},
args: [],
source: "shouldBeAliased\x0a\x09\x22Because we keep track of send indexes, some send nodes need additional care for aliasing. \x0a\x09See IRJSVisitor >> visitIRSend:\x22\x0a\x09\x0a\x09| sends |\x0a\x09\x0a\x09sends := (self method sendIndexes at: self selector) size.\x0a\x09\x0a\x09^ super shouldBeAliased or: [\x0a\x09\x09(sends > 1 and: [ self index < sends ]) and: [ self isReferenced ] ]",
messageSends: ["size", "at:", "sendIndexes", "method", "selector", "or:", "shouldBeAliased", "and:", ">", "<", "index", "isReferenced"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "stopOnStepping",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "stopOnStepping\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "superSend",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@superSend"];
if(($receiver = $2) == nil || $receiver == null){
$1=false;
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"superSend",{},globals.SendNode)})},
args: [],
source: "superSend\x0a\x09^ superSend ifNil: [ false ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "superSend:",
protocol: 'accessing',
fn: function (aBoolean){
var self=this;
self["@superSend"]=aBoolean;
return self},
args: ["aBoolean"],
source: "superSend: aBoolean\x0a\x09superSend := aBoolean",
messageSends: [],
referencedClasses: []
}),
globals.SendNode);

smalltalk.addMethod(
smalltalk.method({
selector: "valueForReceiver:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
function $SendNode(){return globals.SendNode||(typeof SendNode=="undefined"?nil:SendNode)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$5,$4,$6,$1;
$2=_st($SendNode())._new();
_st($2)._position_(self._position());
_st($2)._source_(self._source());
$3=$2;
$5=self._receiver();
$ctx1.sendIdx["receiver"]=1;
if(($receiver = $5) == nil || $receiver == null){
$4=anObject;
} else {
$4=_st(self._receiver())._valueForReceiver_(anObject);
};
_st($3)._receiver_($4);
_st($2)._selector_(self._selector());
_st($2)._arguments_(self._arguments());
$6=_st($2)._yourself();
$1=$6;
return $1;
}, function($ctx1) {$ctx1.fill(self,"valueForReceiver:",{anObject:anObject},globals.SendNode)})},
args: ["anObject"],
source: "valueForReceiver: anObject\x0a\x09^ SendNode new\x0a\x09\x09position: self position;\x0a\x09\x09source: self source;\x0a\x09\x09receiver: (self receiver\x0a\x09\x09ifNil: [ anObject ] \x0a\x09\x09ifNotNil: [ self receiver valueForReceiver: anObject ]);\x0a\x09\x09selector: self selector;\x0a\x09\x09arguments: self arguments;\x0a\x09\x09yourself",
messageSends: ["position:", "new", "position", "source:", "source", "receiver:", "ifNil:ifNotNil:", "receiver", "valueForReceiver:", "selector:", "selector", "arguments:", "arguments", "yourself"],
referencedClasses: ["SendNode"]
}),
globals.SendNode);



smalltalk.addClass('SequenceNode', globals.Node, ['temps', 'scope'], 'Compiler-AST');
globals.SequenceNode.comment="I represent an sequence node. A sequence represent a set of instructions inside the same scope (the method scope or a block scope).";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitSequenceNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.SequenceNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitSequenceNode: self",
messageSends: ["visitSequenceNode:"],
referencedClasses: []
}),
globals.SequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "asBlockSequenceNode",
protocol: 'converting',
fn: function (){
var self=this;
function $BlockSequenceNode(){return globals.BlockSequenceNode||(typeof BlockSequenceNode=="undefined"?nil:BlockSequenceNode)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($BlockSequenceNode())._new();
_st($2)._position_(self._position());
_st($2)._source_(self._source());
_st($2)._nodes_(self._nodes());
_st($2)._temps_(self._temps());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"asBlockSequenceNode",{},globals.SequenceNode)})},
args: [],
source: "asBlockSequenceNode\x0a\x09^ BlockSequenceNode new\x0a\x09\x09position: self position;\x0a\x09\x09source: self source;\x0a\x09\x09nodes: self nodes;\x0a\x09\x09temps: self temps;\x0a\x09\x09yourself",
messageSends: ["position:", "new", "position", "source:", "source", "nodes:", "nodes", "temps:", "temps", "yourself"],
referencedClasses: ["BlockSequenceNode"]
}),
globals.SequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isSequenceNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isSequenceNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.SequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@scope"];
return $1;
},
args: [],
source: "scope\x0a\x09^ scope",
messageSends: [],
referencedClasses: []
}),
globals.SequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "scope:",
protocol: 'accessing',
fn: function (aLexicalScope){
var self=this;
self["@scope"]=aLexicalScope;
return self},
args: ["aLexicalScope"],
source: "scope: aLexicalScope\x0a\x09scope := aLexicalScope",
messageSends: [],
referencedClasses: []
}),
globals.SequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "temps",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@temps"];
if(($receiver = $2) == nil || $receiver == null){
$1=[];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"temps",{},globals.SequenceNode)})},
args: [],
source: "temps\x0a\x09^ temps ifNil: [ #() ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.SequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "temps:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@temps"]=aCollection;
return self},
args: ["aCollection"],
source: "temps: aCollection\x0a\x09temps := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.SequenceNode);



smalltalk.addClass('BlockSequenceNode', globals.SequenceNode, [], 'Compiler-AST');
globals.BlockSequenceNode.comment="I represent an special sequence node for block scopes.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitBlockSequenceNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.BlockSequenceNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitBlockSequenceNode: self",
messageSends: ["visitBlockSequenceNode:"],
referencedClasses: []
}),
globals.BlockSequenceNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockSequenceNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isBlockSequenceNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.BlockSequenceNode);



smalltalk.addClass('ValueNode', globals.Node, ['value'], 'Compiler-AST');
globals.ValueNode.comment="I represent a value node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitValueNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.ValueNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitValueNode: self",
messageSends: ["visitValueNode:"],
referencedClasses: []
}),
globals.ValueNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._value())._isImmutable();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isImmutable",{},globals.ValueNode)})},
args: [],
source: "isImmutable\x0a\x09^ self value isImmutable",
messageSends: ["isImmutable", "value"],
referencedClasses: []
}),
globals.ValueNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isValueNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isValueNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.ValueNode);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@value"];
return $1;
},
args: [],
source: "value\x0a\x09^ value",
messageSends: [],
referencedClasses: []
}),
globals.ValueNode);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@value"]=anObject;
return self},
args: ["anObject"],
source: "value: anObject\x0a\x09value := anObject",
messageSends: [],
referencedClasses: []
}),
globals.ValueNode);



smalltalk.addClass('VariableNode', globals.ValueNode, ['assigned', 'binding'], 'Compiler-AST');
globals.VariableNode.comment="I represent an variable node.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitVariableNode_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.VariableNode)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitVariableNode: self",
messageSends: ["visitVariableNode:"],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "alias",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._binding())._alias();
return $1;
}, function($ctx1) {$ctx1.fill(self,"alias",{},globals.VariableNode)})},
args: [],
source: "alias\x0a\x09^ self binding alias",
messageSends: ["alias", "binding"],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "assigned",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@assigned"];
if(($receiver = $2) == nil || $receiver == null){
$1=false;
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"assigned",{},globals.VariableNode)})},
args: [],
source: "assigned\x0a\x09^ assigned ifNil: [ false ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "assigned:",
protocol: 'accessing',
fn: function (aBoolean){
var self=this;
self["@assigned"]=aBoolean;
return self},
args: ["aBoolean"],
source: "assigned: aBoolean\x0a\x09assigned := aBoolean",
messageSends: [],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "beAssigned",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._binding())._validateAssignment();
self["@assigned"]=true;
return self}, function($ctx1) {$ctx1.fill(self,"beAssigned",{},globals.VariableNode)})},
args: [],
source: "beAssigned\x0a\x09self binding validateAssignment.\x0a\x09assigned := true",
messageSends: ["validateAssignment", "binding"],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "binding",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@binding"];
return $1;
},
args: [],
source: "binding\x0a\x09^ binding",
messageSends: [],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "binding:",
protocol: 'accessing',
fn: function (aScopeVar){
var self=this;
self["@binding"]=aScopeVar;
return self},
args: ["aScopeVar"],
source: "binding: aScopeVar\x0a\x09binding := aScopeVar",
messageSends: [],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isArgument",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._binding())._isArgVar();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isArgument",{},globals.VariableNode)})},
args: [],
source: "isArgument\x0a\x09^ self binding isArgVar",
messageSends: ["isArgVar", "binding"],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isImmutable",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._binding())._isImmutable();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isImmutable",{},globals.VariableNode)})},
args: [],
source: "isImmutable\x0a\x09^ self binding isImmutable",
messageSends: ["isImmutable", "binding"],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isNavigationNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isNavigationNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "isVariableNode",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isVariableNode\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.VariableNode);

smalltalk.addMethod(
smalltalk.method({
selector: "navigationLink",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._value();
return $1;
}, function($ctx1) {$ctx1.fill(self,"navigationLink",{},globals.VariableNode)})},
args: [],
source: "navigationLink\x0a\x09^ self value",
messageSends: ["value"],
referencedClasses: []
}),
globals.VariableNode);


smalltalk.addMethod(
smalltalk.method({
selector: "ast",
protocol: '*Compiler-AST',
fn: function (){
var self=this;
function $Smalltalk(){return globals.Smalltalk||(typeof Smalltalk=="undefined"?nil:Smalltalk)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self._source();
$ctx1.sendIdx["source"]=1;
_st($1)._ifEmpty_((function(){
return smalltalk.withContext(function($ctx2) {
return self._error_("Method source is empty");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$2=_st($Smalltalk())._parse_(self._source());
return $2;
}, function($ctx1) {$ctx1.fill(self,"ast",{},globals.CompiledMethod)})},
args: [],
source: "ast\x0a\x09self source ifEmpty: [ self error: 'Method source is empty' ].\x0a\x09\x0a\x09^ Smalltalk parse: self source",
messageSends: ["ifEmpty:", "source", "error:", "parse:"],
referencedClasses: ["Smalltalk"]
}),
globals.CompiledMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "isNode",
protocol: '*Compiler-AST',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isNode\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.Object);

});

define("amber_core/Compiler-IR", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Compiler-Core", "amber_core/Kernel-Objects", "amber_core/Kernel-Methods"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Compiler-IR');
smalltalk.packages["Compiler-IR"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('IRASTTranslator', globals.NodeVisitor, ['source', 'theClass', 'method', 'sequence', 'nextAlias'], 'Compiler-IR');
globals.IRASTTranslator.comment="I am the AST (abstract syntax tree) visitor responsible for building the intermediate representation graph.";
smalltalk.addMethod(
smalltalk.method({
selector: "alias:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var variable;
function $IRVariable(){return globals.IRVariable||(typeof IRVariable=="undefined"?nil:IRVariable)}
function $AliasVar(){return globals.AliasVar||(typeof AliasVar=="undefined"?nil:AliasVar)}
function $IRAssignment(){return globals.IRAssignment||(typeof IRAssignment=="undefined"?nil:IRAssignment)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$6,$5,$7,$8,$10,$11,$9,$12;
$1=_st(aNode)._isImmutable();
if(smalltalk.assert($1)){
$2=self._visit_(aNode);
$ctx1.sendIdx["visit:"]=1;
return $2;
};
$3=_st($IRVariable())._new();
$ctx1.sendIdx["new"]=1;
$4=$3;
$6=_st($AliasVar())._new();
$ctx1.sendIdx["new"]=2;
$5=_st($6)._name_("$".__comma(self._nextAlias()));
_st($4)._variable_($5);
$7=_st($3)._yourself();
$ctx1.sendIdx["yourself"]=1;
variable=$7;
$8=self._sequence();
$10=_st($IRAssignment())._new();
_st($10)._add_(variable);
$ctx1.sendIdx["add:"]=2;
_st($10)._add_(self._visit_(aNode));
$ctx1.sendIdx["add:"]=3;
$11=_st($10)._yourself();
$9=$11;
_st($8)._add_($9);
$ctx1.sendIdx["add:"]=1;
_st(_st(self._method())._internalVariables())._add_(variable);
$12=variable;
return $12;
}, function($ctx1) {$ctx1.fill(self,"alias:",{aNode:aNode,variable:variable},globals.IRASTTranslator)})},
args: ["aNode"],
source: "alias: aNode\x0a\x09| variable |\x0a\x0a\x09aNode isImmutable ifTrue: [ ^ self visit: aNode ].\x0a\x0a\x09variable := IRVariable new\x0a\x09\x09variable: (AliasVar new name: '$', self nextAlias);\x0a\x09\x09yourself.\x0a\x0a\x09self sequence add: (IRAssignment new\x0a\x09\x09add: variable;\x0a\x09\x09add: (self visit: aNode);\x0a\x09\x09yourself).\x0a\x0a\x09self method internalVariables add: variable.\x0a\x0a\x09^ variable",
messageSends: ["ifTrue:", "isImmutable", "visit:", "variable:", "new", "name:", ",", "nextAlias", "yourself", "add:", "sequence", "internalVariables", "method"],
referencedClasses: ["IRVariable", "AliasVar", "IRAssignment"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "aliasTemporally:",
protocol: 'visiting',
fn: function (aCollection){
var self=this;
var threshold,result;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3,$5;
threshold=(0);
_st(aCollection)._withIndexDo_((function(each,i){
return smalltalk.withContext(function($ctx2) {
$1=_st(each)._subtreeNeedsAliasing();
if(smalltalk.assert($1)){
threshold=i;
return threshold;
};
}, function($ctx2) {$ctx2.fillBlock({each:each,i:i},$ctx1,1)})}));
$ctx1.sendIdx["withIndexDo:"]=1;
result=_st($OrderedCollection())._new();
_st(aCollection)._withIndexDo_((function(each,i){
return smalltalk.withContext(function($ctx2) {
$2=result;
$4=_st(i).__lt_eq(threshold);
if(smalltalk.assert($4)){
$3=self._alias_(each);
} else {
$3=self._visit_(each);
};
return _st($2)._add_($3);
}, function($ctx2) {$ctx2.fillBlock({each:each,i:i},$ctx1,3)})}));
$5=result;
return $5;
}, function($ctx1) {$ctx1.fill(self,"aliasTemporally:",{aCollection:aCollection,threshold:threshold,result:result},globals.IRASTTranslator)})},
args: ["aCollection"],
source: "aliasTemporally: aCollection\x0a\x09\x22https://github.com/NicolasPetton/amber/issues/296\x0a\x09\x0a\x09If a node is aliased, all preceding ones are aliased as well.\x0a\x09The tree is iterated twice. First we get the aliasing dependency,\x0a\x09then the aliasing itself is done\x22\x0a\x0a\x09| threshold result |\x0a\x09threshold := 0.\x0a\x09\x0a\x09aCollection withIndexDo: [ :each :i |\x0a\x09\x09each subtreeNeedsAliasing\x0a\x09\x09\x09ifTrue: [ threshold := i ] ].\x0a\x0a\x09result := OrderedCollection new.\x0a\x09aCollection withIndexDo: [ :each :i |\x0a\x09\x09result add: (i <= threshold\x0a\x09\x09\x09ifTrue: [ self alias: each ]\x0a\x09\x09\x09ifFalse: [ self visit: each ]) ].\x0a\x0a\x09^ result",
messageSends: ["withIndexDo:", "ifTrue:", "subtreeNeedsAliasing", "new", "add:", "ifTrue:ifFalse:", "<=", "alias:", "visit:"],
referencedClasses: ["OrderedCollection"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@method"];
return $1;
},
args: [],
source: "method\x0a\x09^ method",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "method:",
protocol: 'accessing',
fn: function (anIRMethod){
var self=this;
self["@method"]=anIRMethod;
return self},
args: ["anIRMethod"],
source: "method: anIRMethod\x0a\x09method := anIRMethod",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "nextAlias",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self["@nextAlias"];
if(($receiver = $1) == nil || $receiver == null){
self["@nextAlias"]=(0);
self["@nextAlias"];
} else {
$1;
};
self["@nextAlias"]=_st(self["@nextAlias"]).__plus((1));
$2=_st(self["@nextAlias"])._asString();
return $2;
}, function($ctx1) {$ctx1.fill(self,"nextAlias",{},globals.IRASTTranslator)})},
args: [],
source: "nextAlias\x0a\x09nextAlias ifNil: [ nextAlias := 0 ].\x0a\x09nextAlias := nextAlias + 1.\x0a\x09^ nextAlias asString",
messageSends: ["ifNil:", "+", "asString"],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "sequence",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@sequence"];
return $1;
},
args: [],
source: "sequence\x0a\x09^ sequence",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "sequence:",
protocol: 'accessing',
fn: function (anIRSequence){
var self=this;
self["@sequence"]=anIRSequence;
return self},
args: ["anIRSequence"],
source: "sequence: anIRSequence\x0a\x09sequence := anIRSequence",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@source"];
return $1;
},
args: [],
source: "source\x0a\x09^ source",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@theClass"];
return $1;
},
args: [],
source: "theClass\x0a\x09^ theClass",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@theClass"]=aClass;
return self},
args: ["aClass"],
source: "theClass: aClass\x0a\x09theClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitAssignmentNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var left,right,assignment;
function $IRAssignment(){return globals.IRAssignment||(typeof IRAssignment=="undefined"?nil:IRAssignment)}
return smalltalk.withContext(function($ctx1) { 
var $1,$3,$4,$2,$5;
right=self._visit_(_st(aNode)._right());
$ctx1.sendIdx["visit:"]=1;
left=self._visit_(_st(aNode)._left());
$1=self._sequence();
$3=_st($IRAssignment())._new();
_st($3)._add_(left);
$ctx1.sendIdx["add:"]=2;
_st($3)._add_(right);
$4=_st($3)._yourself();
$2=$4;
_st($1)._add_($2);
$ctx1.sendIdx["add:"]=1;
$5=left;
return $5;
}, function($ctx1) {$ctx1.fill(self,"visitAssignmentNode:",{aNode:aNode,left:left,right:right,assignment:assignment},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitAssignmentNode: aNode\x0a\x09| left right assignment |\x0a\x09right := self visit: aNode right.\x0a\x09left := self visit: aNode left.\x0a\x09self sequence add: (IRAssignment new\x0a\x09\x09add: left;\x0a\x09\x09add: right;\x0a\x09\x09yourself).\x0a\x09^ left",
messageSends: ["visit:", "right", "left", "add:", "sequence", "new", "yourself"],
referencedClasses: ["IRAssignment"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitBlockNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var closure;
function $IRClosure(){return globals.IRClosure||(typeof IRClosure=="undefined"?nil:IRClosure)}
function $IRTempDeclaration(){return globals.IRTempDeclaration||(typeof IRTempDeclaration=="undefined"?nil:IRTempDeclaration)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$6,$5,$7,$8,$9;
$1=_st($IRClosure())._new();
$ctx1.sendIdx["new"]=1;
_st($1)._arguments_(_st(aNode)._parameters());
_st($1)._requiresSmalltalkContext_(_st(aNode)._requiresSmalltalkContext());
$2=$1;
$3=_st(aNode)._scope();
$ctx1.sendIdx["scope"]=1;
_st($2)._scope_($3);
$ctx1.sendIdx["scope:"]=1;
$4=_st($1)._yourself();
$ctx1.sendIdx["yourself"]=1;
closure=$4;
$6=_st(aNode)._scope();
$ctx1.sendIdx["scope"]=2;
$5=_st($6)._temps();
_st($5)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$7=_st($IRTempDeclaration())._new();
_st($7)._name_(_st(each)._name());
_st($7)._scope_(_st(aNode)._scope());
$8=_st($7)._yourself();
return _st(closure)._add_($8);
$ctx2.sendIdx["add:"]=1;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$ctx1.sendIdx["do:"]=1;
_st(_st(aNode)._nodes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(closure)._add_(self._visit_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
$9=closure;
return $9;
}, function($ctx1) {$ctx1.fill(self,"visitBlockNode:",{aNode:aNode,closure:closure},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitBlockNode: aNode\x0a\x09| closure |\x0a\x09closure := IRClosure new\x0a\x09\x09arguments: aNode parameters;\x0a\x09\x09requiresSmalltalkContext: aNode requiresSmalltalkContext;\x0a\x09\x09scope: aNode scope;\x0a\x09\x09yourself.\x0a\x09aNode scope temps do: [ :each |\x0a\x09\x09closure add: (IRTempDeclaration new\x0a\x09\x09\x09name: each name;\x0a\x09\x09\x09scope: aNode scope;\x0a\x09\x09\x09yourself) ].\x0a\x09aNode nodes do: [ :each | closure add: (self visit: each) ].\x0a\x09^ closure",
messageSends: ["arguments:", "new", "parameters", "requiresSmalltalkContext:", "requiresSmalltalkContext", "scope:", "scope", "yourself", "do:", "temps", "add:", "name:", "name", "nodes", "visit:"],
referencedClasses: ["IRClosure", "IRTempDeclaration"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitBlockSequenceNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
function $IRBlockSequence(){return globals.IRBlockSequence||(typeof IRBlockSequence=="undefined"?nil:IRBlockSequence)}
function $IRBlockReturn(){return globals.IRBlockReturn||(typeof IRBlockReturn=="undefined"?nil:IRBlockReturn)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$5,$4,$6,$7,$10,$9,$8,$11,$13,$14,$17,$16,$15,$18,$12,$1;
$2=_st($IRBlockSequence())._new();
$ctx1.sendIdx["new"]=1;
$1=self._withSequence_do_($2,(function(){
return smalltalk.withContext(function($ctx2) {
$3=_st(aNode)._nodes();
$ctx2.sendIdx["nodes"]=1;
return _st($3)._ifNotEmpty_((function(){
return smalltalk.withContext(function($ctx3) {
$5=_st(aNode)._nodes();
$ctx3.sendIdx["nodes"]=2;
$4=_st($5)._allButLast();
_st($4)._do_((function(each){
return smalltalk.withContext(function($ctx4) {
$6=self._sequence();
$ctx4.sendIdx["sequence"]=1;
$7=self._visitOrAlias_(each);
$ctx4.sendIdx["visitOrAlias:"]=1;
return _st($6)._add_($7);
$ctx4.sendIdx["add:"]=1;
}, function($ctx4) {$ctx4.fillBlock({each:each},$ctx3,3)})}));
$10=_st(aNode)._nodes();
$ctx3.sendIdx["nodes"]=3;
$9=_st($10)._last();
$ctx3.sendIdx["last"]=1;
$8=_st($9)._isReturnNode();
if(smalltalk.assert($8)){
return _st(self._sequence())._add_(self._visitOrAlias_(_st(_st(aNode)._nodes())._last()));
} else {
$11=self._sequence();
$ctx3.sendIdx["sequence"]=2;
$13=_st($IRBlockReturn())._new();
$14=$13;
$17=_st(aNode)._nodes();
$ctx3.sendIdx["nodes"]=4;
$16=_st($17)._last();
$ctx3.sendIdx["last"]=2;
$15=self._visitOrAlias_($16);
$ctx3.sendIdx["visitOrAlias:"]=2;
_st($14)._add_($15);
$ctx3.sendIdx["add:"]=3;
$18=_st($13)._yourself();
$12=$18;
return _st($11)._add_($12);
$ctx3.sendIdx["add:"]=2;
};
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitBlockSequenceNode:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitBlockSequenceNode: aNode\x0a\x09^ self\x0a\x09\x09withSequence: IRBlockSequence new\x0a\x09\x09do: [\x0a\x09\x09\x09aNode nodes ifNotEmpty: [\x0a\x09\x09\x09\x09aNode nodes allButLast do: [ :each |\x0a\x09\x09\x09\x09\x09self sequence add: (self visitOrAlias: each) ].\x0a\x09\x09\x09\x09aNode nodes last isReturnNode\x0a\x09\x09\x09\x09\x09ifFalse: [ self sequence add: (IRBlockReturn new add: (self visitOrAlias: aNode nodes last); yourself) ]\x0a\x09\x09\x09\x09\x09ifTrue: [ self sequence add: (self visitOrAlias: aNode nodes last) ] ]]",
messageSends: ["withSequence:do:", "new", "ifNotEmpty:", "nodes", "do:", "allButLast", "add:", "sequence", "visitOrAlias:", "ifFalse:ifTrue:", "isReturnNode", "last", "yourself"],
referencedClasses: ["IRBlockSequence", "IRBlockReturn"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitCascadeNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var alias,receiver;
function $VariableNode(){return globals.VariableNode||(typeof VariableNode=="undefined"?nil:VariableNode)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3,$5,$4,$6;
$2=_st(aNode)._receiver();
$ctx1.sendIdx["receiver"]=1;
$1=_st($2)._isImmutable();
if(smalltalk.assert($1)){
receiver=_st(aNode)._receiver();
$ctx1.sendIdx["receiver"]=2;
receiver;
} else {
alias=self._alias_(_st(aNode)._receiver());
$ctx1.sendIdx["alias:"]=1;
alias;
receiver=_st(_st($VariableNode())._new())._binding_(_st(alias)._variable());
receiver;
};
$3=_st(aNode)._nodes();
$ctx1.sendIdx["nodes"]=1;
_st($3)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._receiver_(receiver);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,3)})}));
$ctx1.sendIdx["do:"]=1;
$5=_st(aNode)._nodes();
$ctx1.sendIdx["nodes"]=2;
$4=_st($5)._allButLast();
_st($4)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self._sequence())._add_(self._visit_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,4)})}));
$6=self._alias_(_st(_st(aNode)._nodes())._last());
return $6;
}, function($ctx1) {$ctx1.fill(self,"visitCascadeNode:",{aNode:aNode,alias:alias,receiver:receiver},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitCascadeNode: aNode\x0a\x09| alias receiver |\x0a\x0a\x09aNode receiver isImmutable \x0a\x09\x09ifTrue: [ receiver := aNode receiver ]\x0a\x09\x09ifFalse: [\x0a\x09\x09\x09alias := self alias: aNode receiver.\x0a\x09\x09\x09receiver := VariableNode new binding: alias variable ].\x0a\x09\x0a\x09aNode nodes do: [ :each |\x0a\x09\x09\x09each receiver: receiver ].\x0a\x0a\x09aNode nodes allButLast do: [ :each |\x0a\x09\x09self sequence add: (self visit: each) ].\x0a\x0a\x09^ self alias: aNode nodes last",
messageSends: ["ifTrue:ifFalse:", "isImmutable", "receiver", "alias:", "binding:", "new", "variable", "do:", "nodes", "receiver:", "allButLast", "add:", "sequence", "visit:", "last"],
referencedClasses: ["VariableNode"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitDynamicArrayNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var array;
function $IRDynamicArray(){return globals.IRDynamicArray||(typeof IRDynamicArray=="undefined"?nil:IRDynamicArray)}
return smalltalk.withContext(function($ctx1) { 
var $1;
array=_st($IRDynamicArray())._new();
_st(self._aliasTemporally_(_st(aNode)._nodes()))._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(array)._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=array;
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitDynamicArrayNode:",{aNode:aNode,array:array},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitDynamicArrayNode: aNode\x0a\x09| array |\x0a\x09array := IRDynamicArray new.\x0a\x09(self aliasTemporally: aNode nodes) do: [ :each | array add: each ].\x0a\x09^ array",
messageSends: ["new", "do:", "aliasTemporally:", "nodes", "add:"],
referencedClasses: ["IRDynamicArray"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitDynamicDictionaryNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var dictionary;
function $IRDynamicDictionary(){return globals.IRDynamicDictionary||(typeof IRDynamicDictionary=="undefined"?nil:IRDynamicDictionary)}
return smalltalk.withContext(function($ctx1) { 
var $1;
dictionary=_st($IRDynamicDictionary())._new();
_st(self._aliasTemporally_(_st(aNode)._nodes()))._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(dictionary)._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$1=dictionary;
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitDynamicDictionaryNode:",{aNode:aNode,dictionary:dictionary},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitDynamicDictionaryNode: aNode\x0a\x09| dictionary |\x0a\x09dictionary := IRDynamicDictionary new.\x0a\x09(self aliasTemporally: aNode nodes) do: [ :each | dictionary add: each ].\x0a\x09^ dictionary",
messageSends: ["new", "do:", "aliasTemporally:", "nodes", "add:"],
referencedClasses: ["IRDynamicDictionary"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitJSStatementNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
function $IRVerbatim(){return globals.IRVerbatim||(typeof IRVerbatim=="undefined"?nil:IRVerbatim)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRVerbatim())._new();
_st($2)._source_(_st(_st(aNode)._source())._crlfSanitized());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitJSStatementNode:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitJSStatementNode: aNode\x0a\x09^ IRVerbatim new\x0a\x09\x09source: aNode source crlfSanitized;\x0a\x09\x09yourself",
messageSends: ["source:", "new", "crlfSanitized", "source", "yourself"],
referencedClasses: ["IRVerbatim"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitMethodNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
function $IRMethod(){return globals.IRMethod||(typeof IRMethod=="undefined"?nil:IRMethod)}
function $IRTempDeclaration(){return globals.IRTempDeclaration||(typeof IRTempDeclaration=="undefined"?nil:IRTempDeclaration)}
function $IRReturn(){return globals.IRReturn||(typeof IRReturn=="undefined"?nil:IRReturn)}
function $IRVariable(){return globals.IRVariable||(typeof IRVariable=="undefined"?nil:IRVariable)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$4,$5,$1,$7,$6,$8,$10,$11,$12,$13,$9,$14,$16,$15,$18,$19,$17,$20,$21,$22;
$2=_st($IRMethod())._new();
$ctx1.sendIdx["new"]=1;
_st($2)._source_(_st(self._source())._crlfSanitized());
_st($2)._theClass_(self._theClass());
_st($2)._arguments_(_st(aNode)._arguments());
_st($2)._selector_(_st(aNode)._selector());
_st($2)._sendIndexes_(_st(aNode)._sendIndexes());
_st($2)._superSends_(_st(aNode)._superSends());
_st($2)._requiresSmalltalkContext_(_st(aNode)._requiresSmalltalkContext());
_st($2)._classReferences_(_st(aNode)._classReferences());
$3=$2;
$4=_st(aNode)._scope();
$ctx1.sendIdx["scope"]=1;
_st($3)._scope_($4);
$ctx1.sendIdx["scope:"]=1;
$5=_st($2)._yourself();
$ctx1.sendIdx["yourself"]=1;
$1=$5;
self._method_($1);
$7=_st(aNode)._scope();
$ctx1.sendIdx["scope"]=2;
$6=_st($7)._temps();
_st($6)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$8=self._method();
$ctx2.sendIdx["method"]=1;
$10=_st($IRTempDeclaration())._new();
$ctx2.sendIdx["new"]=2;
_st($10)._name_(_st(each)._name());
$11=$10;
$12=_st(aNode)._scope();
$ctx2.sendIdx["scope"]=3;
_st($11)._scope_($12);
$13=_st($10)._yourself();
$ctx2.sendIdx["yourself"]=2;
$9=$13;
return _st($8)._add_($9);
$ctx2.sendIdx["add:"]=1;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
$ctx1.sendIdx["do:"]=1;
_st(_st(aNode)._nodes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
$14=self._method();
$ctx2.sendIdx["method"]=2;
return _st($14)._add_(self._visit_(each));
$ctx2.sendIdx["add:"]=2;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
$16=_st(aNode)._scope();
$ctx1.sendIdx["scope"]=4;
$15=_st($16)._hasLocalReturn();
if(! smalltalk.assert($15)){
$18=self._method();
$ctx1.sendIdx["method"]=3;
$19=_st($IRReturn())._new();
$ctx1.sendIdx["new"]=3;
$17=_st($18)._add_($19);
$20=_st($IRVariable())._new();
_st($20)._variable_(_st(_st(_st(aNode)._scope())._pseudoVars())._at_("self"));
$21=_st($20)._yourself();
_st($17)._add_($21);
$ctx1.sendIdx["add:"]=3;
};
$22=self._method();
return $22;
}, function($ctx1) {$ctx1.fill(self,"visitMethodNode:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitMethodNode: aNode\x0a\x0a\x09self method: (IRMethod new\x0a\x09\x09source: self source crlfSanitized;\x0a\x09\x09theClass: self theClass;\x0a\x09\x09arguments: aNode arguments;\x0a\x09\x09selector: aNode selector;\x0a\x09\x09sendIndexes: aNode sendIndexes;\x0a\x09\x09superSends: aNode superSends;\x0a\x09\x09requiresSmalltalkContext: aNode requiresSmalltalkContext;\x0a\x09\x09classReferences: aNode classReferences;\x0a\x09\x09scope: aNode scope;\x0a\x09\x09yourself).\x0a\x0a\x09aNode scope temps do: [ :each |\x0a\x09\x09self method add: (IRTempDeclaration new\x0a\x09\x09\x09name: each name;\x0a\x09\x09\x09scope: aNode scope;\x0a\x09\x09\x09yourself) ].\x0a\x0a\x09aNode nodes do: [ :each | self method add: (self visit: each) ].\x0a\x0a\x09aNode scope hasLocalReturn ifFalse: [\x0a\x09\x09(self method add: IRReturn new) add: (IRVariable new\x0a\x09\x09\x09variable: (aNode scope pseudoVars at: 'self');\x0a\x09\x09\x09yourself) ].\x0a\x0a\x09^ self method",
messageSends: ["method:", "source:", "new", "crlfSanitized", "source", "theClass:", "theClass", "arguments:", "arguments", "selector:", "selector", "sendIndexes:", "sendIndexes", "superSends:", "superSends", "requiresSmalltalkContext:", "requiresSmalltalkContext", "classReferences:", "classReferences", "scope:", "scope", "yourself", "do:", "temps", "add:", "method", "name:", "name", "nodes", "visit:", "ifFalse:", "hasLocalReturn", "variable:", "at:", "pseudoVars"],
referencedClasses: ["IRMethod", "IRTempDeclaration", "IRReturn", "IRVariable"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitOrAlias:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(aNode)._shouldBeAliased();
if(smalltalk.assert($2)){
$1=self._alias_(aNode);
} else {
$1=self._visit_(aNode);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitOrAlias:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitOrAlias: aNode\x0a\x09^ aNode shouldBeAliased\x0a\x09\x09ifTrue: [ self alias: aNode ]\x0a\x09\x09ifFalse: [ self visit: aNode ]",
messageSends: ["ifTrue:ifFalse:", "shouldBeAliased", "alias:", "visit:"],
referencedClasses: []
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitReturnNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var return_;
function $IRNonLocalReturn(){return globals.IRNonLocalReturn||(typeof IRNonLocalReturn=="undefined"?nil:IRNonLocalReturn)}
function $IRReturn(){return globals.IRReturn||(typeof IRReturn=="undefined"?nil:IRReturn)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=_st(aNode)._nonLocalReturn();
if(smalltalk.assert($1)){
return_=_st($IRNonLocalReturn())._new();
$ctx1.sendIdx["new"]=1;
} else {
return_=_st($IRReturn())._new();
};
_st(return_)._scope_(_st(aNode)._scope());
_st(_st(aNode)._nodes())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(return_)._add_(self._alias_(each));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,3)})}));
$2=return_;
return $2;
}, function($ctx1) {$ctx1.fill(self,"visitReturnNode:",{aNode:aNode,return_:return_},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitReturnNode: aNode\x0a\x09| return |\x0a\x09return := aNode nonLocalReturn\x0a\x09\x09ifTrue: [ IRNonLocalReturn new ]\x0a\x09\x09ifFalse: [ IRReturn new ].\x0a\x09return scope: aNode scope.\x0a\x09aNode nodes do: [ :each |\x0a\x09\x09return add: (self alias: each) ].\x0a\x09^ return",
messageSends: ["ifTrue:ifFalse:", "nonLocalReturn", "new", "scope:", "scope", "do:", "nodes", "add:", "alias:"],
referencedClasses: ["IRNonLocalReturn", "IRReturn"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitSendNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
var send,all,receiver,arguments;
function $IRSend(){return globals.IRSend||(typeof IRSend=="undefined"?nil:IRSend)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4;
send=_st($IRSend())._new();
$1=send;
_st($1)._selector_(_st(aNode)._selector());
$2=_st($1)._index_(_st(aNode)._index());
$3=_st(aNode)._superSend();
if(smalltalk.assert($3)){
_st(send)._classSend_(_st(self._theClass())._superclass());
};
all=self._aliasTemporally_(_st([_st(aNode)._receiver()]).__comma(_st(aNode)._arguments()));
receiver=_st(all)._first();
arguments=_st(all)._allButFirst();
_st(send)._add_(receiver);
$ctx1.sendIdx["add:"]=1;
_st(arguments)._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(send)._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
$4=send;
return $4;
}, function($ctx1) {$ctx1.fill(self,"visitSendNode:",{aNode:aNode,send:send,all:all,receiver:receiver,arguments:arguments},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitSendNode: aNode\x0a\x09| send all receiver arguments |\x0a\x09send := IRSend new.\x0a\x09send\x0a\x09\x09selector: aNode selector;\x0a\x09\x09index: aNode index.\x0a\x09aNode superSend ifTrue: [ send classSend: self theClass superclass ].\x0a\x09\x0a\x09all := self aliasTemporally: { aNode receiver }, aNode arguments.\x0a\x09receiver := all first.\x0a\x09arguments := all allButFirst.\x0a\x0a\x09send add: receiver.\x0a\x09arguments do: [ :each | send add: each ].\x0a\x0a\x09^ send",
messageSends: ["new", "selector:", "selector", "index:", "index", "ifTrue:", "superSend", "classSend:", "superclass", "theClass", "aliasTemporally:", ",", "receiver", "arguments", "first", "allButFirst", "add:", "do:"],
referencedClasses: ["IRSend"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitSequenceNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
function $IRSequence(){return globals.IRSequence||(typeof IRSequence=="undefined"?nil:IRSequence)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$1=self._withSequence_do_(_st($IRSequence())._new(),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(_st(aNode)._nodes())._do_((function(each){
var instruction;
return smalltalk.withContext(function($ctx3) {
instruction=self._visitOrAlias_(each);
instruction;
$2=_st(instruction)._isVariable();
if(! smalltalk.assert($2)){
return _st(self._sequence())._add_(instruction);
};
}, function($ctx3) {$ctx3.fillBlock({each:each,instruction:instruction},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitSequenceNode:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitSequenceNode: aNode\x0a\x09^ self\x0a\x09\x09withSequence: IRSequence new\x0a\x09\x09do: [\x0a\x09\x09\x09aNode nodes do: [ :each | | instruction |\x0a\x09\x09\x09\x09instruction := self visitOrAlias: each.\x0a\x09\x09\x09\x09instruction isVariable ifFalse: [\x0a\x09\x09\x09\x09\x09self sequence add: instruction ] ]]",
messageSends: ["withSequence:do:", "new", "do:", "nodes", "visitOrAlias:", "ifFalse:", "isVariable", "add:", "sequence"],
referencedClasses: ["IRSequence"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitValueNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
function $IRValue(){return globals.IRValue||(typeof IRValue=="undefined"?nil:IRValue)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRValue())._new();
_st($2)._value_(_st(aNode)._value());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitValueNode:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitValueNode: aNode\x0a\x09^ IRValue new\x0a\x09\x09value: aNode value;\x0a\x09\x09yourself",
messageSends: ["value:", "new", "value", "yourself"],
referencedClasses: ["IRValue"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitVariableNode:",
protocol: 'visiting',
fn: function (aNode){
var self=this;
function $IRVariable(){return globals.IRVariable||(typeof IRVariable=="undefined"?nil:IRVariable)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRVariable())._new();
_st($2)._variable_(_st(aNode)._binding());
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitVariableNode:",{aNode:aNode},globals.IRASTTranslator)})},
args: ["aNode"],
source: "visitVariableNode: aNode\x0a\x09^ IRVariable new\x0a\x09\x09variable: aNode binding;\x0a\x09\x09yourself",
messageSends: ["variable:", "new", "binding", "yourself"],
referencedClasses: ["IRVariable"]
}),
globals.IRASTTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "withSequence:do:",
protocol: 'accessing',
fn: function (aSequence,aBlock){
var self=this;
var outerSequence;
return smalltalk.withContext(function($ctx1) { 
outerSequence=self._sequence();
self._sequence_(aSequence);
$ctx1.sendIdx["sequence:"]=1;
_st(aBlock)._value();
self._sequence_(outerSequence);
return aSequence;
}, function($ctx1) {$ctx1.fill(self,"withSequence:do:",{aSequence:aSequence,aBlock:aBlock,outerSequence:outerSequence},globals.IRASTTranslator)})},
args: ["aSequence", "aBlock"],
source: "withSequence: aSequence do: aBlock\x0a\x09| outerSequence |\x0a\x09outerSequence := self sequence.\x0a\x09self sequence: aSequence.\x0a\x09aBlock value.\x0a\x09self sequence: outerSequence.\x0a\x09^ aSequence",
messageSends: ["sequence", "sequence:", "value"],
referencedClasses: []
}),
globals.IRASTTranslator);



smalltalk.addClass('IRInstruction', globals.Object, ['parent', 'instructions'], 'Compiler-IR');
globals.IRInstruction.comment="I am the abstract root class of the IR (intermediate representation) instructions class hierarchy.\x0aThe IR graph is used to emit JavaScript code using a JSStream.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRInstruction_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInstruction)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRInstruction: self",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "add:",
protocol: 'building',
fn: function (anObject){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(anObject)._parent_(self);
$1=_st(self._instructions())._add_(anObject);
return $1;
}, function($ctx1) {$ctx1.fill(self,"add:",{anObject:anObject},globals.IRInstruction)})},
args: ["anObject"],
source: "add: anObject\x0a\x09anObject parent: self.\x0a\x09^ self instructions add: anObject",
messageSends: ["parent:", "add:", "instructions"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "canBeAssigned",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "canBeAssigned\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "instructions",
protocol: 'accessing',
fn: function (){
var self=this;
function $OrderedCollection(){return globals.OrderedCollection||(typeof OrderedCollection=="undefined"?nil:OrderedCollection)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@instructions"];
if(($receiver = $2) == nil || $receiver == null){
self["@instructions"]=_st($OrderedCollection())._new();
$1=self["@instructions"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"instructions",{},globals.IRInstruction)})},
args: [],
source: "instructions\x0a\x09^ instructions ifNil: [ instructions := OrderedCollection new ]",
messageSends: ["ifNil:", "new"],
referencedClasses: ["OrderedCollection"]
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isClosure",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isClosure\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isInlined",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isInlined\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isLocalReturn",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isLocalReturn\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isMethod",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isMethod\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isReturn",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isReturn\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isSend",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isSend\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isSequence",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isSequence\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isTempDeclaration",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isTempDeclaration\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "isVariable",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isVariable\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._parent())._method();
return $1;
}, function($ctx1) {$ctx1.fill(self,"method",{},globals.IRInstruction)})},
args: [],
source: "method\x0a\x09^ self parent method",
messageSends: ["method", "parent"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "needsBoxingAsReceiver",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "needsBoxingAsReceiver\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "parent",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@parent"];
return $1;
},
args: [],
source: "parent\x0a\x09^ parent",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "parent:",
protocol: 'accessing',
fn: function (anIRInstruction){
var self=this;
self["@parent"]=anIRInstruction;
return self},
args: ["anIRInstruction"],
source: "parent: anIRInstruction\x0a\x09parent := anIRInstruction",
messageSends: [],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "remove",
protocol: 'building',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._parent())._remove_(self);
return self}, function($ctx1) {$ctx1.fill(self,"remove",{},globals.IRInstruction)})},
args: [],
source: "remove\x0a\x09self parent remove: self",
messageSends: ["remove:", "parent"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "remove:",
protocol: 'building',
fn: function (anIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._instructions())._remove_(anIRInstruction);
return self}, function($ctx1) {$ctx1.fill(self,"remove:",{anIRInstruction:anIRInstruction},globals.IRInstruction)})},
args: ["anIRInstruction"],
source: "remove: anIRInstruction\x0a\x09self instructions remove: anIRInstruction",
messageSends: ["remove:", "instructions"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "replace:with:",
protocol: 'building',
fn: function (anIRInstruction,anotherIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
_st(anotherIRInstruction)._parent_(self);
$1=self._instructions();
$ctx1.sendIdx["instructions"]=1;
_st($1)._at_put_(_st(self._instructions())._indexOf_(anIRInstruction),anotherIRInstruction);
return self}, function($ctx1) {$ctx1.fill(self,"replace:with:",{anIRInstruction:anIRInstruction,anotherIRInstruction:anotherIRInstruction},globals.IRInstruction)})},
args: ["anIRInstruction", "anotherIRInstruction"],
source: "replace: anIRInstruction with: anotherIRInstruction\x0a\x09anotherIRInstruction parent: self.\x0a\x09self instructions\x0a\x09\x09at: (self instructions indexOf: anIRInstruction)\x0a\x09\x09put: anotherIRInstruction",
messageSends: ["parent:", "at:put:", "instructions", "indexOf:"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "replaceWith:",
protocol: 'building',
fn: function (anIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._parent())._replace_with_(self,anIRInstruction);
return self}, function($ctx1) {$ctx1.fill(self,"replaceWith:",{anIRInstruction:anIRInstruction},globals.IRInstruction)})},
args: ["anIRInstruction"],
source: "replaceWith: anIRInstruction\x0a\x09self parent replace: self with: anIRInstruction",
messageSends: ["replace:with:", "parent"],
referencedClasses: []
}),
globals.IRInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._parent();
if(($receiver = $2) == nil || $receiver == null){
$1=$2;
} else {
var node;
node=$receiver;
$1=_st(node)._scope();
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"scope",{},globals.IRInstruction)})},
args: [],
source: "scope\x0a\x09^ self parent ifNotNil: [ :node | \x0a\x09\x09node scope ]",
messageSends: ["ifNotNil:", "parent", "scope"],
referencedClasses: []
}),
globals.IRInstruction);


smalltalk.addMethod(
smalltalk.method({
selector: "on:",
protocol: 'instance creation',
fn: function (aBuilder){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=self._new();
_st($2)._builder_(aBuilder);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"on:",{aBuilder:aBuilder},globals.IRInstruction.klass)})},
args: ["aBuilder"],
source: "on: aBuilder\x0a\x09^ self new\x0a\x09\x09builder: aBuilder;\x0a\x09\x09yourself",
messageSends: ["builder:", "new", "yourself"],
referencedClasses: []
}),
globals.IRInstruction.klass);


smalltalk.addClass('IRAssignment', globals.IRInstruction, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRAssignment_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRAssignment)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRAssignment: self",
messageSends: ["visitIRAssignment:"],
referencedClasses: []
}),
globals.IRAssignment);



smalltalk.addClass('IRDynamicArray', globals.IRInstruction, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRDynamicArray_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRDynamicArray)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRDynamicArray: self",
messageSends: ["visitIRDynamicArray:"],
referencedClasses: []
}),
globals.IRDynamicArray);



smalltalk.addClass('IRDynamicDictionary', globals.IRInstruction, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRDynamicDictionary_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRDynamicDictionary)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRDynamicDictionary: self",
messageSends: ["visitIRDynamicDictionary:"],
referencedClasses: []
}),
globals.IRDynamicDictionary);



smalltalk.addClass('IRScopedInstruction', globals.IRInstruction, ['scope'], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@scope"];
return $1;
},
args: [],
source: "scope\x0a\x09^ scope",
messageSends: [],
referencedClasses: []
}),
globals.IRScopedInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "scope:",
protocol: 'accessing',
fn: function (aScope){
var self=this;
self["@scope"]=aScope;
return self},
args: ["aScope"],
source: "scope: aScope\x0a\x09scope := aScope",
messageSends: [],
referencedClasses: []
}),
globals.IRScopedInstruction);



smalltalk.addClass('IRClosureInstruction', globals.IRScopedInstruction, ['arguments', 'requiresSmalltalkContext'], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "arguments",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@arguments"];
if(($receiver = $2) == nil || $receiver == null){
$1=[];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"arguments",{},globals.IRClosureInstruction)})},
args: [],
source: "arguments\x0a\x09^ arguments ifNil: [ #() ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.IRClosureInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "arguments:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@arguments"]=aCollection;
return self},
args: ["aCollection"],
source: "arguments: aCollection\x0a\x09arguments := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.IRClosureInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "locals",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st(self._arguments())._copy();
_st($2)._addAll_(_st(self._tempDeclarations())._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._name();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})})));
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"locals",{},globals.IRClosureInstruction)})},
args: [],
source: "locals\x0a\x09^ self arguments copy\x0a\x09\x09addAll: (self tempDeclarations collect: [ :each | each name ]);\x0a\x09\x09yourself",
messageSends: ["addAll:", "copy", "arguments", "collect:", "tempDeclarations", "name", "yourself"],
referencedClasses: []
}),
globals.IRClosureInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "requiresSmalltalkContext",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@requiresSmalltalkContext"];
if(($receiver = $2) == nil || $receiver == null){
$1=false;
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"requiresSmalltalkContext",{},globals.IRClosureInstruction)})},
args: [],
source: "requiresSmalltalkContext\x0a\x09^ requiresSmalltalkContext ifNil: [ false ]",
messageSends: ["ifNil:"],
referencedClasses: []
}),
globals.IRClosureInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "requiresSmalltalkContext:",
protocol: 'accessing',
fn: function (anObject){
var self=this;
self["@requiresSmalltalkContext"]=anObject;
return self},
args: ["anObject"],
source: "requiresSmalltalkContext: anObject\x0a\x09requiresSmalltalkContext := anObject",
messageSends: [],
referencedClasses: []
}),
globals.IRClosureInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "scope:",
protocol: 'accessing',
fn: function (aScope){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.IRClosureInstruction.superclass.fn.prototype._scope_.apply(_st(self), [aScope]);
_st(aScope)._instruction_(self);
return self}, function($ctx1) {$ctx1.fill(self,"scope:",{aScope:aScope},globals.IRClosureInstruction)})},
args: ["aScope"],
source: "scope: aScope\x0a\x09super scope: aScope.\x0a\x09aScope instruction: self",
messageSends: ["scope:", "instruction:"],
referencedClasses: []
}),
globals.IRClosureInstruction);

smalltalk.addMethod(
smalltalk.method({
selector: "tempDeclarations",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._instructions())._select_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._isTempDeclaration();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"tempDeclarations",{},globals.IRClosureInstruction)})},
args: [],
source: "tempDeclarations\x0a\x09^ self instructions select: [ :each |\x0a\x09\x09each isTempDeclaration ]",
messageSends: ["select:", "instructions", "isTempDeclaration"],
referencedClasses: []
}),
globals.IRClosureInstruction);



smalltalk.addClass('IRClosure', globals.IRClosureInstruction, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRClosure_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRClosure)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRClosure: self",
messageSends: ["visitIRClosure:"],
referencedClasses: []
}),
globals.IRClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "isClosure",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isClosure\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "sequence",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._instructions())._last();
return $1;
}, function($ctx1) {$ctx1.fill(self,"sequence",{},globals.IRClosure)})},
args: [],
source: "sequence\x0a\x09^ self instructions last",
messageSends: ["last", "instructions"],
referencedClasses: []
}),
globals.IRClosure);



smalltalk.addClass('IRMethod', globals.IRClosureInstruction, ['theClass', 'source', 'selector', 'classReferences', 'sendIndexes', 'superSends', 'requiresSmalltalkContext', 'internalVariables'], 'Compiler-IR');
globals.IRMethod.comment="I am a method instruction";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRMethod_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRMethod)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRMethod: self",
messageSends: ["visitIRMethod:"],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "classReferences",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@classReferences"];
return $1;
},
args: [],
source: "classReferences\x0a\x09^ classReferences",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "classReferences:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@classReferences"]=aCollection;
return self},
args: ["aCollection"],
source: "classReferences: aCollection\x0a\x09classReferences := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "internalVariables",
protocol: 'accessing',
fn: function (){
var self=this;
function $Set(){return globals.Set||(typeof Set=="undefined"?nil:Set)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@internalVariables"];
if(($receiver = $2) == nil || $receiver == null){
self["@internalVariables"]=_st($Set())._new();
$1=self["@internalVariables"];
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"internalVariables",{},globals.IRMethod)})},
args: [],
source: "internalVariables\x0a\x09^ internalVariables ifNil: [ internalVariables := Set new ]",
messageSends: ["ifNil:", "new"],
referencedClasses: ["Set"]
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "isMethod",
protocol: 'accessing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isMethod\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "messageSends",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._sendIndexes())._keys();
return $1;
}, function($ctx1) {$ctx1.fill(self,"messageSends",{},globals.IRMethod)})},
args: [],
source: "messageSends\x0a\x09^ self sendIndexes keys",
messageSends: ["keys", "sendIndexes"],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "method",
protocol: 'accessing',
fn: function (){
var self=this;
return self;
},
args: [],
source: "method\x0a\x09^ self",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@selector"];
return $1;
},
args: [],
source: "selector\x0a\x09^ selector",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@selector"]=aString;
return self},
args: ["aString"],
source: "selector: aString\x0a\x09selector := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "sendIndexes",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@sendIndexes"];
return $1;
},
args: [],
source: "sendIndexes\x0a\x09^ sendIndexes",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "sendIndexes:",
protocol: 'accessing',
fn: function (aDictionary){
var self=this;
self["@sendIndexes"]=aDictionary;
return self},
args: ["aDictionary"],
source: "sendIndexes: aDictionary\x0a\x09sendIndexes := aDictionary",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@source"];
return $1;
},
args: [],
source: "source\x0a\x09^ source",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "superSends",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@superSends"];
return $1;
},
args: [],
source: "superSends\x0a\x09^ superSends",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "superSends:",
protocol: 'accessing',
fn: function (aCollection){
var self=this;
self["@superSends"]=aCollection;
return self},
args: ["aCollection"],
source: "superSends: aCollection\x0a\x09superSends := aCollection",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@theClass"];
return $1;
},
args: [],
source: "theClass\x0a\x09^ theClass",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);

smalltalk.addMethod(
smalltalk.method({
selector: "theClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@theClass"]=aClass;
return self},
args: ["aClass"],
source: "theClass: aClass\x0a\x09theClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.IRMethod);



smalltalk.addClass('IRReturn', globals.IRScopedInstruction, [], 'Compiler-IR');
globals.IRReturn.comment="I am a local return instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRReturn_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRReturn)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRReturn: self",
messageSends: ["visitIRReturn:"],
referencedClasses: []
}),
globals.IRReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "canBeAssigned",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "canBeAssigned\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockReturn",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isBlockReturn\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isLocalReturn",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isLocalReturn\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isNonLocalReturn",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._isLocalReturn())._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"isNonLocalReturn",{},globals.IRReturn)})},
args: [],
source: "isNonLocalReturn\x0a\x09^ self isLocalReturn not",
messageSends: ["not", "isLocalReturn"],
referencedClasses: []
}),
globals.IRReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isReturn",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isReturn\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "scope",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self["@scope"];
if(($receiver = $2) == nil || $receiver == null){
$1=_st(self._parent())._scope();
} else {
$1=$2;
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"scope",{},globals.IRReturn)})},
args: [],
source: "scope\x0a\x09^ scope ifNil: [ self parent scope ]",
messageSends: ["ifNil:", "scope", "parent"],
referencedClasses: []
}),
globals.IRReturn);



smalltalk.addClass('IRBlockReturn', globals.IRReturn, [], 'Compiler-IR');
globals.IRBlockReturn.comment="Smalltalk blocks return their last statement. I am a implicit block return instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRBlockReturn_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRBlockReturn)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRBlockReturn: self",
messageSends: ["visitIRBlockReturn:"],
referencedClasses: []
}),
globals.IRBlockReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isBlockReturn",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isBlockReturn\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRBlockReturn);



smalltalk.addClass('IRNonLocalReturn', globals.IRReturn, [], 'Compiler-IR');
globals.IRNonLocalReturn.comment="I am a non local return instruction.\x0aNon local returns are handled using a try/catch JavaScript statement.\x0a\x0aSee `IRNonLocalReturnHandling` class.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRNonLocalReturn_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRNonLocalReturn)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRNonLocalReturn: self",
messageSends: ["visitIRNonLocalReturn:"],
referencedClasses: []
}),
globals.IRNonLocalReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isLocalReturn",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "isLocalReturn\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRNonLocalReturn);



smalltalk.addClass('IRTempDeclaration', globals.IRScopedInstruction, ['name'], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRTempDeclaration_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRTempDeclaration)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRTempDeclaration: self",
messageSends: ["visitIRTempDeclaration:"],
referencedClasses: []
}),
globals.IRTempDeclaration);

smalltalk.addMethod(
smalltalk.method({
selector: "isTempDeclaration",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isTempDeclaration\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRTempDeclaration);

smalltalk.addMethod(
smalltalk.method({
selector: "name",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@name"];
return $1;
},
args: [],
source: "name\x0a\x09^ name",
messageSends: [],
referencedClasses: []
}),
globals.IRTempDeclaration);

smalltalk.addMethod(
smalltalk.method({
selector: "name:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@name"]=aString;
return self},
args: ["aString"],
source: "name: aString\x0a\x09name := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRTempDeclaration);



smalltalk.addClass('IRSend', globals.IRInstruction, ['selector', 'classSend', 'index'], 'Compiler-IR');
globals.IRSend.comment="I am a message send instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRSend_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRSend)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRSend: self",
messageSends: ["visitIRSend:"],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "classSend",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@classSend"];
return $1;
},
args: [],
source: "classSend\x0a\x09^ classSend",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "classSend:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@classSend"]=aClass;
return self},
args: ["aClass"],
source: "classSend: aClass\x0a\x09classSend := aClass",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "index",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@index"];
return $1;
},
args: [],
source: "index\x0a\x09^ index",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "index:",
protocol: 'accessing',
fn: function (anInteger){
var self=this;
self["@index"]=anInteger;
return self},
args: ["anInteger"],
source: "index: anInteger\x0a\x09index := anInteger",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "isSend",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isSend\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "selector",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@selector"];
return $1;
},
args: [],
source: "selector\x0a\x09^ selector",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);

smalltalk.addMethod(
smalltalk.method({
selector: "selector:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@selector"]=aString;
return self},
args: ["aString"],
source: "selector: aString\x0a\x09selector := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRSend);



smalltalk.addClass('IRSequence', globals.IRInstruction, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRSequence_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRSequence)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRSequence: self",
messageSends: ["visitIRSequence:"],
referencedClasses: []
}),
globals.IRSequence);

smalltalk.addMethod(
smalltalk.method({
selector: "isSequence",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isSequence\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRSequence);



smalltalk.addClass('IRBlockSequence', globals.IRSequence, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRBlockSequence_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRBlockSequence)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRBlockSequence: self",
messageSends: ["visitIRBlockSequence:"],
referencedClasses: []
}),
globals.IRBlockSequence);



smalltalk.addClass('IRValue', globals.IRInstruction, ['value'], 'Compiler-IR');
globals.IRValue.comment="I am the simplest possible instruction. I represent a value.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRValue_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRValue)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRValue: self",
messageSends: ["visitIRValue:"],
referencedClasses: []
}),
globals.IRValue);

smalltalk.addMethod(
smalltalk.method({
selector: "needsBoxingAsReceiver",
protocol: 'testing',
fn: function (){
var self=this;
return false;
},
args: [],
source: "needsBoxingAsReceiver\x0a\x09^ false",
messageSends: [],
referencedClasses: []
}),
globals.IRValue);

smalltalk.addMethod(
smalltalk.method({
selector: "value",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@value"];
return $1;
},
args: [],
source: "value\x0a\x09^ value",
messageSends: [],
referencedClasses: []
}),
globals.IRValue);

smalltalk.addMethod(
smalltalk.method({
selector: "value:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@value"]=aString;
return self},
args: ["aString"],
source: "value: aString\x0a\x09value := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRValue);



smalltalk.addClass('IRVariable', globals.IRInstruction, ['variable'], 'Compiler-IR');
globals.IRVariable.comment="I am a variable instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRVariable_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRVariable)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRVariable: self",
messageSends: ["visitIRVariable:"],
referencedClasses: []
}),
globals.IRVariable);

smalltalk.addMethod(
smalltalk.method({
selector: "isVariable",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isVariable\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRVariable);

smalltalk.addMethod(
smalltalk.method({
selector: "needsBoxingAsReceiver",
protocol: 'testing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(self._variable())._isPseudoVar())._not();
return $1;
}, function($ctx1) {$ctx1.fill(self,"needsBoxingAsReceiver",{},globals.IRVariable)})},
args: [],
source: "needsBoxingAsReceiver\x0a\x09^ self variable isPseudoVar not",
messageSends: ["not", "isPseudoVar", "variable"],
referencedClasses: []
}),
globals.IRVariable);

smalltalk.addMethod(
smalltalk.method({
selector: "variable",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@variable"];
return $1;
},
args: [],
source: "variable\x0a\x09^ variable",
messageSends: [],
referencedClasses: []
}),
globals.IRVariable);

smalltalk.addMethod(
smalltalk.method({
selector: "variable:",
protocol: 'accessing',
fn: function (aScopeVariable){
var self=this;
self["@variable"]=aScopeVariable;
return self},
args: ["aScopeVariable"],
source: "variable: aScopeVariable\x0a\x09variable := aScopeVariable",
messageSends: [],
referencedClasses: []
}),
globals.IRVariable);



smalltalk.addClass('IRVerbatim', globals.IRInstruction, ['source'], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRVerbatim_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRVerbatim)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRVerbatim: self",
messageSends: ["visitIRVerbatim:"],
referencedClasses: []
}),
globals.IRVerbatim);

smalltalk.addMethod(
smalltalk.method({
selector: "source",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@source"];
return $1;
},
args: [],
source: "source\x0a\x09^ source",
messageSends: [],
referencedClasses: []
}),
globals.IRVerbatim);

smalltalk.addMethod(
smalltalk.method({
selector: "source:",
protocol: 'accessing',
fn: function (aString){
var self=this;
self["@source"]=aString;
return self},
args: ["aString"],
source: "source: aString\x0a\x09source := aString",
messageSends: [],
referencedClasses: []
}),
globals.IRVerbatim);



smalltalk.addClass('IRVisitor', globals.Object, [], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "visit:",
protocol: 'visiting',
fn: function (anIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(anIRInstruction)._accept_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visit:",{anIRInstruction:anIRInstruction},globals.IRVisitor)})},
args: ["anIRInstruction"],
source: "visit: anIRInstruction\x0a\x09^ anIRInstruction accept: self",
messageSends: ["accept:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRAssignment:",
protocol: 'visiting',
fn: function (anIRAssignment){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRAssignment);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRAssignment:",{anIRAssignment:anIRAssignment},globals.IRVisitor)})},
args: ["anIRAssignment"],
source: "visitIRAssignment: anIRAssignment\x0a\x09^ self visitIRInstruction: anIRAssignment",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRBlockReturn:",
protocol: 'visiting',
fn: function (anIRBlockReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRReturn_(anIRBlockReturn);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRBlockReturn:",{anIRBlockReturn:anIRBlockReturn},globals.IRVisitor)})},
args: ["anIRBlockReturn"],
source: "visitIRBlockReturn: anIRBlockReturn\x0a\x09^ self visitIRReturn: anIRBlockReturn",
messageSends: ["visitIRReturn:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRBlockSequence:",
protocol: 'visiting',
fn: function (anIRBlockSequence){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRSequence_(anIRBlockSequence);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRBlockSequence:",{anIRBlockSequence:anIRBlockSequence},globals.IRVisitor)})},
args: ["anIRBlockSequence"],
source: "visitIRBlockSequence: anIRBlockSequence\x0a\x09^ self visitIRSequence: anIRBlockSequence",
messageSends: ["visitIRSequence:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRClosure:",
protocol: 'visiting',
fn: function (anIRClosure){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRClosure);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRClosure:",{anIRClosure:anIRClosure},globals.IRVisitor)})},
args: ["anIRClosure"],
source: "visitIRClosure: anIRClosure\x0a\x09^ self visitIRInstruction: anIRClosure",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRDynamicArray:",
protocol: 'visiting',
fn: function (anIRDynamicArray){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRDynamicArray);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRDynamicArray:",{anIRDynamicArray:anIRDynamicArray},globals.IRVisitor)})},
args: ["anIRDynamicArray"],
source: "visitIRDynamicArray: anIRDynamicArray\x0a\x09^ self visitIRInstruction: anIRDynamicArray",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRDynamicDictionary:",
protocol: 'visiting',
fn: function (anIRDynamicDictionary){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRDynamicDictionary);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRDynamicDictionary:",{anIRDynamicDictionary:anIRDynamicDictionary},globals.IRVisitor)})},
args: ["anIRDynamicDictionary"],
source: "visitIRDynamicDictionary: anIRDynamicDictionary\x0a\x09^ self visitIRInstruction: anIRDynamicDictionary",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedClosure:",
protocol: 'visiting',
fn: function (anIRInlinedClosure){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRClosure_(anIRInlinedClosure);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedClosure:",{anIRInlinedClosure:anIRInlinedClosure},globals.IRVisitor)})},
args: ["anIRInlinedClosure"],
source: "visitIRInlinedClosure: anIRInlinedClosure\x0a\x09^ self visitIRClosure: anIRInlinedClosure",
messageSends: ["visitIRClosure:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedSequence:",
protocol: 'visiting',
fn: function (anIRInlinedSequence){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRSequence_(anIRInlinedSequence);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedSequence:",{anIRInlinedSequence:anIRInlinedSequence},globals.IRVisitor)})},
args: ["anIRInlinedSequence"],
source: "visitIRInlinedSequence: anIRInlinedSequence\x0a\x09^ self visitIRSequence: anIRInlinedSequence",
messageSends: ["visitIRSequence:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInstruction:",
protocol: 'visiting',
fn: function (anIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(_st(anIRInstruction)._instructions())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return anIRInstruction;
}, function($ctx1) {$ctx1.fill(self,"visitIRInstruction:",{anIRInstruction:anIRInstruction},globals.IRVisitor)})},
args: ["anIRInstruction"],
source: "visitIRInstruction: anIRInstruction\x0a\x09anIRInstruction instructions do: [ :each | self visit: each ].\x0a\x09^ anIRInstruction",
messageSends: ["do:", "instructions", "visit:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRMethod:",
protocol: 'visiting',
fn: function (anIRMethod){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRMethod);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRMethod:",{anIRMethod:anIRMethod},globals.IRVisitor)})},
args: ["anIRMethod"],
source: "visitIRMethod: anIRMethod\x0a\x09^ self visitIRInstruction: anIRMethod",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRNonLocalReturn:",
protocol: 'visiting',
fn: function (anIRNonLocalReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRNonLocalReturn);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRNonLocalReturn:",{anIRNonLocalReturn:anIRNonLocalReturn},globals.IRVisitor)})},
args: ["anIRNonLocalReturn"],
source: "visitIRNonLocalReturn: anIRNonLocalReturn\x0a\x09^ self visitIRInstruction: anIRNonLocalReturn",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRNonLocalReturnHandling:",
protocol: 'visiting',
fn: function (anIRNonLocalReturnHandling){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRNonLocalReturnHandling);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRNonLocalReturnHandling:",{anIRNonLocalReturnHandling:anIRNonLocalReturnHandling},globals.IRVisitor)})},
args: ["anIRNonLocalReturnHandling"],
source: "visitIRNonLocalReturnHandling: anIRNonLocalReturnHandling\x0a\x09^ self visitIRInstruction: anIRNonLocalReturnHandling",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRReturn:",
protocol: 'visiting',
fn: function (anIRReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRReturn);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRReturn:",{anIRReturn:anIRReturn},globals.IRVisitor)})},
args: ["anIRReturn"],
source: "visitIRReturn: anIRReturn\x0a\x09^ self visitIRInstruction: anIRReturn",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRSend:",
protocol: 'visiting',
fn: function (anIRSend){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRSend);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRSend:",{anIRSend:anIRSend},globals.IRVisitor)})},
args: ["anIRSend"],
source: "visitIRSend: anIRSend\x0a\x09^ self visitIRInstruction: anIRSend",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRSequence:",
protocol: 'visiting',
fn: function (anIRSequence){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRSequence);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRSequence:",{anIRSequence:anIRSequence},globals.IRVisitor)})},
args: ["anIRSequence"],
source: "visitIRSequence: anIRSequence\x0a\x09^ self visitIRInstruction: anIRSequence",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRTempDeclaration:",
protocol: 'visiting',
fn: function (anIRTempDeclaration){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRTempDeclaration);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRTempDeclaration:",{anIRTempDeclaration:anIRTempDeclaration},globals.IRVisitor)})},
args: ["anIRTempDeclaration"],
source: "visitIRTempDeclaration: anIRTempDeclaration\x0a\x09^ self visitIRInstruction: anIRTempDeclaration",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRValue:",
protocol: 'visiting',
fn: function (anIRValue){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRValue);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRValue:",{anIRValue:anIRValue},globals.IRVisitor)})},
args: ["anIRValue"],
source: "visitIRValue: anIRValue\x0a\x09^ self visitIRInstruction: anIRValue",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRVariable:",
protocol: 'visiting',
fn: function (anIRVariable){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRVariable);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRVariable:",{anIRVariable:anIRVariable},globals.IRVisitor)})},
args: ["anIRVariable"],
source: "visitIRVariable: anIRVariable\x0a\x09^ self visitIRInstruction: anIRVariable",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRVerbatim:",
protocol: 'visiting',
fn: function (anIRVerbatim){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._visitIRInstruction_(anIRVerbatim);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRVerbatim:",{anIRVerbatim:anIRVerbatim},globals.IRVisitor)})},
args: ["anIRVerbatim"],
source: "visitIRVerbatim: anIRVerbatim\x0a\x09^ self visitIRInstruction: anIRVerbatim",
messageSends: ["visitIRInstruction:"],
referencedClasses: []
}),
globals.IRVisitor);



smalltalk.addClass('IRJSTranslator', globals.IRVisitor, ['stream', 'currentClass'], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "contents",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self._stream())._contents();
return $1;
}, function($ctx1) {$ctx1.fill(self,"contents",{},globals.IRJSTranslator)})},
args: [],
source: "contents\x0a\x09^ self stream contents",
messageSends: ["contents", "stream"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "currentClass",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@currentClass"];
return $1;
},
args: [],
source: "currentClass\x0a\x09^ currentClass",
messageSends: [],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "currentClass:",
protocol: 'accessing',
fn: function (aClass){
var self=this;
self["@currentClass"]=aClass;
return self},
args: ["aClass"],
source: "currentClass: aClass\x0a\x09currentClass := aClass",
messageSends: [],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
function $JSStream(){return globals.JSStream||(typeof JSStream=="undefined"?nil:JSStream)}
return smalltalk.withContext(function($ctx1) { 
globals.IRJSTranslator.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@stream"]=_st($JSStream())._new();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.IRJSTranslator)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09stream := JSStream new.",
messageSends: ["initialize", "new"],
referencedClasses: ["JSStream"]
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "stream",
protocol: 'accessing',
fn: function (){
var self=this;
var $1;
$1=self["@stream"];
return $1;
},
args: [],
source: "stream\x0a\x09^ stream",
messageSends: [],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "stream:",
protocol: 'accessing',
fn: function (aStream){
var self=this;
self["@stream"]=aStream;
return self},
args: ["aStream"],
source: "stream: aStream\x0a\x09stream := aStream",
messageSends: [],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRAssignment:",
protocol: 'visiting',
fn: function (anIRAssignment){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=_st(anIRAssignment)._instructions();
$ctx1.sendIdx["instructions"]=1;
$1=_st($2)._first();
self._visit_($1);
$ctx1.sendIdx["visit:"]=1;
_st(self._stream())._nextPutAssignment();
self._visit_(_st(_st(anIRAssignment)._instructions())._last());
return self}, function($ctx1) {$ctx1.fill(self,"visitIRAssignment:",{anIRAssignment:anIRAssignment},globals.IRJSTranslator)})},
args: ["anIRAssignment"],
source: "visitIRAssignment: anIRAssignment\x0a\x09self visit: anIRAssignment instructions first.\x0a\x09self stream nextPutAssignment.\x0a\x09self visit: anIRAssignment instructions last.",
messageSends: ["visit:", "first", "instructions", "nextPutAssignment", "stream", "last"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRClosure:",
protocol: 'visiting',
fn: function (anIRClosure){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutClosureWith_arguments_((function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
_st($2)._nextPutVars_(_st(_st(anIRClosure)._tempDeclarations())._collect_((function(each){
return smalltalk.withContext(function($ctx3) {
return _st(_st(each)._name())._asVariableName();
}, function($ctx3) {$ctx3.fillBlock({each:each},$ctx2,2)})})));
return _st(self._stream())._nextPutBlockContextFor_during_(anIRClosure,(function(){
return smalltalk.withContext(function($ctx3) {
return globals.IRJSTranslator.superclass.fn.prototype._visitIRClosure_.apply(_st(self), [anIRClosure]);
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,3)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}),_st(anIRClosure)._arguments());
return self}, function($ctx1) {$ctx1.fill(self,"visitIRClosure:",{anIRClosure:anIRClosure},globals.IRJSTranslator)})},
args: ["anIRClosure"],
source: "visitIRClosure: anIRClosure\x0a\x09self stream\x0a\x09\x09nextPutClosureWith: [\x0a\x09\x09\x09self stream nextPutVars: (anIRClosure tempDeclarations collect: [ :each |\x0a\x09\x09\x09\x09\x09each name asVariableName ]).\x0a\x09\x09\x09self stream\x0a\x09\x09\x09\x09nextPutBlockContextFor: anIRClosure\x0a\x09\x09\x09\x09during: [ super visitIRClosure: anIRClosure ] ]\x0a\x09\x09arguments: anIRClosure arguments",
messageSends: ["nextPutClosureWith:arguments:", "stream", "nextPutVars:", "collect:", "tempDeclarations", "asVariableName", "name", "nextPutBlockContextFor:during:", "visitIRClosure:", "arguments"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRDynamicArray:",
protocol: 'visiting',
fn: function (anIRDynamicArray){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutAll_("[");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(_st(anIRDynamicArray)._instructions())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._stream())._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(self["@stream"])._nextPutAll_("]");
return self}, function($ctx1) {$ctx1.fill(self,"visitIRDynamicArray:",{anIRDynamicArray:anIRDynamicArray},globals.IRJSTranslator)})},
args: ["anIRDynamicArray"],
source: "visitIRDynamicArray: anIRDynamicArray\x0a\x09self stream nextPutAll: '['.\x0a\x09anIRDynamicArray instructions\x0a\x09\x09do: [ :each | self visit: each ]\x0a\x09\x09separatedBy: [ self stream nextPutAll: ',' ].\x0a\x09stream nextPutAll: ']'",
messageSends: ["nextPutAll:", "stream", "do:separatedBy:", "instructions", "visit:"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRDynamicDictionary:",
protocol: 'visiting',
fn: function (anIRDynamicDictionary){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutAll_("globals.HashedCollection._newFromPairs_([");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(_st(anIRDynamicDictionary)._instructions())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
return _st($2)._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(self._stream())._nextPutAll_("])");
return self}, function($ctx1) {$ctx1.fill(self,"visitIRDynamicDictionary:",{anIRDynamicDictionary:anIRDynamicDictionary},globals.IRJSTranslator)})},
args: ["anIRDynamicDictionary"],
source: "visitIRDynamicDictionary: anIRDynamicDictionary\x0a\x09self stream nextPutAll: 'globals.HashedCollection._newFromPairs_(['.\x0a\x09\x09anIRDynamicDictionary instructions\x0a\x09\x09\x09do: [ :each | self visit: each ]\x0a\x09\x09\x09separatedBy: [ self stream nextPutAll: ',' ].\x0a\x09self stream nextPutAll: '])'",
messageSends: ["nextPutAll:", "stream", "do:separatedBy:", "instructions", "visit:"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRMethod:",
protocol: 'visiting',
fn: function (anIRMethod){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4,$5,$6,$8,$7,$9,$10;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutMethodDeclaration_with_(anIRMethod,(function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
return _st($2)._nextPutFunctionWith_arguments_((function(){
return smalltalk.withContext(function($ctx3) {
$3=self._stream();
$ctx3.sendIdx["stream"]=3;
$4=_st(_st(anIRMethod)._tempDeclarations())._collect_((function(each){
return smalltalk.withContext(function($ctx4) {
return _st(_st(each)._name())._asVariableName();
}, function($ctx4) {$ctx4.fillBlock({each:each},$ctx3,3)})}));
$ctx3.sendIdx["collect:"]=1;
_st($3)._nextPutVars_($4);
$ctx3.sendIdx["nextPutVars:"]=1;
_st(_st(anIRMethod)._classReferences())._do_((function(each){
return smalltalk.withContext(function($ctx4) {
$5=self._stream();
$ctx4.sendIdx["stream"]=4;
return _st($5)._nextPutClassRefFunction_(each);
}, function($ctx4) {$ctx4.fillBlock({each:each},$ctx3,4)})}));
$6=self._stream();
$ctx3.sendIdx["stream"]=5;
return _st($6)._nextPutContextFor_during_(anIRMethod,(function(){
return smalltalk.withContext(function($ctx4) {
$8=_st(anIRMethod)._internalVariables();
$ctx4.sendIdx["internalVariables"]=1;
$7=_st($8)._notEmpty();
if(smalltalk.assert($7)){
$9=self._stream();
$ctx4.sendIdx["stream"]=6;
_st($9)._nextPutVars_(_st(_st(_st(anIRMethod)._internalVariables())._asArray())._collect_((function(each){
return smalltalk.withContext(function($ctx5) {
return _st(_st(each)._variable())._alias();
}, function($ctx5) {$ctx5.fillBlock({each:each},$ctx4,7)})})));
};
$10=_st(_st(anIRMethod)._scope())._hasNonLocalReturn();
if(smalltalk.assert($10)){
return _st(self._stream())._nextPutNonLocalReturnHandlingWith_((function(){
return smalltalk.withContext(function($ctx5) {
return globals.IRJSTranslator.superclass.fn.prototype._visitIRMethod_.apply(_st(self), [anIRMethod]);
$ctx5.sendIdx["visitIRMethod:"]=1;
}, function($ctx5) {$ctx5.fillBlock({},$ctx4,9)})}));
} else {
return globals.IRJSTranslator.superclass.fn.prototype._visitIRMethod_.apply(_st(self), [anIRMethod]);
};
}, function($ctx4) {$ctx4.fillBlock({},$ctx3,5)})}));
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}),_st(anIRMethod)._arguments());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRMethod:",{anIRMethod:anIRMethod},globals.IRJSTranslator)})},
args: ["anIRMethod"],
source: "visitIRMethod: anIRMethod\x0a\x0a\x09self stream\x0a\x09\x09nextPutMethodDeclaration: anIRMethod\x0a\x09\x09with: [ self stream\x0a\x09\x09\x09nextPutFunctionWith: [\x0a\x09\x09\x09\x09self stream nextPutVars: (anIRMethod tempDeclarations collect: [ :each |\x0a\x09\x09\x09\x09\x09each name asVariableName ]).\x0a\x09\x09\x09\x09anIRMethod classReferences do: [ :each | self stream nextPutClassRefFunction: each ].\x0a\x09\x09\x09\x09self stream nextPutContextFor: anIRMethod during: [\x0a\x09\x09\x09\x09anIRMethod internalVariables notEmpty ifTrue: [\x0a\x09\x09\x09\x09\x09self stream nextPutVars: (anIRMethod internalVariables asArray collect: [ :each |\x0a\x09\x09\x09\x09\x09\x09each variable alias ]) ].\x0a\x09\x09\x09\x09anIRMethod scope hasNonLocalReturn\x0a\x09\x09\x09\x09\x09ifTrue: [\x0a\x09\x09\x09\x09\x09\x09self stream nextPutNonLocalReturnHandlingWith: [\x0a\x09\x09\x09\x09\x09\x09\x09super visitIRMethod: anIRMethod ] ]\x0a\x09\x09\x09\x09\x09ifFalse: [ super visitIRMethod: anIRMethod ] ]]\x0a\x09\x09\x09arguments: anIRMethod arguments ]",
messageSends: ["nextPutMethodDeclaration:with:", "stream", "nextPutFunctionWith:arguments:", "nextPutVars:", "collect:", "tempDeclarations", "asVariableName", "name", "do:", "classReferences", "nextPutClassRefFunction:", "nextPutContextFor:during:", "ifTrue:", "notEmpty", "internalVariables", "asArray", "alias", "variable", "ifTrue:ifFalse:", "hasNonLocalReturn", "scope", "nextPutNonLocalReturnHandlingWith:", "visitIRMethod:", "arguments"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRNonLocalReturn:",
protocol: 'visiting',
fn: function (anIRNonLocalReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._stream())._nextPutNonLocalReturnWith_((function(){
return smalltalk.withContext(function($ctx2) {
return globals.IRJSTranslator.superclass.fn.prototype._visitIRNonLocalReturn_.apply(_st(self), [anIRNonLocalReturn]);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRNonLocalReturn:",{anIRNonLocalReturn:anIRNonLocalReturn},globals.IRJSTranslator)})},
args: ["anIRNonLocalReturn"],
source: "visitIRNonLocalReturn: anIRNonLocalReturn\x0a\x09self stream nextPutNonLocalReturnWith: [\x0a\x09\x09super visitIRNonLocalReturn: anIRNonLocalReturn ]",
messageSends: ["nextPutNonLocalReturnWith:", "stream", "visitIRNonLocalReturn:"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRReturn:",
protocol: 'visiting',
fn: function (anIRReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._stream())._nextPutReturnWith_((function(){
return smalltalk.withContext(function($ctx2) {
return globals.IRJSTranslator.superclass.fn.prototype._visitIRReturn_.apply(_st(self), [anIRReturn]);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRReturn:",{anIRReturn:anIRReturn},globals.IRJSTranslator)})},
args: ["anIRReturn"],
source: "visitIRReturn: anIRReturn\x0a\x09self stream nextPutReturnWith: [\x0a\x09\x09super visitIRReturn: anIRReturn ]",
messageSends: ["nextPutReturnWith:", "stream", "visitIRReturn:"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRSend:",
protocol: 'visiting',
fn: function (anIRSend){
var self=this;
var sends;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
sends=_st(_st(_st(_st(anIRSend)._method())._sendIndexes())._at_(_st(anIRSend)._selector()))._size();
$1=_st(anIRSend)._classSend();
if(($receiver = $1) == nil || $receiver == null){
self._visitSend_(anIRSend);
} else {
self._visitSuperSend_(anIRSend);
};
$2=_st(_st(sends).__gt((1)))._and_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(_st(anIRSend)._index()).__lt(sends);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
if(smalltalk.assert($2)){
_st(self._stream())._nextPutSendIndexFor_(anIRSend);
};
return self}, function($ctx1) {$ctx1.fill(self,"visitIRSend:",{anIRSend:anIRSend,sends:sends},globals.IRJSTranslator)})},
args: ["anIRSend"],
source: "visitIRSend: anIRSend\x0a\x09| sends |\x0a\x09sends := (anIRSend method sendIndexes at: anIRSend selector) size.\x0a\x09\x0a\x09anIRSend classSend\x0a\x09\x09ifNil: [ self visitSend: anIRSend ]\x0a\x09\x09ifNotNil: [ self visitSuperSend: anIRSend ].\x0a\x09\x09\x0a\x09(sends > 1 and: [ anIRSend index < sends ])\x0a\x09\x09ifTrue: [ self stream nextPutSendIndexFor: anIRSend ]",
messageSends: ["size", "at:", "sendIndexes", "method", "selector", "ifNil:ifNotNil:", "classSend", "visitSend:", "visitSuperSend:", "ifTrue:", "and:", ">", "<", "index", "nextPutSendIndexFor:", "stream"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRSequence:",
protocol: 'visiting',
fn: function (anIRSequence){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutSequenceWith_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(_st(anIRSequence)._instructions())._do_((function(each){
return smalltalk.withContext(function($ctx3) {
return _st(self._stream())._nextPutStatementWith_(self._visit_(each));
}, function($ctx3) {$ctx3.fillBlock({each:each},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRSequence:",{anIRSequence:anIRSequence},globals.IRJSTranslator)})},
args: ["anIRSequence"],
source: "visitIRSequence: anIRSequence\x0a\x09self stream nextPutSequenceWith: [\x0a\x09\x09anIRSequence instructions do: [ :each |\x0a\x09\x09\x09self stream nextPutStatementWith: (self visit: each) ] ]",
messageSends: ["nextPutSequenceWith:", "stream", "do:", "instructions", "nextPutStatementWith:", "visit:"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRTempDeclaration:",
protocol: 'visiting',
fn: function (anIRTempDeclaration){
var self=this;
return self},
args: ["anIRTempDeclaration"],
source: "visitIRTempDeclaration: anIRTempDeclaration\x0a\x09\x22self stream\x0a\x09\x09nextPutAll: 'var ', anIRTempDeclaration name asVariableName, ';';\x0a\x09\x09lf\x22",
messageSends: [],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRValue:",
protocol: 'visiting',
fn: function (anIRValue){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._stream())._nextPutAll_(_st(_st(anIRValue)._value())._asJavascript());
return self}, function($ctx1) {$ctx1.fill(self,"visitIRValue:",{anIRValue:anIRValue},globals.IRJSTranslator)})},
args: ["anIRValue"],
source: "visitIRValue: anIRValue\x0a\x09self stream nextPutAll: anIRValue value asJavascript",
messageSends: ["nextPutAll:", "stream", "asJavascript", "value"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRVariable:",
protocol: 'visiting',
fn: function (anIRVariable){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $3,$2,$1,$4;
$3=_st(anIRVariable)._variable();
$ctx1.sendIdx["variable"]=1;
$2=_st($3)._name();
$1=_st($2).__eq("thisContext");
if(smalltalk.assert($1)){
$4=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($4)._nextPutAll_("smalltalk.getThisContext()");
$ctx1.sendIdx["nextPutAll:"]=1;
} else {
_st(self._stream())._nextPutAll_(_st(_st(anIRVariable)._variable())._alias());
};
return self}, function($ctx1) {$ctx1.fill(self,"visitIRVariable:",{anIRVariable:anIRVariable},globals.IRJSTranslator)})},
args: ["anIRVariable"],
source: "visitIRVariable: anIRVariable\x0a\x09anIRVariable variable name = 'thisContext'\x0a\x09\x09ifTrue: [ self stream nextPutAll: 'smalltalk.getThisContext()' ]\x0a\x09\x09ifFalse: [ self stream nextPutAll: anIRVariable variable alias ]",
messageSends: ["ifTrue:ifFalse:", "=", "name", "variable", "nextPutAll:", "stream", "alias"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRVerbatim:",
protocol: 'visiting',
fn: function (anIRVerbatim){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutStatementWith_((function(){
return smalltalk.withContext(function($ctx2) {
return _st(self._stream())._nextPutAll_(_st(anIRVerbatim)._source());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRVerbatim:",{anIRVerbatim:anIRVerbatim},globals.IRJSTranslator)})},
args: ["anIRVerbatim"],
source: "visitIRVerbatim: anIRVerbatim\x0a\x09self stream nextPutStatementWith: [\x0a\x09\x09self stream nextPutAll: anIRVerbatim source ]",
messageSends: ["nextPutStatementWith:", "stream", "nextPutAll:", "source"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitReceiver:",
protocol: 'visiting',
fn: function (anIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3;
$1=_st(anIRInstruction)._needsBoxingAsReceiver();
if(! smalltalk.assert($1)){
$2=self._visit_(anIRInstruction);
$ctx1.sendIdx["visit:"]=1;
return $2;
};
$3=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($3)._nextPutAll_("_st(");
$ctx1.sendIdx["nextPutAll:"]=1;
self._visit_(anIRInstruction);
_st(self._stream())._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"visitReceiver:",{anIRInstruction:anIRInstruction},globals.IRJSTranslator)})},
args: ["anIRInstruction"],
source: "visitReceiver: anIRInstruction\x0a\x09anIRInstruction needsBoxingAsReceiver ifFalse: [ ^ self visit: anIRInstruction ].\x0a\x09\x0a\x09self stream nextPutAll: '_st('.\x0a\x09self visit: anIRInstruction.\x0a\x09self stream nextPutAll: ')'",
messageSends: ["ifFalse:", "needsBoxingAsReceiver", "visit:", "nextPutAll:", "stream"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitSend:",
protocol: 'visiting',
fn: function (anIRSend){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$3,$4,$5;
$2=_st(anIRSend)._instructions();
$ctx1.sendIdx["instructions"]=1;
$1=_st($2)._first();
self._visitReceiver_($1);
$3=self._stream();
$ctx1.sendIdx["stream"]=1;
$4=_st(".".__comma(_st(_st(anIRSend)._selector())._asSelector())).__comma("(");
$ctx1.sendIdx[","]=1;
_st($3)._nextPutAll_($4);
$ctx1.sendIdx["nextPutAll:"]=1;
_st(_st(_st(anIRSend)._instructions())._allButFirst())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
$5=self._stream();
$ctx2.sendIdx["stream"]=2;
return _st($5)._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(self._stream())._nextPutAll_(")");
return self}, function($ctx1) {$ctx1.fill(self,"visitSend:",{anIRSend:anIRSend},globals.IRJSTranslator)})},
args: ["anIRSend"],
source: "visitSend: anIRSend\x0a\x09self visitReceiver: anIRSend instructions first.\x0a\x09self stream nextPutAll: '.', anIRSend selector asSelector, '('.\x0a\x09anIRSend instructions allButFirst\x0a\x09\x09do: [ :each | self visit: each ]\x0a\x09\x09separatedBy: [ self stream nextPutAll: ',' ].\x0a\x09self stream nextPutAll: ')'",
messageSends: ["visitReceiver:", "first", "instructions", "nextPutAll:", "stream", ",", "asSelector", "selector", "do:separatedBy:", "allButFirst", "visit:"],
referencedClasses: []
}),
globals.IRJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitSuperSend:",
protocol: 'visiting',
fn: function (anIRSend){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3,$5,$6;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutAll_(_st(self._currentClass())._asJavascript());
$ctx1.sendIdx["nextPutAll:"]=1;
_st($1)._nextPutAll_(".superclass.fn.prototype.");
$ctx1.sendIdx["nextPutAll:"]=2;
_st($1)._nextPutAll_(_st(_st(_st(anIRSend)._selector())._asSelector()).__comma(".apply("));
$ctx1.sendIdx["nextPutAll:"]=3;
$2=_st($1)._nextPutAll_("_st(");
$ctx1.sendIdx["nextPutAll:"]=4;
$4=_st(anIRSend)._instructions();
$ctx1.sendIdx["instructions"]=1;
$3=_st($4)._first();
self._visit_($3);
$ctx1.sendIdx["visit:"]=1;
$5=self._stream();
$ctx1.sendIdx["stream"]=2;
_st($5)._nextPutAll_("), [");
$ctx1.sendIdx["nextPutAll:"]=5;
_st(_st(_st(anIRSend)._instructions())._allButFirst())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
$6=self._stream();
$ctx2.sendIdx["stream"]=3;
return _st($6)._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=6;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
_st(self._stream())._nextPutAll_("])");
return self}, function($ctx1) {$ctx1.fill(self,"visitSuperSend:",{anIRSend:anIRSend},globals.IRJSTranslator)})},
args: ["anIRSend"],
source: "visitSuperSend: anIRSend\x0a\x09self stream\x0a\x09\x09nextPutAll: self currentClass asJavascript;\x0a\x09\x09nextPutAll: '.superclass.fn.prototype.';\x0a\x09\x09nextPutAll: anIRSend selector asSelector, '.apply(';\x0a\x09\x09nextPutAll: '_st('.\x0a\x09self visit: anIRSend instructions first.\x0a\x09self stream nextPutAll: '), ['.\x0a\x09anIRSend instructions allButFirst\x0a\x09\x09do: [ :each | self visit: each ]\x0a\x09\x09separatedBy: [ self stream nextPutAll: ',' ].\x0a\x09self stream nextPutAll: '])'",
messageSends: ["nextPutAll:", "stream", "asJavascript", "currentClass", ",", "asSelector", "selector", "visit:", "first", "instructions", "do:separatedBy:", "allButFirst"],
referencedClasses: []
}),
globals.IRJSTranslator);



smalltalk.addClass('JSStream', globals.Object, ['stream'], 'Compiler-IR');
smalltalk.addMethod(
smalltalk.method({
selector: "contents",
protocol: 'accessing',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(self["@stream"])._contents();
return $1;
}, function($ctx1) {$ctx1.fill(self,"contents",{},globals.JSStream)})},
args: [],
source: "contents\x0a\x09^ stream contents",
messageSends: ["contents"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "initialize",
protocol: 'initialization',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
globals.JSStream.superclass.fn.prototype._initialize.apply(_st(self), []);
self["@stream"]=""._writeStream();
return self}, function($ctx1) {$ctx1.fill(self,"initialize",{},globals.JSStream)})},
args: [],
source: "initialize\x0a\x09super initialize.\x0a\x09stream := '' writeStream.",
messageSends: ["initialize", "writeStream"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "lf",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@stream"])._lf();
return self}, function($ctx1) {$ctx1.fill(self,"lf",{},globals.JSStream)})},
args: [],
source: "lf\x0a\x09stream lf",
messageSends: ["lf"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPut:",
protocol: 'streaming',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@stream"])._nextPut_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"nextPut:",{aString:aString},globals.JSStream)})},
args: ["aString"],
source: "nextPut: aString\x0a\x09stream nextPut: aString",
messageSends: ["nextPut:"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutAll:",
protocol: 'streaming',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@stream"])._nextPutAll_(aString);
return self}, function($ctx1) {$ctx1.fill(self,"nextPutAll:",{aString:aString},globals.JSStream)})},
args: ["aString"],
source: "nextPutAll: aString\x0a\x09stream nextPutAll: aString",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutAssignment",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@stream"])._nextPutAll_("=");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutAssignment",{},globals.JSStream)})},
args: [],
source: "nextPutAssignment\x0a\x09stream nextPutAll: '='",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutBlockContextFor:during:",
protocol: 'streaming',
fn: function (anIRClosure,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$6,$5,$4,$3,$7,$11,$10,$9,$8,$15,$14,$13,$12,$16,$17,$24,$23,$22,$21,$20,$19,$18;
$1=_st(anIRClosure)._requiresSmalltalkContext();
if(! smalltalk.assert($1)){
$2=_st(aBlock)._value();
$ctx1.sendIdx["value"]=1;
return $2;
};
$6=_st(anIRClosure)._scope();
$ctx1.sendIdx["scope"]=1;
$5=_st($6)._alias();
$ctx1.sendIdx["alias"]=1;
$4="return smalltalk.withContext(function(".__comma($5);
$ctx1.sendIdx[","]=2;
$3=_st($4).__comma(") {");
$ctx1.sendIdx[","]=1;
self._nextPutAll_($3);
$ctx1.sendIdx["nextPutAll:"]=1;
$7=self._lf();
_st(aBlock)._value();
$11=_st(anIRClosure)._scope();
$ctx1.sendIdx["scope"]=2;
$10=_st($11)._alias();
$ctx1.sendIdx["alias"]=2;
$9="}, function(".__comma($10);
$ctx1.sendIdx[","]=4;
$8=_st($9).__comma(") {");
$ctx1.sendIdx[","]=3;
self._nextPutAll_($8);
$ctx1.sendIdx["nextPutAll:"]=2;
$15=_st(anIRClosure)._scope();
$ctx1.sendIdx["scope"]=3;
$14=_st($15)._alias();
$ctx1.sendIdx["alias"]=3;
$13=_st($14).__comma(".fillBlock({");
$ctx1.sendIdx[","]=5;
$12=self._nextPutAll_($13);
$ctx1.sendIdx["nextPutAll:"]=3;
_st(_st(anIRClosure)._locals())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
$16=_st(each)._asVariableName();
$ctx2.sendIdx["asVariableName"]=1;
self._nextPutAll_($16);
$ctx2.sendIdx["nextPutAll:"]=4;
self._nextPutAll_(":");
$ctx2.sendIdx["nextPutAll:"]=5;
$17=self._nextPutAll_(_st(each)._asVariableName());
$ctx2.sendIdx["nextPutAll:"]=6;
return $17;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=7;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
self._nextPutAll_("},");
$ctx1.sendIdx["nextPutAll:"]=8;
$24=_st(anIRClosure)._scope();
$ctx1.sendIdx["scope"]=4;
$23=_st($24)._outerScope();
$22=_st($23)._alias();
$21=_st($22).__comma(",");
$20=_st($21).__comma(_st(_st(_st(anIRClosure)._scope())._blockIndex())._asString());
$ctx1.sendIdx[","]=7;
$19=_st($20).__comma(")})");
$ctx1.sendIdx[","]=6;
$18=self._nextPutAll_($19);
return self}, function($ctx1) {$ctx1.fill(self,"nextPutBlockContextFor:during:",{anIRClosure:anIRClosure,aBlock:aBlock},globals.JSStream)})},
args: ["anIRClosure", "aBlock"],
source: "nextPutBlockContextFor: anIRClosure during: aBlock\x0a\x09anIRClosure requiresSmalltalkContext ifFalse: [ ^ aBlock value ].\x0a\x09self\x0a\x09\x09nextPutAll: 'return smalltalk.withContext(function(', anIRClosure scope alias, ') {'; lf.\x0a\x09\x0a\x09aBlock value.\x0a\x09\x0a\x09self\x0a\x09\x09nextPutAll: '}, function(', anIRClosure scope alias, ') {';\x0a\x09\x09nextPutAll: anIRClosure scope alias, '.fillBlock({'.\x0a\x09\x0a\x09anIRClosure locals\x0a\x09\x09do: [ :each |\x0a\x09\x09\x09self\x0a\x09\x09\x09\x09nextPutAll: each asVariableName;\x0a\x09\x09\x09\x09nextPutAll: ':';\x0a\x09\x09\x09\x09nextPutAll: each asVariableName ]\x0a\x09\x09separatedBy: [ self nextPutAll: ',' ].\x0a\x09\x0a\x09self\x0a\x09\x09nextPutAll: '},';\x0a\x09\x09nextPutAll: anIRClosure scope outerScope alias, ',', anIRClosure scope blockIndex asString, ')})'",
messageSends: ["ifFalse:", "requiresSmalltalkContext", "value", "nextPutAll:", ",", "alias", "scope", "lf", "do:separatedBy:", "locals", "asVariableName", "outerScope", "asString", "blockIndex"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutClassRefFunction:",
protocol: 'streaming',
fn: function (aString){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
$1=self["@stream"];
_st($1)._nextPutAll_("function $");
$ctx1.sendIdx["nextPutAll:"]=1;
_st($1)._nextPutAll_(aString);
$ctx1.sendIdx["nextPutAll:"]=2;
_st($1)._nextPutAll_("(){return globals.");
$ctx1.sendIdx["nextPutAll:"]=3;
_st($1)._nextPutAll_(aString);
$ctx1.sendIdx["nextPutAll:"]=4;
_st($1)._nextPutAll_("||(typeof ");
$ctx1.sendIdx["nextPutAll:"]=5;
_st($1)._nextPutAll_(aString);
$ctx1.sendIdx["nextPutAll:"]=6;
_st($1)._nextPutAll_("==\x22undefined\x22?nil:");
$ctx1.sendIdx["nextPutAll:"]=7;
_st($1)._nextPutAll_(aString);
$ctx1.sendIdx["nextPutAll:"]=8;
_st($1)._nextPutAll_(")}");
$2=_st($1)._lf();
return self}, function($ctx1) {$ctx1.fill(self,"nextPutClassRefFunction:",{aString:aString},globals.JSStream)})},
args: ["aString"],
source: "nextPutClassRefFunction: aString\x0a\x09\x22Creates an inner function $aString into method and called as `$Foo()`whenever the global is accessed.\x0a\x09This ensures that undefined global access will answer `nil`\x22\x0a\x09\x0a\x09stream\x0a\x09\x09nextPutAll: 'function $';\x0a\x09\x09nextPutAll: aString;\x0a\x09\x09nextPutAll: '(){return globals.';\x0a\x09\x09nextPutAll: aString;\x0a\x09\x09nextPutAll: '||(typeof ';\x0a\x09\x09nextPutAll: aString;\x0a\x09\x09nextPutAll: '==\x22undefined\x22?nil:';\x0a\x09\x09nextPutAll: aString;\x0a\x09\x09nextPutAll: ')}';\x0a\x09\x09lf",
messageSends: ["nextPutAll:", "lf"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutClosureWith:arguments:",
protocol: 'streaming',
fn: function (aBlock,anArray){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
_st(self["@stream"])._nextPutAll_("(function(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(anArray)._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPutAll_(_st(each)._asVariableName());
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPut_(",");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
$1=self["@stream"];
_st($1)._nextPutAll_("){");
$ctx1.sendIdx["nextPutAll:"]=3;
$2=_st($1)._lf();
_st(aBlock)._value();
_st(self["@stream"])._nextPutAll_("})");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutClosureWith:arguments:",{aBlock:aBlock,anArray:anArray},globals.JSStream)})},
args: ["aBlock", "anArray"],
source: "nextPutClosureWith: aBlock arguments: anArray\x0a\x09stream nextPutAll: '(function('.\x0a\x09anArray\x0a\x09\x09do: [ :each | stream nextPutAll: each asVariableName ]\x0a\x09\x09separatedBy: [ stream nextPut: ',' ].\x0a\x09stream nextPutAll: '){'; lf.\x0a\x09aBlock value.\x0a\x09stream nextPutAll: '})'",
messageSends: ["nextPutAll:", "do:separatedBy:", "asVariableName", "nextPut:", "lf", "value"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutContextFor:during:",
protocol: 'streaming',
fn: function (aMethod,aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$6,$5,$4,$3,$7,$12,$11,$10,$9,$8,$16,$15,$14,$13,$17,$18,$19;
$1=_st(aMethod)._requiresSmalltalkContext();
if(! smalltalk.assert($1)){
$2=_st(aBlock)._value();
$ctx1.sendIdx["value"]=1;
return $2;
};
$6=_st(aMethod)._scope();
$ctx1.sendIdx["scope"]=1;
$5=_st($6)._alias();
$ctx1.sendIdx["alias"]=1;
$4="return smalltalk.withContext(function(".__comma($5);
$ctx1.sendIdx[","]=2;
$3=_st($4).__comma(") { ");
$ctx1.sendIdx[","]=1;
self._nextPutAll_($3);
$ctx1.sendIdx["nextPutAll:"]=1;
$7=self._lf();
_st(aBlock)._value();
$12=_st(aMethod)._scope();
$ctx1.sendIdx["scope"]=2;
$11=_st($12)._alias();
$ctx1.sendIdx["alias"]=2;
$10="}, function(".__comma($11);
$ctx1.sendIdx[","]=5;
$9=_st($10).__comma(") {");
$ctx1.sendIdx[","]=4;
$8=_st($9).__comma(_st(_st(aMethod)._scope())._alias());
$ctx1.sendIdx[","]=3;
self._nextPutAll_($8);
$ctx1.sendIdx["nextPutAll:"]=2;
$16=_st(_st(aMethod)._selector())._asJavascript();
$ctx1.sendIdx["asJavascript"]=1;
$15=".fill(self,".__comma($16);
$14=_st($15).__comma(",{");
$ctx1.sendIdx[","]=6;
$13=self._nextPutAll_($14);
$ctx1.sendIdx["nextPutAll:"]=3;
_st(_st(aMethod)._locals())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
$17=_st(each)._asVariableName();
$ctx2.sendIdx["asVariableName"]=1;
self._nextPutAll_($17);
$ctx2.sendIdx["nextPutAll:"]=4;
self._nextPutAll_(":");
$ctx2.sendIdx["nextPutAll:"]=5;
$18=self._nextPutAll_(_st(each)._asVariableName());
$ctx2.sendIdx["nextPutAll:"]=6;
return $18;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=7;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
self._nextPutAll_("},");
$ctx1.sendIdx["nextPutAll:"]=8;
self._nextPutAll_(_st(_st(aMethod)._theClass())._asJavascript());
$ctx1.sendIdx["nextPutAll:"]=9;
$19=self._nextPutAll_(")})");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutContextFor:during:",{aMethod:aMethod,aBlock:aBlock},globals.JSStream)})},
args: ["aMethod", "aBlock"],
source: "nextPutContextFor: aMethod during: aBlock\x0a\x09aMethod requiresSmalltalkContext ifFalse: [ ^ aBlock value ].\x0a\x09\x0a\x09self\x0a\x09\x09nextPutAll: 'return smalltalk.withContext(function(', aMethod scope alias, ') { '; lf.\x0a\x09aBlock value.\x0a\x09\x0a\x09self\x0a\x09\x09nextPutAll: '}, function(', aMethod scope alias, ') {', aMethod scope alias;\x0a\x09\x09nextPutAll: '.fill(self,', aMethod selector asJavascript, ',{'.\x0a\x0a\x09aMethod locals\x0a\x09\x09do: [ :each |\x0a\x09\x09\x09self\x0a\x09\x09\x09\x09nextPutAll: each asVariableName;\x0a\x09\x09\x09\x09nextPutAll: ':';\x0a\x09\x09\x09\x09nextPutAll: each asVariableName ]\x0a\x09\x09separatedBy: [ self nextPutAll: ',' ].\x0a\x09\x0a\x09self\x0a\x09\x09nextPutAll: '},';\x0a\x09\x09nextPutAll: aMethod theClass asJavascript;\x0a\x09\x09nextPutAll: ')})'",
messageSends: ["ifFalse:", "requiresSmalltalkContext", "value", "nextPutAll:", ",", "alias", "scope", "lf", "asJavascript", "selector", "do:separatedBy:", "locals", "asVariableName", "theClass"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutFunctionWith:arguments:",
protocol: 'streaming',
fn: function (aBlock,anArray){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4;
_st(self["@stream"])._nextPutAll_("fn: function(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(anArray)._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPutAll_(_st(each)._asVariableName());
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPut_(",");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
$1=self["@stream"];
_st($1)._nextPutAll_("){");
$ctx1.sendIdx["nextPutAll:"]=3;
$2=_st($1)._lf();
$ctx1.sendIdx["lf"]=1;
$3=self["@stream"];
_st($3)._nextPutAll_("var self=this;");
$ctx1.sendIdx["nextPutAll:"]=4;
$4=_st($3)._lf();
_st(aBlock)._value();
_st(self["@stream"])._nextPutAll_("}");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutFunctionWith:arguments:",{aBlock:aBlock,anArray:anArray},globals.JSStream)})},
args: ["aBlock", "anArray"],
source: "nextPutFunctionWith: aBlock arguments: anArray\x0a\x09stream nextPutAll: 'fn: function('.\x0a\x09anArray\x0a\x09\x09do: [ :each | stream nextPutAll: each asVariableName ]\x0a\x09\x09separatedBy: [ stream nextPut: ',' ].\x0a\x09stream nextPutAll: '){'; lf.\x0a\x09stream nextPutAll: 'var self=this;'; lf.\x0a\x09aBlock value.\x0a\x09stream nextPutAll: '}'",
messageSends: ["nextPutAll:", "do:separatedBy:", "asVariableName", "nextPut:", "lf", "value"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutIf:with:",
protocol: 'streaming',
fn: function (aBlock,anotherBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
_st(self["@stream"])._nextPutAll_("if(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aBlock)._value();
$ctx1.sendIdx["value"]=1;
$1=self["@stream"];
_st($1)._nextPutAll_("){");
$ctx1.sendIdx["nextPutAll:"]=2;
$2=_st($1)._lf();
_st(anotherBlock)._value();
_st(self["@stream"])._nextPutAll_("}");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutIf:with:",{aBlock:aBlock,anotherBlock:anotherBlock},globals.JSStream)})},
args: ["aBlock", "anotherBlock"],
source: "nextPutIf: aBlock with: anotherBlock\x0a\x09stream nextPutAll: 'if('.\x0a\x09aBlock value.\x0a\x09stream nextPutAll: '){'; lf.\x0a\x09anotherBlock value.\x0a\x09stream nextPutAll: '}'",
messageSends: ["nextPutAll:", "value", "lf"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutIfElse:with:with:",
protocol: 'streaming',
fn: function (aBlock,ifBlock,elseBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4;
_st(self["@stream"])._nextPutAll_("if(");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aBlock)._value();
$ctx1.sendIdx["value"]=1;
$1=self["@stream"];
_st($1)._nextPutAll_("){");
$ctx1.sendIdx["nextPutAll:"]=2;
$2=_st($1)._lf();
$ctx1.sendIdx["lf"]=1;
_st(ifBlock)._value();
$ctx1.sendIdx["value"]=2;
$3=self["@stream"];
_st($3)._nextPutAll_("} else {");
$ctx1.sendIdx["nextPutAll:"]=3;
$4=_st($3)._lf();
_st(elseBlock)._value();
_st(self["@stream"])._nextPutAll_("}");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutIfElse:with:with:",{aBlock:aBlock,ifBlock:ifBlock,elseBlock:elseBlock},globals.JSStream)})},
args: ["aBlock", "ifBlock", "elseBlock"],
source: "nextPutIfElse: aBlock with: ifBlock with: elseBlock\x0a\x09stream nextPutAll: 'if('.\x0a\x09aBlock value.\x0a\x09stream nextPutAll: '){'; lf.\x0a\x09ifBlock value.\x0a\x09stream nextPutAll: '} else {'; lf.\x0a\x09elseBlock value.\x0a\x09stream nextPutAll: '}'",
messageSends: ["nextPutAll:", "value", "lf"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutMethodDeclaration:with:",
protocol: 'streaming',
fn: function (aMethod,aBlock){
var self=this;
function $String(){return globals.String||(typeof String=="undefined"?nil:String)}
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$5,$4,$3,$6,$9,$8,$7,$10,$11,$12,$15,$14,$13,$16,$19,$18,$17,$20,$23,$22,$21,$24,$25,$26;
$1=self["@stream"];
_st($1)._nextPutAll_("smalltalk.method({");
$ctx1.sendIdx["nextPutAll:"]=1;
_st($1)._lf();
$ctx1.sendIdx["lf"]=1;
$2=$1;
$5=_st(_st(aMethod)._selector())._asJavascript();
$ctx1.sendIdx["asJavascript"]=1;
$4="selector: ".__comma($5);
$ctx1.sendIdx[","]=2;
$3=_st($4).__comma(",");
$ctx1.sendIdx[","]=1;
_st($2)._nextPutAll_($3);
$ctx1.sendIdx["nextPutAll:"]=2;
_st($1)._lf();
$ctx1.sendIdx["lf"]=2;
$6=$1;
$9=_st(_st(aMethod)._source())._asJavascript();
$ctx1.sendIdx["asJavascript"]=2;
$8="source: ".__comma($9);
$ctx1.sendIdx[","]=4;
$7=_st($8).__comma(",");
$ctx1.sendIdx[","]=3;
_st($6)._nextPutAll_($7);
$ctx1.sendIdx["nextPutAll:"]=3;
$10=_st($1)._lf();
$ctx1.sendIdx["lf"]=3;
_st(aBlock)._value();
$ctx1.sendIdx["value"]=1;
$11=self["@stream"];
$12=$11;
$15=_st($String())._lf();
$ctx1.sendIdx["lf"]=4;
$14=",".__comma($15);
$ctx1.sendIdx[","]=6;
$13=_st($14).__comma("messageSends: ");
$ctx1.sendIdx[","]=5;
_st($12)._nextPutAll_($13);
$ctx1.sendIdx["nextPutAll:"]=4;
$16=$11;
$19=_st(_st(aMethod)._messageSends())._asArray();
$ctx1.sendIdx["asArray"]=1;
$18=_st($19)._asJavascript();
$ctx1.sendIdx["asJavascript"]=3;
$17=_st($18).__comma(",");
$ctx1.sendIdx[","]=7;
_st($16)._nextPutAll_($17);
$ctx1.sendIdx["nextPutAll:"]=5;
_st($11)._lf();
$ctx1.sendIdx["lf"]=5;
$20=$11;
$23=_st(_st(_st(_st(aMethod)._arguments())._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(each)._value();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})})))._asArray())._asJavascript();
$ctx1.sendIdx["asJavascript"]=4;
$22="args: ".__comma($23);
$21=_st($22).__comma(",");
$ctx1.sendIdx[","]=8;
_st($20)._nextPutAll_($21);
$ctx1.sendIdx["nextPutAll:"]=6;
_st($11)._lf();
$24=_st($11)._nextPutAll_("referencedClasses: [");
$ctx1.sendIdx["nextPutAll:"]=7;
_st(_st(aMethod)._classReferences())._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPutAll_(_st(each)._asJavascript());
$ctx2.sendIdx["nextPutAll:"]=8;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=9;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
$25=self["@stream"];
_st($25)._nextPutAll_("]");
$ctx1.sendIdx["nextPutAll:"]=10;
$26=_st($25)._nextPutAll_("})");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutMethodDeclaration:with:",{aMethod:aMethod,aBlock:aBlock},globals.JSStream)})},
args: ["aMethod", "aBlock"],
source: "nextPutMethodDeclaration: aMethod with: aBlock\x0a\x09stream\x0a\x09\x09nextPutAll: 'smalltalk.method({'; lf;\x0a\x09\x09nextPutAll: 'selector: ', aMethod selector asJavascript, ','; lf;\x0a\x09\x09nextPutAll: 'source: ', aMethod source asJavascript, ',';lf.\x0a\x09aBlock value.\x0a\x09stream\x0a\x09\x09nextPutAll: ',', String lf, 'messageSends: ';\x0a\x09\x09nextPutAll: aMethod messageSends asArray asJavascript, ','; lf;\x0a\x09\x09nextPutAll: 'args: ', (aMethod arguments collect: [ :each | each value ]) asArray asJavascript, ','; lf;\x0a\x09\x09nextPutAll: 'referencedClasses: ['.\x0a\x09aMethod classReferences\x0a\x09\x09do: [ :each | stream nextPutAll: each asJavascript ]\x0a\x09\x09separatedBy: [ stream nextPutAll: ',' ].\x0a\x09stream\x0a\x09\x09nextPutAll: ']';\x0a\x09\x09nextPutAll: '})'",
messageSends: ["nextPutAll:", "lf", ",", "asJavascript", "selector", "source", "value", "asArray", "messageSends", "collect:", "arguments", "do:separatedBy:", "classReferences"],
referencedClasses: ["String"]
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutNonLocalReturnHandlingWith:",
protocol: 'streaming',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$3,$4;
$1=self["@stream"];
_st($1)._nextPutAll_("var $early={};");
$ctx1.sendIdx["nextPutAll:"]=1;
_st($1)._lf();
$ctx1.sendIdx["lf"]=1;
_st($1)._nextPutAll_("try {");
$ctx1.sendIdx["nextPutAll:"]=2;
$2=_st($1)._lf();
$ctx1.sendIdx["lf"]=2;
_st(aBlock)._value();
$3=self["@stream"];
_st($3)._nextPutAll_("}");
$ctx1.sendIdx["nextPutAll:"]=3;
_st($3)._lf();
$ctx1.sendIdx["lf"]=3;
_st($3)._nextPutAll_("catch(e) {if(e===$early)return e[0]; throw e}");
$4=_st($3)._lf();
return self}, function($ctx1) {$ctx1.fill(self,"nextPutNonLocalReturnHandlingWith:",{aBlock:aBlock},globals.JSStream)})},
args: ["aBlock"],
source: "nextPutNonLocalReturnHandlingWith: aBlock\x0a\x09stream\x0a\x09\x09nextPutAll: 'var $early={};'; lf;\x0a\x09\x09nextPutAll: 'try {'; lf.\x0a\x09aBlock value.\x0a\x09stream\x0a\x09\x09nextPutAll: '}'; lf;\x0a\x09\x09nextPutAll: 'catch(e) {if(e===$early)return e[0]; throw e}'; lf",
messageSends: ["nextPutAll:", "lf", "value"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutNonLocalReturnWith:",
protocol: 'streaming',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@stream"])._nextPutAll_("throw $early=[");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aBlock)._value();
_st(self["@stream"])._nextPutAll_("]");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutNonLocalReturnWith:",{aBlock:aBlock},globals.JSStream)})},
args: ["aBlock"],
source: "nextPutNonLocalReturnWith: aBlock\x0a\x09stream nextPutAll: 'throw $early=['.\x0a\x09aBlock value.\x0a\x09stream nextPutAll: ']'",
messageSends: ["nextPutAll:", "value"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutReturn",
protocol: 'streaming',
fn: function (){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self["@stream"])._nextPutAll_("return ");
return self}, function($ctx1) {$ctx1.fill(self,"nextPutReturn",{},globals.JSStream)})},
args: [],
source: "nextPutReturn\x0a\x09stream nextPutAll: 'return '",
messageSends: ["nextPutAll:"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutReturnWith:",
protocol: 'streaming',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._nextPutReturn();
_st(aBlock)._value();
return self}, function($ctx1) {$ctx1.fill(self,"nextPutReturnWith:",{aBlock:aBlock},globals.JSStream)})},
args: ["aBlock"],
source: "nextPutReturnWith: aBlock\x0a\x09self nextPutReturn.\x0a\x09aBlock value",
messageSends: ["nextPutReturn", "value"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutSendIndexFor:",
protocol: 'streaming',
fn: function (anIRSend){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
self._nextPutAll_(";");
$ctx1.sendIdx["nextPutAll:"]=1;
self._lf();
self._nextPutAll_(_st(_st(anIRSend)._scope())._alias());
$ctx1.sendIdx["nextPutAll:"]=2;
self._nextPutAll_(".sendIdx[");
$ctx1.sendIdx["nextPutAll:"]=3;
self._nextPutAll_(_st(_st(anIRSend)._selector())._asJavascript());
$ctx1.sendIdx["nextPutAll:"]=4;
self._nextPutAll_("]=");
$ctx1.sendIdx["nextPutAll:"]=5;
$1=self._nextPutAll_(_st(_st(anIRSend)._index())._asString());
return self}, function($ctx1) {$ctx1.fill(self,"nextPutSendIndexFor:",{anIRSend:anIRSend},globals.JSStream)})},
args: ["anIRSend"],
source: "nextPutSendIndexFor: anIRSend\x0a\x09self \x0a\x09\x09nextPutAll: ';'; lf;\x0a\x09\x09nextPutAll: anIRSend scope alias;\x0a\x09\x09nextPutAll: '.sendIdx[';\x0a\x09\x09nextPutAll: anIRSend selector asJavascript;\x0a\x09\x09nextPutAll: ']=';\x0a\x09\x09nextPutAll: anIRSend index asString",
messageSends: ["nextPutAll:", "lf", "alias", "scope", "asJavascript", "selector", "asString", "index"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutSequenceWith:",
protocol: 'streaming',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aBlock)._value();
return self}, function($ctx1) {$ctx1.fill(self,"nextPutSequenceWith:",{aBlock:aBlock},globals.JSStream)})},
args: ["aBlock"],
source: "nextPutSequenceWith: aBlock\x0a\x09\x22stream\x0a\x09\x09nextPutAll: 'switch(smalltalk.thisContext.pc){'; lf.\x22\x0a\x09aBlock value.\x0a\x09\x22stream\x0a\x09\x09nextPutAll: '};'; lf\x22",
messageSends: ["value"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutStatementWith:",
protocol: 'streaming',
fn: function (aBlock){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
_st(aBlock)._value();
$1=self["@stream"];
_st($1)._nextPutAll_(";");
$2=_st($1)._lf();
return self}, function($ctx1) {$ctx1.fill(self,"nextPutStatementWith:",{aBlock:aBlock},globals.JSStream)})},
args: ["aBlock"],
source: "nextPutStatementWith: aBlock\x0a\x09aBlock value.\x0a\x09stream nextPutAll: ';'; lf",
messageSends: ["value", "nextPutAll:", "lf"],
referencedClasses: []
}),
globals.JSStream);

smalltalk.addMethod(
smalltalk.method({
selector: "nextPutVars:",
protocol: 'streaming',
fn: function (aCollection){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2;
var $early={};
try {
_st(aCollection)._ifEmpty_((function(){
throw $early=[self];
}));
_st(self["@stream"])._nextPutAll_("var ");
$ctx1.sendIdx["nextPutAll:"]=1;
_st(aCollection)._do_separatedBy_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPutAll_(each);
$ctx2.sendIdx["nextPutAll:"]=2;
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return _st(self["@stream"])._nextPutAll_(",");
$ctx2.sendIdx["nextPutAll:"]=3;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
$1=self["@stream"];
_st($1)._nextPutAll_(";");
$2=_st($1)._lf();
return self}
catch(e) {if(e===$early)return e[0]; throw e}
}, function($ctx1) {$ctx1.fill(self,"nextPutVars:",{aCollection:aCollection},globals.JSStream)})},
args: ["aCollection"],
source: "nextPutVars: aCollection\x0a\x09aCollection ifEmpty: [ ^ self ].\x0a\x09\x0a\x09stream nextPutAll: 'var '.\x0a\x09aCollection\x0a\x09\x09do: [ :each | stream nextPutAll: each ]\x0a\x09\x09separatedBy: [ stream nextPutAll: ',' ].\x0a\x09stream nextPutAll: ';'; lf",
messageSends: ["ifEmpty:", "nextPutAll:", "do:separatedBy:", "lf"],
referencedClasses: []
}),
globals.JSStream);


smalltalk.addMethod(
smalltalk.method({
selector: "appendToInstruction:",
protocol: '*Compiler-IR',
fn: function (anIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(anIRInstruction)._appendBlock_(self);
return self}, function($ctx1) {$ctx1.fill(self,"appendToInstruction:",{anIRInstruction:anIRInstruction},globals.BlockClosure)})},
args: ["anIRInstruction"],
source: "appendToInstruction: anIRInstruction\x0a\x09anIRInstruction appendBlock: self",
messageSends: ["appendBlock:"],
referencedClasses: []
}),
globals.BlockClosure);

});

define("amber_core/Compiler-Inlining", ["amber_vm/smalltalk", "amber_vm/nil", "amber_vm/_st", "amber_vm/globals", "amber_core/Compiler-IR", "amber_core/Kernel-Objects", "amber_core/Compiler-Core"], function(smalltalk,nil,_st, globals){
smalltalk.addPackage('Compiler-Inlining');
smalltalk.packages["Compiler-Inlining"].transport = {"type":"amd","amdNamespace":"amber_core"};

smalltalk.addClass('IRInlinedAssignment', globals.IRAssignment, [], 'Compiler-Inlining');
globals.IRInlinedAssignment.comment="I represent an inlined assignment instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRInlinedAssignment_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedAssignment)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRInlinedAssignment: self",
messageSends: ["visitIRInlinedAssignment:"],
referencedClasses: []
}),
globals.IRInlinedAssignment);

smalltalk.addMethod(
smalltalk.method({
selector: "isInlined",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isInlined\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInlinedAssignment);



smalltalk.addClass('IRInlinedClosure', globals.IRClosure, [], 'Compiler-Inlining');
globals.IRInlinedClosure.comment="I represent an inlined closure instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitIRInlinedClosure_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedClosure)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitIRInlinedClosure: self",
messageSends: ["visitIRInlinedClosure:"],
referencedClasses: []
}),
globals.IRInlinedClosure);

smalltalk.addMethod(
smalltalk.method({
selector: "isInlined",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isInlined\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInlinedClosure);



smalltalk.addClass('IRInlinedReturn', globals.IRReturn, [], 'Compiler-Inlining');
globals.IRInlinedReturn.comment="I represent an inlined local return instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(aVisitor)._visitIRInlinedReturn_(self);
return $1;
}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedReturn)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09^ aVisitor visitIRInlinedReturn: self",
messageSends: ["visitIRInlinedReturn:"],
referencedClasses: []
}),
globals.IRInlinedReturn);

smalltalk.addMethod(
smalltalk.method({
selector: "isInlined",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isInlined\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInlinedReturn);



smalltalk.addClass('IRInlinedSend', globals.IRSend, [], 'Compiler-Inlining');
globals.IRInlinedSend.comment="I am the abstract super class of inlined message send instructions.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitInlinedSend_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedSend)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitInlinedSend: self",
messageSends: ["visitInlinedSend:"],
referencedClasses: []
}),
globals.IRInlinedSend);

smalltalk.addMethod(
smalltalk.method({
selector: "isInlined",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isInlined\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInlinedSend);



smalltalk.addClass('IRInlinedIfFalse', globals.IRInlinedSend, [], 'Compiler-Inlining');
globals.IRInlinedIfFalse.comment="I represent an inlined `#ifFalse:` message send instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitIRInlinedIfFalse_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedIfFalse)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitIRInlinedIfFalse: self",
messageSends: ["visitIRInlinedIfFalse:"],
referencedClasses: []
}),
globals.IRInlinedIfFalse);



smalltalk.addClass('IRInlinedIfNilIfNotNil', globals.IRInlinedSend, [], 'Compiler-Inlining');
globals.IRInlinedIfNilIfNotNil.comment="I represent an inlined `#ifNil:ifNotNil:` message send instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitIRInlinedIfNilIfNotNil_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedIfNilIfNotNil)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitIRInlinedIfNilIfNotNil: self",
messageSends: ["visitIRInlinedIfNilIfNotNil:"],
referencedClasses: []
}),
globals.IRInlinedIfNilIfNotNil);



smalltalk.addClass('IRInlinedIfTrue', globals.IRInlinedSend, [], 'Compiler-Inlining');
globals.IRInlinedIfTrue.comment="I represent an inlined `#ifTrue:` message send instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitIRInlinedIfTrue_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedIfTrue)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitIRInlinedIfTrue: self",
messageSends: ["visitIRInlinedIfTrue:"],
referencedClasses: []
}),
globals.IRInlinedIfTrue);



smalltalk.addClass('IRInlinedIfTrueIfFalse', globals.IRInlinedSend, [], 'Compiler-Inlining');
globals.IRInlinedIfTrueIfFalse.comment="I represent an inlined `#ifTrue:ifFalse:` message send instruction.";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitIRInlinedIfTrueIfFalse_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedIfTrueIfFalse)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitIRInlinedIfTrueIfFalse: self",
messageSends: ["visitIRInlinedIfTrueIfFalse:"],
referencedClasses: []
}),
globals.IRInlinedIfTrueIfFalse);



smalltalk.addClass('IRInlinedSequence', globals.IRBlockSequence, [], 'Compiler-Inlining');
globals.IRInlinedSequence.comment="I represent a (block) sequence inside an inlined closure instruction (instance of `IRInlinedClosure`).";
smalltalk.addMethod(
smalltalk.method({
selector: "accept:",
protocol: 'visiting',
fn: function (aVisitor){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(aVisitor)._visitIRInlinedSequence_(self);
return self}, function($ctx1) {$ctx1.fill(self,"accept:",{aVisitor:aVisitor},globals.IRInlinedSequence)})},
args: ["aVisitor"],
source: "accept: aVisitor\x0a\x09aVisitor visitIRInlinedSequence: self",
messageSends: ["visitIRInlinedSequence:"],
referencedClasses: []
}),
globals.IRInlinedSequence);

smalltalk.addMethod(
smalltalk.method({
selector: "isInlined",
protocol: 'testing',
fn: function (){
var self=this;
return true;
},
args: [],
source: "isInlined\x0a\x09^ true",
messageSends: [],
referencedClasses: []
}),
globals.IRInlinedSequence);



smalltalk.addClass('IRInliner', globals.IRVisitor, [], 'Compiler-Inlining');
globals.IRInliner.comment="I visit an IR tree, inlining message sends and block closures.\x0a\x0aMessage selectors that can be inlined are answered by `IRSendInliner >> #inlinedSelectors`";
smalltalk.addMethod(
smalltalk.method({
selector: "assignmentInliner",
protocol: 'factory',
fn: function (){
var self=this;
function $IRAssignmentInliner(){return globals.IRAssignmentInliner||(typeof IRAssignmentInliner=="undefined"?nil:IRAssignmentInliner)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRAssignmentInliner())._new();
_st($2)._translator_(self);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"assignmentInliner",{},globals.IRInliner)})},
args: [],
source: "assignmentInliner\x0a\x09^ IRAssignmentInliner new\x0a\x09\x09translator: self;\x0a\x09\x09yourself",
messageSends: ["translator:", "new", "yourself"],
referencedClasses: ["IRAssignmentInliner"]
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "returnInliner",
protocol: 'factory',
fn: function (){
var self=this;
function $IRReturnInliner(){return globals.IRReturnInliner||(typeof IRReturnInliner=="undefined"?nil:IRReturnInliner)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRReturnInliner())._new();
_st($2)._translator_(self);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"returnInliner",{},globals.IRInliner)})},
args: [],
source: "returnInliner\x0a\x09^ IRReturnInliner new\x0a\x09\x09translator: self;\x0a\x09\x09yourself",
messageSends: ["translator:", "new", "yourself"],
referencedClasses: ["IRReturnInliner"]
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "sendInliner",
protocol: 'factory',
fn: function (){
var self=this;
function $IRSendInliner(){return globals.IRSendInliner||(typeof IRSendInliner=="undefined"?nil:IRSendInliner)}
return smalltalk.withContext(function($ctx1) { 
var $2,$3,$1;
$2=_st($IRSendInliner())._new();
_st($2)._translator_(self);
$3=_st($2)._yourself();
$1=$3;
return $1;
}, function($ctx1) {$ctx1.fill(self,"sendInliner",{},globals.IRInliner)})},
args: [],
source: "sendInliner\x0a\x09^ IRSendInliner new\x0a\x09\x09translator: self;\x0a\x09\x09yourself",
messageSends: ["translator:", "new", "yourself"],
referencedClasses: ["IRSendInliner"]
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldInlineAssignment:",
protocol: 'testing',
fn: function (anIRAssignment){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $4,$3,$2,$1;
$1=_st(_st(_st(anIRAssignment)._isInlined())._not())._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=_st(anIRAssignment)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._last();
$ctx2.sendIdx["last"]=1;
$2=_st($3)._isSend();
return _st($2)._and_((function(){
return smalltalk.withContext(function($ctx3) {
return self._shouldInlineSend_(_st(_st(anIRAssignment)._instructions())._last());
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$ctx1.sendIdx["and:"]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldInlineAssignment:",{anIRAssignment:anIRAssignment},globals.IRInliner)})},
args: ["anIRAssignment"],
source: "shouldInlineAssignment: anIRAssignment\x0a\x09^ anIRAssignment isInlined not and: [\x0a\x09\x09anIRAssignment instructions last isSend and: [\x0a\x09\x09\x09self shouldInlineSend: (anIRAssignment instructions last) ]]",
messageSends: ["and:", "not", "isInlined", "isSend", "last", "instructions", "shouldInlineSend:"],
referencedClasses: []
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldInlineReturn:",
protocol: 'testing',
fn: function (anIRReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $4,$3,$2,$1;
$1=_st(_st(_st(anIRReturn)._isInlined())._not())._and_((function(){
return smalltalk.withContext(function($ctx2) {
$4=_st(anIRReturn)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._first();
$ctx2.sendIdx["first"]=1;
$2=_st($3)._isSend();
return _st($2)._and_((function(){
return smalltalk.withContext(function($ctx3) {
return self._shouldInlineSend_(_st(_st(anIRReturn)._instructions())._first());
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
$ctx1.sendIdx["and:"]=1;
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldInlineReturn:",{anIRReturn:anIRReturn},globals.IRInliner)})},
args: ["anIRReturn"],
source: "shouldInlineReturn: anIRReturn\x0a\x09^ anIRReturn isInlined not and: [\x0a\x09\x09anIRReturn instructions first isSend and: [\x0a\x09\x09\x09self shouldInlineSend: (anIRReturn instructions first) ]]",
messageSends: ["and:", "not", "isInlined", "isSend", "first", "instructions", "shouldInlineSend:"],
referencedClasses: []
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "shouldInlineSend:",
protocol: 'testing',
fn: function (anIRSend){
var self=this;
function $IRSendInliner(){return globals.IRSendInliner||(typeof IRSendInliner=="undefined"?nil:IRSendInliner)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=_st(_st(_st(anIRSend)._isInlined())._not())._and_((function(){
return smalltalk.withContext(function($ctx2) {
return _st($IRSendInliner())._shouldInline_(anIRSend);
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
return $1;
}, function($ctx1) {$ctx1.fill(self,"shouldInlineSend:",{anIRSend:anIRSend},globals.IRInliner)})},
args: ["anIRSend"],
source: "shouldInlineSend: anIRSend\x0a\x09^ anIRSend isInlined not and: [\x0a\x09\x09IRSendInliner shouldInline: anIRSend ]",
messageSends: ["and:", "not", "isInlined", "shouldInline:"],
referencedClasses: ["IRSendInliner"]
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "transformNonLocalReturn:",
protocol: 'visiting',
fn: function (anIRNonLocalReturn){
var self=this;
var localReturn;
function $IRReturn(){return globals.IRReturn||(typeof IRReturn=="undefined"?nil:IRReturn)}
return smalltalk.withContext(function($ctx1) { 
var $2,$1,$4,$3,$5,$6,$7,$8,$9;
$2=_st(anIRNonLocalReturn)._scope();
$ctx1.sendIdx["scope"]=1;
$1=_st($2)._canInlineNonLocalReturns();
if(smalltalk.assert($1)){
$4=_st(anIRNonLocalReturn)._scope();
$ctx1.sendIdx["scope"]=2;
$3=_st($4)._methodScope();
$5=_st(anIRNonLocalReturn)._scope();
$ctx1.sendIdx["scope"]=3;
_st($3)._removeNonLocalReturn_($5);
$6=_st($IRReturn())._new();
_st($6)._scope_(_st(anIRNonLocalReturn)._scope());
$7=_st($6)._yourself();
localReturn=$7;
localReturn;
_st(_st(anIRNonLocalReturn)._instructions())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(localReturn)._add_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
_st(anIRNonLocalReturn)._replaceWith_(localReturn);
$8=localReturn;
return $8;
};
$9=globals.IRInliner.superclass.fn.prototype._visitIRNonLocalReturn_.apply(_st(self), [anIRNonLocalReturn]);
return $9;
}, function($ctx1) {$ctx1.fill(self,"transformNonLocalReturn:",{anIRNonLocalReturn:anIRNonLocalReturn,localReturn:localReturn},globals.IRInliner)})},
args: ["anIRNonLocalReturn"],
source: "transformNonLocalReturn: anIRNonLocalReturn\x0a\x09\x22Replace a non local return into a local return\x22\x0a\x0a\x09| localReturn |\x0a\x09anIRNonLocalReturn scope canInlineNonLocalReturns ifTrue: [\x0a\x09\x09anIRNonLocalReturn scope methodScope removeNonLocalReturn: anIRNonLocalReturn scope.\x0a\x09\x09localReturn := IRReturn new\x0a\x09\x09\x09scope: anIRNonLocalReturn scope;\x0a\x09\x09\x09yourself.\x0a\x09\x09anIRNonLocalReturn instructions do: [ :each |\x0a\x09\x09\x09localReturn add: each ].\x0a\x09\x09anIRNonLocalReturn replaceWith: localReturn.\x0a\x09\x09^ localReturn ].\x0a\x09^ super visitIRNonLocalReturn: anIRNonLocalReturn",
messageSends: ["ifTrue:", "canInlineNonLocalReturns", "scope", "removeNonLocalReturn:", "methodScope", "scope:", "new", "yourself", "do:", "instructions", "add:", "replaceWith:", "visitIRNonLocalReturn:"],
referencedClasses: ["IRReturn"]
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRAssignment:",
protocol: 'visiting',
fn: function (anIRAssignment){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._shouldInlineAssignment_(anIRAssignment);
if(smalltalk.assert($2)){
$1=_st(self._assignmentInliner())._inlineAssignment_(anIRAssignment);
} else {
$1=globals.IRInliner.superclass.fn.prototype._visitIRAssignment_.apply(_st(self), [anIRAssignment]);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRAssignment:",{anIRAssignment:anIRAssignment},globals.IRInliner)})},
args: ["anIRAssignment"],
source: "visitIRAssignment: anIRAssignment\x0a\x09^ (self shouldInlineAssignment: anIRAssignment)\x0a\x09\x09ifTrue: [ self assignmentInliner inlineAssignment: anIRAssignment ]\x0a\x09\x09ifFalse: [ super visitIRAssignment: anIRAssignment ]",
messageSends: ["ifTrue:ifFalse:", "shouldInlineAssignment:", "inlineAssignment:", "assignmentInliner", "visitIRAssignment:"],
referencedClasses: []
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRNonLocalReturn:",
protocol: 'visiting',
fn: function (anIRNonLocalReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._transformNonLocalReturn_(anIRNonLocalReturn);
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRNonLocalReturn:",{anIRNonLocalReturn:anIRNonLocalReturn},globals.IRInliner)})},
args: ["anIRNonLocalReturn"],
source: "visitIRNonLocalReturn: anIRNonLocalReturn\x0a\x09^ self transformNonLocalReturn: anIRNonLocalReturn",
messageSends: ["transformNonLocalReturn:"],
referencedClasses: []
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRReturn:",
protocol: 'visiting',
fn: function (anIRReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._shouldInlineReturn_(anIRReturn);
if(smalltalk.assert($2)){
$1=_st(self._returnInliner())._inlineReturn_(anIRReturn);
} else {
$1=globals.IRInliner.superclass.fn.prototype._visitIRReturn_.apply(_st(self), [anIRReturn]);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRReturn:",{anIRReturn:anIRReturn},globals.IRInliner)})},
args: ["anIRReturn"],
source: "visitIRReturn: anIRReturn\x0a\x09^ (self shouldInlineReturn: anIRReturn)\x0a\x09\x09ifTrue: [ self returnInliner inlineReturn: anIRReturn ]\x0a\x09\x09ifFalse: [ super visitIRReturn: anIRReturn ]",
messageSends: ["ifTrue:ifFalse:", "shouldInlineReturn:", "inlineReturn:", "returnInliner", "visitIRReturn:"],
referencedClasses: []
}),
globals.IRInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRSend:",
protocol: 'visiting',
fn: function (anIRSend){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $2,$1;
$2=self._shouldInlineSend_(anIRSend);
if(smalltalk.assert($2)){
$1=_st(self._sendInliner())._inlineSend_(anIRSend);
} else {
$1=globals.IRInliner.superclass.fn.prototype._visitIRSend_.apply(_st(self), [anIRSend]);
};
return $1;
}, function($ctx1) {$ctx1.fill(self,"visitIRSend:",{anIRSend:anIRSend},globals.IRInliner)})},
args: ["anIRSend"],
source: "visitIRSend: anIRSend\x0a\x09^ (self shouldInlineSend: anIRSend)\x0a\x09\x09ifTrue: [ self sendInliner inlineSend: anIRSend ]\x0a\x09\x09ifFalse: [ super visitIRSend: anIRSend ]",
messageSends: ["ifTrue:ifFalse:", "shouldInlineSend:", "inlineSend:", "sendInliner", "visitIRSend:"],
referencedClasses: []
}),
globals.IRInliner);



smalltalk.addClass('IRInliningJSTranslator', globals.IRJSTranslator, [], 'Compiler-Inlining');
globals.IRInliningJSTranslator.comment="I am a specialized JavaScript translator able to write inlined IR instructions to JavaScript stream (`JSStream` instance).";
smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedAssignment:",
protocol: 'visiting',
fn: function (anIRInlinedAssignment){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._visit_(_st(_st(anIRInlinedAssignment)._instructions())._last());
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedAssignment:",{anIRInlinedAssignment:anIRInlinedAssignment},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedAssignment"],
source: "visitIRInlinedAssignment: anIRInlinedAssignment\x0a\x09self visit: anIRInlinedAssignment instructions last",
messageSends: ["visit:", "last", "instructions"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedClosure:",
protocol: 'visiting',
fn: function (anIRInlinedClosure){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(self._stream())._nextPutVars_(_st(_st(anIRInlinedClosure)._tempDeclarations())._collect_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(_st(each)._name())._asVariableName();
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})})));
_st(_st(anIRInlinedClosure)._instructions())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return self._visit_(each);
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedClosure:",{anIRInlinedClosure:anIRInlinedClosure},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedClosure"],
source: "visitIRInlinedClosure: anIRInlinedClosure\x0a\x09self stream nextPutVars: (anIRInlinedClosure tempDeclarations collect: [ :each |\x0a\x09\x09each name asVariableName ]).\x0a\x09anIRInlinedClosure instructions do: [ :each |\x0a\x09\x09self visit: each ]",
messageSends: ["nextPutVars:", "stream", "collect:", "tempDeclarations", "asVariableName", "name", "do:", "instructions", "visit:"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedIfFalse:",
protocol: 'visiting',
fn: function (anIRInlinedIfFalse){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutIf_with_((function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
_st($2)._nextPutAll_("! smalltalk.assert(");
$ctx2.sendIdx["nextPutAll:"]=1;
$4=_st(anIRInlinedIfFalse)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._first();
self._visit_($3);
$ctx2.sendIdx["visit:"]=1;
return _st(self._stream())._nextPutAll_(")");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._visit_(_st(_st(anIRInlinedIfFalse)._instructions())._last());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedIfFalse:",{anIRInlinedIfFalse:anIRInlinedIfFalse},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedIfFalse"],
source: "visitIRInlinedIfFalse: anIRInlinedIfFalse\x0a\x09self stream nextPutIf: [\x0a\x09\x09self stream nextPutAll: '! smalltalk.assert('.\x0a\x09\x09self visit: anIRInlinedIfFalse instructions first.\x0a\x09\x09self stream nextPutAll: ')' ]\x0a\x09\x09with: [ self visit: anIRInlinedIfFalse instructions last ]",
messageSends: ["nextPutIf:with:", "stream", "nextPutAll:", "visit:", "first", "instructions", "last"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedIfNil:",
protocol: 'visiting',
fn: function (anIRInlinedIfNil){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutIf_with_((function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
_st($2)._nextPutAll_("($receiver = ");
$ctx2.sendIdx["nextPutAll:"]=1;
$4=_st(anIRInlinedIfNil)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._first();
self._visit_($3);
$ctx2.sendIdx["visit:"]=1;
return _st(self._stream())._nextPutAll_(") == nil || $receiver == null");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._visit_(_st(_st(anIRInlinedIfNil)._instructions())._last());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedIfNil:",{anIRInlinedIfNil:anIRInlinedIfNil},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedIfNil"],
source: "visitIRInlinedIfNil: anIRInlinedIfNil\x0a\x09self stream nextPutIf: [\x0a\x09\x09self stream nextPutAll: '($receiver = '.\x0a\x09\x09self visit: anIRInlinedIfNil instructions first.\x0a\x09\x09self stream nextPutAll: ') == nil || $receiver == null' ]\x0a\x09\x09with: [ self visit: anIRInlinedIfNil instructions last ]",
messageSends: ["nextPutIf:with:", "stream", "nextPutAll:", "visit:", "first", "instructions", "last"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedIfNilIfNotNil:",
protocol: 'visiting',
fn: function (anIRInlinedIfNilIfNotNil){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3,$6,$5;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutIfElse_with_with_((function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
_st($2)._nextPutAll_("($receiver = ");
$ctx2.sendIdx["nextPutAll:"]=1;
$4=_st(anIRInlinedIfNilIfNotNil)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._first();
self._visit_($3);
$ctx2.sendIdx["visit:"]=1;
return _st(self._stream())._nextPutAll_(") == nil || $receiver == null");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
$6=_st(anIRInlinedIfNilIfNotNil)._instructions();
$ctx2.sendIdx["instructions"]=2;
$5=_st($6)._second();
return self._visit_($5);
$ctx2.sendIdx["visit:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._visit_(_st(_st(anIRInlinedIfNilIfNotNil)._instructions())._third());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedIfNilIfNotNil:",{anIRInlinedIfNilIfNotNil:anIRInlinedIfNilIfNotNil},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedIfNilIfNotNil"],
source: "visitIRInlinedIfNilIfNotNil: anIRInlinedIfNilIfNotNil\x0a\x09self stream\x0a\x09\x09nextPutIfElse: [\x0a\x09\x09\x09self stream nextPutAll: '($receiver = '.\x0a\x09\x09\x09self visit: anIRInlinedIfNilIfNotNil instructions first.\x0a\x09\x09\x09self stream nextPutAll: ') == nil || $receiver == null' ]\x0a\x09\x09with: [ self visit: anIRInlinedIfNilIfNotNil instructions second ]\x0a\x09\x09with: [ self visit: anIRInlinedIfNilIfNotNil instructions third ]",
messageSends: ["nextPutIfElse:with:with:", "stream", "nextPutAll:", "visit:", "first", "instructions", "second", "third"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedIfTrue:",
protocol: 'visiting',
fn: function (anIRInlinedIfTrue){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutIf_with_((function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
_st($2)._nextPutAll_("smalltalk.assert(");
$ctx2.sendIdx["nextPutAll:"]=1;
$4=_st(anIRInlinedIfTrue)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._first();
self._visit_($3);
$ctx2.sendIdx["visit:"]=1;
return _st(self._stream())._nextPutAll_(")");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._visit_(_st(_st(anIRInlinedIfTrue)._instructions())._last());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedIfTrue:",{anIRInlinedIfTrue:anIRInlinedIfTrue},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedIfTrue"],
source: "visitIRInlinedIfTrue: anIRInlinedIfTrue\x0a\x09self stream nextPutIf: [\x0a\x09\x09self stream nextPutAll: 'smalltalk.assert('.\x0a\x09\x09self visit: anIRInlinedIfTrue instructions first.\x0a\x09\x09self stream nextPutAll: ')' ]\x0a\x09\x09with: [ self visit: anIRInlinedIfTrue instructions last ]",
messageSends: ["nextPutIf:with:", "stream", "nextPutAll:", "visit:", "first", "instructions", "last"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedIfTrueIfFalse:",
protocol: 'visiting',
fn: function (anIRInlinedIfTrueIfFalse){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1,$2,$4,$3,$6,$5;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutIfElse_with_with_((function(){
return smalltalk.withContext(function($ctx2) {
$2=self._stream();
$ctx2.sendIdx["stream"]=2;
_st($2)._nextPutAll_("smalltalk.assert(");
$ctx2.sendIdx["nextPutAll:"]=1;
$4=_st(anIRInlinedIfTrueIfFalse)._instructions();
$ctx2.sendIdx["instructions"]=1;
$3=_st($4)._first();
self._visit_($3);
$ctx2.sendIdx["visit:"]=1;
return _st(self._stream())._nextPutAll_(")");
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}),(function(){
return smalltalk.withContext(function($ctx2) {
$6=_st(anIRInlinedIfTrueIfFalse)._instructions();
$ctx2.sendIdx["instructions"]=2;
$5=_st($6)._second();
return self._visit_($5);
$ctx2.sendIdx["visit:"]=2;
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,2)})}),(function(){
return smalltalk.withContext(function($ctx2) {
return self._visit_(_st(_st(anIRInlinedIfTrueIfFalse)._instructions())._third());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,3)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedIfTrueIfFalse:",{anIRInlinedIfTrueIfFalse:anIRInlinedIfTrueIfFalse},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedIfTrueIfFalse"],
source: "visitIRInlinedIfTrueIfFalse: anIRInlinedIfTrueIfFalse\x0a\x09self stream\x0a\x09\x09nextPutIfElse: [\x0a\x09\x09\x09self stream nextPutAll: 'smalltalk.assert('.\x0a\x09\x09\x09self visit: anIRInlinedIfTrueIfFalse instructions first.\x0a\x09\x09\x09self stream nextPutAll: ')' ]\x0a\x09\x09with: [ self visit: anIRInlinedIfTrueIfFalse instructions second ]\x0a\x09\x09with: [ self visit: anIRInlinedIfTrueIfFalse instructions third ]",
messageSends: ["nextPutIfElse:with:with:", "stream", "nextPutAll:", "visit:", "first", "instructions", "second", "third"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedNonLocalReturn:",
protocol: 'visiting',
fn: function (anIRInlinedReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._stream();
$ctx1.sendIdx["stream"]=1;
_st($1)._nextPutStatementWith_((function(){
return smalltalk.withContext(function($ctx2) {
return self._visit_(_st(_st(anIRInlinedReturn)._instructions())._last());
}, function($ctx2) {$ctx2.fillBlock({},$ctx1,1)})}));
_st(self._stream())._nextPutNonLocalReturnWith_((function(){
}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedNonLocalReturn:",{anIRInlinedReturn:anIRInlinedReturn},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedReturn"],
source: "visitIRInlinedNonLocalReturn: anIRInlinedReturn\x0a\x09self stream nextPutStatementWith: [\x0a\x09\x09self visit: anIRInlinedReturn instructions last ].\x0a\x09self stream nextPutNonLocalReturnWith: [ ]",
messageSends: ["nextPutStatementWith:", "stream", "visit:", "last", "instructions", "nextPutNonLocalReturnWith:"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedReturn:",
protocol: 'visiting',
fn: function (anIRInlinedReturn){
var self=this;
return smalltalk.withContext(function($ctx1) { 
self._visit_(_st(_st(anIRInlinedReturn)._instructions())._last());
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedReturn:",{anIRInlinedReturn:anIRInlinedReturn},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedReturn"],
source: "visitIRInlinedReturn: anIRInlinedReturn\x0a\x09self visit: anIRInlinedReturn instructions last",
messageSends: ["visit:", "last", "instructions"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);

smalltalk.addMethod(
smalltalk.method({
selector: "visitIRInlinedSequence:",
protocol: 'visiting',
fn: function (anIRInlinedSequence){
var self=this;
return smalltalk.withContext(function($ctx1) { 
_st(_st(anIRInlinedSequence)._instructions())._do_((function(each){
return smalltalk.withContext(function($ctx2) {
return _st(self._stream())._nextPutStatementWith_((function(){
return smalltalk.withContext(function($ctx3) {
return self._visit_(each);
}, function($ctx3) {$ctx3.fillBlock({},$ctx2,2)})}));
}, function($ctx2) {$ctx2.fillBlock({each:each},$ctx1,1)})}));
return self}, function($ctx1) {$ctx1.fill(self,"visitIRInlinedSequence:",{anIRInlinedSequence:anIRInlinedSequence},globals.IRInliningJSTranslator)})},
args: ["anIRInlinedSequence"],
source: "visitIRInlinedSequence: anIRInlinedSequence\x0a\x09anIRInlinedSequence instructions do: [ :each |\x0a\x09\x09self stream nextPutStatementWith: [ self visit: each ]]",
messageSends: ["do:", "instructions", "nextPutStatementWith:", "stream", "visit:"],
referencedClasses: []
}),
globals.IRInliningJSTranslator);



smalltalk.addClass('IRSendInliner', globals.Object, ['send', 'translator'], 'Compiler-Inlining');
globals.IRSendInliner.comment="I inline some message sends and block closure arguments. I heavily rely on #perform: to dispatch inlining methods.";
smalltalk.addMethod(
smalltalk.method({
selector: "ifFalse:",
protocol: 'inlining',
fn: function (anIRInstruction){
var self=this;
function $IRInlinedIfFalse(){return globals.IRInlinedIfFalse||(typeof IRInlinedIfFalse=="undefined"?nil:IRInlinedIfFalse)}
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._inlinedSend_with_(_st($IRInlinedIfFalse())._new(),anIRInstruction);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifFalse:",{anIRInstruction:anIRInstruction},globals.IRSendInliner)})},
args: ["anIRInstruction"],
source: "ifFalse: anIRInstruction\x0a\x09^ self inlinedSend: IRInlinedIfFalse new with: anIRInstruction",
messageSends: ["inlinedSend:with:", "new"],
referencedClasses: ["IRInlinedIfFalse"]
}),
globals.IRSendInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "ifFalse:ifTrue:",
protocol: 'inlining',
fn: function (anIRInstruction,anotherIRInstruction){
var self=this;
return smalltalk.withContext(function($ctx1) { 
var $1;
$1=self._perform_withArguments_("ifTrue:ifFalse:",[anotherIRInstruction,anIRInstruction]);
return $1;
}, function($ctx1) {$ctx1.fill(self,"ifFalse:ifTrue:",{anIRInstruction:anIRInstruction,anotherIRInstruction:anotherIRInstruction},globals.IRSendInliner)})},
args: ["anIRInstruction", "anotherIRInstruction"],
source: "ifFalse: anIRInstruction ifTrue: anotherIRInstruction\x0a\x09^ self perform: #ifTrue:ifFalse: withArguments: { anotherIRInstruction. anIRInstruction }",
messageSends: ["perform:withArguments:"],
referencedClasses: []
}),
globals.IRSendInliner);

smalltalk.addMethod(
smalltalk.method({
selector: "ifNil:",
protocol: 'inlining',
fn: function (anIRInstruction){
var self=this;
function $IRInlinedIfNilIfNotNil(){return globals.IRInlinedIfNilIfNotNil||(typeof IRInlinedIfNilIfNotNil=="undefined"?nil:IRInlinedIfNilIfNotNil)}
function $IRClosure(){return globals.IRClosure||(typeof IRClosure=="undefined"?nil:IRClosure)}
function $IRBlockSequence(){return globals.IRBlockSequence||(typeof IRBlockSequence=="undefined"?nil:IRBlockSequence)}
return smalltalk.withContext(function($ctx1) { 
var $2,$4,$5,$7,$8,$6,$9,$3,$1;
$2=_st($IRInlinedIfNilIfNotNil())._new();
$ctx1.sendIdx["new"]=1;
$4=_st($IRClosure())._new();
$ctx1.sendIdx["new"]=2;
_st($4)._scope_(_st(_st(anI